/*
* Copyright (C) 2013 Canonical Ltd
*
* This program is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License version 3 as
* published by the Free Software Foundation.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program.  If not, see <http://www.gnu.org/licenses/>.
*
*/
#ifndef _PARSER_H
#define _PARSER_H

#include <string>

#include <boost/config/warning_disable.hpp>
#include <boost/fusion/include/adapt_struct.hpp>
#include <boost/spirit/include/phoenix_object.hpp>
#include <boost/spirit/include/phoenix_operator.hpp>
#include <boost/spirit/include/qi.hpp>

#include "xpathquerypart.h"

// this allows spirit to lazily construct these two structs...
BOOST_FUSION_ADAPT_STRUCT(
    xpathselect::XPathQueryPart,
    (std::string, node_name_)
    (boost::optional<xpathselect::XPathQueryParam>, parameter)
    );

BOOST_FUSION_ADAPT_STRUCT(
    xpathselect::XPathQueryParam,
    (std::string, param_name)
    (std::string, param_value)
    );

namespace xpathselect
{
namespace parser
{
namespace qi = boost::spirit::qi;
namespace phoenix = boost::phoenix;

    // This is the main XPath grammar. It looks horrible, until you emerse yourself in it for a few
    // days, then the beauty of boost::spirit creeps into your brain. To help future programmers,
    // I've heavily commented this.
    //
    // The first template parameter to this grammar defines the type of iterator the grammer will operate
    // on - it must adhere to std::forward_iterator. The second template parameter is the type
    // that this grammar will produce (in this case: a list of XPathQueryPart objects).
    template <typename Iterator>
    struct xpath_grammar : qi::grammar<Iterator, QueryList()>
    {
        xpath_grammar() : xpath_grammar::base_type(node_sequence) // node_sequence is the start rule.
        {
            using namespace qi::labels;

            // node names can either be a text string (no spaces), or a wildcard.
            node_name = ( +qi::char_("a-zA-Z0-9_\\-") | qi::char_('*') );
            // a search node is '//' as long as it's not followed by '/' or '*' or end of input.
            search_node = "//" >> !(qi::lit("*") || "/" || qi::eoi)[qi::_val = XPathQueryPart()];
            // a normal separator is a '/' as long as it's followed by something other than another '/'
            normal_sep = '/' >> !qi::lit('/');
            separator = normal_sep || search_node;  // nodes can be separated by normal_sep or search_node.

            // parameter name and values can contain text (no spaces)
            param_name = +qi::char_("a-zA-Z0-9_\\-");
            param_value = +qi::char_("a-zA-Z0-9_\\-");
            // parameter specification is simple: name=value
            param %= param_name >> '=' >> param_value;

            // a node consists of a node name, followed by an *optional* parameter inside '[' and ']'.
            node %= node_name >> -('[' >> param >> ']');
            // this is the money shot: a node sequence is one or more of a separator, followed by an
            // optional node.
            node_sequence %= +(separator >> -node);

            // DEBUGGING SUPPORT:
            // define DEBUG in order to have boost::spirit spit out useful debug information:
#ifdef DEBUG
            // this gives english names to all the grammar rules:
            node_name.name("node_name");
            search_node.name("search_node");
            normal_sep.name("normal_separator");
            separator.name("separator");
            param_name.name("param_name");
            param_value.name("param_value");
            param.name("parameter");
            node.name("node");
            node_sequence.name("node_sequence");

            // set up error logging:
            qi::on_error<qi::fail>(
                node_sequence,
                std::cout
                    << phoenix::val("Error! Expecting ")
                    << qi::_4                               // what failed?
                    << phoenix::val(" here: \"")
                    << phoenix::construct<std::string>(qi::_3, qi::_2)   // iterators to error-pos, end
                    << phoenix::val("\"")
                    << std::endl
            );
            // specify which rules we want debug info about (all of them):
            qi::debug(node_name);
            qi::debug(search_node);
            qi::debug(normal_sep);
            qi::debug(separator);
            qi::debug(param_name);
            qi::debug(param_value);
            qi::debug(param);
            qi::debug(node);
            qi::debug(node_sequence);
#endif
        }
        // declare all the rules. The second template parameter is the type they produce.
        qi::rule<Iterator, std::string()> node_name;
        qi::rule<Iterator, XPathQueryPart()> search_node;
        qi::rule<Iterator> normal_sep;
        qi::rule<Iterator, xpathselect::QueryList()> separator;

        qi::rule<Iterator, std::string()> param_name;
        qi::rule<Iterator, std::string()> param_value;
        qi::rule<Iterator, XPathQueryParam()> param;

        qi::rule<Iterator, XPathQueryPart()> node;
        qi::rule<Iterator, xpathselect::QueryList()> node_sequence;
    };

}
}

#endif
