/*  Og(g)re - Ogg-Output-Plugin
 *  (C) copyright 2002 Lars Siebold <khandha5@gmx.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <time.h>
#include <stdlib.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>
#include <xmms/dirbrowser.h>
#include <xmms/configfile.h>
#include <xmms/util.h>

#include <vorbis/vorbisenc.h>

#define ENCBUFFER_SIZE 35000
#define OGGRE_VER "0.2"


GtkWidget *configure_win = NULL, *path_vbox;
GtkWidget *path_hbox, *path_label, *path_entry, *path_browse, *path_dirbrowser = NULL;
GtkWidget *configure_separator, *configure_bbox, *configure_ok, *configure_cancel;
GtkWidget *vbox;
GtkWidget *quality_frame, *quality_vbox, *quality_hbox1, *quality_spin, *quality_label, *quality_explain_label;
GtkObject *quality_adj;
GtkWidget *extention_toggle, *extention_vbox;
GtkWidget *filename_frame, *filename_vbox, *filename_radio1, *filename_radio2;

static gchar *file_path = NULL;
static FILE *output_file = NULL;
static guint64 written = 0;
static guint64 olen = 0;
static AFormat afmt;
gint ctrlsocket_get_session_id( void );
static gint srate;
static gint inch;

static void oggre_init( void );
static void oggre_about( void );
static gint oggre_open( AFormat fmt, gint rate, gint nch );
static void oggre_write( void *ptr, gint length );
static void oggre_close( void );
static void oggre_flush( gint time );
static void oggre_pause( short p );
static gint oggre_free( void );
static gint oggre_playing( void );
static gint oggre_get_written_time( void );
static gint oggre_get_output_time( void );
static void oggre_configure( void );

static int filename_type = 0;
static int dont_remove_extention = 0;

static float v_base_quality = 0.3;

static ogg_stream_state os;
static ogg_page og;
static ogg_packet op;

static vorbis_dsp_state vd;
static vorbis_block vb;
static vorbis_info vi;
static vorbis_comment  vc;

int encout;
static float **encbuffer;

OutputPlugin oggre_op =
{
	NULL,
	NULL,
	NULL,			/* Description */
	oggre_init,
	oggre_about,
	oggre_configure,
	NULL,			/* get_volume */
	NULL,			/* set_volume */
	oggre_open,
	oggre_write,
	oggre_close,
	oggre_flush,
	oggre_pause,
	oggre_free,
	oggre_playing,
	oggre_get_output_time,
	oggre_get_written_time,
};

OutputPlugin *get_oplugin_info( void )
{
	oggre_op.description = g_strdup_printf( "Og(g)re %s", OGGRE_VER );
	return &oggre_op;
}

static void oggre_init( void )
{
	ConfigFile *cfgfile;
	gchar *filename;

	filename = g_strconcat( g_get_home_dir(), "/.xmms/config", NULL );
	cfgfile = xmms_cfg_open_file( filename );
	if( cfgfile )
	{
		xmms_cfg_read_string( cfgfile, "oggre", "file_path", &file_path );
		xmms_cfg_read_float( cfgfile, "oggre", "base_quality", &v_base_quality );
		xmms_cfg_read_int( cfgfile, "oggre", "dont_remove_extention", &dont_remove_extention );
		xmms_cfg_read_int( cfgfile, "oggre", "filename_type", &filename_type );
		xmms_cfg_free( cfgfile );
	}
	g_free( filename );
	if( !file_path )
		file_path = g_strdup( g_get_home_dir() );

}


void oggre_about( void )
{
	static GtkWidget *dialog;

	if( dialog != NULL )
		return;
	
	dialog = xmms_show_message(
		"About Og(g)re-Output-Plugin",
		"Og(g)re-Output-Plugin\n\n "
		  "This program is free software; you can redistribute it and/or modify\n"
		  "it under the terms of the GNU General Public License as published by\n"
		  "the Free Software Foundation; either version 2 of the License, or\n"
		  "(at your option) any later version.\n"
		  "\n"
		  "This program is distributed in the hope that it will be useful,\n"
		  "but WITHOUT ANY WARRANTY; without even the implied warranty of\n"
		  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"
		  "GNU General Public License for more details.\n"
		  "\n"
		  "You should have received a copy of the GNU General Public License\n"
		  "along with this program; if not, write to the Free Software\n"
		  "Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307,\n"
		  "USA.", "Ok", FALSE, NULL, NULL );
	gtk_signal_connect( GTK_OBJECT( dialog ), "destroy", GTK_SIGNAL_FUNC( gtk_widget_destroyed ), &dialog );
}


static gint oggre_open( AFormat fmt, gint rate, gint nch )
{

	gchar *filename, *title, *temp;
	gint pos;
	int result;

	ogg_packet header;
	ogg_packet header_comm;
	ogg_packet header_code;

	/* So all the values will be reset to the ones saved */
	/* Easier than to implement a tmp variable for every value */
	oggre_init();

	written = 0;
	afmt = fmt;

	if( xmms_check_realtime_priority() )
	{
		xmms_show_message( "Error",
				  "You cannot use the Og(g)re-Output plugin\n"
				    "when you're running in realtime mode.",
				  "Ok", FALSE, NULL, NULL );
		return 0;
	}

	/* Choose filename */
	pos = xmms_remote_get_playlist_pos( ctrlsocket_get_session_id() );

	/* We want original filename */
	if( filename_type == 0 )
	{
		title = xmms_remote_get_playlist_file( ctrlsocket_get_session_id(), pos );

		if( dont_remove_extention == 0 )
		{
			if( title != NULL && ( temp = strrchr( title, '.' ) ) != NULL )
				*temp = '\0';
		}
	}

	/* No filename, lets try title instead */
	if( filename_type == 1 || title == NULL || strlen( g_basename( title ) ) == 0 )
	{
		title = xmms_remote_get_playlist_title( ctrlsocket_get_session_id(), pos );
		
	/* This causes a nasty crash - don't know why this was there anyway */
	/*	
		while( title != NULL && ( temp = strchr( title, '/' ) ) != NULL )
			*temp = '-';
	*/
	}

	/* No title either.  Just set it to something. */
	if( title == NULL || strlen( g_basename( title ) ) == 0 )
	{
		title = g_strdup_printf( "xmms-%d", pos );
	}

	filename = g_strdup_printf( "%s/%s.ogg", file_path, g_basename( title ) );
	g_free( title );


	output_file = fopen( filename , "r" );
	
	if( output_file )
	{
		xmms_show_message( "Warning", "File exists", "Ok", TRUE, NULL, NULL );
		fclose( output_file );
		output_file = NULL;
	}
	
	output_file = fopen( filename , "w" );
	g_free( filename );

	if( !output_file )
		return 0;

	srate = rate;
	inch = nch;

	/* Ok, now init vorbis stuff */
	vorbis_info_init( &vi );
	vorbis_comment_init( &vc );

	vorbis_comment_add_tag( &vc, "title", xmms_remote_get_playlist_title( ctrlsocket_get_session_id(), pos ) );

	if( ( result = vorbis_encode_init_vbr( &vi, ( long )nch, ( long )rate, v_base_quality ) ) != 0 )
	{
		printf( "vorbis_encode_init_vbr result: %i\n", result );
		vorbis_info_clear(&vi);
		return 0;
	}

	vorbis_analysis_init( &vd, &vi );
	vorbis_block_init( &vd, &vb );

	srand( time( NULL ) );
	ogg_stream_init( &os, rand() );
	
	vorbis_analysis_headerout( &vd, &vc, &header, &header_comm, &header_code );

	ogg_stream_packetin( &os, &header );
	ogg_stream_packetin( &os, &header_comm );
	ogg_stream_packetin( &os, &header_code );

	while( ( result = ogg_stream_flush( &os, &og ) ) ) {

		if( !result ) break;

		written += fwrite( og.header, 1, og.header_len, output_file );
		written += fwrite( og.body, 1, og.body_len, output_file );
	}


	return 1;
}

static void oggre_write( void *ptr, gint length )
{
	int i;
	int result;
	short int *tmpdata;
	encbuffer = vorbis_analysis_buffer(&vd, length);

	tmpdata = ptr;

	/* 
	   Deinterleaving stuff.
	   Dunno where the 32768.0 comes from, but... it works :)
	   Shamelessly ripped form kio_audiocd.
	*/
	if( inch == 1)
	{
		for( i = 0; i < ( length / 2 ); i++ ){
			encbuffer[0][i] = tmpdata[i] / 32768.0;
			encbuffer[0][i] = tmpdata[i] / 32768.0;
		}
	}
	else
	{
		for( i = 0; i < ( length / 4 ); i++ ){
			encbuffer[0][i] = tmpdata[ 2 * i ] / 32768.0;
			encbuffer[1][i] = tmpdata[ 2 * i + 1 ] / 32768.0;
		}
	}

	vorbis_analysis_wrote( &vd, i );

	while( vorbis_analysis_blockout( &vd, &vb ) == 1 )
	{

		/* Do the main analysis, creating a packet */
		vorbis_analysis( &vb, &op );
		vorbis_bitrate_addblock( &vb );

		while( vorbis_bitrate_flushpacket( &vd, &op ) )
		{
			/* Add packet to bitstream */
			ogg_stream_packetin( &os, &op );

			/* If we've gone over a page boundary, we can do actual output,
			so do so (for however many pages are available) */

			while( ( result = ogg_stream_pageout( &os, &og ) ) )
			{
				if( !result ) break;

				written += fwrite( og.header, 1, og.header_len, output_file );
				written += fwrite( og.body, 1, og.body_len, output_file );
			}
		}
	}

	olen += length;

}

static void oggre_close( void )
{
	if( output_file )
	{
		fclose( output_file );
	}

	ogg_stream_clear( &os );

	vorbis_block_clear( &vb );
	vorbis_dsp_clear( &vd );
	vorbis_info_clear( &vi );

	written = 0;
	olen = 0;

	output_file = NULL;

}

static void oggre_flush( gint time )
{
}

static void oggre_pause( short p )
{
}

static gint oggre_free( void )
{
	return 1000000;
}

static gint oggre_playing( void )
{
	return 0;
}

static gint oggre_get_written_time( void )
{
	if( srate && inch )
		return ( gint ) ( ( olen * 1000 ) / ( srate * 2 * inch ) );
	return 0;
}

static gint oggre_get_output_time( void )
{
	return oggre_get_written_time();
}


/*****************/
/* Configuration */
/*****************/

static void quality_change( GtkAdjustment *adjustment, gpointer user_data)
{
	if( gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( quality_spin ) ) )
		v_base_quality = gtk_spin_button_get_value_as_float( GTK_SPIN_BUTTON( quality_spin ) ) / 10;
	else
		v_base_quality = 0.0;

}

static void filename_toggle( GtkToggleButton *togglebutton, gpointer user_data )
{
	if( user_data == "1" )
	{
		gtk_widget_set_sensitive( extention_vbox, FALSE );
		filename_type = 1;
	}
	else if( user_data == "0" )
	{
		gtk_widget_set_sensitive( extention_vbox, TRUE );
		filename_type = 0;
	}
}

static void toggle_extention( GtkToggleButton *togglebutton, gpointer user_data )
{

	if( gtk_toggle_button_get_active( GTK_TOGGLE_BUTTON( extention_toggle ) ) == TRUE )
		dont_remove_extention = 1;
	else
		dont_remove_extention = 0;

}


static void path_dirbrowser_cb( gchar *dir )
{
	gtk_entry_set_text( GTK_ENTRY( path_entry ), dir );
}

static void path_browse_cb( GtkWidget *w, gpointer data )
{
	if( !path_dirbrowser )
	{
		path_dirbrowser = xmms_create_dir_browser( "Select the directory where you want to store the output files:", file_path, GTK_SELECTION_SINGLE, path_dirbrowser_cb );
		gtk_signal_connect( GTK_OBJECT( path_dirbrowser ), "destroy", GTK_SIGNAL_FUNC( gtk_widget_destroyed ), &path_dirbrowser );
		gtk_window_set_transient_for( GTK_WINDOW( path_dirbrowser ), GTK_WINDOW( configure_win ) );
		gtk_widget_show( path_dirbrowser );
	}
}

/* Save Configuration */

static void configure_ok_cb( gpointer data )
{
	gchar *filename;
	ConfigFile *cfgfile;

	filename = g_strconcat( g_get_home_dir(), "/.xmms/config", NULL );

	if( file_path )
		g_free( file_path );

	file_path = g_strdup( gtk_entry_get_text( GTK_ENTRY( path_entry ) ) );

	cfgfile = xmms_cfg_open_file( filename );

	if( !cfgfile )
		cfgfile = xmms_cfg_new();

	xmms_cfg_write_string( cfgfile, "oggre", "file_path", file_path );
	xmms_cfg_write_float( cfgfile, "oggre", "base_quality", v_base_quality );
	xmms_cfg_write_int( cfgfile, "oggre", "dont_remove_extention", dont_remove_extention );
	xmms_cfg_write_int( cfgfile, "oggre", "filename_type", filename_type );

	xmms_cfg_write_file( cfgfile, filename );
	xmms_cfg_free( cfgfile );

	g_free( filename );

	gtk_widget_destroy( configure_win );
	if( path_dirbrowser )
		gtk_widget_destroy( path_dirbrowser );
}

static void configure_destroy(void)
{
	if (path_dirbrowser)
		gtk_widget_destroy(path_dirbrowser);
}


/************************/
/* Configuration Widget */
/************************/


static void oggre_configure( void )
{
	if( !configure_win )
	{
		configure_win = gtk_window_new( GTK_WINDOW_DIALOG );

		gtk_signal_connect( GTK_OBJECT( configure_win ), "destroy", GTK_SIGNAL_FUNC( configure_destroy ), NULL );
		gtk_signal_connect( GTK_OBJECT( configure_win ), "destroy", GTK_SIGNAL_FUNC( gtk_widget_destroyed ), &configure_win );
		gtk_window_set_title( GTK_WINDOW( configure_win ), "Og(g)re Configuration" );
		gtk_window_set_position( GTK_WINDOW( configure_win ), GTK_WIN_POS_MOUSE );
		gtk_window_set_policy( GTK_WINDOW( configure_win ), FALSE, TRUE, FALSE );
		gtk_container_set_border_width( GTK_CONTAINER( configure_win ), 5 );

 		vbox = gtk_vbox_new( FALSE, 5 );
		gtk_container_add( GTK_CONTAINER( configure_win ), vbox );


/* Quality control */


		/* Quality Options */

		quality_frame = gtk_frame_new( "Quality:" );
		gtk_container_set_border_width( GTK_CONTAINER( quality_frame ), 5 );
		gtk_box_pack_start( GTK_BOX( vbox ), quality_frame, FALSE, FALSE, 2 );

		quality_vbox = gtk_vbox_new( FALSE, 5 );
		gtk_container_set_border_width( GTK_CONTAINER( quality_vbox ), 10 );
		gtk_container_add( GTK_CONTAINER( quality_frame ), quality_vbox );

		/* Quality Level */

		quality_hbox1 = gtk_hbox_new( FALSE, 5 );
		gtk_container_set_border_width( GTK_CONTAINER( quality_hbox1 ), 10 );
		gtk_container_add( GTK_CONTAINER( quality_vbox ), quality_hbox1 );

		quality_label = gtk_label_new( "Quality (0-10):" );
		gtk_misc_set_alignment( GTK_MISC( quality_label ), 0, 0.5 );
		gtk_box_pack_start( GTK_BOX( quality_hbox1 ), quality_label, TRUE, TRUE, 0 );

		quality_adj = gtk_adjustment_new( 3, 0, 10, 0.1, 1, 1 );
		quality_spin = gtk_spin_button_new( GTK_ADJUSTMENT( quality_adj ), 1, 2 );
		gtk_widget_set_usize( quality_spin, 20, 20 );
		gtk_box_pack_start( GTK_BOX( quality_hbox1 ), quality_spin, TRUE, TRUE, 0 );
		gtk_signal_connect( GTK_OBJECT( quality_adj ), "value-changed", GTK_SIGNAL_FUNC( quality_change ), NULL );

		gtk_spin_button_set_value( GTK_SPIN_BUTTON( quality_spin ), ( v_base_quality * 10 ) );

		quality_explain_label = gtk_label_new( "Quality 3 is the standard recommended quality. It compares about in Quality to a 160 kbps encoded mp3. Use a value of 4.99 for a slightly higher quality and bitrate." );
		gtk_label_set_justify( GTK_LABEL( quality_explain_label ), GTK_JUSTIFY_FILL );
		gtk_label_set_line_wrap( GTK_LABEL( quality_explain_label ), TRUE );
		gtk_box_pack_start( GTK_BOX( quality_vbox ), quality_explain_label, TRUE, TRUE, 0 );


/* Filename Config */

		filename_frame = gtk_frame_new( "Set Filename to" );
		gtk_container_set_border_width( GTK_CONTAINER( filename_frame ), 5 );
		gtk_box_pack_start( GTK_BOX( vbox ), filename_frame, FALSE, FALSE, 2 );

		filename_vbox = gtk_vbox_new( FALSE, 5 );
		gtk_container_set_border_width( GTK_CONTAINER( filename_vbox ), 5 );
		gtk_container_add( GTK_CONTAINER( filename_frame ), filename_vbox );

		filename_radio1 = gtk_radio_button_new_with_label( NULL, "Playlist Title" );
		gtk_box_pack_start( GTK_BOX( filename_vbox ), filename_radio1, TRUE, TRUE, 0 );
		if( filename_type == 1 )
			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(filename_radio1), TRUE);

		filename_radio2 = gtk_radio_button_new_with_label_from_widget( GTK_RADIO_BUTTON( filename_radio1 ), "Original Filename" );
		gtk_box_pack_start( GTK_BOX( filename_vbox ), filename_radio2, TRUE, TRUE, 0 );
		if( filename_type == 0 )
 			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( filename_radio2 ), TRUE );

		gtk_signal_connect( GTK_OBJECT( filename_radio1 ), "toggled", GTK_SIGNAL_FUNC( filename_toggle ), "1" );
		gtk_signal_connect( GTK_OBJECT( filename_radio2 ), "toggled", GTK_SIGNAL_FUNC( filename_toggle ), "0" );

		extention_vbox = gtk_vbox_new( FALSE, 5 );
		gtk_container_set_border_width( GTK_CONTAINER( extention_vbox ), 0 );
		gtk_container_add( GTK_CONTAINER( filename_vbox ), extention_vbox );

		extention_toggle = gtk_check_button_new_with_label( "Don't remove Extention" );
		gtk_box_pack_start( GTK_BOX( extention_vbox ), extention_toggle, FALSE, FALSE, 0 );
		gtk_signal_connect( GTK_OBJECT( extention_toggle ), "toggled", GTK_SIGNAL_FUNC( toggle_extention ), NULL );

		if( dont_remove_extention == 1 )
			gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( extention_toggle ), TRUE );

		if( filename_type == 1 )
			gtk_widget_set_sensitive( extention_vbox, FALSE );


/* Path Config */

		path_hbox = gtk_hbox_new( FALSE, 5 );
		gtk_container_set_border_width( GTK_CONTAINER( path_hbox ), 10 );
		gtk_box_pack_start( GTK_BOX( vbox ), path_hbox, FALSE, FALSE, 0 );

		path_label = gtk_label_new( "Path:" );
		gtk_box_pack_start( GTK_BOX( path_hbox ), path_label, FALSE, FALSE, 0 );
		gtk_widget_show( path_label );

		path_entry = gtk_entry_new();
		if( file_path )
			gtk_entry_set_text( GTK_ENTRY( path_entry ), file_path );
		gtk_widget_set_usize( path_entry, 200, -1 );
		gtk_box_pack_start( GTK_BOX( path_hbox ), path_entry, FALSE, FALSE, 0 );

		path_browse = gtk_button_new_with_label( "Browse" );
		gtk_signal_connect( GTK_OBJECT( path_browse ), "clicked", GTK_SIGNAL_FUNC( path_browse_cb ), NULL );
		gtk_box_pack_start( GTK_BOX( path_hbox ), path_browse, FALSE, FALSE, 0 );


/* The Rest */

		/* Seperator */

		configure_separator = gtk_hseparator_new();
		gtk_box_pack_start( GTK_BOX( vbox ), configure_separator, FALSE, FALSE, 0 );
		gtk_widget_show( configure_separator );

		/* Buttons */

		configure_bbox = gtk_hbutton_box_new();
		gtk_button_box_set_layout( GTK_BUTTON_BOX( configure_bbox ), GTK_BUTTONBOX_END );
		gtk_button_box_set_spacing( GTK_BUTTON_BOX( configure_bbox ), 5 );
		gtk_box_pack_start( GTK_BOX( vbox ), configure_bbox, FALSE, FALSE, 0 );

		configure_ok = gtk_button_new_with_label( "Ok" );
		gtk_signal_connect( GTK_OBJECT( configure_ok ), "clicked", GTK_SIGNAL_FUNC( configure_ok_cb ), NULL );
		GTK_WIDGET_SET_FLAGS( configure_ok, GTK_CAN_DEFAULT );
		gtk_box_pack_start( GTK_BOX( configure_bbox ), configure_ok, TRUE, TRUE, 0 );
		gtk_widget_show( configure_ok );
		gtk_widget_grab_default( configure_ok );

		configure_cancel = gtk_button_new_with_label( "Cancel" );
		gtk_signal_connect_object( GTK_OBJECT( configure_cancel ), "clicked", GTK_SIGNAL_FUNC( gtk_widget_destroy ), GTK_OBJECT( configure_win ) );
		GTK_WIDGET_SET_FLAGS( configure_cancel, GTK_CAN_DEFAULT );
		gtk_box_pack_start( GTK_BOX( configure_bbox ), configure_cancel, TRUE, TRUE, 0 );

		/* Show it! */

		gtk_widget_show_all( configure_win );

	}
}
