/*
  Copyright (C) 2000-2008

  Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane

  This file is part of xmds.

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*
  $Id: xmds_breakpoint.cc 1885 2008-03-18 15:24:56Z paultcochrane $
*/

/*! @file xmds_breakpoint.cc
  @brief Breakpoint classes and methods

  More detailed explanation...
*/

#include <xmds_common.h>
#include <xmds_simulation.h>
#include <xmds_vector.h>
#include <xmds_breakpoint.h>
#include <string>
#include <iostream>
#include <fstream>
#include <vector>
#include "version.h"

// **************************************************************************
// **************************************************************************
//        xmdsBreakpoint public
// **************************************************************************
// **************************************************************************

extern bool debugFlag;
extern vector<string> simHeaderText, simBodyText, simFooterText;

long nxmdsBreakpoints = 0;  //!< Number of xmds breakpoint objects

// **************************************************************************
xmdsBreakpoint::xmdsBreakpoint(
                               const xmdsSimulation *const yourSimulation,
                               const bool& yourVerboseMode) :
  xmdsSegment(yourSimulation, yourVerboseMode) {
  if (debugFlag) {
    nxmdsBreakpoints++;
    printf("xmdsBreakpoint::xmdsBreakpoint\n");
    printf("nxmdsBreakpoints=%li\n", nxmdsBreakpoints);
  }
}

// **************************************************************************
xmdsBreakpoint::~xmdsBreakpoint() {
  if (debugFlag) {
    nxmdsBreakpoints--;
    printf("xmdsBreakpoint::~xmdsBreakpoint\n");
    printf("nxmdsBreakpoints=%li\n", nxmdsBreakpoints);
  }
}

// **************************************************************************
bool xmdsBreakpoint::runsInOvertime() const
{
  return true;
}

// **************************************************************************
void xmdsBreakpoint::processElement(const Element *const yourElement)
{
  if (debugFlag) {
    printf("xmdsBreakpoint::processElement\n");
  }

  if (simulation()->parameters()->stochastic &&
      simulation()->parameters()->nPaths != "1") {
    printf(
        "Warning: For stochastic simulations, each time a breakpoint element\n"
        "is reached for a path, the same file will be overwritten.\n");
  }

  const long unsigned int nDims = simulation()->field()->geometry()->nDims();
  const NodeList* candidateElements;
  list<XMLString> bpFnameList;

  // ************************************
  // find filename element

  candidateElements = yourElement->getElementsByTagName("filename", NOT_DEEP);

  if (candidateElements->length() == 0) {
    printf("Breakpoint names defaulting to the sequence 1.xsil, 2.xsil, etc.\n");
    autoIncrementFilenames = true;
  }
  else if (candidateElements->length()>1) {
    throw xmdsException(yourElement, "Multiple <filename> elements defined");
  }
  else {
    // find out what the filename is, and if no string is specified then
    // barf, because the tag has been specified, but no name
    getAssignmentStrings(yourElement, "filename", NOT_REQD, 1, bpFnameList);
    list<XMLString>::const_iterator tmpIter = bpFnameList.begin();
    breakpointFilename = *tmpIter;
    autoIncrementFilenames = false;

  }

  if (nDims>0) {

    // ************************************
    // find space

    list<bool> aSpaceList;

    getAssignmentBools(yourElement, "fourier_space", 0,
        simulation()->field()->geometry()->nDims(), aSpaceList);

    if (aSpaceList.size() == 0) {
      printf("Breakpoint space for defaulting to x-space.\n");
      outputSpace = 0;
    }
    else {
      list<bool>::const_iterator pBool = aSpaceList.begin();
      for (long unsigned int i = 0;
          i < simulation()->field()->geometry()->nDims(); i++) {
        if (verbose()) {
          if (*pBool)
            printf("Breakpoint will be written with dimension #%li in fourier space\n", i+1);
          else
            printf("Breakpoint will be written with dimension #%li in normal space\n", i+1);
        }
        pBool++;
      }
      outputSpace = spaceList2ULong(aSpaceList);
    }
  }

  // ************************************
  // find vectors

  // Vector names are required
  getAssignmentStrings(yourElement, "vectors", 1, 0, myVectorNamesList);

  simulation()->field()->processVectors(myVectorNamesList, outputSpace);

  if (verbose()) {
    printf("The breakpoint options are:\n");
    printf("   filename = %s\n", breakpointFilename.c_str());
  }

}

// **************************************************************************
// **************************************************************************
//                              xmdsBreakpoint private
// **************************************************************************
// **************************************************************************

// **************************************************************************
void xmdsBreakpoint::writePrototypes(FILE *const outfile) const
{
  if (debugFlag) {
    printf("xmdsBreakpoint::writePrototypes\n");
  }

  if (verbose()) {
    printf("Writing breakpoint prototypes ...\n");
  }

  fprintf(outfile, "void _segment%li(unsigned long cycle); // breakpoint\n",
      segmentNumber);
}

// **************************************************************************
void xmdsBreakpoint::writeRoutines(FILE *const outfile) const
{
  if (debugFlag) {
    printf("xmdsBreakpoint::writeRoutines\n");
  }

  const unsigned long nDims = simulation()->field()->geometry()->nDims();

  long unsigned int i;

  if (verbose()) {
    printf("Writing breakpoint routines ...\n");
  }

  fprintf(outfile,
      "// ********************************************************\n"
      "// segment %li (breakpoint) routines\n"
      "\n", segmentNumber);

  fprintf(outfile,
      "// *************************\n"
      "void _segment%li(unsigned long cycle) {\n"
      "\n", segmentNumber);

  if (autoIncrementFilenames) {
    fprintf(outfile,
        "_breakpointAutoIncrementCounter++;\n"
        "char *_baseFilename = (char*)malloc(100);\n"
        "snprintf(_baseFilename, 100, \"%%li\", _breakpointAutoIncrementCounter);\n");
  }
  else {
    // get the xsil filename, rip off the extension if it equals '.xsil'
    // otherwise, leave it alone
    XMLString xsilFilename = breakpointFilename;
    XMLString xsilExtension;
    xsilExtension += ".xsil";
    XMLString testExtension;
    xsilFilename.subString(testExtension, xsilFilename.length()-5, xsilFilename.length());

    if (testExtension == xsilExtension) {
      xsilFilename.deleteData(xsilFilename.length()-5, 5);
    }

    fprintf(outfile,
        "char *_baseFilename = \"%s\";\n", xsilFilename.c_str());
  }

  if (simulation()->parameters()->usempi &&
      !simulation()->parameters()->stochastic)
    fprintf(outfile, "MPI_Barrier(MPI_COMM_WORLD);\n");

  simulation()->field()->vectors2space(outfile, outputSpace, myVectorNamesList, "");

  bool swapped = false;
  if (simulation()->parameters()->usempi &&
      !simulation()->parameters()->stochastic) {
    if ((outputSpace & 1) & ((outputSpace >> 1) & 1)) {
      swapped = true;
    }
    fprintf(outfile,
            "// Only write to file if you are rank 0, as we cannot assume\n"
            "// that the nodes have equal access to the filesystem\n"
            "if (rank == 0) {\n");
  }

  fprintf(outfile,
      "char *_xsilFilename = (char*)malloc(200), *_datFilename = (char*)malloc(200);\n"
      "strncpy(_xsilFilename, _baseFilename, 200);\n"
      "strncpy(_datFilename, _baseFilename, 200);\n"
      "strcat(_xsilFilename, \".xsil\");\n"
      "strcat(_datFilename, \".dat\");\n");

  fprintf(outfile,
          "    FILE* _outfile = fopen(_xsilFilename, \"w\");\n"
          "\n"
          "    if (_outfile == 0) {\n"
          "      printf(\"Unable to open output file %%s\\n\", _xsilFilename);\n"
          "      printf(\"Exiting.\\n\");\n"
          "      return;\n"
          "    }\n"
          "\n");

  // put the header text into the output from the xmds script
  fprintf(outfile, "\n");
  for (unsigned int i = 0; i < simHeaderText.size(); i++) {
    fprintf(outfile, "      fprintf(_outfile, \"%%s\\n\", \"%s\");\n",
            simHeaderText[i].c_str());
  }

  // put the body text into the output from the xmds script
  fprintf(outfile, "\n");
  for (unsigned int i = 0; i < simBodyText.size(); i++) {
    fprintf(outfile, "      fprintf(_outfile, \"%%s\\n\", \"%s\");\n",
            simBodyText[i].c_str());
  }

  // biff out some informative text about xmds - this is the footer
  fprintf(outfile,
      "    fprintf(_outfile, \"\\n<info>\\n\");\n"
      "    fprintf(_outfile, \"Output generated with xmds version %s (%s).\\n\");\n",
          simulation()->parameters()->version.c_str(),
          simulation()->parameters()->revision.c_str());
  fprintf(outfile,
      "    fprintf(_outfile, "
      "\"See http://www.xmds.org for more information.\\n\");\n");

  if (simulation()->argStruct()->nameList.size() != 0) {
    fprintf(outfile,
        "    fprintf(_outfile, \"  "
        "Variables that can be specified on the command line: \\n\");\n");

    list<string>::const_iterator inameList =
      simulation()->argStruct()->nameList.begin();
    list<string>::const_iterator itypeList =
      simulation()->argStruct()->typeList.begin();
    for (long unsigned int
        i = 0; i < simulation()->argStruct()->nameList.size(); i++) {
        string theType = itypeList->c_str();
        if (theType == "double ") {
          fprintf(outfile,
              "    fprintf(_outfile, \"    "
              "Command line argument '%s' = %%e \\n\", %s);\n",
              inameList->c_str(), inameList->c_str());
        }
        else if (theType == "long ") {
          fprintf(outfile,
              "    fprintf(_outfile, \"    "
              "Command line argument '%s' = %%li \\n\", %s);\n",
              inameList->c_str(), inameList->c_str());
        }
        else if (theType == "float ") {
          fprintf(outfile,
              "    fprintf(_outfile, \"    "
              "Command line argument '%s' = %%e \\n\", %s);\n",
              inameList->c_str(), inameList->c_str());
        }
        else if (theType == "int ") {
          fprintf(outfile,
              "    fprintf(_outfile, \"    "
              "Command line argument '%s' = %%i \\n\", %s);\n",
              inameList->c_str(), inameList->c_str());
        }
        else {
          fprintf(outfile,
              "    fprintf(_outfile, \"    "
              "Command line argument '%s' is an unimplemented output type '%s'\\n\");\n",
              inameList->c_str(), itypeList->c_str());
        }
        itypeList++;
        inameList++;
      }
    fprintf(outfile, "\n");
  }

  fprintf(outfile, "    fprintf(_outfile, \"</info>\\n\");\n");
  // end of informative text


  // this is where the moment groups are written
  fprintf(outfile,
      "fprintf(_outfile, \"\\n\");\n"
      "fprintf(_outfile, \"<XSIL Name=\\\"breakpoint\\\">\\n\");\n");

  unsigned long nVariables =0;
  list <XMLString> variableNames;

  for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
      pXMLString != myVectorNamesList.end(); pXMLString++) {
    const xmdsVector *vector;
    simulation()->field()->getVector(*pXMLString, vector);
    if (vector->vectorType() == DOUBLE) {
      nVariables += vector->nComponents();
      for (long unsigned int i = 0; i < vector->nComponents(); i++) {
        XMLString realPartName = *vector->componentName(i);
        realPartName += "R";
        variableNames.push_back(realPartName);
      }
    }
    else {
      // complex vector
      nVariables += 2*vector->nComponents();
      for (long unsigned int i = 0; i < vector->nComponents(); i++) {
        XMLString baseName = *vector->componentName(i);
        XMLString realPartName = baseName;
        realPartName += "R";
        XMLString imagPartName = baseName;
        imagPartName += "I";
        variableNames.push_back(realPartName);
        variableNames.push_back(imagPartName);
      }
    }
  }

  fprintf(outfile,
      "fprintf(_outfile, "
      "\"  <Param Name=\\\"n_independent\\\">%li</Param>\\n\");\n", nDims);
  fprintf(outfile,
      "fprintf(_outfile, "
      "\"  <Array Name=\\\"variables\\\" Type=\\\"Text\\\">\\n\");\n");
  fprintf(outfile,
      "fprintf(_outfile, "
      "\"    <Dim>%li</Dim>\\n\");\n", nVariables+nDims);
  fprintf(outfile,
      "fprintf(_outfile, "
      "\"    <Stream><Metalink Format=\\\"Text\\\" Delimiter=\\\" \\\\n\\\"/>\\n\");\n");

  fprintf(outfile, "fprintf(_outfile, \"");
  for (i = 0; i < nDims; i++) {
    if (space(i)) {
      fprintf(outfile, "k");
    }
    fprintf(outfile, "%s ",
        simulation()->field()->geometry()->dimension(i)->name.c_str());
  }

  for (list<XMLString>::const_iterator pXMLString = variableNames.begin();
      pXMLString != variableNames.end(); pXMLString++)
    fprintf(outfile, "%s ", pXMLString->c_str());

  fprintf(outfile, "\\n\");\n");
  fprintf(outfile, "fprintf(_outfile, \"      </Stream>\\n\");\n");
  fprintf(outfile, "fprintf(_outfile, \"    </Array>\\n\");\n");
  fprintf(outfile, "fprintf(_outfile, "
      "\"  <Array Name=\\\"data\\\" Type=\\\"double\\\">\\n\");\n");

  for (i = 0; i < nDims; i++)
    fprintf(outfile,
        "fprintf(_outfile, "
        "\"    <Dim>%li</Dim>\\n\");\n",
        simulation()->field()->geometry()->dimension(i)->lattice);

  fprintf(outfile,
      "fprintf(_outfile, "
      "\"    <Dim>%li</Dim>\\n\");\n",
      nVariables+nDims);

  // the "precision" variable is for the xsil output, and for matlab
  string precisionString, precision;

  fprintf(outfile,
      "\nstring encodingStr;\n"
      "if (CPU_IS_BIG_ENDIAN) {\n"
      "    // big endian\n"
      "    encodingStr = \"BigEndian\";\n"
      "}\n"
      "else if (CPU_IS_LITTLE_ENDIAN) {\n"
      "    // little endian\n"
      "    encodingStr = \"LittleEndian\";\n"
      "}\n"
      "else {\n"
      "    // dunno what the byte order is\n"
      "    cout << \"I don't know what kind of byte ordering you're using\\n\";\n"
      "    cout << \"Using \\\"Binary\\\" as encoding\\n\";\n"
      "    encodingStr = \"Binary\";\n"
      "}\n"
      "\nstring ulongTypeStr;\n"
      "if (SIZEOF_UNSIGNED_LONG == 4) {\n"
      "    // 32 bit\n"
      "    ulongTypeStr = \"uint32\";\n"
      "}\n"
      "else if (SIZEOF_UNSIGNED_LONG == 8) {\n"
      "    // 64 bit\n"
      "    ulongTypeStr = \"uint64\";\n"
      "}\n"
      "else {\n"
      "    // dunno what the default size of the unsigned long is\n"
      "    ulongTypeStr = \"ulong\";\n"
      "}\n"
      "fprintf(_outfile, "
      "\"  <Stream><Metalink Format=\\\"Binary\\\" UnsignedLong=\\\"%%s\\\" \", "
      "ulongTypeStr.c_str());\n"
      "fprintf(_outfile, "
      "\"precision=\\\"double\\\" Type=\\\"Remote\\\" Encoding=\\\"%%s\\\"/>\", "
      "encodingStr.c_str());\n");

  fprintf(outfile,
      "fprintf(_outfile, \"\\n%%s\\n\", _datFilename);\n"
      "fprintf(_outfile, \"                </Stream>\\n\");\n"
      "fprintf(_outfile, \"</Array>\\n</XSIL>\\n\");\n");

  // put the footer text into the xmds simulation output
  fprintf(outfile, "\n");
  for (unsigned int i = 0; i < simFooterText.size(); i++) {
    fprintf(outfile, "      fprintf(_outfile, \"%%s\\n\", \"%s\");\n",
            simFooterText[i].c_str());
  }

  // close things down and tidy up a bit
  fprintf(outfile,
      "\n"
      "    fclose(_outfile);\n"
      "\n");

  fprintf(outfile,
      "FILE *fpBinary;\n"
      "if ((fpBinary = fopen(_datFilename, \"wb\")) == NULL) {\n"
      "  printf(\"Unable to open output file %%s\\n\", _datFilename);\n"
      "  printf(\"Chucking a spack....\\n\");\n"
      "  exit(255);\n"
      "}\n");

  fprintf(outfile,
      "unsigned long dataSize;\n"
      "off_t fieldOffset = 0;\n"
      "double coordinate;\n");

  for (long unsigned int i = 0; i < nDims; i++) {
    fprintf(outfile,
        "dataSize = _main_lattice%li;\n"
        "fwrite(&dataSize, sizeof(unsigned long), 1, fpBinary);\n", i);
    if (space(i)) {
      // k-space
      fprintf(outfile,
          "coordinate = -(_main_lattice%1$li)/2 * _main_dk%1$li;\n"
          "for (long _i0 = -(_main_lattice%1$li)/2; _i0 < (_main_lattice%1$li+1)/2; _i0++, coordinate += _main_dk%1$li) {\n", i);
    }
    else {
      // x-space
      fprintf(outfile,
          "coordinate = _main_xmin%1$li;\n"
          "for (long _i0 = 0; _i0 < _main_lattice%1$li; _i0++, coordinate += _main_dx%1$li) {\n", i);
    }
    fprintf(outfile,
        "\tfwrite(&coordinate, sizeof(double), 1, fpBinary);\n"
        "}\n"
        "fieldOffset += sizeof(unsigned long) + sizeof(double)*_main_lattice%li;\n", i);
  }

  fprintf(outfile,
      "off_t vectorFieldSize = _main_size*sizeof(double) + sizeof(unsigned long);\n"
      "dataSize = _main_size;\n\n");

  for (long unsigned int i = 0; i < nVariables; i++)
    fprintf(outfile,
        "fseeko(fpBinary, fieldOffset + %li*vectorFieldSize, SEEK_SET);\n"
        "fwrite(&dataSize, sizeof(unsigned long), 1, fpBinary);\n\n", i);

  // Insert code that does the magic of accumulating all the data
  if (simulation()->parameters()->usempi &&
      !simulation()->parameters()->stochastic) {
    /* BEGIN MPI-SPECIFIC CODE */
    // nightmare. But this is the bit I care about...
    fprintf(outfile,
        "int my_local_nx = local_nx;\n"
        "int my_local_x_start = local_x_start;\n"
        "int my_local_ny_after_transpose = local_ny_after_transpose;\n"
        "int my_local_y_start_after_transpose = local_y_start_after_transpose;\n"
        "int my_total_local_size = total_local_size;\n"
        "for (int dataForRank = 0; dataForRank < size; dataForRank++) {\n"
        "int local_nx;\n"
        "int local_x_start;\n"
        "int local_ny_after_transpose;\n"
        "int local_y_start_after_transpose;\n"
        "int total_local_size;\n");

    for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
        pXMLString != myVectorNamesList.end(); pXMLString++) {
      const xmdsVector *vector;
      simulation()->field()->getVector(*pXMLString, vector);
      fprintf(outfile, "%s *_outputfield_%s;\n",
          vector->vectorType() == DOUBLE ? "double" : "complex",
          pXMLString->c_str());
    }

    fprintf(outfile,
        "if (dataForRank == 0) {\n"
        "    local_nx = my_local_nx;\n"
        "    local_x_start = my_local_x_start;\n"
        "    local_ny_after_transpose = my_local_ny_after_transpose;\n"
        "    local_y_start_after_transpose = my_local_y_start_after_transpose;\n"
        "    total_local_size = my_total_local_size;\n");

    for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
        pXMLString != myVectorNamesList.end(); pXMLString++) {
      fprintf(outfile,
          "    _active_main_%1$s = _main_%1$s;\n", pXMLString->c_str());
    }
    fprintf(outfile,
        "}\n"
        "else {\n"
        "    MPI_Status status;\n"
        "    MPI_Recv(&local_nx, 1, MPI_INT, dataForRank, MPI_ANY_TAG, MPI_COMM_WORLD, &status);\n"
        "    MPI_Recv(&local_x_start, 1, MPI_INT, dataForRank, MPI_ANY_TAG, MPI_COMM_WORLD, &status);\n"
        "    MPI_Recv(&local_ny_after_transpose, 1, MPI_INT, dataForRank, MPI_ANY_TAG, MPI_COMM_WORLD, &status);\n"
        "    MPI_Recv(&local_y_start_after_transpose, 1, MPI_INT, dataForRank, MPI_ANY_TAG, MPI_COMM_WORLD, &status);\n"
        "    MPI_Recv(&total_local_size, 1, MPI_INT, dataForRank, MPI_ANY_TAG, MPI_COMM_WORLD, &status);\n"
        "    // Now allocate the space needed locally, and receive the entire buffer\n\n");

    for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
        pXMLString != myVectorNamesList.end(); pXMLString++) {
      const xmdsVector *vector;
      simulation()->field()->getVector(*pXMLString, vector);

      fprintf(outfile,
          "    _outputfield_%1$s = (%2$s *)fftw_malloc(sizeof(%2$s)*total_local_size*_main_%1$s_ncomponents);\n",
          pXMLString->c_str(),
          vector->vectorType() == DOUBLE ? "double" : "complex");
      fprintf(outfile,
          "    MPI_Recv(_outputfield_%1$s, %2$s*total_local_size*_main_%1$s_ncomponents, MPI_DOUBLE, dataForRank, MPI_ANY_TAG, MPI_COMM_WORLD, &status);\n",
          pXMLString->c_str(), vector->vectorType() == DOUBLE ? "1" : "2");
      fprintf(outfile,
          "    _active_main_%1$s = _outputfield_%1$s;\n\n",
          pXMLString->c_str());
    }
    fprintf(outfile, "}\n\n");

    /* END MPI-SPECIFIC CODE */
  }

  int variablesInEarlierVectors = 0;
  for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
      pXMLString != myVectorNamesList.end(); pXMLString++) {
    const xmdsVector *vector;
    simulation()->field()->getVector(*pXMLString, vector);

    fprintf(outfile,
        "{ // Begin block for vector '%1$s'\n", pXMLString->c_str());
    fprintf(outfile,
        "off_t _outputfield_index_pointer, _outputfield_old_index_pointer;\n"
        "for (unsigned int _component = 0; _component < %li; _component++) { \n"
        "_outputfield_index_pointer = -42; // Just so that we always seek the first time\n",
        vector->nComponents() * (vector->vectorType() == DOUBLE ? 1 : 2));

    // write vector data
    simulation()->field()->openLoops(outfile, outputSpace, myVectorNamesList, DO_NOT_PARALLELISE_LOOP);

    // write data
    fprintf(outfile,
        "_outputfield_old_index_pointer = _outputfield_index_pointer;\n"
        "_outputfield_index_pointer = 0;\n");

    for (long unsigned int
        i = 0; i < simulation()->field()->geometry()->nDims(); i++) {

      // Calculate the output field index pointer
      if ((outputSpace >> i) & 1) {
        fprintf(outfile,
            "_outputfield_index_pointer += (lround(k%2$s/_main_dk%1$li) + (_main_lattice%1$li/2))",
            i, simulation()->field()->geometry()->dimension(i)->name.c_str());
      }
      else {
        fprintf(outfile,
            "_outputfield_index_pointer += lround((%2$s - _main_xmin%1$li)/_main_dx%1$li)",
            i, simulation()->field()->geometry()->dimension(i)->name.c_str());
      }

      for (long unsigned int
          j = i+1; j < simulation()->field()->geometry()->nDims(); j++) {
        fprintf(outfile, "*_main_lattice%li", j);
      }
      fprintf(outfile, ";\n\n");

    }

    fprintf(outfile,
        "if (_outputfield_index_pointer != _outputfield_old_index_pointer + 1)\n"
        "\t fseeko(fpBinary, fieldOffset + _outputfield_index_pointer*sizeof(double) + (%i + _component)*vectorFieldSize + sizeof(unsigned long), SEEK_SET);\n", variablesInEarlierVectors);

    if (vector->vectorType() == DOUBLE) {
      fprintf(outfile,
          "fwrite(&_active_main_%1$s[_main_%1$s_index_pointer + _component], sizeof(double), 1, fpBinary);\n",
          vector->name()->c_str());
    }
    else { // COMPLEX
      fprintf(outfile,
          "if (_component & 1) \n"
          "\t fwrite(&_active_main_%1$s[_main_%1$s_index_pointer + _component/2].im, sizeof(double), 1, fpBinary);\n"
          "else \n"
          "\t fwrite(&_active_main_%1$s[_main_%1$s_index_pointer + _component/2].re, sizeof(double), 1, fpBinary);\n",
          vector->name()->c_str());
    }

    simulation()->field()->closeLoops(outfile, outputSpace, myVectorNamesList);
    fprintf(outfile,
        "} // end for loop over components of vector '%s'\n",
        vector->name()->c_str());
    fprintf(outfile,
        "} // end block for vector '%s'\n",
        vector->name()->c_str());
    variablesInEarlierVectors += vector->nComponents();

    if (vector->vectorType() == COMPLEX) {
      variablesInEarlierVectors += vector->nComponents();
    }
  } // end for loop

  if (simulation()->parameters()->usempi &&
      !simulation()->parameters()->stochastic) {
    /* BEGIN MPI-SPECIFIC CODE */
    fprintf(outfile, "if (dataForRank != 0) {\n");
    for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
        pXMLString != myVectorNamesList.end(); pXMLString++) {
      fprintf(outfile,
          //"    fprintf(stderr, \"Freeing _outputfield_%1$s\\n\");\n"
              "    fftw_free(_outputfield_%1$s);\n", pXMLString->c_str());
    }

    fprintf(outfile,
        "}\n"
        "// End looping over the ranks\n"
        "}\n"
        "local_nx = my_local_nx;\n"
        "local_x_start = my_local_x_start;\n"
        "local_ny_after_transpose = my_local_ny_after_transpose;\n"
        "local_y_start_after_transpose = my_local_y_start_after_transpose;\n"
        "total_local_size = my_total_local_size;\n");

    for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
        pXMLString != myVectorNamesList.end(); pXMLString++) {
      fprintf(outfile,
          "_active_main_%1$s = _main_%1$s;\n", pXMLString->c_str());
    }
    /* END MPI-SPECIFIC CODE */
  }

  // close the binary file
  fprintf(outfile, "fclose(fpBinary);\n");

  // close the main file-writing code for rank == 0
  if (simulation()->parameters()->usempi &&
      !simulation()->parameters()->stochastic) {

    if ((outputSpace & 1) & ((outputSpace >> 1) & 1)) {
      swapped = true;
    }

    fprintf(outfile,
        "// Only write to file if you are rank 0, as we cannot assume\n"
        "// that the nodes have equal access to the filesystem\n"
        "}\n"
        "else {\n"
        "// We are some other rank that isn't 0\n"
        "    MPI_Ssend(&local_nx, 1, MPI_INT, 0, 0, MPI_COMM_WORLD);\n"
        "    MPI_Ssend(&local_x_start, 1, MPI_INT, 0, 0, MPI_COMM_WORLD);\n"
        "    MPI_Ssend(&local_ny_after_transpose, 1, MPI_INT, 0, 0, MPI_COMM_WORLD);\n"
        "    MPI_Ssend(&local_y_start_after_transpose, 1, MPI_INT, 0, 0, MPI_COMM_WORLD);\n"
        "    MPI_Ssend(&total_local_size, 1, MPI_INT, 0, 0, MPI_COMM_WORLD);\n\n");

    for (list<XMLString>::const_iterator pXMLString = myVectorNamesList.begin();
        pXMLString != myVectorNamesList.end(); pXMLString++) {
      const xmdsVector *vector;
      simulation()->field()->getVector(*pXMLString, vector);
      fprintf(outfile,
          "    MPI_Ssend(_main_%1$s, %2$s*total_local_size*_main_%1$s_ncomponents, MPI_DOUBLE, 0, 0, MPI_COMM_WORLD);\n",
          pXMLString->c_str(), vector->vectorType() == DOUBLE ? "1" : "2");
    }

    fprintf(outfile,
        "}\n"
        "MPI_Barrier(MPI_COMM_WORLD);\n");
  }

  fprintf(outfile, "}\n\n");
  xmdsElement::writeRoutines(outfile);
}

// **************************************************************************
//! Returns space at index.  Don't ask me why this returns boolean
bool xmdsBreakpoint::space(
                           const long unsigned int& index) const {
  if (debugFlag) {
    printf("xmdsFilter::space\n");
  }

  if (index >= simulation()->field()->geometry()->nDims()) {
    throw xmdsException("Internal range error in xmdsBreakpoint::space()");
  }

  return (outputSpace >> index) & 1;
}

/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
