/*
 Copyright (C) 2000-2007
 
 Code contributed by Greg Collecutt, Joseph Hope and Paul Cochrane
 Modifications to output in Mathematica format by Andrew Reid
 
 This file is part of xmds.
 
 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

/*
 $Id: xsil_field.cc 1631 2007-12-22 12:56:11Z joehope $
 */

/*! @file xsil_field.cc
@brief XSIL format parsing classes and methods

More detailed explanation...
*/

#include <cstring>
#include <cstdlib>
#include <string>
#include <ctype.h>
#include <xmds_common.h>
#include <xsil_field.h>

//! Turn debugging on or off
#define DEBUG 0

// **************************************************************************
// **************************************************************************
//                              xsilField public
// **************************************************************************
// **************************************************************************

long nxsilFields = 0; //!< The number of xsil fields

// **************************************************************************
xsilField::xsilField() {
	if (DEBUG) {
		nxsilFields++;
		printf("xsilField::xsilField\n");
		printf("nxsilFields=%li\n", nxsilFields);
	}
}

// **************************************************************************
xsilField::~xsilField() {
	if (DEBUG) {
		nxsilFields--;
		printf("xsilField::~xsilField\n");
		printf("nxsilFields=%li\n", nxsilFields);
	}
}

// **************************************************************************
void xsilField::processElement(const Element *const yourElement)
{
	if (DEBUG) {
		printf("xsilField::processElement\n");
	}
	
	fieldName = "";
	nIndependentVariables = 0;
	variableNamesList.clear();
	latticeList.clear();
	
	const NodeList* candidateElements;
	const NamedNodeMap* elementAttributes;
	const Node* attributeNode;
	list<XMLString> anXMLStringList;
	
	// ************************************
	// find name
	
	elementAttributes = yourElement->attributes();
	
	attributeNode = elementAttributes->getNamedItem("Name");
	
	if (attributeNode != 0) {
		fieldName = *attributeNode->nodeValue();
	}
	else {
		throw xmdsException(yourElement, "Where is my Name='...' attribute?");
	}
	
	// ************************************
	// find n_independent Param assignment
	
	candidateElements = yourElement->getElementsByTagName("Param", 0);
	
	if (candidateElements->length() > 1) {
		throw xmdsException(yourElement, "Only one <Param> element expected ");
	}
	
	elementAttributes = candidateElements->item(0)->attributes();
	
	attributeNode = elementAttributes->getNamedItem("Name");
	
	if (attributeNode != 0) {
		if (*attributeNode->nodeValue() == "n_independent") {
			if (!candidateElements->item(0)->textContent(0)->asULong(nIndependentVariables)) {
				throw xmdsException(candidateElements->item(0),
									"Invalid positive integer format");
			}
		}
		else {
			throw xmdsException(candidateElements->item(0), "Unknown <Param> element");
		}
	}
	else {
		throw xmdsException(candidateElements->item(0),
							"Where is my Name='...' attribute?");
	}
	
	// ************************************
	// find Arrays
	
	candidateElements = yourElement->getElementsByTagName("Array", 0);
	
	if (candidateElements->length() != 2) {
		throw xmdsException(yourElement, "Exactly two <Array> elements expected ");
	}
	
	// ************************************
	// find variables Array
	
	elementAttributes = candidateElements->item(0)->attributes();
	
	attributeNode = elementAttributes->getNamedItem("Name");
	
	if (attributeNode != 0) {
		if (*attributeNode->nodeValue() != "variables") {
			sprintf(errorMessage(), "Unknown <Array> element '%s'",
					attributeNode->nodeValue()->c_str());
			throw xmdsException(yourElement, errorMessage());
		}
	}
	else {
		throw xmdsException(candidateElements->item(0),
							"Where is my Name='...' attribute?");
	}
	
	const Element* myVariablesArrayElement =
		dynamic_cast<const Element*>(candidateElements->item(0));
	
	// ************************************
	// find data Array
	
	elementAttributes = candidateElements->item(1)->attributes();
	
	attributeNode = elementAttributes->getNamedItem("Name");
	
	if (attributeNode != 0) {
		if (*attributeNode->nodeValue() != "data") {
			sprintf(errorMessage(), "Unknown <Array> element '%s'",
					attributeNode->nodeValue()->c_str());
			throw xmdsException(yourElement, errorMessage());
		}
	}
	else {
		throw xmdsException(candidateElements->item(1),
							"Where is my Name='...' attribute?");
	}
	
	const Element* myDataArrayElement =
		dynamic_cast<const Element*>(candidateElements->item(1));
	
	// ************************************
	// process variables Array
	
	unsigned long nVariables;
	
	candidateElements = myVariablesArrayElement->getElementsByTagName("Dim", 0);
	
	if (candidateElements->length() > 1) {
		throw xmdsException(myVariablesArrayElement, "Only one <Dim> element expected ");
	}
	
	if (!candidateElements->item(0)->textContent(0)->asULong(nVariables)) {
		throw xmdsException(candidateElements->item(0),
							"Invalid positive integer format");
	}
	
	getAssignmentStrings(myVariablesArrayElement, "Stream", 1, nVariables, variableNamesList);
	
	
	// ************************************
	// process data Array
	
	candidateElements = myDataArrayElement->getElementsByTagName("Dim", 0);
	
	if (candidateElements->length() != nIndependentVariables+1) {
		sprintf(errorMessage(), "Exactly %li <Dim> elements expected",
				nIndependentVariables + 1);
		throw xmdsException(myDataArrayElement, errorMessage());
	}
	
	for (unsigned long i=0; i<candidateElements->length(); i++) {
		
		unsigned long nextDim;
		
		if (!candidateElements->item(i)->textContent(0)->asULong(nextDim)) {
			throw xmdsException(candidateElements->item(i),
								"Invalid positive integer format");
		}
		
		latticeList.push_back(nextDim);
	}
	
	candidateElements = myDataArrayElement->getElementsByTagName("Stream", 0);
	
	if (candidateElements->length() != 1) {
		throw xmdsException(myDataArrayElement, "A <Stream> element expected");
	}
	
	const Element* myStreamElement =
		dynamic_cast<const Element*>(candidateElements->item(0));
	
	// get the Metalink tags
	candidateElements = myDataArrayElement->getElementsByTagName("Metalink", 1);
	
	if (candidateElements->length() != 1) {
		throw xmdsException(myDataArrayElement, "<Metalink> element expected");
	}
	
	// get the Format attribute
	elementAttributes = candidateElements->item(0)->attributes();
	
	attributeNode = elementAttributes->getNamedItem("Format");
	
	if (attributeNode != 0) {
		if (*attributeNode->nodeValue() != "Text" &&
			*attributeNode->nodeValue() != "Binary") {
			sprintf(errorMessage(), "Unknown <Metalink> attribute '%s'",
					attributeNode->nodeValue()->c_str());
			throw xmdsException(yourElement, errorMessage());
		}
	}
	else {
		throw xmdsException(candidateElements->item(0),
							"Where is my Format='...' attribute?");
	}
	
	//  determine the stream format and work out where the data is
	streamFormat = *attributeNode->nodeValue();
	if (streamFormat == "Text") {
		// ok then, the data is ascii, go get it tiger!
		data = *myStreamElement->textContent(0);
	}
	else if (streamFormat == "Binary") {
		// data is binary, the textContent of the Stream element is the datafile filename
		binDatFname = *myStreamElement->textContent(0);
		
		// try to remove whitespace chars
		std::string tmp = binDatFname.c_str();
		std::string tmp2;
		for (unsigned long int i=0; i<tmp.length(); i++) {
			if (!isspace(tmp[i])) {
				tmp2 += tmp[i];
			}
		}
		
		// biff the temporary variable into the binary data filename
		binDatFname = tmp2.c_str();
		
		// need to check what encoding the data is in
		// is it big or little endian??
		attributeNode = elementAttributes->getNamedItem("Encoding");
		if (attributeNode != 0) {
			if (*attributeNode->nodeValue() != "LittleEndian" &&
				*attributeNode->nodeValue() != "BigEndian") {
				sprintf(errorMessage(), "Unknown <Metalink> attribute '%s'",
						attributeNode->nodeValue()->c_str());
				throw xmdsException(yourElement, errorMessage());
			}
		}
		else {
			throw xmdsException(candidateElements->item(0),
								"Where is my Encoding='...' attribute?");
		}
		
		// the binary encoding of the data file
		binEncoding = *attributeNode->nodeValue();
		
		// get the precision attribute
		elementAttributes = candidateElements->item(0)->attributes();
		
		attributeNode = elementAttributes->getNamedItem("precision");
		
		if (attributeNode != 0) {
			if (*attributeNode->nodeValue() != "single" &&
				*attributeNode->nodeValue() != "double") {
				sprintf(errorMessage(), "Unknown <Metalink> attribute '%s'",
						attributeNode->nodeValue()->c_str());
				throw xmdsException(yourElement, errorMessage());
			}
		}
		else {
			throw xmdsException(candidateElements->item(0),
								"Where is my precison='...' attribute?");
		}
		
		// the precison of the binary data
		binPrecision = *attributeNode->nodeValue();
		
		// get the UnsignedLong attribute
		elementAttributes = candidateElements->item(0)->attributes();
		
		attributeNode = elementAttributes->getNamedItem("UnsignedLong");
		
		if (attributeNode != 0) {
			if (*attributeNode->nodeValue() != "ulong" &&
				*attributeNode->nodeValue() != "uint32" &&
				*attributeNode->nodeValue() != "uint64") {
				sprintf(errorMessage(), "Unknown <Metalink> attribute '%s'",
						attributeNode->nodeValue()->c_str());
				throw xmdsException(yourElement, errorMessage());
			}
			ulongType = *attributeNode->nodeValue();
		}
		else {
			ulongType = "ulong";
			printf("Defaulting to ulong\n");
		}
	}
}

// **************************************************************************
void xsilField::writeAsFormat(
                              FILE *const outfile,
                              const outputFormatEnum& format,
                              const long& iD,
                              const char *datFileNameBase) {
	if (DEBUG) {
		printf("xsilField::writeAsFormat\n");
	}
	
    // In order to avoid having any underscores in the variable names for
    // Mathematica one needs to avoid using the xsil filename, in case it is in
    // a disallowed format, therefore a new variable will be created which
    // contains the name minus any underscores, to use throughout the
    // Mathematica portions of the code.  It's a string because that makes
    // removing just the underscores easier.
	
    string datFileNameClear(datFileNameBase);
	
    if ((format == FORMAT_MATHEMATICA)||(format == FORMAT_MATHEMATICA5)) {
		
		for (unsigned long i=0; i<datFileNameClear.size(); i++) {
			if (datFileNameClear[i] == '_') {
				for (unsigned long j=i; j<datFileNameClear.size(); j++) {
					datFileNameClear[j] = datFileNameClear[j+1];
				}
				datFileNameClear.erase(datFileNameClear.size()-1);
				i--;
			}
		}
    }
	
	if (streamFormat == "Text") {
		
		// dump data as ascii file
		
		char datFileName[64];
		
		sprintf(datFileName, "%s%li.dat", datFileNameBase, iD);
		
		FILE *tempfile = fopen(datFileName, "w");
		
		// write header row (this doesn't work for matlab or mathematica!)
		if (format != FORMAT_MATLAB &&
			format != FORMAT_MATHEMATICA5 &&
			format != FORMAT_MATHEMATICA &&
			format != FORMAT_R) {
			for (list<XMLString>::const_iterator
				 pXMLString = variableNamesList.begin();
				 pXMLString != variableNamesList.end();
				 pXMLString++) {
				fprintf(tempfile, " %s ", pXMLString->c_str());
			}
		}
		else if (format == FORMAT_R) {
			// R likes to have the names the same in the header as in the data
			for (list<XMLString>::const_iterator
				 pXMLString = variableNamesList.begin();
				 pXMLString != variableNamesList.end();
				 pXMLString++) {
				fprintf(tempfile, " %s_%li ", pXMLString->c_str(), iD);
			}
		}
		
		// write data
		fprintf(tempfile, "%s", data.c_str());
		fclose(tempfile);
		
		// there is no file used by Gnuplot to load the data, so we can just
		// return here now
		if (format == FORMAT_GNUPLOT) {
			fprintf(outfile,
					"# Gnuplot requires no variable loading commands.\n"
					"# Please use the relevant plot commands to display your data.\n"
					"# This comment is just a placeholder.\n");
			return;  // returning from a void function, duh.
		}
		
		/*! \todo Writing of the files for data loading into the various
			* environments should really be in separate routines
			*/
		
		// If the output format is Mathematica, an initial line is needed to load
		// an addon package with some necessary commands
		if (format == FORMAT_MATHEMATICA5) {
			fprintf(outfile, "<< LinearAlgebra`MatrixManipulation`\n");
		}
		
		// initialise variables
		list<XMLString>::iterator pXMLString = variableNamesList.begin();
		for (unsigned long i=0; i<variableNamesList.size(); i++) {
			
			// need to format variable names to remove any non alpha-numeric
			// characters.  Mathematica disallows '_' in variable names, so they also
			// need to be removed for that case.
			// Except this isn't working, grrrr.
			if ((format == FORMAT_MATHEMATICA)||(format == FORMAT_MATHEMATICA5)) {
				pXMLString->goLatinAlphaNumericNoUnderScore();
			}
			else{
				pXMLString->goLatinAlphaNumeric();
			}
			
			char tempString[64];
			// Mathematica doesn't like an '_' in variable names.
			if ((format == FORMAT_MATHEMATICA)||(format == FORMAT_MATHEMATICA5)) {
				sprintf(tempString, "%li", iD);
			}
			else {
				sprintf(tempString, "_%li", iD);
			}
			*pXMLString += tempString;
			
			// Since Mathematica has a very different format to Scilab and Matlab,
			// this needs to be split off, even though the purpose is very similar.
			
			if (format == FORMAT_MATHEMATICA5) {
				
				if (nIndependentVariables == 0) {
					fprintf(outfile, "%s = Table[0, {i1, 1}];\n", pXMLString->c_str());
				}
				else if (nIndependentVariables == 1) {
					if (i == 0) {
						fprintf(outfile, "tempd1 = Table[0, {i1, %li}];\n", lattice(0));
					}
					else {
						fprintf(outfile, "%s = Table[0, {i1, %li}];\n",
								pXMLString->c_str(), lattice(0));
					}
				}
				else {
					if (i < nIndependentVariables) {
						fprintf(outfile, "tempd%li = Table[0, {i1, %li}",
								i+1, lattice(nIndependentVariables-1));
					}
					else {
						fprintf(outfile, "%s = Table[0, {i1, %li}",
								pXMLString->c_str(), lattice(nIndependentVariables-1));
					}
					
					for (unsigned long j=nIndependentVariables-1; j>0; j--) {
						fprintf(outfile, ", {i%li, %li}",
								(nIndependentVariables - j + 1) , lattice(j-1));
					}
					
					fprintf(outfile, "];\n");
				}
				
				if (i < nIndependentVariables) {
					fprintf(outfile, "%s = Table[0, {i1, %li}];\n",
							pXMLString->c_str(), lattice(i));
				}
			}
			else if (format == FORMAT_MATHEMATICA) {
				
				// Version 6+ needs no headers
			}
			// The section for Matlab and Scilab
			else if (format == FORMAT_MATLAB ||
					 format == FORMAT_SCILAB) {
				
				if (nIndependentVariables == 0) {
					fprintf(outfile, "%s = zeros(1, 1);\n", pXMLString->c_str());
				}
				else if (nIndependentVariables == 1) {
					if (i == 0) {
						fprintf(outfile, "temp_d1 = zeros(1, %li);\n", lattice(0));
					}
					else {
						fprintf(outfile, "%s = zeros(1, %li);\n",
								pXMLString->c_str(), lattice(0));
					}
				}
				else {
					if (i < nIndependentVariables) {
						fprintf(outfile, "temp_d%li = zeros(%li",
								i+1, lattice(nIndependentVariables-1));
					}
					else {
						fprintf(outfile, "%s = zeros(%li",
								pXMLString->c_str(), lattice(nIndependentVariables-1));
					}
					
					for (unsigned long j=nIndependentVariables-1; j>0; j--) {
						fprintf(outfile, ", %li", lattice(j-1));
					}
					
					fprintf(outfile, ");\n");
				}
				
				if (i < nIndependentVariables) {
					fprintf(outfile, "%s = zeros(1, %li);\n",
							pXMLString->c_str(), lattice(i));
				}
			}
			else if (format == FORMAT_R) {
				// I don't think we need to add anything here for R, but this is
				// just a placeholder just in case
			}
			else {
				throw xmdsException("Unknown ASCII output format!\n");
			}
			
			pXMLString++;
		}
		fprintf(outfile, "\n");
		
		// load in temp file
		if (format == FORMAT_SCILAB) {
			fprintf(outfile, "%s%li = fscanfMat('%s');\n",
					datFileNameBase, iD, datFileName);
		}
		else if (format == FORMAT_MATHEMATICA5) {
			fprintf(outfile, "%s%li = Import[\"%s\", \"Table\"];\n\n",
					datFileNameClear.c_str(), iD, datFileName);
		}
		else if (format == FORMAT_MATHEMATICA) {
			fprintf(outfile, "%s%li = Drop[Drop[Import[\"%s\", \"Table\"],1],-1];\n\n",
					datFileNameClear.c_str(), iD, datFileName);
		}
		else if (format == FORMAT_R) {
			fprintf(outfile, "%s%li <- read.table(\"%s\", header=TRUE)\n",
					datFileNameBase, iD, datFileName);
		}
		else {
			fprintf(outfile, "load %s -ascii\n", datFileName);
		}
		
		if (format == FORMAT_MATHEMATICA) {
			// Initialising
			for (unsigned long i=nIndependentVariables; i<variableNamesList.size(); i++) {
				// Now start construcing the Do
				fprintf(outfile, "%s = Table[Take[%s%li,All,{%li,%li}][[", variableName(i)->c_str(),datFileNameClear.c_str(), iD,i+1,i+1);
								
				for (unsigned long j=0; j<nIndependentVariables; j++) {
					if (j == 0) {
						fprintf(outfile, "i%li", j+1);
					}
					else {
						// To build up the  index properly we need a value that is the multiple
						// of the maximum possible value for each previous index, potentially a
						// very large number.
						unsigned long mult = 1;
						for (unsigned long k=(nIndependentVariables-j); k<nIndependentVariables; k++) {
							mult *= lattice(k);
						}
						fprintf(outfile, " + %li* (i%li - 1)", mult, j+1);
					}
				}
				fprintf(outfile, ",1]]");
				for (unsigned long j=0; j<nIndependentVariables; j++) {
					fprintf(outfile, ", {i%li, 1, %li}",
							(nIndependentVariables - j), lattice(j));
				}
				fprintf(outfile, "];\n");
				
			}
		}
		else if (format == FORMAT_MATHEMATICA5) {
			// Initialising the multidimensional arrays in Mathematica is less
			// straightforward than in matlab and scilab.  Bulding the elaborate Do
			// command that is necessary requires all of the for loops below.
			
			for (unsigned long i=0; i<nIndependentVariables; i++) {
				// To read the data in correctly we need to run along the column of
				// numbers provided by the xsil for each variable.  To do this effficently
				// we first split out the column
				fprintf(outfile,
						"%s%licolumn = Flatten[TakeColumns[%s%li , {%li, %li}]];\n",
						datFileNameClear.c_str(), iD, datFileNameClear.c_str(), iD, i+1, i+1);
				
				// Now write the Do command to assign the data
				fprintf(outfile, "Do[tempd%li[[", i+1);
				for (unsigned long j=0; j<nIndependentVariables; j++) {
					if (j == 0) {
						fprintf(outfile, "i%li", j+1);
					}
					else {
						fprintf(outfile, ", i%li", j+1);
					}
				}
				
				fprintf(outfile, "]] = %s%licolumn[[", datFileNameClear.c_str(), iD);
				
				for (unsigned long j=0; j<nIndependentVariables; j++) {
					if (j == 0) {
						fprintf(outfile, "i%li", j+1);
					}
					else {
						// To build up the  index properly we need a value that is the multiple
						// of the maximum possible value for each previous index, potentially a
						// very large number.
						unsigned long mult = 1;
						for (unsigned long k=(nIndependentVariables-j) ; k<nIndependentVariables; k++) {
							mult *= lattice(k);
						}
						fprintf(outfile, " + %li* (i%li - 1)", mult, j+1);
					}
				}
				fprintf(outfile, "]]");
				for (unsigned long j=nIndependentVariables-1; j>0; j--) {
					fprintf(outfile, ", {i%li, 1, %li}",
							(nIndependentVariables - j), lattice(j));
				}
				fprintf(outfile, ", {i%li, 1, %li}]\n\n",
						nIndependentVariables, lattice(0));
				
			}
			// Initialising
			for (unsigned long i=nIndependentVariables; i<variableNamesList.size(); i++) {
				// Again, start by seperating the column of interest
				fprintf(outfile,
						"%s%licolumn = Flatten[TakeColumns[%s%li , {%li, %li}]];\n",
						datFileNameClear.c_str(), iD, datFileNameClear.c_str(), iD, i+1, i+1);
				
				// Now start construcing the Do
				fprintf(outfile, "Do[%s[[", variableName(i)->c_str());
				
				for (unsigned long j=0; j<nIndependentVariables; j++) {
					if (j == 0) {
						fprintf(outfile, "i%li", j+1);
					}
					else {
						fprintf(outfile, ", i%li", j+1);
					}
				}
				
				fprintf(outfile, "]] = %s%licolumn[[", datFileNameClear.c_str(), iD);
				
				for (unsigned long j=0; j<nIndependentVariables; j++) {
					if (j == 0) {
						fprintf(outfile, "i%li", j+1);
					}
					else {
						// To build up the  index properly we need a value that is the multiple
						// of the maximum possible value for each previous index, potentially a
						// very large number.
						unsigned long mult = 1;
						for (unsigned long k=(nIndependentVariables-j); k<nIndependentVariables; k++) {
							mult *= lattice(k);
						}
						fprintf(outfile, " + %li* (i%li - 1)", mult, j+1);
					}
				}
				fprintf(outfile, "]]");
				for (unsigned long j=nIndependentVariables-1; j>0; j--) {
					fprintf(outfile, ", {i%li, 1, %li}",
							(nIndependentVariables - j), lattice(j));
				}
				fprintf(outfile, ", {i%li, 1, %li}]\n\n",
						nIndependentVariables, lattice(0));
				
			}
		}
		else if (format == FORMAT_MATLAB ||
				 format == FORMAT_SCILAB) { // Matlab, Octave and Scilab
			for (unsigned long i=0; i<nIndependentVariables; i++) {
				fprintf(outfile, "temp_d%li(:) = %s%li(:, %li);\n",
						i+1, datFileNameBase, iD, i+1);
			}
			
			for (unsigned long i=nIndependentVariables; i<variableNamesList.size(); i++) {
				fprintf(outfile, "%s(:) = %s%li(:, %li);\n",
						variableName(i)->c_str(), datFileNameBase, iD, i+1);
			}
		}
		else if (format == FORMAT_R) {
			for (unsigned long i=nIndependentVariables; i<variableNamesList.size(); i++) {
				fprintf(outfile, "%s <- %s%li$%s\n",
						variableName(i)->c_str(), datFileNameBase, iD, variableName(i)->c_str());
			}
		}
		
		// work out coordinates, again Mathematica has a different required format
		for (unsigned long i=0; i<nIndependentVariables; i++) {
			if (format == FORMAT_MATHEMATICA) {
				fprintf(outfile, "%s = Flatten[Take[%s%li,{1,%li",variableName(i)->c_str(),datFileNameClear.c_str(), iD,lattice(i));
				if (i+1<nIndependentVariables) {
					for (unsigned long j=i+1; j<nIndependentVariables; j++) {
						fprintf(outfile, " %li",lattice(j));					
					}
					fprintf(outfile, ",");					
					for (unsigned long j=i+1; j<nIndependentVariables; j++) {
						fprintf(outfile, " %li",lattice(j));					
					}
				}
				fprintf(outfile, "}, {%li,%li}]];\n",i+1,i+1);
			}
			else {
				if (format == FORMAT_MATHEMATICA5) {
					fprintf(outfile, "Do[%s[[i]] = tempd%li[[", variableName(i)->c_str(), i+1);
				}
				else if (format == FORMAT_MATLAB ||
						 format == FORMAT_SCILAB) {
					fprintf(outfile, "%s(:) = temp_d%li(", variableName(i)->c_str(), i+1);
				}
				else if (format == FORMAT_R) {
					fprintf(outfile, "%s <- %s%li$%s\n",
							variableName(i)->c_str(), datFileNameBase, iD, variableName(i)->c_str());
				}
				else {
					throw xmdsException("Unknown output data format!\n");
				}
				
				if (i == (nIndependentVariables-1)) {
					if (format == FORMAT_MATHEMATICA5) {
						fprintf(outfile, "i");
					}
					else if (format == FORMAT_MATLAB ||
							 format == FORMAT_SCILAB) {
						fprintf(outfile, ":");
					}
				}
				else {
					if (format != FORMAT_R) {
						fprintf(outfile, "1");
					}
				}
				
				for (unsigned long j=nIndependentVariables-1; j>0; j--) {
					if ((j-1) == i) {
						if (format == FORMAT_MATHEMATICA5) {
							fprintf(outfile, ", i");
						}
						else if (format == FORMAT_MATLAB ||
								 format == FORMAT_SCILAB) {
							fprintf(outfile, ", :");
						}
					}
					else {
						if (format != FORMAT_R) {
							fprintf(outfile, ", 1");
						}
					}
				}
				
				if (format == FORMAT_MATHEMATICA5) {
					fprintf(outfile, "]], {i, 1, %li}]\n", lattice(i));
				}
				else if (format == FORMAT_MATLAB ||
						 format == FORMAT_SCILAB) {
					fprintf(outfile, ");\n");
				}
			}
		}
		
		
		fprintf(outfile, "\n");
		
		// clear excess variables
		
		if (format == FORMAT_MATHEMATICA5) {
			fprintf(outfile, "Clear[ %s%li, %s%licolumn",
					datFileNameClear.c_str(), iD, datFileNameClear.c_str(), iD);
			for (unsigned long i=0; i<nIndependentVariables; i++) {
				fprintf(outfile, ", tempd%li", i+1);
			}
			fprintf(outfile, "]");
		}
		else if (format == FORMAT_MATHEMATICA) {
			fprintf(outfile, "Clear[ %s%li]\n",
					datFileNameClear.c_str(), iD);
		}
		else if (format == FORMAT_MATLAB ||
				 format == FORMAT_SCILAB) {
			fprintf(outfile, "clear %s%li", datFileNameBase, iD);
			for (unsigned long i=0; i<nIndependentVariables; i++) {
				fprintf(outfile, " temp_d%li", i+1);
			}
		}
		else if (format == FORMAT_R) {
			// I don't think there's any output needed here for R, but this is a
			// placeholder just in case
		}
		else {
			throw xmdsException("Unknown output format!\n");
		}
		
		fprintf(outfile, "\n");
		
		// There's no straightforward way to output the defined variables in
		// Mathematica, so it's a good idea to output the variables to screen.
		if (format == FORMAT_MATHEMATICA5) {
			// The Print[] command will print an expression, including a string to
			// screen.
			fprintf(outfile, "Print[\"%s ", variableName(0)->c_str());
			
			for (unsigned long i=1; i<variableNamesList.size(); i++) {
				fprintf(outfile, ", %s", variableName(i)->c_str());
			}
			fprintf(outfile, "\"]\n");
		}
		
		fprintf(outfile, "\n");
		
	}
	else if (streamFormat == "Binary") {
		
		if (format == FORMAT_MATLAB) {
			
			// work out how matlab will interpret endian-ness
			std::string machineFormat;
			if (binEncoding == "BigEndian") {
				machineFormat = "ieee-be";
			}
			else if (binEncoding == "LittleEndian") {
				machineFormat = "ieee-le";
			}
			else {
				machineFormat = "native";
			}
			
			fprintf(outfile, "fpDat = fopen('%s', 'r', '%s');\n",
					binDatFname.c_str(), machineFormat.c_str());
			fprintf(outfile, "if (fpDat < 0)\n");
			fprintf(outfile, "  disp('Cannot open binary data file: %s')\n",
					binDatFname.c_str());
			fprintf(outfile, "  return\n");
			fprintf(outfile, "end\n");
			
			unsigned long int k = 0;
			for (list<XMLString>::iterator
				 pXMLString = variableNamesList.begin();
				 pXMLString != variableNamesList.end();
				 pXMLString++) {
				char tempString[64];
				sprintf(tempString, "_%li", iD);
				*pXMLString += tempString;
				
				if (k < nIndependentVariables) {
					fprintf(outfile, "%sLen = fread(fpDat, 1, '%s');\n",
							pXMLString->c_str(), ulongType.c_str());
					fprintf(outfile, "%s = zeros(1, %sLen);\n",
							pXMLString->c_str(), pXMLString->c_str());
					fprintf(outfile, "%s(:) = fread(fpDat, %sLen, '%s');\n",
							pXMLString->c_str(), pXMLString->c_str(), binPrecision.c_str());
				}
				else if (k >= nIndependentVariables) {
					fprintf(outfile, "%sLen = fread(fpDat, 1, '%s');\n",
							pXMLString->c_str(), ulongType.c_str());
					if (nIndependentVariables <= 1) {
						fprintf(outfile, "%s = fread(fpDat, %sLen, '%s');\n",
								pXMLString->c_str(), variableNamesList.begin()->c_str(),
								binPrecision.c_str());
					}
					else if (nIndependentVariables == 2) {
						fprintf(outfile, "%s = fread(fpDat, [%sLen, %sLen], '%s');\n",
								pXMLString->c_str(), (++variableNamesList.begin())->c_str(),
								variableNamesList.begin()->c_str(), binPrecision.c_str());
					}
					else if (nIndependentVariables > 2) {
						// now we need to create a multi-dimensional matrix,
						// and this is harder to do...
						// we need to read in a matrix-sized (ie 2D) block at a time,
						// and append this to the other dimensions the number of
						// independent variables determines the dimensions of the
						// N-D matrix to produce
						
						// construct the for loop to loop over the third and subsequent dimensions
						list<XMLString>::iterator pIndepVars = variableNamesList.begin();
						for (unsigned long int inumIndepVars=2; inumIndepVars<nIndependentVariables; inumIndepVars++) {
							fprintf(outfile, "for index%li = 1:%sLen\n",
									inumIndepVars-2, pIndepVars->c_str());
							pIndepVars++;
						}
						
						// generate the first part of the string, which is the array to be assigned into
						fprintf(outfile, "%s(:, :, ", pXMLString->c_str());
						pIndepVars = variableNamesList.begin();
						for (unsigned long int inumIndepVars=nIndependentVariables-1; inumIndepVars>=2; inumIndepVars--) {
							fprintf(outfile, "index%li", inumIndepVars-2);
							// need to append a comma if not the last index to append
							if (inumIndepVars != 2) {
								fprintf(outfile, ", ");
							}
						}
						
						// generate the fread statement
						// to do this, I have to work out what the last and second-to-last
						// independent variable names are this is because, for some reason,
						// one can't inspect a given element of a list
						
						// first, the last one
						pIndepVars = variableNamesList.begin();
						XMLString lastIndepVar;
						for (unsigned long int inumIndepVars=0; inumIndepVars<nIndependentVariables; inumIndepVars++) {
							lastIndepVar = *pIndepVars;
							pIndepVars++;
						}
						
						// now the second to last one
						XMLString secondLastIndepVar;
						pIndepVars = variableNamesList.begin();
						for (unsigned long int inumIndepVars=1; inumIndepVars<nIndependentVariables; inumIndepVars++) {
							secondLastIndepVar = *pIndepVars;
							pIndepVars++;
						}
						
						fprintf(outfile, ") = fread(fpDat, [%sLen, %sLen], '%s');\n",
								lastIndepVar.c_str(), secondLastIndepVar.c_str(),
								binPrecision.c_str());
						
						// finish off the for loop
						for (unsigned long int inumIndepVars=2; inumIndepVars<nIndependentVariables; inumIndepVars++) {
							fprintf(outfile, "end\n");
						}
						
					}
					
				}
				k++;
				
			}
			
			// clean up a bit
			fprintf(outfile, "fclose(fpDat);\n");
			fprintf(outfile, "clear fpDat ");
			
			for (list<XMLString>::iterator
				 pXMLString = variableNamesList.begin();
				 pXMLString != variableNamesList.end();
				 pXMLString++) {
				fprintf(outfile, "%sLen ", pXMLString->c_str());
			}
			for (unsigned long int inumIndepVars=2; inumIndepVars<nIndependentVariables; inumIndepVars++) {
				fprintf(outfile, "index%li ", inumIndepVars-2);
			}
			
			fprintf(outfile, "\n");
			
		}
		else if (format == FORMAT_SCILAB) {
			
			// as far as I can tell, scilab can't handle binary data input,
			// so barf, and tell why, and give some alternative.
			
			printf("\nFatal error: Sorry, but at the time of this version of xmds,\n"
				   "scilab cannot handle binary input.  To be able to use scilab,\n"
				   "please change your output format to ascii (in the <output> tag).\n"
				   "Exiting...\n");
			exit(254);
		}
		else if (format == FORMAT_MATHEMATICA5) {
			
			// I have no idea if binary data works with Mathematica,
			// so for the moment, barf, and tell why, and give some alternative.
			
			printf("\nFatal error: Sorry, but at the moment reading binary data \n"
				   "into older versions of Mathematica is not implemented.  To be able to use Mathematica,\n"
				   "please change your output format to ascii (in the <output> tag), or use version 6.0 or later.\n"
				   "Exiting...\n");
			exit(254);
		}
		else if (format == FORMAT_MATHEMATICA) {
			
			// work out how Mathematica will interpret endian-ness
			std::string machineFormat;
			if (binEncoding == "BigEndian") {
				machineFormat = "1";
			}
			else if (binEncoding == "LittleEndian") {
				machineFormat = "-1";
			}
			else {
				machineFormat = "$ByteOrdering";
			}

			std::string uLongFormat;
			if (ulongType.c_str() == "uint32") {
				uLongFormat = "UnsignedInteger32";
			}
			else if (ulongType.c_str() == "uint64") {
				uLongFormat = "UnsignedInteger64";
			}
			else {
			  // Have to guess
				uLongFormat = "UnsignedInteger32";
			}

			std::string outputPrecision;
			if (binPrecision.c_str() == "single") {
				outputPrecision = "Real32";
			}
			else if (binPrecision.c_str() == "double") {
				outputPrecision = "Real64";
			}
			else {
			  // Have to guess
				outputPrecision = "Real64";
			}
			
			fprintf(outfile, "fpDat = OpenRead[\"%s\",BinaryFormat -> True];\n",
					binDatFname.c_str());
			
			unsigned long int k = 0;
			for (list<XMLString>::iterator
				 pXMLString = variableNamesList.begin();
				 pXMLString != variableNamesList.end();
				 pXMLString++) {
				
				pXMLString->goLatinAlphaNumericNoUnderScore();
				
				char tempString[64];
				sprintf(tempString, "%li", iD);
				*pXMLString += tempString;
				
				if (k < nIndependentVariables) {
					fprintf(outfile, "%sLen = BinaryRead[fpDat, \"%s\", ByteOrdering->%s];\n",
							pXMLString->c_str(), uLongFormat.c_str(), machineFormat.c_str());
					fprintf(outfile, "%s = BinaryReadList[fpDat, {\"%s\"}, %sLen, ByteOrdering->%s];\n",
							pXMLString->c_str(), outputPrecision.c_str(), pXMLString->c_str(), machineFormat.c_str());
				}
				else if (k >= nIndependentVariables) {
					fprintf(outfile, "%sLen = BinaryRead[fpDat, \"%s\", ByteOrdering->%s];\n",
							pXMLString->c_str(), uLongFormat.c_str(), machineFormat.c_str());
					
					if (nIndependentVariables == 1) {
						fprintf(outfile, "%s = BinaryReadList[fpDat, {\"%s\"}, %sLen, ByteOrdering->%s];\n",
								pXMLString->c_str(), outputPrecision.c_str(), pXMLString->c_str(), machineFormat.c_str());
					}
					else {
						list<XMLString>::iterator getLastNameXMLString = variableNamesList.begin();
						for (unsigned long int indepVars = 0; indepVars<nIndependentVariables-1; indepVars++) {
							getLastNameXMLString++;
						}
						fprintf(outfile, "%s = Flatten[Table[BinaryReadList[fpDat, {\"%s\"}, %sLen, ByteOrdering->%s]",
								pXMLString->c_str(), outputPrecision.c_str(), getLastNameXMLString->c_str(), machineFormat.c_str());
						list<XMLString>::iterator tempXMLString = variableNamesList.begin();
						for (unsigned long int indepVars = 0; indepVars<nIndependentVariables-1; indepVars++) {
							fprintf(outfile, ",{j%li,1,%sLen}",indepVars+1,tempXMLString->c_str());
							tempXMLString++;
						}
						fprintf(outfile, "],{");
						for (unsigned long int indepVars = 0; indepVars<nIndependentVariables-1; indepVars++) {
							fprintf(outfile, "{%li},",indepVars+1);
						}
						fprintf(outfile, "{%li,%li}}];\n",nIndependentVariables,nIndependentVariables+1);
					}
				}
				
				
				k++;
				
			}
			
			// clean up a bit
			fprintf(outfile, "Close[fpDat];\n");
			
/*			for (list<XMLString>::iterator
				 pXMLString = variableNamesList.begin();
				 pXMLString != variableNamesList.end();
				 pXMLString++) {
				fprintf(outfile, "%sLen\n ", pXMLString->c_str());
			}
			for (unsigned long int inumIndepVars=2; inumIndepVars<nIndependentVariables; inumIndepVars++) {
				fprintf(outfile, "index%li\n ", inumIndepVars-2);
			}*/
			
			fprintf(outfile, "\n");
			
		}
		else if (format == FORMAT_GNUPLOT) {
			//! \todo why doesn't throwing an xmdsException produce any output here?
			printf("Error:\nGnuplot is unable to load binary xmds data files\n");
			exit(253);
		}
		else if (format == FORMAT_R) {
			//! \todo why doesn't throwing an xmdsException produce any output here?
			printf("Error:\nR is unable to load binary xmds data files\n");
			exit(253);
		}
		else {
			throw(xmdsException("Unknown format.  I only accept Mathematica, Matlab, Octave, Gnuplot or Scilab at present\n"));
		}
		
	}
	else {
		throw(xmdsException("Stream format is neither Text or Binary, something has seriously gone wrong!\n"));
	}
	
}

// **************************************************************************
// **************************************************************************
//                              xsilField private
// **************************************************************************
// **************************************************************************

// **************************************************************************
unsigned long xsilField::lattice(
                                 const unsigned long& index) const {
	if (DEBUG) {
		printf("xsilField::lattice\n");
	}
	
	if (index >= latticeList.size()) {
		throw xmdsException("Internal range error in xsilField::lattice");
	}
	
	list<unsigned long>::const_iterator pULong = latticeList.begin();
	for (unsigned long i=0; i<index; i++) {
		pULong++;
	}
	
	return *pULong;
								 }

// **************************************************************************
const XMLString* xsilField::variableName(
                                         const unsigned long& index) const {
	if (DEBUG) {
		printf("xsilField::varaibleName\n");
	}
	
	if (index >= variableNamesList.size()) {
		throw xmdsException("Internal range error in xsilField::variableName");
	}
	
	list<XMLString>::const_iterator pXMLString = variableNamesList.begin();
	for (unsigned long i=0; i<index; i++) {
		pXMLString++;
	}
	
	return &*pXMLString;
										 }

#include <kissdom.h>
#include <xml_parser.h>
#include <config.h>

bool initialiseFieldFromXSILFile(
								 const char *filename, const char *mgName, unsigned long dimension,
								 char **dimNames, char **componentNames,
								 // output variables
								 char**binaryDataFilename, int *unsignedLongSize,
								 bool *dataEncodingIsNative, bool *isPrecisionDouble,
								 unsigned long *nDataComponents, unsigned long **inputLattice,
								 int **componentIndicesPtr)
{
	XMLParser myXMLParser;
	Document *theDocument = 0;
	try {
		theDocument = myXMLParser.parseFromFile(filename);
	}
	catch(XMLParserException XMLRoutinesErr) {
		printf("Could not load XSIL file '%s'\n", filename);
		printf("Due to the following XMLParser exception:\n");
		printf("%s\n", XMLRoutinesErr.getError());
		return false;
	}
	
	// ************************************
	// find and process xsil children
	// ************************************
	
	const NodeList* candidateElements;
	list<const Node*> xsilNodeList;
	
	if (*theDocument->documentElement()->nodeName() == "simulation") {
		
		candidateElements = theDocument->documentElement()->getElementsByTagName("XSIL", 0);
		
		if (candidateElements->length() < 1) {
			printf("Error: no <xsil> elements from within <simulation> element.\n");
			return false;
		}
		
		for (unsigned long i=0; i<candidateElements->length(); i++) {
			xsilNodeList.push_back(candidateElements->item(i));
		}
	}
	else if (*theDocument->documentElement()->nodeName() == "XSIL") {
		xsilNodeList.push_back(theDocument->documentElement());
	}
	else {
		printf("Error: expecting root element to be <XSIL> or <simulation>.\n");
		return false;
	}
	
	
	for (list<const Node*>::const_iterator
		 ppNode = xsilNodeList.begin();
		 ppNode != xsilNodeList.end();
		 ppNode++)
    {
		xsilField myxsilField;
		const Element* nextElement = dynamic_cast<const Element*>(*ppNode);
		try {
			myxsilField.processElement(nextElement);
		} catch(xmdsException xmdsErr) {
			printf("Could not load XSIL data container\n");
			printf("due to the following XSIL exception:\n");
			printf("%s", xmdsErr.getError());
			return false;
		}
		
		if (strcmp("NULL", mgName) == 0) {
			// We only want to use the first XSIL element if there is only one XSIL element
			if (xsilNodeList.size() != 1) {
				printf("This XSIL file contains more than one moment group, and no moment group number\n"
					   "was specified in the <filename format=\"xsil\"> tag. Try adding a moment_group=\"X\"\n"
					   "attribute (e.g. <filename format=\"xsil\" moment_group=\"X\">)\n\n");
				return false;
			}
		}
		else if (myxsilField.fieldName != mgName) {
			// If the moment group name isn't the same as the current one, keep looking
			continue;
		}
		// Check dimension
		if (myxsilField.nIndependentVariables != dimension) {
			printf("The dimension of the XSIL moment group does not match the dimension of the\nvector being initialised\n");
			printf("Are you sure that you specified the correct moment group number?\n\n");
			return false;
		}
		// Check dimension names
		for (long unsigned int i=0; i<dimension; i++) {
			if (*myxsilField.variableName(i) != dimNames[i]) {
				printf("The names of the dimensions in the XSIL file do not match up with the names specified in the XMDS script.\n");
				printf("Specifically, '%s' != '%s'\n",
					   myxsilField.variableName(i)->c_str(), dimNames[i]);
				printf("Are you sure that you set the initial space (whether each dimension is in fourier space or not) correctly in your script?\n\n");
				return false;
			}
		}
		unsigned long numVariables = myxsilField.lattice(dimension);
		
		// !!!!! this for loop doesn't do anything!!!!!
		// it looks like it determines the number of components, if so, then
		// there should be a comment here to that effect.
		unsigned long numComponents;
		for (numComponents=0; componentNames[numComponents]!=NULL; numComponents++)
			;
		
		// Check that all the variable names are there
		int *componentIndices = new int[numComponents];
		*componentIndicesPtr = componentIndices;
		
		for (long unsigned int i=0; i<numComponents; i++) {
			componentIndices[i] = -1;
			for (long unsigned int j=dimension; j<numVariables; j++) {
				if (*myxsilField.variableName(j) == componentNames[i]) {
					componentIndices[i] = j;
					break;
				}
			}
		}
		bool foundAnyVariables = false;
		for (long unsigned int i=0; i<numComponents; i++) {
			if (componentIndices[i] == -1) {
				printf("Warning: Unable to find variable name '%s' in XSIL file\n",
					   componentNames[i]);
			}
			else {
				foundAnyVariables = true;
			}
		}
		if (foundAnyVariables == false) {
			printf("Error: We didn't find any of the variables for this vector.\n"
				   "Are you sure that you have selected the correct moment group?\n"
				   "This moment group contains the variables:\n");
			for (long unsigned int j=dimension; j<numVariables; j++) {
				printf("%s", myxsilField.variableName(j)->c_str());
				if (j != numVariables) {
					printf(", ");
				}
			}
			printf("\n");
			
			return false;
		}
		
		// Check that the data format is binary
		if (myxsilField.streamFormat != "Binary") {
			printf("The XSIL input interpreter does not support XSIL files with "
				   "the data in ASCII. Use binary instead\n");
			return false;
		}
		
		// Check that the binary encoding format is native
		*dataEncodingIsNative = true;
		if (CPU_IS_LITTLE_ENDIAN) {
			if (myxsilField.binEncoding != "LittleEndian") {
				*dataEncodingIsNative = false;
			}
		}
		else if (CPU_IS_BIG_ENDIAN) {
			if (myxsilField.binEncoding != "BigEndian") {
				*dataEncodingIsNative = false;
			}
		}
		else {
			printf("Warning: I don't know what my endian-ness is... "
				   "I'm going to assume that the data from the XSIL file is in "
				   "my native endian encoding\n");
		}
		
		*isPrecisionDouble = myxsilField.binPrecision == "double" ? true : false;
		
		*nDataComponents = numVariables;
		unsigned long *lattice = new unsigned long[dimension];
		for (long unsigned int i=0; i<dimension; i++)
			lattice[i] = myxsilField.lattice(i);
		
		*inputLattice = lattice;
		
		*binaryDataFilename = strdup(myxsilField.binDatFname.c_str());
		
		if (myxsilField.ulongType == "ulong") {
			*unsignedLongSize = 0;
		}
		else if (myxsilField.ulongType == "uint32") {
			*unsignedLongSize = 4;
		}
		else if (myxsilField.ulongType == "uint64") {
			*unsignedLongSize = 8;
		}
		else {
			printf("Unsigned long type '%s' is not understood\n",
				   myxsilField.ulongType.c_str());
			return false;
		}
		
		return true;
    }
	
	printf("Unable to find XSIL element with name: '%s'\n", mgName);
	return false;
	
}


/*
 * Local variables:
 * c-indentation-style: bsd
 * c-basic-offset: 2
 * indent-tabs-mode: nil
 * End:
 *
 * vim: tabstop=2 expandtab shiftwidth=2:
 */
