/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <libxfce4util/libxfce4util.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include "jumptofilewin.h"
#include <xfmedia/xfmedia-playlist.h>
#include <xfmedia/xfmedia-settings.h>

typedef struct
{
    JTFActivateFunc activate_func;
    gpointer callback_data;
    gint column;
    
    GtkWidget *jtf_box;
    GtkWidget *entry;
    GtkWidget *treeview;
    GtkTreeSelection *treesel;
    GtkTreeModelFilter *filter;
    guint idle_id;
    
    gchar **words;
} JtfData;

static void
jtf_widget_about_to_destroy(gpointer data, GObject *where_the_object_was)
{
    JtfData *jtfdata = data;
    
    if(jtfdata->idle_id)
        g_source_remove(jtfdata->idle_id);
    if(jtfdata->words)
        g_strfreev(jtfdata->words);
    g_free(jtfdata);
}

static void
jtf_run_callback(JtfData *jtfdata, GtkTreePath *path)
{
    if(jtfdata->activate_func)
        jtfdata->activate_func(path, jtfdata->callback_data);
}

static void
jtf_get_words(JtfData *jtfdata)
{
    gchar *text, *tmp, **words, *str_norm;
    gint i;
    
    if(jtfdata->words)
        g_strfreev(jtfdata->words);
    
    text = gtk_editable_get_chars(GTK_EDITABLE(jtfdata->entry), 0, -1);
    tmp = g_utf8_strdown(text, -1);
    g_free(text);
    words = g_strsplit(tmp, " ", -1);
    g_free(tmp);
    
    for(i = 0; words[i]; i++) {
        str_norm = g_utf8_normalize(words[i], -1, G_NORMALIZE_ALL);
        g_free(words[i]);
        if(!str_norm)
            str_norm = g_strdup("");
        words[i] = str_norm;
    }
    
    jtfdata->words = words;
}

static gboolean
jtf_refilter_idled(gpointer data)
{
    JtfData *jtfdata = data;
    GtkTreePath *path;
    
    gdk_threads_enter();
    
    jtfdata->idle_id = 0;
    
    gtk_tree_model_filter_refilter(GTK_TREE_MODEL_FILTER(jtfdata->filter));
    
    gtk_tree_selection_unselect_all(jtfdata->treesel);
    path = gtk_tree_path_new_first();
    gtk_tree_selection_select_path(jtfdata->treesel, path);
    gtk_tree_path_free(path);
    
    gtk_tree_view_scroll_to_point(GTK_TREE_VIEW(jtfdata->treeview), 0, 0);
    
    gdk_threads_leave();
    
    return FALSE;
}

static gboolean
jtf_entry_keyrelease_cb(GtkWidget *w, GdkEventKey *evt, gpointer user_data)
{
    JtfData *jtfdata = user_data;
    GList *selected;
    
    if(evt->keyval == GDK_Escape) {
        gtk_widget_destroy(jtfdata->jtf_box);
        return TRUE;
    } else if(evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter) {
        selected = gtk_tree_selection_get_selected_rows(jtfdata->treesel, NULL);
        if(selected) {
            GtkTreePath *path = selected->data;
            GtkTreeIter itr, child_itr;
            GtkTreeModel *child_model;
            
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(jtfdata->filter), &itr, path)) {
                gtk_tree_model_filter_convert_iter_to_child_iter(jtfdata->filter,
                        &child_itr, &itr);
                child_model = gtk_tree_model_filter_get_model(jtfdata->filter);
                path = gtk_tree_model_get_path(child_model, &child_itr);
                jtf_run_callback(jtfdata, path);
                gtk_tree_path_free(path);
                gtk_widget_destroy(jtfdata->jtf_box);
            }
            
            return TRUE;
        }
    } else if(evt->keyval == GDK_Page_Up || evt->keyval == GDK_Page_Down
            || evt->keyval == GDK_KP_Page_Up || evt->keyval == GDK_KP_Page_Down
            || evt->keyval == GDK_Down || evt->keyval == GDK_KP_Down)
    {
        return TRUE;
    } else {
        if(jtfdata->idle_id)
            g_source_remove(jtfdata->idle_id);
        jtf_get_words(jtfdata);
        jtfdata->idle_id = g_timeout_add(500, jtf_refilter_idled, jtfdata);
    }
    
    return FALSE;
}

gboolean
xfmedia_jump_to_file_filter_visible_func(GtkTreeModel *model, GtkTreeIter *itr,
        gpointer data)
{
    JtfData *jtfdata = data;
    gchar *title = NULL, *tmp, *title_norm, **words;
    gint i;
    gboolean showme = TRUE;
    
    gtk_tree_model_get(model, itr, jtfdata->column, &title, -1);
    if(!title)
        return FALSE;
    tmp = g_utf8_strdown(title, -1);
    g_free(title);
    if(!tmp)
        return FALSE;
    title_norm = g_utf8_normalize(tmp, -1, G_NORMALIZE_ALL);
    g_free(tmp);
    if(!title_norm)
        return FALSE;
    
    if(!jtfdata->words)
        jtf_get_words(jtfdata);
    words = jtfdata->words;
    
    for(i = 0; words[i]; i++) {
        if(strstr(title_norm, words[i]))
            showme = TRUE;
        else {
            showme = FALSE;
            break;
        }
    }
    
    g_free(title_norm);
    
    return showme;
}

GtkWidget *
xfmedia_jump_to_file_get_widget(GtkTreeView *treeview, GtkTreeModel *model,
        gint column, JTFActivateFunc activate_func, gpointer callback_data)
{
    JtfData *jtfdata = g_new0(JtfData, 1);
    GtkWidget *hbox, *entry, *btn;
    
    jtfdata->treeview = GTK_WIDGET(treeview);
    jtfdata->treesel = gtk_tree_view_get_selection(treeview);
    jtfdata->filter = GTK_TREE_MODEL_FILTER(model);
    jtfdata->column = column;
    jtfdata->activate_func = activate_func;
    jtfdata->callback_data = callback_data;
    
    jtfdata->jtf_box = hbox = gtk_hbox_new(FALSE, BORDER/2);
    g_object_weak_ref(G_OBJECT(hbox), jtf_widget_about_to_destroy, jtfdata);
    
    jtfdata->entry = entry = gtk_entry_new();
    gtk_widget_add_events(entry, GDK_KEY_PRESS|GDK_KEY_RELEASE);
    gtk_widget_show(entry);
    gtk_box_pack_start(GTK_BOX(hbox), entry, TRUE, TRUE, 0);
    g_signal_connect(G_OBJECT(entry), "key-release-event",
            G_CALLBACK(jtf_entry_keyrelease_cb), jtfdata);
    
    btn = xfmedia_custom_button_new(NULL, GTK_STOCK_CLOSE);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect_swapped(G_OBJECT(btn), "clicked",
            G_CALLBACK(gtk_widget_destroy), hbox);
    
    g_object_set_data(G_OBJECT(hbox), "xfmedia-jtfdata", jtfdata);
    
    return hbox;
}
