/*
 *  This file is part of X-File Manager XFM
 *  ----------------------------------------------------------------------
 FmMain.c

 (c) S.Marlow 1990-92
 (c) A.Graef 1994
 (c) R.Vogelgesang 1994 (`Xfm.BourneShells' stuff)

  modified 7-1997 by strauman@sun6hft.ee.tu-berlin.de to add
  different enhancements (see README-1.4).

  modified 2003,2004,2005,2006,2007 by Bernhard R. Link (see Changelog)

 main module for file manager    
 *  ----------------------------------------------------------------------
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.

 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.

 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */
#include <xfmconfig.h>

#include "FmVersion.h"

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <fcntl.h>
#include <ctype.h>

#ifdef ENHANCE_USERINFO
#include <sys/stat.h>
#include <unistd.h>
#endif

#ifdef ENHANCE_TRANSLATIONS
#include <string.h>
#endif

#ifdef _AIX
#include <sys/resource.h>
#endif

#include <sys/wait.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw3d/Cardinals.h>
#include <X11/Shell.h>

#include "global.h"
#include "Am.h"
#include "Fm.h"
#include "mime.h"
#include "mime_start.h"

#ifdef ENHANCE_HISTORY
#include "FmHistory.h"
#endif

#ifdef ENHANCE_SELECTION
#include "FmSelection.h"
#endif

#ifdef ENHANCE_LOG
#include "FmLog.h"
#endif
#include "execute.h"

#define XtRDisplayType "DisplayType"
#define XtRSortType "SortType"
#define XtRExpandedString "ExpandedString"

#ifdef TESTRESOURCES
#define XFM_APP_CLASS "TestXfm"
#else
#define XFM_APP_CLASS "Xfm"
#endif

#define XFM_BITMAP_PATH PKGDATADIR "/bitmaps:/usr/include/X11/bitmaps"
#define XFM_PIXMAP_PATH PKGDATADIR "/pixmaps:/usr/include/X11/pixmaps:/usr/share/pixmaps"
#define XFM_ICON_PATH PKGDATADIR "/icons"

/*---------------------------------------------------------------------------
  Public variables
---------------------------------------------------------------------------*/

/* program name */
char *progname;

/* information about the user */
UserInfo user;

/* application resource values */
Resources resources;

/* application context */
XtAppContext app_context;

/* Update semaphor */
int freeze = False;

#ifdef ENHANCE_HISTORY
HistoryList path_history;
#endif


/*---------------------------------------------------------------------------
  Command line options
---------------------------------------------------------------------------*/

static XrmOptionDescRec options[] = {
  { "-appmgr", ".appmgr", XrmoptionNoArg, "True" },
  { "-filemgr", ".filemgr", XrmoptionNoArg, "True" },
  { "-version", ".version", XrmoptionNoArg, "True" }
};

/*---------------------------------------------------------------------------
  Application Resources
---------------------------------------------------------------------------*/

static XtResource resource_list[] = {
  { "appmgr", "Appmgr", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, appmgr), XtRImmediate, (XtPointer) False },
  { "filemgr", "Filemgr", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, filemgr), XtRImmediate, (XtPointer) False },
  { "version", "Version", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, version), XtRImmediate, (XtPointer) False },
  { "initGeometry", "InitGeometry", XtRString, sizeof(String),
      XtOffsetOf(Resources, init_geometry), XtRString, NULL },
  { "iconFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, icon_font), XtRString, XtDefaultFont },
  { "buttonFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, button_font), XtRString, XtDefaultFont },
  { "menuFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, menu_font), XtRString, XtDefaultFont },
  { "labelFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, label_font), XtRString, XtDefaultFont },
  { "statusFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, status_font), XtRString, XtDefaultFont },
  { "boldFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, bold_font), XtRString, XtDefaultFont },
  { "cellFont", XtCFont, XtRFontStruct, sizeof(XFontStruct *), 
      XtOffsetOf(Resources, cell_font), XtRString, XtDefaultFont },
  { "appIconWidth", "Width", XtRInt, sizeof(int),
      XtOffsetOf(Resources, app_icon_width), XtRImmediate, (XtPointer) 48 },
  { "appIconHeight", "Height", XtRInt, sizeof(int),
      XtOffsetOf(Resources, app_icon_height), XtRImmediate, (XtPointer) 40 },
  { "fileIconWidth", "Width", XtRInt, sizeof(int),
      XtOffsetOf(Resources, file_icon_width), XtRImmediate, (XtPointer) 48 },
  { "fileIconHeight", "Height", XtRInt, sizeof(int),
      XtOffsetOf(Resources, file_icon_height), XtRImmediate, (XtPointer) 40 },
  { "treeIconWidth", "Width", XtRInt, sizeof(int),
      XtOffsetOf(Resources, tree_icon_width), XtRImmediate, (XtPointer) 48 },
  { "treeIconHeight", "Height", XtRInt, sizeof(int),
      XtOffsetOf(Resources, tree_icon_height), XtRImmediate, (XtPointer) 32 },
  { "bitmapPath", "Path", XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, bitmap_path), XtRString, XFM_BITMAP_PATH },
  { "pixmapPath", "Path", XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, pixmap_path), XtRString, XFM_PIXMAP_PATH },
  { "iconPath", "Path", XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, icon_path), XtRString, XFM_ICON_PATH },
  { "symlinkDir", "ConfigDir",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, symlinkdir), XtRString, "/tmp/xfm-${USER}-${PID}" },
  { "alwaysSymlink", "Safety", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, alwaysSymlink), XtRImmediate, (XtPointer)False },
  { "applicationDataFile", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, main_app_file), XtRString, "~/.xfm/Apps.xfm" },
  { "applicationDataDir", "ConfigDir",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, app_dir), XtRString, "~/.xfm" },
  { "applicationDataClip", "File",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, app_clip), XtRString, "~/.xfm/.XfmClip" },
  { "devFile", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, dev_file), XtRString, "~/.xfm/xfm_dev" },
  { "mailcapFile", "ConfigFile", XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, mailcap_file), XtRString, "~/.xfm/xfm_mailcap" },
  { "mimeTypesFile", "ConfigFile", XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, mimetypes_file), XtRString, "~/.xfm/xfm_mime.types" },
#ifdef SYSTEMWIDE_DEFAULTS
  { "systemwideApplicationDataFile", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, system_main_app_file), XtRString, XFM_APPDIR "/Apps.xfm" },
  { "systemwideApplicationDataDir", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, system_app_dir), XtRString, XFM_APPDIR },
  { "systemwideDevFile", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, system_dev_file), XtRString, XFM_CONFDIR "/xfm_dev" },
  { "systemwideMailcapFile", "ConfigFile", XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, system_mailcap_file), XtRString, XFM_CONFDIR "/xfm_mailcap" },
  { "systemwideMagicFile", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, system_magic_file), XtRString, XFM_CONFDIR "/xfm_magic" },
  { "systemwideMimeTypesFile", "ConfigFile", XtRString, sizeof(String),
      XtOffsetOf(Resources, system_mimetypes_file), XtRString, XFM_CONFDIR "/xfm_mime.types" },
#endif
  { "magicFile", "ConfigFile",  XtRExpandedString, sizeof(String),
      XtOffsetOf(Resources, magic_file), XtRString, "~/.xfm/xfm_magic" },
  { "maxMagicRead", XtCValue, XtRCardinal, sizeof(Cardinal),
      XtOffsetOf(Resources, maxMagicRead), XtRImmediate, (XtPointer) 1024 },
  { "confirmDeletes", "Confirm", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, confirm_deletes), XtRImmediate, (XtPointer) True },
  { "confirmDeleteFolder", "Confirm", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, confirm_delete_folder), XtRImmediate,
      (XtPointer) True },
  { "confirmMoves", "Confirm", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, confirm_moves), XtRImmediate, (XtPointer) True },
  { "confirmCopies", "Confirm", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, confirm_copies), XtRImmediate, (XtPointer) True },
  { "confirmOverwrite", "Confirm", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, confirm_overwrite), XtRImmediate,
      (XtPointer) True },
  { "confirmQuit", "Confirm", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, confirm_quit), XtRImmediate, (XtPointer) True },
  { "echoActions", "Echo", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, echo_actions), XtRImmediate, (XtPointer) False },
  { "echoMimeSearch", "Echo", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, echo_mime_search), XtRImmediate, (XtPointer) False },
  { "showOwner", "ShowOwner", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_owner), XtRImmediate, (XtPointer) True },
  { "showDate", "ShowDate", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_date), XtRImmediate, (XtPointer) True },
  { "showPermissions", "ShowPermissions", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_perms), XtRImmediate, (XtPointer) True },
  { "showLength", "ShowLength", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_length), XtRImmediate, (XtPointer) True },
  { "showInode", "ShowInode", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_inode), XtRImmediate, (XtPointer) True },
  { "showType", "ShowType", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_type), XtRImmediate, (XtPointer) True },
  { "showGroup", "ShowGroup", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_group), XtRImmediate, (XtPointer) True },
  { "showLinks", "ShowLinks", XtRBoolean, sizeof(Boolean),
      XtOffsetOf(Resources, show_links), XtRImmediate, (XtPointer) True },
  { "defaultDisplayType", "DefaultDisplayType", XtRDisplayType, 
      sizeof(DisplayType), XtOffsetOf(Resources, default_display_type),
      XtRImmediate, (XtPointer) Icons },
  { "initialDisplayType", "InitialDisplayType", XtRDisplayType, 
      sizeof(DisplayType), XtOffsetOf(Resources, initial_display_type),
      XtRImmediate, (XtPointer) Icons },
  { "defaultSortType", "DefaultSortType", XtRSortType, 
      sizeof(SortType), XtOffsetOf(Resources, default_sort_type),
      XtRImmediate, (XtPointer) SortByName },
  { "doubleClickTime", "DoubleClickTime", XtRInt, sizeof(int),
      XtOffsetOf(Resources, double_click_time), XtRImmediate,
      (XtPointer) 300 },
  { "updateInterval", "UpdateInterval", XtRInt, sizeof(int),
      XtOffsetOf(Resources, update_interval), XtRImmediate,
      (XtPointer) 10000 },
#ifdef ENHANCE_PERMS
  { "hardUpdateTicks", "HardUpdateTicks", XtRInt, sizeof(int),
      XtOffsetOf(Resources, hard_update_ticks), XtRImmediate, (XtPointer)0 },
#endif
  { "editor", "Editor", XtRCommandArgArray, sizeof(String),
      XtOffsetOf(Resources, editor), XtRString, NULL },
  { "viewer", "Viewer", XtRCommandArgArray, sizeof(String),
      XtOffsetOf(Resources, viewer), XtRString, NULL },
  { "xTerminal", "XTerminal", XtRCommandArgArray, sizeof(String*),
      XtOffsetOf(Resources, xterminal), XtRString, "x-terminal-emulator -e" },
  { "xTerminalAlone", "XTerminalAlone", XtRCommandArgArray, sizeof(String*),
      XtOffsetOf(Resources, xtermalone), XtRString, "x-terminal-emulator" },
  { "shell", "Shell", XtRCommandArgArray, sizeof(String),
      XtOffsetOf(Resources, shell), XtRString, NULL },
#ifdef ENHANCE_HISTORY
  { "historyMaxN", "HistoryMaxN", XtRInt, sizeof(int),
      XtOffsetOf(Resources, history_max_n), XtRImmediate, (XtPointer)30 },
#endif
#ifdef ENHANCE_TRANSLATIONS
  { "appDefsVersion","AppDefsVersion",XtRString,sizeof(String),
      XtOffsetOf(Resources, app_defs_version), XtRString, (XtPointer)"NOT SET"},
#endif
#ifdef ENHANCE_CMAP
  { "colorCloseness","ColorCloseness",XtRInt,sizeof(int),
      XtOffsetOf(Resources, color_closeness), XtRImmediate, (XtPointer)40000},
#endif
#ifdef ENHANCE_SELECTION
  { "highlightColor","HighlightColor",XtRPixel,sizeof(Pixel),
      XtOffsetOf(Resources, highlight_pixel), XtRString, (XtPointer)XtDefaultForeground},
  { "selectionPathsSeparator","SelectionPathsSeparator",XtRString,sizeof(String),
      XtOffsetOf(Resources, selection_paths_separator), XtRString, (XtPointer)" "},
#endif
};

/*---------------------------------------------------------------------------
 Fallback resources
---------------------------------------------------------------------------*/

static String fallback_resources[] = {
#include "../defaults/fallback.res"
NULL,
};

/*---------------------------------------------------------------------------
  Widget argument lists
---------------------------------------------------------------------------*/

static Arg shell_args[] = {
  { XtNtitle, (XtArgVal) "Applications" }
};

static struct sigaction sigterm;

/*---------------------------------------------------------------------------
  Resource converter functions
---------------------------------------------------------------------------*/

static Boolean CvtStringToDisplayType(UNUSED(Display *display),
		UNUSED(XrmValue *args), UNUSED(Cardinal *n_args),
		XrmValue *fromVal, XrmValue *toVal,
		UNUSED(XtPointer *priv_data))
{
  static DisplayType d;

  if( strcmp(fromVal->addr, "Tree") == 0)
	  d = Tree;
  else if( strcmp(fromVal->addr, "Icons") == 0)
	  d = Icons;
  else if( strcmp(fromVal->addr, "Text") == 0)
	  d = Text;
  else {
	  XtStringConversionWarning(fromVal->addr, XtRDisplayType);
	  return False;
  }

  if( sizeof(DisplayType) > toVal->size ) {
	  toVal->addr = (caddr_t) &d;
  } else {
	  *(DisplayType*)toVal->addr = d;
  }
  toVal->size = sizeof(DisplayType);
  return True;
}

/*---------------------------------------------------------------------------*/
 
static Boolean CvtStringToSortType(UNUSED(Display *display),
		UNUSED(XrmValue *args), UNUSED(Cardinal *n_args),
		XrmValue *fromVal, XrmValue *toVal,
		UNUSED(XtPointer *priv_data))
{
  static SortType d;

  if (strcmp(fromVal->addr, "SortByName") == 0)
    d = SortByName;
  else if (strcmp(fromVal->addr, "SortBySize") == 0)
    d = SortBySize;
  else if (strcmp(fromVal->addr, "SortByDate") == 0)
    d = SortByMTime;
  else {
    XtStringConversionWarning(fromVal->addr, XtRSortType);
    return False;
  }
  
  if( sizeof(SortType) > toVal->size ) {
	  toVal->addr = (caddr_t) &d;
  } else {
	  *(SortType*)toVal->addr = d;
  }
  toVal->size = sizeof(SortType);
  return True;
}

/*---------------------------------------------------------------------------*/
 
static Boolean CvtStringToExpandedString(
		UNUSED(Display *dpy), 
		UNUSED(XrmValue *args), UNUSED(Cardinal *n_args),
		XrmValue *fromVal, XrmValue *toVal, XtPointer *data)
{
  char *n;

  if( fromVal->size > 0 && fromVal->addr[fromVal->size-1] == '\0' &&
		  toVal->size == sizeof(String)) {
	  n = home_and_env_expand(fromVal->addr);
	  if( n != NULL ) {
		  *data = NULL;
		  *(String*)toVal->addr = n;
		  return True;
	  }
  }
  *data = NULL;
  XtStringConversionWarning(fromVal->addr, XtRExpandedString);
  return False;
} 

static void CvtFreeExpandedString(
		UNUSED(XtAppContext app), UNUSED(XrmValue *to),
		XtPointer data,
		UNUSED(XrmValue *args), UNUSED(Cardinal *n_args)) {
	XtFree(data);
}

/*---------------------------------------------------------------------------
  Main function
---------------------------------------------------------------------------*/

static void noWarnings(UNUSED(String msg))
{
}

int main(int argc, char *argv[])
{
  int r;
  char *s;
  const char *xfmversion = XFMVERSION;

  progname = argv[0];

  /* get some information about the user */
  user.uid = getuid();
  user.gid = getgid();

  if ((s = getenv("HOME")))
    strcpy(user.home,s);
  else
    getwd(user.home);
  getwd(user.cwd);

#ifndef ENHANCE_USERINFO /* read app-resources first */
  if ((s = getenv("SHELL")))
    strcpy(user.shell,s);
  else
    strcpy(user.shell,"/bin/sh");
#endif

  user.umask = umask(0);
  umask(user.umask);
  user.umask = 0777777 ^ user.umask;

  /* initialise the application and create the application shell */

  aw.shell = XtAppInitialize(&app_context, XFM_APP_CLASS, options, XtNumber(options),
			     &argc, argv, fallback_resources, shell_args,
			     XtNumber(shell_args) );

  /* First, initialize the atoms */
  /* initialise the communications module */
  initComms();

  /* make sure we can close-on-exec the display connection */
  if (fcntl(ConnectionNumber(XtDisplay(aw.shell)), F_SETFD, 1) == -1)
    abortXfm("Couldn't mark display connection as close-on-exec");

  /* register resource converters */
  XtSetTypeConverter(XtRString, XtRDisplayType, 
		    CvtStringToDisplayType, NULL, ZERO,
		    XtCacheAll, NULL);
  XtSetTypeConverter(XtRString, XtRSortType, 
		    CvtStringToSortType, NULL, ZERO,
		    XtCacheAll, NULL);
  XtSetTypeConverter(XtRString, XtRExpandedString, 
		    CvtStringToExpandedString, NULL, ZERO,
		    XtCacheAll|XtCacheRefCount,CvtFreeExpandedString);

  /* get the application resources */
  XtGetApplicationResources(aw.shell, &resources, resource_list,
			    XtNumber(resource_list), NULL, ZERO);

  /* -version: print version number and exit: */
  if (resources.version) {
    printf("xfm version %s.%s\n", xfmversion,XFMMINORVERSION);
    exit(0);
  }

  /* initialise the utilities module */
  initUtils();

  resources.symlinkdir = home_and_env_expand(resources.symlinkdir);
  if( !mime_start_init(resources.symlinkdir) )
	  abortXfm("error creating start directory");

#ifdef ENHANCE_USERINFO
  if (resources.shell == NULL || resources.shell[0] == NULL ||
		  resources.shell[0][0] == '\0') {
	  const char *shell = getenv("SHELL");

	  resources.shell = (const char **)XtMalloc(3*sizeof(char*));
	  if (shell != NULL)
	  	resources.shell[0] = (String)shell;
	  else
	  	resources.shell[0] = "/bin/sh";
	  resources.shell[1] = "-c";
	  resources.shell[2] = NULL;
  }
#endif

  r = mime_init();
  if( r <= -2 )
	  abortXfm("Out of Memory");
  readBitmaps();

#ifdef ENHANCE_LOG
  (void)FmCreateLog(aw.shell,resources.button_font);
#endif

  /* default is to launch both application and file manager */
  if (!resources.appmgr && !resources.filemgr)
    resources.appmgr = resources.filemgr = True;

  /* set the multi-click time */
  XtSetMultiClickTime(XtDisplay(aw.shell), resources.double_click_time);

  /* set up signal handlers */
  sigterm.sa_handler = quit;
  sigemptyset(&sigterm.sa_mask);
  sigterm.sa_flags = 0;
  sigaction(SIGTERM,&sigterm,NULL);
  if( !execute_init(app_context) ) {
	  exit(EXIT_FAILURE);
  }

  /* Set the icon for the application manager */
  XtVaSetValues(aw.shell, XtNiconPixmap, bm_appmgr, NULL);
  XtVaSetValues(aw.shell, XtNiconMask, bm_appmgrmsk, NULL);

  /* Create Path History menu, must be before any file window is created */
#ifdef ENHANCE_HISTORY
  char *fixed[5];
  int	nfixed;

  /* Path History menu is initialized
   * (need to do this first because the history feature
   * provides an additional action that must be added to the action table
   * so the different popups can use it :-)
   */

  nfixed=0;
  fixed[nfixed++]=user.home;
#ifdef ENHANCE_USERINFO
  fixed[nfixed++]=user.cwd;
#endif
  fixed[nfixed++]=0;
  path_history=FmCreateHistoryList(
	0, /* let the menu's name be "fm_history" */
	aw.shell,
	fixed);
#endif

  /* some additional Popup Windows to initialize (move them into their windows?) */
  /* Errors */
  createErrorPopup();

#ifdef ENHANCE_TRANSLATIONS
  if (strcmp(resources.app_defs_version, XFMVERSION)) {
    if (resources.app_defs_version[0] == '\0')
	error("Sorry: Xfm.appDefsVersion was not defined!\n","perhaps Appl. Defaults are broken or too old!\n");
    else
	error("Appl. Defaults: wrong version number (need " XFMVERSION "): ",resources.app_defs_version);
    exit(1);
  }
#endif

#ifdef SYSTEMWIDE_DEFAULTS
  /* check if config-dir is missing but should contain clipfile */
  if (!exists(resources.app_dir)) {
    if (strncmp(resources.app_dir,resources.app_clip,
                                  strlen(resources.app_dir))==0 
        && ( resources.app_clip[strlen(resources.app_dir)] =='/' ||
	    (strlen(resources.app_dir)>0 && 
	     resources.app_clip[strlen(resources.app_dir)-1] == '/'))) {
      mkdir(resources.app_dir,0777);
    }
  }
#else
  /* check for config dir and run xfm.install if appropriate */
  if (!exists(resources.app_dir)) {
    Boolean aborted;
    if (amConfirm("It appears that you are running xfm for the first time.",
		"Would you like me to run the xfm.install script now",
		"to create your personal setup?", &aborted))
      system("xfm.install -n");
    else if (aborted)
      exit(1);
  }
#endif

#ifdef SYSTEMWIDE_DEFAULTS
  /* if the main_app_file is not found, use the system_main_app_file
   * instead. (Only change app_file and not main_app_file, to
   * allow adding a user-file while running). */
  if (!exists(resources.main_app_file))
    aw.app_file = XtNewString(resources.system_main_app_file);
  else
#endif
    aw.app_file = XtNewString(resources.main_app_file);
  
  /* initialise the applications module & create the window */
  if (resources.appmgr) {
    readApplicationData();
    createApplicationWindow();
    createApplicationDisplay();
  } else {
	/* create a dummy window to store the WM_COMMAND property */
	XtVaSetValues(aw.shell,
		XtNmappedWhenManaged,False,
		(char*)0);
  }
  /* still realize and set the WM properties. the WM needs
   * a window around and kwm is even able to restart an app
   * without a mapped toplevel window.
   */
  XtRealizeWidget(aw.shell);
  setApplicationWindowName();
  setWMProps(aw.shell);
  XtAddEventHandler(aw.shell, (EventMask)0L, True,
	    (XtEventHandler)clientMessageHandler, (XtPointer)NULL);

#ifdef ENHANCE_SELECTION
  /* initialize the selection ICCM */
  FmSelectionInit(XtDisplay(aw.shell));
#endif

  /* initialise the file windows module & create a file window */
  initFileWindows();
  if (resources.filemgr)
#ifdef ENHANCE_USERINFO
    newFileWindow(user.cwd,resources.initial_display_type,False);
#else
    newFileWindow(user.home,resources.initial_display_type,False);
#endif
  resources.init_geometry = NULL;

  /* start up window refresh timer */
  if (resources.update_interval > 0)
    XtAppAddTimeOut(app_context, resources.update_interval, timeoutCb, NULL);

  /* those 'text cursor out of range' messages from the TextField widget kept
     annoying me, so I switch Xt warnings off here --AG */
  XtAppSetWarningHandler(app_context, noWarnings);

  /* collect & process events */
  XtAppMainLoop(app_context);
  mime_start_done(resources.symlinkdir);
  return 0;
}

/*---------------------------------------------------------------------------*/

void quit(UNUSED(int unused))
{
  unsigned int d;

  for (d = 0; d < n_devices; d++)
    if (devs[d].mounted) {
      devs[d].mounted = 1;
      umountDev(d);
    }

  mime_start_done(resources.symlinkdir);
  XtDestroyApplicationContext(app_context);
  exit(0);
}
