#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <inttypes.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/mman.h>

#include <xen/elfnote.h>
#include <xen/elfstructs.h>

#include "xenner.h"
#include "mm.h"

/* ------------------------------------------------------------------ */

static const char *corepath[] = {
    VMCORE_PATH,
};

int vmcore_init(struct xenvm *xen)
{
    int i;

    for (i = 0; i < sizeof(corepath)/sizeof(corepath[0]); i++) {
	snprintf(xen->corefile, sizeof(xen->corefile), "%s/vmcore.%d",
		 corepath[i], xen->domid);
	xen->corefd = open(xen->corefile, O_CREAT | O_EXCL | O_RDWR, 0600);
	if (xen->corefd >= 0)
	    break;
	if (EEXIST == errno)
	    goto fail;
    }
    if (i == sizeof(corepath)/sizeof(corepath[0]))
	goto fail;

    /* create and map sparse file */
    xen->coresize = VMCORE_HEADER + 4096 * xen->pg_total;
    lseek(xen->corefd, xen->coresize -1, SEEK_SET);
    write(xen->corefd, "\0", 1);
    xen->coremap = mmap(NULL, xen->coresize, PROT_READ | PROT_WRITE,
			MAP_SHARED | MAP_POPULATE, xen->corefd, 0);
    if ((void*)-1 == xen->coremap) {
	fprintf(stderr, "mmap: %s\n", strerror(errno));
	return -1;
    }
    xen->memory = xen->coremap + VMCORE_HEADER;
    return 0;

fail:
    fprintf(stderr, "open %s: %s\n", xen->corefile, strerror(errno));
    return -1;
}

void vmcore_fini(struct xenvm *xen)
{
    unlink(xen->corefile);
    munmap(xen->coremap, xen->coresize);
}

/* ------------------------------------------------------------------ */

#define VMCORE_PHOFF  (PAGE_SIZE * 1)
#define VMCORE_SHOFF  (PAGE_SIZE * 2)
#define VMCORE_STROFF (PAGE_SIZE * 3)

static Elf64_Phdr phdrs[] = {
    {
	/* physical memory */
	.p_type    = PT_LOAD,
	.p_flags   = PF_R | PF_W | PF_X,
	.p_offset  = VMCORE_HEADER,
	.p_paddr   = 0,
	.p_filesz  = 0, /* to be filled */
    }
};

static char *sh_name[] = {
    ".addr.shared_info",
    ".addr.grant_table",
    ".addr.vminfo",
    ELF_SHSTRTAB,
};

static Elf64_Shdr shdrs[] = {
    {
	/* shared_info */
	.sh_name      = 0,
	.sh_type      = SHT_PROGBITS,
        .sh_addr      = 0,
        .sh_offset    = 0,
    },{
	/* grant_table */
	.sh_name      = 0,
	.sh_type      = SHT_PROGBITS,
        .sh_addr      = 0,
        .sh_offset    = 0,
    },{
	/* vminfo */
	.sh_name      = 0,
	.sh_type      = SHT_PROGBITS,
        .sh_addr      = 0,
        .sh_offset    = 0,
    },{
	/* shstrtab */
	.sh_name      = 0,
	.sh_type      = SHT_STRTAB,
        .sh_offset    = VMCORE_STROFF,
    }
};

static Elf64_Ehdr ehdr = {
    .e_ident     = {
	[ EI_MAG0 ]       = ELFMAG0,
	[ EI_MAG1 ]       = ELFMAG1,
	[ EI_MAG2 ]       = ELFMAG2,
	[ EI_MAG3 ]       = ELFMAG3,
	[ EI_CLASS ]      = ELFCLASS64,
	[ EI_DATA ]       = ELFDATA2LSB,
 	[ EI_VERSION ]    = EV_CURRENT,
	[ EI_OSABI ]      = ELFOSABI_SYSV,
	[ EI_ABIVERSION ] = EV_CURRENT,
    },
    .e_type      = ET_CORE,
    .e_machine   = EM_X86_64,
    .e_version   = EV_CURRENT,
    .e_ehsize    = sizeof(Elf64_Ehdr),

    .e_phoff     = VMCORE_PHOFF,
    .e_phentsize = sizeof(phdrs[0]),
    .e_phnum     = sizeof(phdrs)/sizeof(phdrs[0]),

    .e_shoff     = VMCORE_SHOFF,
    .e_shentsize = sizeof(shdrs[0]),
    .e_shnum     = sizeof(shdrs)/sizeof(shdrs[0]),
};

#define elf_file_offset(_x, _p) ((uintptr_t)_p - (uintptr_t)_x->coremap)
#define elf_virt_addr(_x, _p)   ((uintptr_t)_p - (uintptr_t)_x->memory + _x->emu_vs)

int vmcore_write_headers(struct xenvm *xen)
{
    int s,stroff;
    void *ptr;

    /* phdrs */
    if (ehdr.e_phnum) {
	phdrs[0].p_filesz = xen->pg_total * PAGE_SIZE;

	lseek(xen->corefd, ehdr.e_phoff, SEEK_SET);
	write(xen->corefd, phdrs, sizeof(phdrs));
    }

    /* shdrs */
    if (ehdr.e_shnum) {
	lseek(xen->corefd, VMCORE_STROFF, SEEK_SET);
	stroff = 0; s = 0; 
	
	shdrs[s].sh_name   = stroff;
	stroff += write(xen->corefd, sh_name[s], strlen(sh_name[s])+1);
	ptr = mfn_to_ptr(xen, xen->e.config[EMUDEV_CONF_SHARED_INFO_PFN]);
	shdrs[s].sh_addr   = elf_virt_addr(xen, ptr);
	shdrs[s].sh_offset = elf_file_offset(xen, ptr);
	shdrs[s].sh_size   = PAGE_SIZE;
	s++;
	
	shdrs[s].sh_name   = stroff;
	stroff += write(xen->corefd, sh_name[s], strlen(sh_name[s])+1);
	ptr = mfn_to_ptr(xen, xen->e.gnttab[0]);
	shdrs[s].sh_addr   = elf_virt_addr(xen, ptr);
	shdrs[s].sh_offset = elf_file_offset(xen, ptr);
	shdrs[s].sh_size   = PAGE_SIZE * GRANT_FRAMES_MAX;
	s++;
	
	shdrs[s].sh_name   = stroff;
	stroff += write(xen->corefd, sh_name[s], strlen(sh_name[s])+1);
	ptr = mfn_to_ptr(xen, xen->e.config[EMUDEV_CONF_VMINFO_PFN]);
	shdrs[s].sh_addr   = elf_virt_addr(xen, ptr);
	shdrs[s].sh_offset = elf_file_offset(xen, ptr);
	shdrs[s].sh_size   = sizeof(struct xenner_info);
	s++;

	/* shstrtab */
	shdrs[s].sh_name   = stroff;
	stroff += write(xen->corefd, sh_name[s], strlen(sh_name[s])+1);
	shdrs[s].sh_size   = stroff;
	ehdr.e_shstrndx    = s,

	lseek(xen->corefd, ehdr.e_shoff, SEEK_SET);
	write(xen->corefd, shdrs, sizeof(shdrs));
    }

    /* ehdr */
    lseek(xen->corefd, 0, SEEK_SET);
    write(xen->corefd, &ehdr, sizeof(ehdr));

    return 0;
}
