 /*
 *  file      : split_shuttle.c
 *  project   : xcfa
 *  with      : Gtk-2
 *
 *  copyright : (C) 2003 - 2010 by Claude Bulin
 *
 *  xcfa - GTK+ implementation of the GNU shell command
 *  GNU General Public License
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  OLD ADRESS:
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *  NEW ADRESS:
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 *  ---
 *
 *  Widget pour la fonction SPLIT de XCFA qui permet la simplification
 *  de selection de debut et de fin de plage pour la decoupe ainsi que
 *  pour l'ecoute audio.
 *  J'ai developpe ce widget car je ne l'ai trouve nulle part !!!
 *
 *  Representation du spectre frequentiel (amplitude des différentes frequences)
 *  d'un fichier de type wav
 *
 *  Documentation:
 *
 *  	.)
 *	Adaptation d'un widget externe:
 *	http://franckh.developpez.com/tutoriels/gtk/creation-widget/
 *	Test d'integration:
 *	http://www.xcfa.tuxfamily.org/gtkbuilder.html
 *
 *  	.)
 *  	Les secrets des formats de compression audio
 *  	par Saad Bennani
 *  	GNU/Linux Magazine France no 101 page 50
 *
 *  	.)
 *  	Analyse code depuis: wavbreaker
 *  	par Timothy Robinson et Thomas Perl
 *  	http://wavbreaker.sourceforge.net/
 *
 *  	.)
 *	Analyse de Fourier à l'aide d'un script PERL.
 *  	http://perso.ensc-rennes.fr/jimmy.roussel/tice/perl/tuto_FFT.html
 *	par Jimmy ROUSSEL
 *
 */

#include "global.h"
#include "split_shuttle.h"
#include <math.h>
#include <string.h>
#include <limits.h>

#include "utils.h"



static gboolean SplitShuttle_button_press (GtkWidget *widget, GdkEventButton *event);
static gboolean SplitShuttle_button_release (GtkWidget * widget, GdkEventButton *event);
static void     SplitShuttle_class_init (SplitShuttleClass *klass);
static void     SplitShuttle_init (SplitShuttle *led);
static void     SplitShuttle_size_request (GtkWidget *widget, GtkRequisition *requisition);
static void     SplitShuttle_size_allocate (GtkWidget *widget, GtkAllocation *allocation);
static void     SplitShuttle_realize (GtkWidget *widget);
static gboolean SplitShuttle_expose (GtkWidget *widget, GdkEventExpose *event);
static void     SplitShuttle_destroy (GtkObject *object);
static gboolean SplitShuttle_motion_notify (GtkWidget *wheel, GdkEventMotion *event);
static void     splitshuttle_draw_lines (GtkWidget *wheel, cairo_t *cr);

#define SPLIT_SHUTTLE_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), SPLIT_TYPE_SHUTTLE, SplitShuttlePrivate))

enum
{
	SPLIT_VALUE_CHANGED = 0,
	SPLIT_LAST_SIGNAL
};

#define	SAMPLE_SHADES	3
#define BLOCK_SIZE 2352

typedef struct {							// DEFINITION D'UN POINT
	float		Max [ SAMPLE_SHADES ];				// Partie haute
	float		Min [ SAMPLE_SHADES ];				// Partie basse
} POINTS_SCREEN;

typedef struct {							// DEFINITION D'UN POINT
	float		Max;						// Partie haute
	float		Min;						// Partie basse
} POINTS_FILE;

typedef struct {
	guint		 Signals[ SPLIT_LAST_SIGNAL ];			// Signal
	GtkWidget	*GtkWidgetWHEEL;				// Adresse du widget
	GtkWidget	*GtkWidgetRedraw;				// Adresse du conteneur
	POINTS_FILE	*Tab;						// Tableau des amplitudes du fichier actif
	glong		 TotalAllocation;				// Longeur du tableau Tab
	POINTS_SCREEN	*TabScreen;					// Tableau des amplitudes pour l'ecran
	glong		 TotalAllocationTabScreen;			// Longeur du tableau TabScreen
	gshort		 nBitsPerSample;				// 
	gint		 nTotalChunckSize;				// 
	TYPE_FILE_IS	 TypeFileIs;					// 
	
} VAR_SPLITSHUTTLE;

static VAR_SPLITSHUTTLE VarSplitShuttle = { { 0 }, NULL, NULL, NULL, 0, NULL, 0, 0, 0, FALSE};


typedef struct _SplitShuttlePrivate SplitShuttlePrivate;
struct _SplitShuttlePrivate
{
	gboolean	 pressed;					// TRUE si curseur souris presse sinon FALSE

	gint		 OldX;						// Ancienne position de la souris

	gdouble		 CadreX, CadreY, CadreW, CadreH;		// Cadre du widget

	gdouble		 LigneVieX, LigneVieY, LigneVieW, LigneVieH;	// Pour info/debug  -->  Cadre reel du widget

	gdouble		 Zone_X_Before,					// Zone avant
			 Zone_X_In,					// Zone pendant
			 Zone_X_After,					// Zone apres
			 ZoneY,						//
			 Zone_W_Before,					// Zone avant
			 Zone_W_In,					// Zone pendant
			 Zone_W_After,					// Zone apres
			 ZoneH;						//

	gboolean	 BoolInDebut;					// TRUE si curseur souris dans le widget
	GdkPixbuf	*PixbufBegin;					// Selecteur de debut
	GdkPixbuf	*PixbufBeginPress;				// Selecteur de debut
	gdouble		 PixbufBeginX;					// Pos X
	gdouble		 PixbufBeginY;					// Pos Y
	gdouble		 PixbufBeginHeight;				// Hauteur
	gdouble		 PixbufBeginWidth;				// Largeur

	gboolean	 BoolInFin;					// TRUE si curseur souris dans le widget
	GdkPixbuf	*PixbufEnd;					// Selecteur de fin
	GdkPixbuf	*PixbufEndPress;				// Selecteur de fin
	gdouble		 PixbufEndX;					// Pos X
	gdouble		 PixbufEndY;					// Pos Y
	gdouble		 PixbufEndHeight;				// Hauteur
	gdouble		 PixbufEndWidth;				// Largeur

	gboolean	 BoolInPlay;					// TRUE is dedans else FALSE
	gdouble		 ZonePlayBeginX;
	gdouble		 ZonePlayEndX;
	gdouble		 ZonePlayBeginPercent;
	gdouble		 ZonePlayEndPercent;
	gdouble		 ZonePlayX, ZonePlayY, ZonePlayW, ZonePlayH;	//
	gdouble		 ActivePlayPercent;				// DebutPercent .. ActivePlayPercent
	gboolean	 BoolPlayerProgress;				// TRUE is progress else FALSE
	gdouble		 PlayerProgressX,				//
			 PlayerProgressY,				//
			 PlayerProgressW,				//
			 PlayerProgressH;				//

									// Selecteur de debut
	gdouble		 PosBeginX;					// Pos curseur debut pour X
	gdouble		 SelectBeginX, SelectBeginY;			// Pos selecteur de debut
	gdouble		 SelectBeginPercent;				// Pourcentage de debut

									// Selecteur de fin
	gdouble		 PosEndX;					// Pos curseur fin pour X
	gdouble		 SelectEndX, SelectEndY;			// Pos selecteur de fin
	gdouble		 SelectEndPercent;				// Pourcentage de fin

	gdouble		 TimeSongSec;					// Duree totale du fichier en secondes

	void		(*FuncExternBegin) (gdouble Percent);		// Rappel modif pointeur DEBUT
	void		(*FuncExternEnd) (gdouble Percent);		// Rappel modif pointeur FIN
	void		(*FuncExternWaitPlay) (void);			// Rappel selection d'ecoute
	void		(*FuncExternStopPlay) (void);			// Rappel fin d'ecoute
};

enum {
	SplitShuttle_SWITCH_STATE_SIGNAL,
	SplitShuttle_NB_SIGNALS
};

static guint SplitShuttle_signals [ SplitShuttle_NB_SIGNALS ] = { 0 };



/* Renvoie TRUE si le curseur souris est dans PLAY sinon FALSE
 */
static gboolean splitshuttle_get_in_play (GtkWidget *wheel, gint CurX, gint CurY)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (wheel);

	priv->BoolInPlay = FALSE;
	if (CurX > priv->CadreX +1 && CurX < priv->CadreW -1) {
		// if (CurY > priv->PlayVisual.Y && CurY < (priv->PlayVisual.Y + priv->PlayVisual.H) -1) {
		if (CurY > 0 && CurY < priv->LigneVieH -1) {
			priv->BoolInPlay = TRUE;
		}
	}
	return (priv->BoolInPlay);
}


static gboolean splitshuttle_get_in_play_selected (GtkWidget *wheel, gint CurX, gint CurY)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (wheel);

	if (priv->ZonePlayBeginX > -1 && priv->ZonePlayEndX > -1) {
		if (CurX > priv->ZonePlayX && CurX < priv->ZonePlayX + priv->ZonePlayW) {
			if (CurY > priv->ZonePlayY && CurY < priv->ZonePlayY + priv->ZonePlayH) {
				return (TRUE);
			}
		}
	}
	return (FALSE);
	// cairo_rectangle (cr, priv->ZonePlayX, priv->ZonePlayY, priv->ZonePlayW, priv->ZonePlayH);
}

/* Renvoie TRUE si le curseur souris est dans DEBUT sinon FALSE
 */
static gboolean splitshuttle_get_in_debut (GtkWidget *wheel, gint CurX, gint CurY)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (wheel);

	priv->BoolInDebut = FALSE;
	if (CurX > priv->SelectBeginX  +1 && CurX < (priv->SelectBeginX + priv->PixbufBeginWidth) -1) {
		if (CurY > priv->SelectBeginY && CurY < (priv->SelectBeginY + priv->PixbufBeginHeight) -1) {
			priv->BoolInDebut = TRUE;
		}
	}
	return (priv->BoolInDebut);
}


/* Renvoie TRUE si le curseur souris est dans FIN sinon FALSE
 */
static gboolean splitshuttle_get_in_fin (GtkWidget *wheel, gint CurX, gint CurY)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (wheel);

	priv->BoolInFin = FALSE;
	if (CurX > priv->SelectEndX  +1 && CurX < (priv->SelectEndX + priv->PixbufEndWidth) -1) {
		if (CurY > priv->SelectEndY && CurY < (priv->SelectEndY + priv->PixbufEndHeight) -1) {
			priv->BoolInFin = TRUE;
		}
	}
	return (priv->BoolInFin);
}

static void splitshuttle_get_values (GtkWidget *wheel)
{
	SplitShuttlePrivate	*priv   = SPLIT_SHUTTLE_GET_PRIVATE (wheel);

	// COORDONNEES DU CADRE

	priv->CadreX = 0;
	priv->CadreY = 0;
	priv->CadreW = wheel->allocation.width -1;
	priv->CadreH = wheel->allocation.height -1;

	// LA SURFACE REELLE

	priv->LigneVieX    = priv->PixbufBeginWidth / 2;
	priv->LigneVieY    = 0;
	priv->LigneVieW    = priv->CadreW - priv->PixbufBeginWidth;
	priv->LigneVieH    = priv->CadreH;

	// POS REELLE DEBUT
	priv->PosBeginX    = ((priv->LigneVieW -1) * priv->SelectBeginPercent) / 100.0;
	// POS REELLE FIN
	priv->PosEndX      = ((priv->LigneVieW -1) * priv->SelectEndPercent) / 100.0;

	// COORDONNEES SELECTEUR DEBUT

	priv->SelectBeginX = priv->PosBeginX;
	priv->SelectBeginY = wheel->allocation.height - priv->PixbufBeginHeight;

	// COORDONNEES SELECTEUR FIN

	priv->SelectEndX   = priv->PosEndX;
	priv->SelectEndY   = 0;

	priv->ZoneY = priv->PixbufEndHeight;
	priv->ZoneH = priv->CadreH - (priv->PixbufBeginHeight + priv->PixbufEndHeight);

	priv->Zone_X_Before = priv->PixbufBeginWidth / 2;
	priv->Zone_W_Before = priv->PosBeginX;

	priv->Zone_X_In     = (priv->PixbufBeginWidth / 2) + priv->PosBeginX;
	priv->Zone_W_In     = priv->PosEndX - priv->PosBeginX;

	priv->Zone_X_After  = (priv->PixbufBeginWidth / 2) + priv->PosEndX;
	priv->Zone_W_After  = priv->LigneVieW - priv->PosEndX;

	if (priv->ZonePlayBeginPercent > -1. && priv->ZonePlayEndPercent> -1.) {

		gdouble S_ZonePlayEndX         = priv->ZonePlayEndX;
		gdouble S_ZonePlayBeginX       = priv->ZonePlayBeginX;
		gdouble S_ZonePlayBeginPercent = priv->ZonePlayBeginPercent;
		gdouble S_ZonePlayEndPercent   = priv->ZonePlayEndPercent;

		if (priv->ZonePlayEndX < priv->ZonePlayBeginX) {
			gdouble        debut = priv->ZonePlayBeginX;
			priv->ZonePlayBeginX = priv->ZonePlayEndX;
			priv->ZonePlayEndX   = debut;

			priv->ZonePlayBeginPercent = ((priv->ZonePlayBeginX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
			priv->ZonePlayEndPercent   = ((priv->ZonePlayEndX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
		}

		// ZONE DE SELECTION POUR LE PLAY
		priv->ZonePlayX = priv->ZonePlayBeginX;
		priv->ZonePlayX = ((priv->LigneVieW * priv->ZonePlayBeginPercent) / 100.0) + priv->LigneVieX;
		priv->ZonePlayY = priv->LigneVieY;
		priv->ZonePlayW = priv->ZonePlayEndX - priv->ZonePlayBeginX;
		priv->ZonePlayW = (((priv->LigneVieW * priv->ZonePlayEndPercent) / 100.0) + priv->LigneVieX) - (((priv->LigneVieW * priv->ZonePlayBeginPercent) / 100.0) + priv->LigneVieX);
		if (priv->ZonePlayEndPercent == 100.0) priv->ZonePlayW --;
		priv->ZonePlayH = priv->LigneVieH;

		// ZONE DE PROGRESSION DU PLAYER

		if (priv->ActivePlayPercent > -1.0 && priv->BoolPlayerProgress == TRUE) {
			priv->PlayerProgressX    = priv->ZonePlayX;
			priv->PlayerProgressY    = priv->ZonePlayY +15;
			priv->PlayerProgressW    = (priv->LigneVieW * priv->ActivePlayPercent) / 100.0;
			priv->PlayerProgressH    = priv->ZonePlayH -30;
			priv->PlayerProgressH    = 2;
		}

		priv->ZonePlayEndX         = S_ZonePlayEndX;
		priv->ZonePlayBeginX       = S_ZonePlayBeginX;
		priv->ZonePlayBeginPercent = S_ZonePlayBeginPercent;
		priv->ZonePlayEndPercent   = S_ZonePlayEndPercent;
	}
}

GtkType SplitShuttle_get_type (void)
{
	static GtkType SplitShuttle_type = 0;

	if ( ! SplitShuttle_type) {
		static const GtkTypeInfo SplitShuttle_info = {
			"SplitShuttle",
			sizeof (SplitShuttle),
			sizeof (SplitShuttleClass),
			(GtkClassInitFunc) SplitShuttle_class_init,
			(GtkObjectInitFunc) SplitShuttle_init,
			NULL,
			NULL,
			(GtkClassInitFunc) NULL
			};
		SplitShuttle_type = gtk_type_unique (GTK_TYPE_WIDGET, &SplitShuttle_info);
	}
	return SplitShuttle_type;
}


GtkWidget *SplitShuttle_new (void)
{
	SplitShuttle	*NewWidget = NULL;

	NewWidget = gtk_type_new (SplitShuttle_get_type ());
	if (NewWidget != NULL) {
	}
	return GTK_WIDGET (NewWidget);
}


static gboolean SplitShuttle_motion_notify (GtkWidget *wheel, GdkEventMotion *event)
{
	SplitShuttlePrivate	*priv = SPLIT_SHUTTLE_GET_PRIVATE (wheel);
	GtkWidget		*widget = GTK_WIDGET (wheel);
	GdkRegion		*region;

	if (priv->OldX == -1) {
		priv->OldX = (gint)event->x;
	}

	if (priv->pressed == TRUE) {
		if (priv->BoolInDebut == TRUE) {

			priv->PosBeginX = (priv->LigneVieW * priv->SelectBeginPercent) / 100.0;

			if ((gint)event->x > priv->OldX) {
				priv->PosBeginX += (gint)event->x - priv->OldX;
			} else if ((gint)event->x < priv->OldX) {
				priv->PosBeginX -= priv->OldX - (gint)event->x;
			}

			priv->SelectBeginPercent = (priv->PosBeginX / priv->LigneVieW) * 100.0;
			if (priv->SelectBeginPercent < 0.0) priv->SelectBeginPercent =  0.0;
			if (priv->SelectBeginPercent > priv->SelectEndPercent) priv->SelectBeginPercent = priv->SelectEndPercent;

			if (priv->FuncExternBegin != NULL) {
				(priv->FuncExternBegin) (priv->SelectBeginPercent);
			}
		}
		else if (priv->BoolInFin == TRUE) {

			priv->PosEndX = (priv->LigneVieW * priv->SelectEndPercent) / 100.0;

			if ((gint)event->x > priv->OldX) {
				priv->PosEndX += (gint)event->x - priv->OldX;
			} else if ((gint)event->x < priv->OldX) {
				priv->PosEndX -= priv->OldX - (gint)event->x;
			}

			priv->SelectEndPercent = (priv->PosEndX / priv->LigneVieW) * 100.0;
			if (priv->SelectEndPercent > 100.0 ) priv->SelectEndPercent = 100.0;
			if (priv->SelectEndPercent < priv->SelectBeginPercent ) priv->SelectEndPercent = priv->SelectBeginPercent;

			if (priv->FuncExternEnd != NULL) {
				(priv->FuncExternEnd) (priv->SelectEndPercent);
			}
		}
		else if (priv->BoolInPlay == TRUE) {

			if (priv->ZonePlayBeginX == -1.0) {
				priv->ZonePlayBeginX = event->x;
				if (priv->ZonePlayBeginX < priv->LigneVieX)	priv->ZonePlayBeginX = priv->LigneVieX;
				if (priv->ZonePlayBeginX > priv->LigneVieW + priv->LigneVieX)	priv->ZonePlayBeginX = priv->LigneVieW + priv->LigneVieX;

				priv->ZonePlayEndPercent   =
				priv->ZonePlayBeginPercent = ((priv->ZonePlayBeginX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
			}
			else {
				priv->ZonePlayEndX = event->x;
				if (priv->ZonePlayEndX < priv->LigneVieX)			priv->ZonePlayEndX = priv->LigneVieX;
				if (priv->ZonePlayEndX > priv->LigneVieW + priv->LigneVieX)	priv->ZonePlayEndX = priv->LigneVieW + priv->LigneVieX;

				priv->ZonePlayEndPercent   = ((priv->ZonePlayEndX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
			}
		}
	}
	else {
		priv->BoolInDebut = splitshuttle_get_in_debut (wheel, (gint)event->x, (gint)event->y);
		priv->BoolInFin   = splitshuttle_get_in_fin (wheel, (gint)event->x, (gint)event->y);
		priv->BoolInPlay  = splitshuttle_get_in_play (wheel, (gint)event->x, (gint)event->y);

		utils_puts_statusbar_global ("");
		if (priv->BoolInDebut) {
			// ZERO
			utils_puts_statusbar_global ( _("Selection du debut de la plage"));
			// utils_puts_statusbar_global ( translate_get_mess (0));
		} else if (priv->BoolInFin) {
			// ONE
			utils_puts_statusbar_global ( _("Selection de la fin de la plage"));
			// utils_puts_statusbar_global ( translate_get_mess (1));
		} else if (splitshuttle_get_in_play_selected (wheel, (gint)event->x, (gint)event->y)) {
			// TWO
			utils_puts_statusbar_global ( _("Plage d'ecoute. Suppression = click droit"));
			// utils_puts_statusbar_global ( translate_get_mess (2));
		} else if (priv->BoolInPlay) {
			// TREE
			utils_puts_statusbar_global ( _("Click = selection de la plage pour ecoute / Click + Ctrl = selection globale de la plage pour ecoute"));
			// utils_puts_statusbar_global ( translate_get_mess (3));
		}
	}

	if (priv->pressed == TRUE) {
		region = gdk_drawable_get_clip_region (widget->window);
		gdk_window_invalidate_region (widget->window, region, TRUE);
		gdk_window_process_updates (widget->window, TRUE);
		gdk_region_destroy (region);
	}
	
	priv->OldX = (gint)event->x;

	return TRUE;
}

static gboolean SplitShuttle_button_press (GtkWidget *widget, GdkEventButton *event)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (widget);
	gboolean	bool_click_droit = (event->button == 3);

	if (priv == NULL) return (FALSE);
	priv->BoolInDebut = splitshuttle_get_in_debut (widget, (gint)event->x, (gint)event->y);
	priv->BoolInFin   = splitshuttle_get_in_fin (widget, (gint)event->x, (gint)event->y);
	if (priv->BoolInDebut == TRUE || priv->BoolInFin == TRUE) {
		priv->pressed = TRUE;
		return FALSE;
	}

	priv->ZonePlayBeginX    = -1.;
	priv->ZonePlayEndX      = -1.;
	priv->ActivePlayPercent = -1.;
	priv->ZonePlayX         = -1;
	priv->ZonePlayY         = -1;
	priv->ZonePlayW         = -1;
	priv->ZonePlayH         = -1;

	if (priv->FuncExternStopPlay != NULL) {
		(priv->FuncExternStopPlay) ();
	}

	splitshuttle_force_redraw ();

	if (keys.BoolGDK_Control_L == TRUE || keys.BoolGDK_Control_R == TRUE) {
		if (priv->BoolInPlay == TRUE) {
			if (event->x < priv->Zone_X_In) {
				priv->ZonePlayBeginX = priv->LigneVieX;
				priv->ZonePlayEndX   = priv->Zone_X_In;
				priv->ZonePlayBeginPercent = ((priv->ZonePlayBeginX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
				priv->ZonePlayEndPercent   = ((priv->ZonePlayEndX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
				return FALSE;
			} else if (event->x < priv->Zone_X_After) {
				priv->ZonePlayBeginX = priv->Zone_X_In;
				priv->ZonePlayEndX   = priv->Zone_X_After;
				priv->ZonePlayBeginPercent = ((priv->ZonePlayBeginX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
				priv->ZonePlayEndPercent   = ((priv->ZonePlayEndX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
				return FALSE;
			} else {
				priv->ZonePlayBeginX = priv->Zone_X_After;
				priv->ZonePlayEndX   = priv->Zone_X_After + priv->Zone_W_After;
				priv->ZonePlayBeginPercent = ((priv->ZonePlayBeginX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
				priv->ZonePlayEndPercent   = ((priv->ZonePlayEndX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
				return FALSE;
			}
		}
	}

	if (bool_click_droit == TRUE) return FALSE;

	priv->pressed = TRUE;

	return FALSE;
}


static gboolean SplitShuttle_button_release (GtkWidget *widget, GdkEventButton *event)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (widget);
	
	if (priv == NULL) return (FALSE);
	priv->pressed = FALSE;
	if (priv->BoolInDebut == TRUE || priv->BoolInFin == TRUE) {
		return FALSE;
	}

	if (priv->ZonePlayEndX == priv->ZonePlayBeginX) {
		priv->ZonePlayBeginX    = -1.;
		priv->ZonePlayEndX      = -1.;
		priv->ActivePlayPercent = -1.;
		priv->ZonePlayX         = -1;
		priv->ZonePlayY         = -1;
		priv->ZonePlayW         = -1;
		priv->ZonePlayH         = -1;
		if (priv->FuncExternStopPlay != NULL) {
			(priv->FuncExternStopPlay) ();
		}
		return FALSE;
	}

	if (priv->ZonePlayEndX < priv->ZonePlayBeginX) {
		gdouble             debut = priv->ZonePlayBeginX;
		priv->ZonePlayBeginX = priv->ZonePlayEndX;
		priv->ZonePlayEndX   = debut;
	}
	priv->ZonePlayBeginPercent = ((priv->ZonePlayBeginX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
	priv->ZonePlayEndPercent   = ((priv->ZonePlayEndX - priv->LigneVieX) / priv->LigneVieW) * 100.0;
	if (priv->ZonePlayBeginPercent < 0.0) priv->ZonePlayBeginPercent = 0.0;
	if (priv->ZonePlayEndPercent > 100.0) priv->ZonePlayEndPercent   = 100.0;

	if (priv->FuncExternWaitPlay != NULL) {
		(priv->FuncExternWaitPlay) ();
	}

	return FALSE;
}


static void SplitShuttle_class_init (SplitShuttleClass *klass)
{
	GtkWidgetClass	*widget_class;
	GtkObjectClass	*object_class;

	widget_class				= (GtkWidgetClass *) klass;
	object_class				= (GtkObjectClass *) klass;
	widget_class->realize			= SplitShuttle_realize;
	widget_class->size_request		= SplitShuttle_size_request;
	widget_class->size_allocate		= SplitShuttle_size_allocate;
	widget_class->expose_event		= SplitShuttle_expose;
	object_class->destroy			= SplitShuttle_destroy;
	widget_class->motion_notify_event	= SplitShuttle_motion_notify;
	// widget_class->button_press_event	= SplitShuttle_button_press;
	// widget_class->button_release_event	= SplitShuttle_button_release;

	SplitShuttle_signals [SplitShuttle_SWITCH_STATE_SIGNAL] = g_signal_new (
		"value-changed",
		G_TYPE_FROM_CLASS (klass),
		G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
		G_STRUCT_OFFSET (SplitShuttleClass, value_changed),
		NULL, NULL,
		// g_cclosure_marshal_VOID__VOID,
		gtk_marshal_NONE__NONE,
		G_TYPE_NONE,
		0
		);

	VarSplitShuttle.Tab = NULL;
	VarSplitShuttle.TabScreen = NULL;

	g_type_class_add_private (object_class, sizeof (SplitShuttlePrivate));
}


GdkPixbuf *splitshuttle_init_pixbufs (char *NameFilePixbuf)
{
	GdkPixbuf *NewPixbuf = NULL;
	GError    *error = NULL;
	gchar     *Pathname_Pixbuf = NULL;

	Pathname_Pixbuf = utils_get_pathname (NameFilePixbuf);

	NewPixbuf = gdk_pixbuf_new_from_file(Pathname_Pixbuf, &error);
 	if (error) {
 		GDK_PIXBUF_ERROR;
		g_critical ("Could not load pixbuf: %s\n", error->message);
		g_error_free (error);
		g_free (Pathname_Pixbuf);
		Pathname_Pixbuf = NULL;
		return (NULL);
	}
	g_free (Pathname_Pixbuf);
	Pathname_Pixbuf = NULL;

	return (NewPixbuf);
}

static void SplitShuttle_init (SplitShuttle *wheel)
{
	SplitShuttlePrivate *priv = SPLIT_SHUTTLE_GET_PRIVATE (wheel);

	VarSplitShuttle.GtkWidgetWHEEL = GTK_WIDGET (wheel);

	priv->PixbufBegin      = splitshuttle_init_pixbufs ("shuttle_begin.png");
	priv->PixbufBeginPress = splitshuttle_init_pixbufs ("shuttle_begin_press.png");
	priv->PixbufEnd        = splitshuttle_init_pixbufs ("shuttle_end.png");
	priv->PixbufEndPress   = splitshuttle_init_pixbufs ("shuttle_end_press.png");

	priv->SelectBeginPercent = 0.0;
	priv->SelectEndPercent   = 100.0;

	priv->BoolInDebut = FALSE;
	priv->BoolInFin   = FALSE;
	priv->BoolInPlay  = FALSE;

	priv->PixbufBeginHeight = gdk_pixbuf_get_height (priv->PixbufBegin);
	priv->PixbufBeginWidth  = gdk_pixbuf_get_width (priv->PixbufBegin);

	priv->PixbufEndHeight   = gdk_pixbuf_get_height (priv->PixbufEnd);
	priv->PixbufEndWidth    = gdk_pixbuf_get_width (priv->PixbufEnd);

	// Stocke pos curseur
	priv->OldX = -1;

	priv->FuncExternBegin = NULL;
	priv->FuncExternEnd   = NULL;

	priv->ZonePlayBeginX       = -1;
	priv->ZonePlayEndX         = -1;
	priv->ZonePlayBeginPercent = -1;
	priv->ZonePlayEndPercent   = -1;
	priv->ZonePlayX            = -1;
	priv->ZonePlayY            = -1;
	priv->ZonePlayW            = -1;
	priv->ZonePlayH            = -1;

	
	g_signal_connect (
		G_OBJECT (wheel),
		"button_release_event",
		G_CALLBACK (SplitShuttle_button_release),
		wheel
		);
	g_signal_connect (
		G_OBJECT (wheel),
		"button_press_event",
		G_CALLBACK (SplitShuttle_button_press),
		wheel
	);
	
	gtk_widget_add_events (
		GTK_WIDGET (wheel),
		GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK
		);
}


static void SplitShuttle_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SHUTTLE (widget));
	g_return_if_fail (requisition != NULL);

	requisition->width = GTK_SHUTTLE (widget)->width;
	requisition->height = GTK_SHUTTLE (widget)->height;
}


static void SplitShuttle_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SHUTTLE (widget));
	g_return_if_fail (allocation != NULL);

	widget->allocation = *allocation;
	if (GTK_WIDGET_REALIZED (widget)) {
		gdk_window_move_resize (widget->window, allocation->x, allocation->y, allocation->width, allocation->height);
	}
}


static void SplitShuttle_realize (GtkWidget *widget)
{
	GdkWindowAttr	 attributes;
	guint		 attributes_mask;
	SplitShuttle	*NewWidget = GTK_SHUTTLE (widget);


	g_return_if_fail (widget != NULL);
	g_return_if_fail (GTK_IS_SHUTTLE (widget));

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	attributes.window_type	= GDK_WINDOW_CHILD;
	attributes.x		= widget->allocation.x;
	attributes.y		= widget->allocation.y;
	attributes.width	= widget->allocation.width;
	attributes.height	= widget->allocation.height;
	attributes.wclass	= GDK_INPUT_OUTPUT;
	attributes.event_mask	= gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK | GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK;

	attributes_mask = GDK_WA_X | GDK_WA_Y;

	widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), &attributes, attributes_mask);

	gdk_window_set_user_data (widget->window, widget);

	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
	
	NewWidget->width  = widget->allocation.width;
	NewWidget->height = widget->allocation.height;
}


static gboolean SplitShuttle_expose (GtkWidget *widget, GdkEventExpose *event)
{
	cairo_t	*cr = NULL;
	
	g_return_val_if_fail (widget != NULL, FALSE);
	g_return_val_if_fail (GTK_IS_SHUTTLE (widget), FALSE);
	g_return_val_if_fail (event != NULL, FALSE);

	cr = gdk_cairo_create (widget->window);
	splitshuttle_get_values (widget);

	cairo_rectangle (cr, event->area.x, event->area.y, event->area.width, event->area.height);
	cairo_clip (cr);
	cairo_set_antialias (cr, CAIRO_ANTIALIAS_NONE);
	splitshuttle_draw_lines (widget, cr);
	cairo_destroy (cr);

	return FALSE;
}

#define	RED	1.0, 0.0, 0.0
#define	BLUE	0.0, 0.0, 1.0
#define	BLUE1	0.0, 0.5, 1.0
#define	MAGENTA	1.0, 0.0, 1.0
#define	YELLOW	1.0, 1.0, 0.0
#define	GREEN	0.0, 1.0, 0.0
#define	CYAN	0.0, 1.0, 1.0
#define	WHITE	1.0, 1.0, 1.0
#define	BLACK	0.0, 0.0, 0.0
#define	GRAY	0.6, 0.6, 0.6
#define	GRAY1	0.2, 0.2, 0.2


#define FOND_BLANC \
	gdk_color.red = 65535; \
	gdk_color.green = 65535; \
	gdk_color.blue = 65535; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_window_set_background (wheel->window, &gdk_color);

#define FOND_NOIR \
	gdk_color.red = 0; \
	gdk_color.green = 0; \
	gdk_color.blue = 0; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_window_set_background (wheel, &gdk_color);

#define FOND_BLEU \
	gdk_color.red = 0; \
	gdk_color.green = 0; \
	gdk_color.blue = 65535; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_window_set_background (wheel->window, &gdk_color);

#define CRAYON_BLANC \
	gdk_color.red = 65535; \
	gdk_color.green = 65535; \
	gdk_color.blue = 65535; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_NOIR \
	gdk_color.red = 0; \
	gdk_color.green = 0; \
	gdk_color.blue = 0; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_BLEU \
	gdk_color.red = 0; \
	gdk_color.green = 0; \
	gdk_color.blue = 65535; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_BLEU_CLAIR \
	gdk_color.red = 0; \
	gdk_color.green = 32000; \
	gdk_color.blue = 65535; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_BLEU_CLAIR_PLUS \
	gdk_color.red = 0; \
	gdk_color.green = 55000; \
	gdk_color.blue = 65535; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_ROUGE \
	gdk_color.red = 65535; \
	gdk_color.green = 0; \
	gdk_color.blue = 0; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_VERT \
	gdk_color.red = 0; \
	gdk_color.green = 65535; \
	gdk_color.blue = 0; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

#define CRAYON_JAUNE \
	gdk_color.red = 65535; \
	gdk_color.green = 65535; \
	gdk_color.blue = 0; \
	gdk_color_alloc (colormap, &gdk_color); \
	gdk_gc_set_foreground (gc, &gdk_color);

//
// RAPIDITE DE REAFFICHAGE DE L'AMPLITUDE ACCRUE AVEC LES FONCTIONS gdk_draw_ (...)
// 13 juillet 2009
//
static void splitshuttle_draw_lines (GtkWidget *wheel, cairo_t *cr)
{
	SplitShuttlePrivate	*priv         = SPLIT_SHUTTLE_GET_PRIVATE (wheel);
	GdkColor		 gdk_color;
	GdkColormap		*colormap;
	GdkGC			*gc;
	gboolean		BOOLSensitive = GTK_WIDGET_SENSITIVE(GTK_WIDGET (VarSplitShuttle.GtkWidgetRedraw));
	gint			y_min;
	gint			y_max;
	gint			xaxis;
	gint			scale;
	gint			points;
	gint			maxpoints = VarSplitShuttle.nTotalChunckSize / BLOCK_SIZE;
	gint			shade;
	gint			k;
	gdouble			Percent;
	gboolean		BOOLToogleAroundActivate = gtk_toggle_button_get_active ( GTK_TOGGLE_BUTTON (XCFA_GET_OBJECT("checkbutton_contour_split")));

	if (VarSplitShuttle.nBitsPerSample < 8) {
		return;
	}
	colormap = gdk_window_get_colormap (wheel->window);
	gc = gdk_gc_new (wheel->window);

	// PRINT_FUNC_LF();
	
	// DESSIN ZONE DE PLAY
	if (priv->ZonePlayBeginX > -1 &&
	    priv->ZonePlayEndX > -1 &&
	    priv->ZonePlayX > -1 &&
	    priv->ZonePlayY > -1 &&
	    priv->ZonePlayW > -1 &&
	    priv->ZonePlayH > -1
	    ) {
		CRAYON_JAUNE
		gdk_draw_rectangle (wheel->window,
			gc,
			TRUE,
			priv->ZonePlayX,
			priv->ZonePlayY,
			priv->ZonePlayW,
			priv->ZonePlayH
			);
		CRAYON_NOIR
		gdk_draw_rectangle (wheel->window,
			gc,
			FALSE,
			priv->ZonePlayX,
			priv->ZonePlayY,
			priv->ZonePlayW,
			priv->ZonePlayH
			);
	}
	
	// WAV if bitrate <> 8 16 24 32
	// MP3, OGG, FLAC, SHN, WAVPACK

	if (VarSplitShuttle.Tab == NULL &&
		(VarSplitShuttle.TypeFileIs == FILE_IS_WAV ||
		VarSplitShuttle.TypeFileIs == FILE_IS_MP3 ||
		VarSplitShuttle.TypeFileIs == FILE_IS_OGG ||
		VarSplitShuttle.TypeFileIs == FILE_IS_FLAC ||
		VarSplitShuttle.TypeFileIs == FILE_IS_SHN ||
		VarSplitShuttle.TypeFileIs == FILE_IS_WAVPACK)) {

		gint x = priv->LigneVieX +1;
		gint y = priv->LigneVieY + ((priv->LigneVieH / 2) -5);
		gint w = priv->LigneVieW -1;
		gint h = 10;
		
		// SI AUCUNE SELECTION
		if (BOOLSensitive == FALSE) {
			CRAYON_BLANC
			gdk_draw_rectangle (
				wheel->window,
				gc,
				! BOOLToogleAroundActivate,
				x,
				y,
				w,
				h
				);
		}
		else {
			// LA REPRESENTATION GRAPHIQUE COMPLETE DU FICHIER
			CRAYON_NOIR
			gdk_draw_rectangle (
				wheel->window,
				gc,
				! BOOLToogleAroundActivate,
				x,
				y,
				w,
				h
				);
			// SI POINTEUR DEBUT ET POINTEUR DE FIN SONT DIFFERENTS
			if (priv->Zone_X_In != priv->Zone_X_After) {
				CRAYON_BLEU
				gdk_draw_rectangle (
					wheel->window,
					gc,
					! BOOLToogleAroundActivate,
					priv->Zone_X_In,
					y,
					priv->Zone_X_After - priv->Zone_X_In +1,
					h
					);
				CRAYON_BLEU_CLAIR
				gdk_draw_rectangle (
					wheel->window,
					gc,
					! BOOLToogleAroundActivate,
					priv->Zone_X_In,
					y +2,
					priv->Zone_X_After - priv->Zone_X_In +1,
					h -4
					);
				CRAYON_BLEU_CLAIR_PLUS
				gdk_draw_rectangle (
					wheel->window,
					gc,
					! BOOLToogleAroundActivate,
					priv->Zone_X_In,
					y +4,
					priv->Zone_X_After - priv->Zone_X_In +1,
					h -8
					);
			}
		}
	}
	
	// WAV if bitrate == 8 16 24 32
	
	else if (VarSplitShuttle.Tab != NULL && VarSplitShuttle.TypeFileIs == FILE_IS_WAV) {

		if (VarSplitShuttle.TabScreen != NULL) {
			g_free (VarSplitShuttle.TabScreen);
			VarSplitShuttle.TabScreen = NULL;
		}
		VarSplitShuttle.TotalAllocationTabScreen = sizeof(POINTS_SCREEN) * priv->CadreW;
		VarSplitShuttle.TabScreen = (POINTS_SCREEN *)g_malloc0 (VarSplitShuttle.TotalAllocationTabScreen + 2);
		
		// LA LIGNE MEDIANE DU GRAPHE
		// xaxis = HAUTEUR_WIDGET / 2;
		xaxis = (priv->CadreH +1) / 2;

		if (xaxis != 0) {
			if (VarSplitShuttle.nBitsPerSample == 16)
				scale = SHRT_MAX  / (xaxis -1);
			else	scale = UCHAR_MAX / (xaxis -1);
			if (scale == 0)
				scale = 1;
		} else
			scale = 1;

		cairo_set_source_rgb (cr, RED);
		cairo_set_line_width (cr, 1);
		
		// 
		// Pas terrible cet algo hein  :/
		// Bahhhhhh !
		// Le prochain sera mieux ... patience ...
		// 
		for (points = 0; points < priv->LigneVieW; points ++) {
						
			Percent = (points / priv->LigneVieW) * 100;
			k = (Percent * maxpoints) / 100;
						
			y_max = xaxis - VarSplitShuttle.Tab[ k ].Max / scale;
			y_min = xaxis + fabs(VarSplitShuttle.Tab[ k ].Min) / scale;

			for(shade = 2; shade >= 0; shade --) {

				if (VarSplitShuttle.TabScreen[ points ].Max[shade]  == 0.) {
					VarSplitShuttle.TabScreen[ points ].Max[shade] = y_max-(y_max-xaxis)*(shade+0)/SAMPLE_SHADES;
				} else {
					if (y_max-(y_max-xaxis)*(shade+0)/SAMPLE_SHADES <= VarSplitShuttle.TabScreen[ points ].Max[shade])
						VarSplitShuttle.TabScreen[ points ].Max[shade] = y_max-(y_max-xaxis)*(shade+0)/SAMPLE_SHADES;
				}

				if (VarSplitShuttle.TabScreen[ points ].Min[shade] == 0.) {
					VarSplitShuttle.TabScreen[ points ].Min[shade] = y_min+(xaxis-y_min)*(shade+0)/SAMPLE_SHADES;
				} else {
					if (y_min+(xaxis-y_min)*(shade+0)/SAMPLE_SHADES >= VarSplitShuttle.TabScreen[ points ].Min[shade])
						VarSplitShuttle.TabScreen[ points ].Min[shade] = y_min+(xaxis-y_min)*(shade+0)/SAMPLE_SHADES;
				}
			}
		}
		
		// CONTOUR DE L ONDE
		
		if (BOOLToogleAroundActivate == TRUE) {
			
			shade = 0;
			for (points = 0; points < priv->LigneVieW -1; points ++) {
				
				if (BOOLSensitive == FALSE) {
					CRAYON_BLANC
				} else if (points + priv->LigneVieX >= priv->Zone_X_In && points + priv->LigneVieX < priv->Zone_X_After) {
					if (shade == 0) {
						CRAYON_BLEU
					} else if (shade == 1) {
						CRAYON_BLEU_CLAIR
					} else if (shade == 2) {
						CRAYON_BLEU_CLAIR_PLUS
					}
				} else {
					CRAYON_NOIR
				}

				gdk_draw_line (wheel->window,
						gc,
						points + priv->LigneVieX,
						VarSplitShuttle.TabScreen[ points ].Max[ shade ],
						points + priv->LigneVieX,
						VarSplitShuttle.TabScreen[ points +1 ].Max[ shade ]
						);
						
				gdk_draw_line (wheel->window,
						gc,
						points + priv->LigneVieX,
						VarSplitShuttle.TabScreen[ points ].Min[ shade ],
						points + priv->LigneVieX,
						VarSplitShuttle.TabScreen[ points +1 ].Min[ shade ]
						);
			}
		}
		
		// LES ONDES
		
		else {
			for (points = 0; points < priv->LigneVieW; points ++) {
				for (shade = 0; shade < SAMPLE_SHADES; shade ++) {
					if (BOOLSensitive == FALSE) {
						CRAYON_BLANC
					} else if (points + priv->LigneVieX >= priv->Zone_X_In && points + priv->LigneVieX < priv->Zone_X_After) {
						if (shade == 0) {
							CRAYON_BLEU
						} else if (shade == 1) {
							CRAYON_BLEU_CLAIR
						} else if (shade == 2) {
							CRAYON_BLEU_CLAIR_PLUS
						}
					} else {
						CRAYON_NOIR
					}

					gdk_draw_line (wheel->window,
								gc,
								points + priv->LigneVieX,
								VarSplitShuttle.TabScreen[ points ].Max[shade],
								points + priv->LigneVieX,
								VarSplitShuttle.TabScreen[ points ].Min[ shade ]
								);
				}
			}
		}
	}

	// ZONE DE PROGRESSION DU PLAYER
	if (priv->ActivePlayPercent > 0.0 &&
	    priv->BoolPlayerProgress == TRUE &&
	    priv->PlayerProgressX > 0.0

	    ) {
		if (priv->PlayerProgressX + priv->PlayerProgressW <= priv->ZonePlayX + priv->ZonePlayW) {
			CRAYON_ROUGE
			gdk_draw_line (wheel->window,
					gc,
					priv->PlayerProgressX + priv->PlayerProgressW,
					0,
					priv->PlayerProgressX + priv->PlayerProgressW,
					priv->ZonePlayH
					);
		}
	}

	// AFTER
	if (BOOLSensitive == FALSE) {
		CRAYON_BLANC
	} else {
		CRAYON_ROUGE
	}
	gdk_draw_line (wheel->window,
			gc,
			priv->Zone_X_After,
			0,
			priv->Zone_X_After,
			priv->CadreH
			);

	// IN
	if (BOOLSensitive == FALSE) {
		CRAYON_BLANC
	} else {
		CRAYON_ROUGE
	}
	gdk_draw_line (wheel->window,
			gc,
			priv->Zone_X_In,
			0,
			priv->Zone_X_In,
			priv->CadreH
			);

	// SELECTEUR DE DEBUT
	if (priv->BoolInDebut == TRUE)
		gdk_cairo_set_source_pixbuf (cr, priv->PixbufBeginPress, priv->SelectBeginX, priv->SelectBeginY);
	else	gdk_cairo_set_source_pixbuf (cr, priv->PixbufBegin, priv->SelectBeginX, priv->SelectBeginY);
	cairo_paint (cr);

	// SELECTEUR DE FIN
	if (priv->BoolInFin == TRUE)
		gdk_cairo_set_source_pixbuf (cr, priv->PixbufEndPress, priv->SelectEndX, priv->SelectEndY);
	else	gdk_cairo_set_source_pixbuf (cr, priv->PixbufEnd, priv->SelectEndX, priv->SelectEndY);
	cairo_paint (cr);

	// g_signal_emit (wheel, VarSplitShuttle.Signals[ SPLIT_VALUE_CHANGED ], 0);

	gdk_gc_destroy (gc);
}


static void SplitShuttle_destroy (GtkObject * object)
{
	SplitShuttle		*led;
	SplitShuttleClass	*klass;

	g_return_if_fail (object != NULL);
	g_return_if_fail (GTK_IS_SHUTTLE (object));

	led = GTK_SHUTTLE (object);

	klass = gtk_type_class (gtk_widget_get_type ());

	if (GTK_OBJECT_CLASS (klass)->destroy) {
		(* GTK_OBJECT_CLASS (klass)->destroy) (object);
	}
}

/*******************************************************************/

void splitshuttle_set_func_begin (void *FuncExternBegin)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->FuncExternBegin = FuncExternBegin;
}
void splitshuttle_set_percent_begin (gdouble Percent)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->SelectBeginPercent = Percent;

	if (priv->FuncExternBegin != NULL) {
		(priv->FuncExternBegin) (priv->SelectBeginPercent);
	}

	splitshuttle_force_redraw ();
}
gdouble splitshuttle_get_percent_begin (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return (0.0);
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return (0.0);

	return (priv->SelectBeginPercent);
}
void splitshuttle_set_func_end (void *FuncExternEnd)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->FuncExternEnd = FuncExternEnd;
}
void splitshuttle_set_percent_end (gdouble Percent)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->SelectEndPercent = Percent;

	if (priv->FuncExternEnd != NULL) {
		(priv->FuncExternEnd) (priv->SelectEndPercent);
	}

	splitshuttle_force_redraw ();
}
gdouble splitshuttle_get_percent_end (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return (0.0);
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return (0.0);

	return (priv->SelectEndPercent);
}
void splitshuttle_set_TimeSongSec (guint TimeSongSec)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	if (priv != NULL) {
		priv->TimeSongSec = (gdouble)TimeSongSec;
	}
}
gdouble splitshuttle_get_TimeSongSec (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return (0.0);
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return (0.0);

	return (priv->TimeSongSec);
}
gchar *splitshuttle_get_time (gdouble Percent, gboolean BoolTag, gboolean BoolPercent, gboolean BoolHours, gboolean BoolMin)
{
	gchar    *Mess = NULL;
	gint      H, M, S, C;

	splitshuttle_get_time_from_percent (Percent, &H, &M, &S, &C);

	if (BoolPercent == TRUE) {
		Mess = g_strdup_printf ("%s%3.3f%s", BoolTag ? "<b>" : "", Percent, BoolTag ? "</b>" : "");
	}
	else if (BoolHours == TRUE) {
		Mess = g_strdup_printf ("%s%02d%s", BoolTag ? "<b>" : "", H, BoolTag ? "</b>" : "");
	}
	else if (BoolMin == TRUE) {
		Mess = g_strdup_printf ("%s%02d%s", BoolTag ? "<b>" : "", M, BoolTag ? "</b>" : "");
	}
	else {
		Mess = g_strdup_printf ("%s%02d:%02d:%02d.%02d%s", BoolTag ? "<b>" : "", H, M, S, C, BoolTag ? "</b>" : "");
	}

	return ((gchar *)Mess);
}
void splitshuttle_get_time_from_percent (gdouble Percent, gint *H, gint *M, gint *S, gint *C)
{
	gint      sec;
	gdouble   dsec;
	gint      hundr;
	gdouble   TimeSongSec =  splitshuttle_get_TimeSongSec ();

	sec = (gint) ((TimeSongSec * Percent) / 100.0);
	dsec  = (TimeSongSec * Percent) / 100.0;
	hundr = (dsec - (gdouble)sec) * 100.0;

	if (hundr >= 100) hundr = 99;
	if (sec < 0) sec = 0;
	if (hundr < 0) hundr = 0;

	*H = (sec / 60) / 60;
	*M = (sec / 60) % 60;
	*S = sec % 60;
	*C = hundr;
}
gdouble splitshuttle_get_percent (gint H, gint M, gint S, gint C)
{
	gdouble		SecondesTempsActuel = 0.0;
	gdouble		PercentTempsActuel = 0.0;

	// TEMPS ACTUEL EN SECONDES
	// SecondesTempsActuel = (gdouble)((H * 60) * 60) + (gdouble)(M * 60) + S + ((gdouble)C / 100.);
	SecondesTempsActuel = GETTIMESEC (H,M,S,C);

	// POURCENTAGE
	// PercentTempsActuel = (SecondesTempsActuel / splitshuttle_get_TimeSongSec () ) * 100.0;
	PercentTempsActuel = GETTIMEPERCENT(SecondesTempsActuel,splitshuttle_get_TimeSongSec ());

	return ((gdouble)PercentTempsActuel);
}
void splitshuttle_set_ActivePlayPercent (gdouble Percent)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	if (priv->BoolPlayerProgress == FALSE) {
		priv->ActivePlayPercent = -1;
	} else {
		priv->ActivePlayPercent = Percent;
	}

	splitshuttle_force_redraw ();
}
gboolean splitshuttle_if_play (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return (FALSE);
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return (FALSE);

	return (priv->ActivePlayPercent > -0.0 ? TRUE : FALSE);
}
gdouble splitshuttle_get_percent_play_begin (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return (0.0);
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return (0.0);

	return (priv->ZonePlayBeginPercent);
}
gdouble splitshuttle_get_percent_play_end (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return (0.0);
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return (0.0);

	return (priv->ZonePlayEndPercent);
}
void splitshuttle_set_BoolActivePlayPercent (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->BoolPlayerProgress = TRUE;
}
void splitshuttle_set_BoolStopPlayPercent (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->BoolPlayerProgress = FALSE;
	splitshuttle_force_redraw ();
}
void splitshuttle_set_func_WaitPlay (void *FuncExternWaitPlay)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->FuncExternWaitPlay = FuncExternWaitPlay;
}
void splitshuttle_set_func_StopPlay (void *FuncExternStopPlay)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->FuncExternStopPlay = FuncExternStopPlay;
}
void splitshuttle_set_adr_redraw (GtkWidget *Widget)
{
	VarSplitShuttle.GtkWidgetRedraw = Widget;
}
void splitshuttle_reset_play (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	priv->ZonePlayBeginX    = -1.;
	priv->ZonePlayEndX      = -1.;
	priv->ActivePlayPercent = -1.;
	priv->ZonePlayX         = -1;
	priv->ZonePlayY         = -1;
	priv->ZonePlayW         = -1;
	priv->ZonePlayH         = -1;

	splitshuttle_force_redraw ();
	if (priv->FuncExternStopPlay != NULL) {
		(priv->FuncExternStopPlay) ();
	}

	splitshuttle_force_redraw ();
}
void splitshuttle_set_out (void)
{
	SplitShuttlePrivate	*priv = NULL;
	
	if (VarSplitShuttle.GtkWidgetWHEEL == NULL) return;
	priv = SPLIT_SHUTTLE_GET_PRIVATE (VarSplitShuttle.GtkWidgetWHEEL);
	if (priv == NULL) return;

	// DESACTIVE LE CURSEUR SUR LES OBJETS
	priv->BoolInDebut = FALSE;
	priv->BoolInFin   = FALSE;
	priv->BoolInPlay  = FALSE;
	// REDRAW
	splitshuttle_force_redraw ();
}

gboolean splitshuttle_read_file_spectre (DETAIL *Detail)
{
	FILE 		*pFile = NULL;
	gchar		 pbType[ 4 ];
	gint		 nSampleRate;
	gshort		 nChannels;
	gshort		 nBitsPerSample;
	gint		 nSamples;
	gint		 nTotalChunckSize = 0;
	gboolean	 found = FALSE;
	gint		 nRead;
	gint		 Ret;
	gint		 Indice;
	gint		 ChannelNext;
	gint		 min, max;
	gint		 tmp = 0;
	gint		 points;
	unsigned char 	 devbuf[ BLOCK_SIZE +10 ];
	size_t		 Dummy;
	gchar		*NameFile = Detail->namefile;
	
	
	VarSplitShuttle.TypeFileIs = Detail->type_infosong_file_is;
	
	// PRINT_FUNC_LF();
	
	// TODO: ANALYSE DES TYPES: MP3, OGG, FLAC, SHN, WAVPACK
	// TODO: fileanalyze_thread_do_verif_analyze_file_ (void *arg)
	
	if (VarSplitShuttle.TypeFileIs == FILE_IS_MP3 ||
	    VarSplitShuttle.TypeFileIs == FILE_IS_OGG ||
	    VarSplitShuttle.TypeFileIs == FILE_IS_FLAC ||
	    VarSplitShuttle.TypeFileIs == FILE_IS_SHN ||
	    VarSplitShuttle.TypeFileIs == FILE_IS_WAVPACK) {
	
		// DEL ALLOC MEMORY
		if (VarSplitShuttle.Tab != NULL) {
			g_free (VarSplitShuttle.Tab);
			VarSplitShuttle.Tab = NULL;
		}
		if (VarSplitShuttle.TabScreen != NULL) {
			g_free (VarSplitShuttle.TabScreen);
			VarSplitShuttle.TabScreen = NULL;
		}
		
		return (TRUE);
	}

	// TODO: Ne doit pas arriver
	if (VarSplitShuttle.TypeFileIs != FILE_IS_WAV) {
		PRINT ("VarSplitShuttle.TypeFileIs != FILE_IS_WAV");
		return FALSE;
	}
	
	pFile = fopen (NameFile, "rb");
	if (pFile == NULL) {
		PRINT_FUNC_LF();
		g_print ("\nImpossible d'ouvrir le fichier : \"%s\"\n\n", NameFile);
		return FALSE;
	}

	rewind (pFile);									// INDICATEUR DE FLUX AU DEBUT DU FICHIER
	fseek (pFile, 4, SEEK_CUR);							// Caractères 'RIFF' [52h 49h 46h 46h] identifiant le format.
	fseek (pFile, 4, SEEK_CUR);							// Longueur du groupe de données au format WAV = [30h.00h.01h.00h]
	fseek (pFile, 8, SEEK_CUR);							// Caractères 'WAVEfmt ' identifiant le format WAV.
	fseek (pFile, 4, SEEK_CUR);							// [10.00.00.00] = 00000010h = 16 = nombre d'octets utilisés après pour définir le format.
	fseek (pFile, 2, SEEK_CUR);							// [01.00] = 0001h = 1 = numéro de format du fichier (pas de compression, format PCM classique).
	Dummy = fread (&nChannels, 2, 1, pFile);					// Nombre de canaux
	Dummy = fread (&nSampleRate, 4, 1, pFile);					// Frequence d'echantillonange
	fseek (pFile, 4, SEEK_CUR);							// [11.2B.00.00] = 00002B11h = 11025, nombres d'octets par seconde, ce qui revient au même car un
											// échantillon mesure un octet et l'on est en mono.
	fseek (pFile, 2, SEEK_CUR);							// [01.00] = 0001h = 1 = Produit du nombre de canaux par le nombre d'octets par échantillon (ici 1x1=1).
	Dummy = fread (&nBitsPerSample, 2, 1, pFile);					// Lecture du nombre de bits par echantillon
	
	if (nBitsPerSample <= 0) nBitsPerSample = 8;
	VarSplitShuttle.nBitsPerSample = nBitsPerSample;

	switch (nBitsPerSample) {
	case 8 :
	case 16 :
	case 24 :
	case 32 :
		break;
	default :
		fclose (pFile);
		PRINT_FUNC_LF();
		g_print("Seuls ces formats de bitrates : 8 16 24 et 32 sont pris en charge pour la lecture\n"
			"Format de bitrate [ %d ] non prit en charge\n", 
			nBitsPerSample);
	
		// DEL ALLOC MEMORY
		if (VarSplitShuttle.Tab != NULL) {
			g_free (VarSplitShuttle.Tab);
			VarSplitShuttle.Tab = NULL;
		}
		if (VarSplitShuttle.TabScreen != NULL) {
			g_free (VarSplitShuttle.TabScreen);
			VarSplitShuttle.TabScreen = NULL;
		}
		return FALSE;
	
	}
	// CHERCHER LE: ChunckData

	rewind (pFile);									// INDICATEUR DE FLUX AU DEBUT DU FICHIER
	Dummy = fread (devbuf, 1, BLOCK_SIZE, pFile);					// LECTURE
	for (points = 0; points < BLOCK_SIZE; points ++) {
		if (devbuf [ points +0 ] == 'd' &&					// CHERCHER LE MOT data
		    devbuf [ points +1 ] == 'a' &&
		    devbuf [ points +2 ] == 't' &&
		    devbuf [ points +3 ] == 'a') {
			rewind (pFile);							// INDICATEUR DE FLUX AU DEBUT DU FICHIER
			fseek (pFile, points, SEEK_CUR);				// POSITION SUR 'data'
			do {
				nRead = fread (pbType, 4, 1, pFile);			// 'data' : annonce l'arrivée des données.
				nRead = fread (&nTotalChunckSize, 4, 1, pFile);		// [00.00.01.00] = 00010000h = 65536 : taille des données.
				VarSplitShuttle.nTotalChunckSize = nTotalChunckSize;
				if (pbType[ 0 ] == 'd' &&				// CHERCHER LE MOT data
				    pbType[ 1 ] == 'a' &&
				    pbType[ 2 ] == 't' &&
				    pbType[ 3 ] == 'a') {
					found = TRUE;					// OK
					nRead = 0;
					break;						// EXIT DO WHILE
				}
				fseek (pFile, nTotalChunckSize, SEEK_CUR);
			} while (nRead > 0);
			break;								// EXIT FOR
		}
	}
		
	// DEL ALLOC MEMORY
	if (VarSplitShuttle.Tab != NULL) {
		g_free (VarSplitShuttle.Tab);
		VarSplitShuttle.Tab = NULL;
	}
	if (VarSplitShuttle.TabScreen != NULL) {
		g_free (VarSplitShuttle.TabScreen);
		VarSplitShuttle.TabScreen = NULL;
	}

	if (found == FALSE) {
		/*
		printf ("\n");
		printf("ANALYSE DE: %s\n", NameFile);
		printf ("\n");
		printf("max_sample = %d\n", max_sample);
		printf("min_sample = %d\n", min_sample);
		printf ("nChannels             = %d\n", nChannels);
		printf ("nSampleRate           = %d\n", nSampleRate);
		printf ("nBitsPerSample        = %d\n", nBitsPerSample);
		printf ("nTotalChunckSize      = %d\n", nTotalChunckSize);
		printf ("nSamples              = %d\n", nSamples);
		printf ("TotalAllocation        = %lu\n", TotalAllocation);
		TotalAllocation = (sizeof(POINTS) * (nTotalChunckSize / BLOCK_SIZE));
		printf ("Allocation memoire    = %lu Ko\n", TotalAllocation);
		printf ("Allocation memoire    = %lu Mo\n", TotalAllocation / 1024);
		printf ("Allocation memoire    = %lu Go\n", (TotalAllocation / 1024) / 1024);
		printf ("\n");
		*/
		fclose (pFile);
		return FALSE;
	}
	else if (found == TRUE) {
		
		VarSplitShuttle.TotalAllocation = sizeof(POINTS_FILE) * ((nTotalChunckSize / BLOCK_SIZE) +10);
		// VarSplitShuttle.TotalAllocation = sizeof(POINTS_FILE) * (nTotalChunckSize);
		VarSplitShuttle.Tab = (POINTS_FILE *)g_malloc0 (VarSplitShuttle.TotalAllocation + 10);

		points = 0;
		if (nBitsPerSample <= 0) nBitsPerSample = 8;
		nSamples = (nTotalChunckSize) / (nBitsPerSample / 8 * nChannels);

		// TRAITEMENT UNIQUES: 8, 16, 24 et 32 bits

		switch (nBitsPerSample) {
			case 8 :
			case 16 :
			case 24 :
			case 32 :
				// PRE - CALCUL

				ChannelNext = (nChannels -1) * (nBitsPerSample / 8);

				// LECTURE DES DATAS

				while ((Ret = fread (devbuf, 1, BLOCK_SIZE, pFile)) > 0) {

					min = max = 0;

					for (Indice = 0; Indice < Ret; Indice++) {

						if (nBitsPerSample == 8) {
							tmp = devbuf [ Indice ];
							tmp -= 128;
						}
						else if (nBitsPerSample == 16) {
							tmp = (gchar)devbuf [ Indice +1 ] << 8 | (gchar)devbuf [ Indice ];
							
							Indice ++;
						}
						else if (nBitsPerSample == 24) {
							tmp  = (gchar)devbuf [ Indice ];
							tmp += (gchar)devbuf [ Indice +1 ] >> 8;
							tmp += (gchar)devbuf [ Indice +2 ];
							
							// if (tmp > 0x7FFFFF) tmp = 0x7FFFFF;  else if (tmp < -0x800000) tmp = -0x800000;
							
							Indice += 2;
						}
						else if (nBitsPerSample == 32) {
							tmp  = (gchar)devbuf [ Indice ];
							tmp |= (gchar)devbuf [ Indice +2 ];
							tmp += (gchar)devbuf [ Indice +1 ];
							tmp |= (gchar)devbuf [ Indice +3 ];
							
							// if (tmp > 0x7FFFFF) tmp = 0x7FFFFF;  else if (tmp < -0x800000) tmp = -0x800000;
							
							Indice += 2;
						}
						
						if (tmp > max) {
							
							max = tmp;
						} else if (tmp <= min) {
							
							min = tmp;
						}

						// skip over any extra channels
						Indice += ChannelNext;
					}
					VarSplitShuttle.Tab [ points ].Min = min;
					VarSplitShuttle.Tab [ points ].Max = max;
					points ++;
				}
				break;
			default :
				PRINT("Ne doit pas arriver ici  !!!!");
				g_print ("Ce format n'est pas pris en charge : %d\n", nBitsPerSample);
				fclose (pFile);
				return FALSE;
		}
	}
	fclose (pFile);
	/*
	printf ("\n");
	printf("ANALYSE DE: %s\n", NameFile);
	printf ("\n");
	printf ("nChannels             = %d\n", nChannels);
	printf ("nSampleRate           = %d\n", nSampleRate);
	printf ("nBitsPerSample        = %d\n", nBitsPerSample);
	printf ("nTotalChunckSize      = %d\n", nTotalChunckSize);
	printf ("nSamples              = %d\n", nSamples);
	printf ("Allocation memoire    = %5lu Ko\n", VarSplitShuttle.TotalAllocation);
	printf ("Allocation memoire    = %5lu,%lu Mo\n", VarSplitShuttle.TotalAllocation / 1024, VarSplitShuttle.TotalAllocation % 1024);
	printf ("Allocation memoire    = %5lu,%lu Go\n", (VarSplitShuttle.TotalAllocation / 1024) / 1024, (VarSplitShuttle.TotalAllocation / 1024) % 1024);
	printf ("\n");
	*/
	// max = 0;
	// min = 0;
	// while (points > 0) {
	// 	if (VarSplitShuttle.Tab [ points ].Max > max) max = VarSplitShuttle.Tab [ points ].Max;
	// 	if (VarSplitShuttle.Tab [ points ].Min < min) min = VarSplitShuttle.Tab [ points ].Min;
	// 	points --;
	// }
	// g_print ("max = %d,    min = %d\n", max, min);
	
	return TRUE;
}
void splitshuttle_remove_data (void)
{
	if (VarSplitShuttle.Tab != NULL) {
		g_free (VarSplitShuttle.Tab);
		VarSplitShuttle.Tab = NULL;
	}
	if (VarSplitShuttle.TabScreen != NULL) {
		g_free (VarSplitShuttle.TabScreen);
		VarSplitShuttle.TabScreen = NULL;
	}
}
void splitshuttle_force_redraw (void)
{
	GtkWidget	*WD = VarSplitShuttle.GtkWidgetRedraw;

	// DESSIN DU WIDGET
	gtk_widget_queue_draw_area (
				WD,
				WD->allocation.x,
				WD->allocation.y,
				WD->allocation.width,
				WD->allocation.height
				);
}

