#!/usr/bin/env python
# -*- coding: utf-8 -*-

import string, os, sys

#This file is part of PLCOpenEditor, a library implementing an IEC 61131-3 editor
#based on the plcopen standard. 
#
#Copyright (C): Edouard TISSERANT and Laurent BESSARD
#
#See COPYING file for copyrights details.
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#Lesser General Public License for more details.
#
#You should have received a copy of the GNU General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


LANGUAGES = ["IL","ST","FBD","LD","SFC"]


#-------------------------------------------------------------------------------
#                        Function Block Types definitions
#-------------------------------------------------------------------------------


"""
Ordored list of common Function Blocks defined in the IEC 61131-3
Each block have this attributes:
    - "name" : The block name
    - "type" : The block type. It can be "function", "functionBlock" or "program"
    - "extensible" : Boolean that define if the block is extensible
    - "inputs" : List of the block inputs
    - "outputs" : List of the block outputs
    - "comment" : Comment that will be displayed in the block popup
Inputs and outputs are a tuple of characteristics that are in order:
    - The name
    - The data type
    - The default modifier which can be "none", "negated", "rising" or "falling"
"""

oldBlockTypes = [{"name" : "Standard function blocks", "list":
               [{"name" : "SR", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("S1","BOOL","none"),("R","BOOL","none")], 
                    "outputs" : [("Q1","BOOL","none")],
                    "comment" : "SR bistable\nThe SR bistable is a latch where the Set dominates."},
                {"name" : "RS", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("S","BOOL","none"),("R1","BOOL","none")], 
                    "outputs" : [("Q1","BOOL","none")],
                    "comment" : "RS bistable\nThe RS bistable is a latch where the Reset dominates."},
                {"name" : "SEMA", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("CLAIM","BOOL","none"),("RELEASE","BOOL","none")], 
                    "outputs" : [("BUSY","BOOL","none")],
                    "comment" : "Semaphore\nThe semaphore provides a mechanism to allow software elements mutually exclusive access to certain ressources."},
                {"name" : "R_TRIG", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("CLK","BOOL","none")], 
                    "outputs" : [("Q","BOOL","none")],
                    "comment" : "Rising edge detector\nThe output produces a single pulse when a rising edge is detected."},
                {"name" : "F_TRIG", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("CLK","BOOL","none")], 
                    "outputs" : [("Q","BOOL","none")],
                    "comment" : "Falling edge detector\nThe output produces a single pulse when a falling edge is detected."},
                {"name" : "CTU", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("CU","BOOL","rising"),("R","BOOL","none"),("PV","INT","none")], 
                    "outputs" : [("Q","BOOL","none"),("CV","INT","none")],
                    "comment" : "Up-counter\nThe up-counter can be used to signal when a count has reached a maximum value."},
                {"name" : "CTD", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("CD","BOOL","rising"),("LD","BOOL","none"),("PV","INT","none")], 
                    "outputs" : [("Q","BOOL","none"),("CV","INT","none")],
                    "comment" : "Down-counter\nThe down-counter can be used to signal when a count has reached zero, on counting down from a preset value."},
                {"name" : "CTUD", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("CU","BOOL","rising"),("CD","BOOL","rising"),("R","BOOL","none"),("LD","BOOL","none"),("PV","INT","none")], 
                    "outputs" : [("QU","BOOL","none"),("QD","BOOL","none"),("CV","INT","none")],
                    "comment" : "Up-down counter\nThe up-down counter has two inputs CU and CD. It can be used to both count up on one input ans down on the other."},
                {"name" : "TP", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("IN","BOOL","none"),("PT","TIME","none")], 
                    "outputs" : [("Q","BOOL","none"),("ET","TIME","none")],
                    "comment" : "Pulse timer\nThe pulse timer can be used to generate output pulses of a given time duration."},
                {"name" : "TOF", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("IN","BOOL","none"),("PT","TIME","none")], 
                    "outputs" : [("Q","BOOL","none"),("ET","TIME","none")],
                    "comment" : "On-delay timer\nThe on-delay timer can be used to delay setting an output true, for fixed period after an input becomes true."},
                {"name" : "TON", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("IN","BOOL","none"),("PT","TIME","none")], 
                    "outputs" : [("Q","BOOL","none"),("ET","TIME","none")],
                    "comment" : "Off-delay timer\nThe off-delay timer can be used to delay setting an output false, for fixed period after input goes false."},
                {"name" : "RTC", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("EN","BOOL","none"),("PDT","DATE_AND_TIME","none")], 
                    "outputs" : [("Q","BOOL","none"),("CDT","DATE_AND_TIME","none")],
                    "comment" : "Real time clock\nThe real time clock has many uses including time stamping, setting dates and times of day in batch reports, in alarm messages and so on."},
                {"name" : "INTEGRAL", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("RUN","BOOL","none"),("R1","BOOL","none"),("XIN","REAL","none"),("X0","REAL","none"),("CYCLE","TIME","none")], 
                    "outputs" : [("Q","BOOL","none"),("XOUT","REAL","none")],
                    "comment" : "Integral\nThe integral function block integrates the value of input XIN over time."},
                {"name" : "DERIVATIVE", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("RUN","BOOL","none"),("XIN","REAL","none"),("CYCLE","TIME","none")], 
                    "outputs" : [("XOUT","REAL","none")],
                    "comment" : "Derivative\nThe derivative function block produces an output XOUT proportional to the rate of change of the input XIN."},
                {"name" : "PID", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("AUTO","BOOL","none"),("PV","REAL","none"),("SP","REAL","none"),("X0","REAL","none"),("KP","REAL","none"),("TR","REAL","none"),("TD","REAL","none"),("CYCLE","TIME","none")], 
                    "outputs" : [("XOUT","REAL","none")],
                    "comment" : "PID\nThe PID (proportional, Integral, Derivative) function block provides the classical three term controller for closed loop control."},
                {"name" : "RAMP", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("RUN","BOOL","none"),("X0","REAL","none"),("X1","REAL","none"),("TR","TIME","none"),("CYCLE","TIME","none"),("HOLDBACK","BOOL","none"),("ERROR","REAL","none"),("PV","REAL","none")], 
                    "outputs" : [("RAMP","BOOL","none"),("XOUT","REAL","none")],
                    "comment" : "Ramp\nThe RAMP function block is modelled on example given in the standard but with the addition of a 'Holdback' feature."},
                {"name" : "HYSTERESIS", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("XIN1","REAL","none"),("XIN2","REAL","none"),("EPS","REAL","none")], 
                    "outputs" : [("Q","BOOL","none")],
                    "comment" : "Hysteresis\nThe hysteresis function block provides a hysteresis boolean output driven by the difference of two floating point (REAL) inputs XIN1 and XIN2."},
                {"name" : "RATIO_MONITOR", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("PV1","REAL","none"),("PV2","REAL","none"),("RATIO","REAL","none"),("TIMON","TIME","none"),("TIMOFF","TIME","none"),("TOLERANCE","BOOL","none"),("RESET","BOOL","none"),("CYCLE","TIME","none")], 
                    "outputs" : [("ALARM","BOOL","none"),("TOTAL_ERR","BOOL","none")],
                    "comment" : "Ratio monitor\nThe ratio_monitor function block checks that one process value PV1 is always a given ratio (defined by input RATIO) of a second process value PV2."},
               ]}
             ]

BlockTypes = [{"name" : "Standard function blocks", "list":
               [{"name" : "Container", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("Show","BOOL","none"),("Set State","BOOL","none")], 
                    "outputs" : [("Show","BOOL","none"),("State Changed","BOOL","none")],
                    "comment" : "SVGUI Container"},
                {"name" : "Button", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("Show","BOOL","none"),("Toggle","BOOL","none")], 
                    "outputs" : [("Visible","BOOL","none"),("State","BOOL","none")],
                    "comment" : "SVGUI Button"},
                {"name" : "TextCtrl", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("Text","STRING","none"),("Set Text","BOOL","none")], 
                    "outputs" : [("Text","STRING","none"),("Text Changed","BOOL","none")],
                    "comment" : "SVGUI Text Control"},
                {"name" : "ScrollBar", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("Position","UINT","none"),("Set Position","BOOL","none")], 
                    "outputs" : [("Position","UINT","none"),("Position Changed","BOOL","none")],
                    "comment" : "SVGUI ScrollBar"},
                {"name" : "NoteBook", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("Selected","UINT","none"),("Set Selected","BOOL","none")], 
                    "outputs" : [("Selected","UINT","none"),("Selected Changed","BOOL","none")],
                    "comment" : "SVGUI Notebook"},
                {"name" : "RotatingCtrl", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("Angle","REAL","none"),("Set Angle","BOOL","none")], 
                    "outputs" : [("Angle","REAL","none"),("Angle changed","BOOL","none")],
                    "comment" : "SVGUI Rotating Control"},
                {"name" : "Transform", "type" : "functionBlock", "extensible" : False, 
                    "inputs" : [("X","REAL","none"),("Y","REAL","none"),("Scale X","REAL","none"),("Scale Y","REAL","none"),("Angle","REAL","none"),("Set","BOOL","none")], 
                    "outputs" : [("X","REAL","none"),("Y","REAL","none"),("Scale X","REAL","none"),("Scale Y","REAL","none"),("Angle","REAL","none"),("Changed","BOOL","none")],
                    "comment" : "SVGUI Transform"},
               ]}
             ]

"""
Function that returns the block definition associated to the block type given
"""

def GetBlockType(type):
    for category in BlockTypes:
        for blocktype in category["list"]:
            if blocktype["name"] == type:
                return blocktype
    return None

    