# -*- Mode: Python; coding: utf-8; indent-tabs-mode: nil; tab-width: 4 -*-
# Copyright 2012 Canonical
# Author: Thomi Richards
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.


import os
from mock import patch
from StringIO import StringIO
from testtools import TestCase
from testtools.matchers import raises

import windowmocker


class TestFilePathLoading(TestCase):

    """Tests the ability to create apps from a file path."""

    def test_create_application_from_path_raises_IOError_on_missing_file(self):
        """create_application_from_path function must raise IOError when asked
        to open a file that does not exist.
        """
        self.assertThat(lambda: windowmocker.create_application_from_path("_does_not_exist"),
            raises(IOError))

    def test_create_application_from_path_raises_IOError_on_bad_perms(self):
        """create_application_from_path function must raise IOError when asked
        to open a file that does not exist.
        """
        file_name = '_bad_perms'
        open(file_name, 'w')
        os.chmod(file_name, 0)
        self.addCleanup(os.remove, file_name)

        self.assertThat(lambda: windowmocker.create_application_from_path(file_name),
            raises(IOError))


class TestFileObjectLoading(TestCase):

    """Tests the ability to create apps from a file-like object."""

    def test_raises_attributeerror_on_bad_parameter_type(self):
        """create_application_from_file must raise a AttributeError if we give it
        something other than a file-like object.

        """
        self.assertThat(lambda: windowmocker.create_application_from_file("string"),
            raises(AttributeError))
        self.assertThat(lambda: windowmocker.create_application_from_file(123),
            raises(AttributeError))
        self.assertThat(lambda: windowmocker.create_application_from_file(123.456),
            raises(AttributeError))
        self.assertThat(lambda: windowmocker.create_application_from_file(True),
            raises(AttributeError))

    @patch('windowmocker.create_application_from_data')
    def test_can_load_ascii_json(self, load_mock):
        windowmocker.create_application_from_file(StringIO(b'["Hello", "World"]'))
        load_mock.assert_called_once_with(["Hello", "World"], None)

    @patch('windowmocker.create_application_from_data')
    def test_can_load_unicode_json(self, load_mock):
        windowmocker.create_application_from_file(StringIO(u'["ɥǝןןo", "ʍoɹןp"]'))
        load_mock.assert_called_once_with([u'\u0265\u01dd\u05df\u05dfo', u'\u028do\u0279\u05dfp'], None)

    def test_raises_runtime_on_empty_file(self):
        fn = lambda: windowmocker.create_application_from_file(StringIO())
        self.assertThat(fn, raises(RuntimeError("No JSON object could be decoded")))
