/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "definitionparser.h"

#include <QStringList>

DefinitionParser::DefinitionParser()
{
    QString string( "\"(?:\\\\[\"\\\\]|[^\"\\\\])*\"" );
    QString attribute( "([-a-z]+)=(-?\\d+|%1|\\{%2(?:,%3)*\\})" );

    m_stringRegExp.setPattern( string );
    m_attributeRegExp.setPattern( attribute.arg( string, string, string ) );
}

DefinitionParser::~DefinitionParser()
{
}

bool DefinitionParser::parse( const QString& text )
{
    if ( text.isEmpty() )
        return false;

    QString attrs;

    int index = text.indexOf( ' ' );
    if ( index == -1 ) {
        m_type = text;
    } else {
        m_type = text.left( index );
        attrs = text.mid( index + 1 );
    }

    int pos = 0;
    while ( ( pos = m_attributeRegExp.indexIn( attrs, pos ) ) != -1 ) {
        QString key = m_attributeRegExp.cap( 1 );
        QString value = m_attributeRegExp.cap( 2 );

        if ( value[ 0 ] == QLatin1Char( '\"' ) )
            m_metadata.insert( key, parseString( value ) );
        else if ( value[ 0 ] == QLatin1Char( '{' ) )
            m_metadata.insert( key, parseStringArray( value ) );
        else
            m_metadata.insert( key, value.toInt() );

        pos += m_attributeRegExp.matchedLength();
    }

    return true;
}

QStringList DefinitionParser::parseStringArray( const QString& value )
{
    QStringList result;
    int pos = 0;
    while ( ( pos = m_stringRegExp.indexIn( value, pos ) ) != -1 ) {
        result.append( parseString( m_stringRegExp.cap( 0 ) ) );
        pos += m_stringRegExp.matchedLength();
    }
    return result;
}

QString DefinitionParser::parseString( const QString& value )
{
    QString result;
    int len = value.length() - 1;
    for ( int i = 1; i < len; i++ ) {
        QChar ch = value[ i ];
        if ( ch == QLatin1Char( '\\' ) )
            ch = value[ ++i ];
        result += ch;
    }
    return result;
}
