/**************************************************************************
* Simple XML-based UI builder for Qt 4
* Copyright (C) 2007 Michał Męciński
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef XMLUI_CLIENT_H
#define XMLUI_CLIENT_H

#include <QString>
#include <QMap>

#include "node.h"

class QAction;
class QDomElement;

namespace XmlUi
{

class Builder;

/**
* The UI client.
*
* This class is a container for actions. It also describes the layout
* of the menu bar, toolbars and context menus for a main window, view,
* component or plug-in.
*
* One ore more clients can be added to a Builder object.
*
* Window, view or component classes  should inherit the Client class
* using multiple inheritance. They should add actions to the container,
* set menu and toolbar titles and load the XML file describing
* the layout of the UI.
*
* The XML file should have a @c xmlui root element which can contain
* one @c menubar and a number of @c toolbar elements and @c menu elements
* for context menus. Toolbars and menus can contain @c action elements
* and @c separator elements. Menus can be nested. The @c toolbar, @c menu
* and @c action elements have an @c id attribute.
*
* The @c merge element without attributes marks the position where elements
* from the next client are inserted. If it's not present, they are appended
* to the end.
*
* The @c merge element with an @c id attribute inserts all elements from
* a @c group element with the same @c id and the same location.
*
* Example:
*
* @code
* <xmlui>
*   <menubar>
*     <menu id="menu_file">
*       <merge/>
*       <separator/>
*       <action id="file_quit"/>
*     </menu>
*     <menu id="menu_edit">
*       <action id="edit_cut"/>
*       <action id="edit_copy"/>
*       <action id="edit_paste"/>
*       <merge id="merge_paste"/>
*       <separator/>
*       <action id="edit_select_all"/>
*       <separator/>
*       <merge/>
*     <menu>
*   <menubar>
*   <toolbar id="toolbar_main">
*     <action id="edit_cut"/>
*     <action id="edit_copy"/>
*     <action id="edit_paste"/>
*   </toolbar>
*   <menu id="context_menu">
*     <action id="edit_select_all"/>
*   <menu>
* <xmlui>
* @endcode
*/
class Client
{
public:
    /**
    * Default constructor.
    */
    Client();

    /**
    * Destructor.
    */
    ~Client();

public:
    /**
    * Add an action to the collection.
    *
    * @param id The identifier of the action.
    *
    * @param action The action to add.
    */
    void setAction( const QString& id, QAction* action );

    /**
    * Find an action with the given identifier.
    *
    * @param id The identifier of the action.
    *
    * @return The action or @c NULL if it wasn't found.
    */
    QAction* action( const QString& id ) const;

    /**
    * Set the title of a menu or toolbar.
    *
    * @param id The identifier of the menu or toolbar.
    *
    * @param title The user-visible name of the element.
    */
    void setTitle( const QString& id, const QString& title );

    /**
    * Return the title of a menu or toolbar.
    *
    * @param id The identifier of the menu or toolbar.
    *
    * @return The user-visible name of the element.
    */
    QString title( const QString& id ) const;

    /**
    * Load the UI layout from the given XML file.
    *
    * @param path The path of the file.
    */
    void loadXmlFile( const QString& path );

    /**
    * Return the root node of the UI layout.
    */
    const Node& rootNode() const { return m_rootNode; }

    /**
    * Set the associated UI builder.
    *
    * This method is called automatically when the client
    * is added to a builder.
    */
    void setBuilder( Builder* builder );

    /**
    * Return the associated UI builder.
    */
    Builder* builder() const { return m_builder; }

private:
    Node createNode( const QDomElement& element );

    NodeType typeFromTag( const QString& tag );

private:
    QMap<QString, QAction*> m_actions;

    QMap<QString, QString> m_titles;

    Node m_rootNode;

    Builder* m_builder;
};

}

#endif
