/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "rowfilters.h"

#include "attributes/datetimehelper.h"
#include "data/datamanager.h"
#include "data/datarows.h"
#include "tablemodelshelper.h"

using namespace WebIssues;

IssueRowFilter::IssueRowFilter( QObject* parent ) : AbstractRowFilter( parent )
{
}

IssueRowFilter::~IssueRowFilter()
{
}

void IssueRowFilter::setConditions( const QList<ColumnCondition>& conditions )
{
    m_conditions = conditions;

    emit conditionsChanged();
} 

void IssueRowFilter::setQuickSearch( const QString& search )
{
    m_search = search;

    emit conditionsChanged();
}

bool IssueRowFilter::filterRow( int id )
{
    for ( int i = 0; i < m_conditions.count(); i++ ) {
        QString value = extractValue( id, m_conditions.at( i ).column() );
        if ( !m_conditions.at( i ).match( value ) )
            return false;
    }

    if ( !m_search.isEmpty() ) {
        const IssueRow* row = dataManager->issues()->find( id );
        if ( !row )
            return false;
        if ( !row->name().contains( m_search, Qt::CaseInsensitive ) )
            return false;
    }

    return true;
}

QString IssueRowFilter::extractValue( int id, Column column )
{
    const IssueRow* row = dataManager->issues()->find( id );
    if ( !row )
        return QString();

    if ( column > Column_UserDefined ) {
        int attributeId = column - Column_UserDefined;
        const ValueRow* value = dataManager->values()->find( attributeId, id );
        return value ? value->value() : QString();
    }

    switch ( column ) {
        case Column_Name:
            return row->name();
        case Column_ID:
            return QString::number( id );
        case Column_CreatedDate:
            return DateTimeHelper::formatDateTime( row->createdDate() );
        case Column_CreatedBy:
            return TableModelsHelper::userName( row->createdUser() );
        case Column_ModifiedDate:
            return DateTimeHelper::formatDateTime( row->modifiedDate() );
        case Column_ModifiedBy:
            return TableModelsHelper::userName( row->modifiedUser() );
        default:
            return QString();
    }
}

#include "rowfilters.moc"
