/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef USERSBATCH_H
#define USERSBATCH_H

#include "commands/abstractbatch.h"
#include "batchjob.h"
#include "datarows.h"

namespace WebIssues
{

/**
* Batch for operations on users.
*
* This batch executes a series of commands creating and modifying users
* and their rights. After making the modifications, the batch updates
* the users data.
*/
class UsersBatch : public AbstractBatch
{
    Q_OBJECT
public:
    /**
    * Constructor.
    */
    UsersBatch();

    /**
    * Destructor.
    */
    ~UsersBatch();

public:
    /**
    * Add the <tt>ADD USER</tt> command to the batch.
    *
    * @param login Login of the user to create.
    *
    * @param name Name of the user.
    *
    * @param password Password for the user.
    */
    void addUser( const QString& login, const QString& name, const QString& password );

    /**
    * Add the <tt>SET PASSWORD</tt> command to the batch.
    *
    * @param userId Identifier of the user.
    *
    * @param password The new password for the user.
    */
    void setPassword( int userId, const QString& password );

    /**
    * Add the <tt>GRANT USER</tt> command to the batch.
    *
    * @param userId Identifier of the user.
    *
    * @param access The new access level of the user.
    */
    void grantUser( int userId, Access access );

    /**
    * Add the <tt>RENAME USER</tt> command to the batch.
    *
    * @param userId Identifier of the user.
    *
    * @param newName The new name of the user.
    */
    void renameUser( int userId, const QString& newName );

    /**
    * Add the <tt>GRANT MEMBER</tt> command to the batch.
    *
    * @param userId Identifier of the user.
    *
    * @param projectId Identifier of the project.
    *
    * @param access The new access level to the project.
    */
    void grantMember( int userId, int projectId, Access access );

public: // overrides
    Command* fetchNext();

private:
    typedef BatchJob<UsersBatch> Job;
    typedef BatchJobQueue<UsersBatch> JobQueue;

private:
    Command* addUserJob( const Job& job );
    Command* setPasswordJob( const Job& job );
    Command* grantUserJob( const Job& job );
    Command* renameUserJob( const Job& job );
    Command* grantMemberJob( const Job& job );

private slots:
    void setUpdate();

private:
    JobQueue m_queue;

    bool m_update;
};

}

#endif
