/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef ROWHELPERS_H
#define ROWHELPERS_H

#include <QList>
#include <QString>

namespace WebIssues
{

/**
* Sort rows of a table using given property.
*
* The values of the property are compared using operator >.
*
* @param it Row iterator of any kind.
*
* @param order Getter method of the property to use for sorting.
*
* @return The list of rows ordered by the property value.
*
* Example:
*
* @code
* TableConstIterator<MyRow> it( &myTable );
* QValueList<const MyRow*> rows = sortRows( it, &MyRow::size );
* @endcode
*/
template<class T, typename IT, typename R>
inline QList<const T*> sortRows( IT it, R (T::*order)() const )
{
    QList<const T*> list;
    while ( it.next() ) {
        const T* row = it.get();
        typename QList<const T*>::iterator jt = list.begin();
        for ( ; jt != list.end(); ++jt ) {
            if ( ( ( *jt )->*order )() > ( row->*order )() )
                break;
        }
        list.insert( jt, row );
    }
    return list;
}

/**
* Sort rows of a table using given property.
*
* The values of the property are compared using QString::localeAwareCompare().
*
* @param it Row iterator of any kind.
*
* @param order Getter method of the property to use for sorting.
*
* @return The list of rows ordered by the property value.
*
* @see sortRows()
*/
template<class T, typename IT, typename R>
inline QList<const T*> localeAwareSortRows( IT it, R (T::*order)() const )
{
    QList<const T*> list;
    while ( it.next() ) {
        const T* row = it.get();
        typename QList<const T*>::iterator jt = list.begin();
        for ( ; jt != list.end(); ++jt ) {
            if ( QString::localeAwareCompare( ( ( *jt )->*order )() , ( row->*order )() ) > 0 )
                break;
        }
        list.insert( jt, row );
    }
    return list;
}

}

#endif
