/**************************************************************************
* Generic, native, relational database
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "tablecore.h"

using namespace RDB;

TableCore::TableCore( TableData* data, uint dims ) :
    m_data( data ),
    m_dims( dims )
{
}

TableCore::~TableCore()
{
}

void TableCore::insert( Row* row )
{
    TableData::Bucket* bucket = m_data[ 0 ].allocBucket( m_dims );
    bucket->m_row = row;

    for ( uint i = 0; i < m_dims; i++ )
        m_data[ i ].linkBucket( bucket );
}

Row* TableCore::find( uint dim, int key ) const
{
    TableData::Bucket* bucket = m_data[ dim ].findBucket( key );
    if ( bucket )
        return bucket->m_row;
    return 0;
}

Row* TableCore::find( uint dim, int key, uint otherDim, int otherKey ) const
{
    TableData::Bucket* bucket = m_data[ dim ].findBucket( key, otherDim, otherKey );
    if ( bucket )
        return bucket->m_row;
    return 0;
}

void TableCore::remove( uint dim, int key )
{
    TableData::Bucket* bucket = m_data[ dim ].findBucket( key );
    if ( bucket )
        removeBucket( bucket );
}

void TableCore::remove( uint dim, int key, uint otherDim, int otherKey )
{
    TableData::Bucket* bucket = m_data[ dim ].findBucket( key, otherDim, otherKey );
    if ( bucket )
        removeBucket( bucket );
}

void TableCore::removeAll( uint dim, int key )
{
    uint hash = key % m_data[ dim ].m_size;
    TableData::Bucket* bucket = m_data[ dim ].m_buckets[ hash ];
    while ( bucket ) {
        TableData::Bucket* next = bucket->m_links[ dim ].m_next;
        if ( bucket->m_row->key( dim ) == key )
            removeBucket( bucket );
        bucket = next;
    }
}

void TableCore::clear()
{
    freeAllBuckets();

    for ( uint i = 0; i < m_dims; i++ )
        m_data[ i ].clear();
}

void TableCore::freeAllBuckets()
{
    for ( uint i = 0; i < m_data[ 0 ].m_size; i++ ) {
        TableData::Bucket* bucket = m_data[ 0 ].m_buckets[ i ];
        while ( bucket ) {
            TableData::Bucket* next = bucket->m_links[ 0 ].m_next;
            delete bucket->m_row;
            m_data[ 0 ].freeBucket( bucket );
            bucket = next;
        }
    }
}

void TableCore::removeBucket( TableData::Bucket* bucket )
{
    for ( uint i = 0; i < m_dims; i++ )
        m_data[ i ].unlinkBucket( bucket );
    delete bucket->m_row;
    m_data[ 0 ].freeBucket( bucket );
}

