#include "texteditor.h"

#include <iostream>
#include <cstdlib>
#include <typeinfo>

#include "noteobj.h"
#include "settings.h"

extern int statusbarTime;
extern Settings settings;

extern QAction *actionViewToggleNoteEditor;

extern QString iconPath;
extern QString vymName;

using namespace std;


///////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////


TextEditor::TextEditor()
{
    printer = new QPrinter( QPrinter::HighResolution );
	printer->setPrinterName (settings.value("/mainwindow/printerName",printer->printerName()).toString());

    e = new QTextEdit( this);
    e->setFocus();
	e->setTextFormat(Qt::RichText);		// default
	e->setTabStopWidth (20);		// unit is pixel
	e->setColor (Qt::black);
	e->setAutoFillBackground (true);
	connect (e, SIGNAL( textChanged() ), this, SLOT( editorChanged() ) );
    setCentralWidget( e );
    statusBar()->message( tr("Ready","Statusbar message"), statusbarTime);
	setCaption(vymName +" - " +tr ("Note Editor","Window caption"));


	connect(e, SIGNAL(currentCharFormatChanged(const QTextCharFormat &)),
            this, SLOT(formatChanged(const QTextCharFormat &)));



	// Toolbars
	setupFileActions();
	setupEditActions();
	setupFormatActions();
	setupSettingsActions();
	
	// Various states
	blockChangedSignal=false;
	setInactive();

	// Load Settings
	resize (settings.value ( "/satellite/noteeditor/geometry/size", QSize(450,600)).toSize());
	move   (settings.value ( "/satellite/noteeditor/geometry/pos", QPoint (250,50)).toPoint());
	
	setShowWithMain (settings.value ( "/satellite/noteeditor/showWithMain",true).toBool());

	varFont.fromString( settings.value
		("/satellite/noteeditor/fonts/varFont",
		"Nimbus Sans l,14,-1,5,48,0,0,0,0,0").toString() 
	);
	fixedFont.fromString (settings.value(
		"/satellite/noteeditor/fonts/fixedFont",
		"Courier,14,-1,5,48,0,0,0,1,0").toString() 
	);
	QString s=settings.value ("/satellite/noteeditor/fonts/fonthintDefault","variable").toString();
	if (s == "fixed")
	{	
		actionSettingsFonthintDefault->setOn (true);
		e->setCurrentFont (fixedFont);
	} else	
	{
		actionSettingsFonthintDefault->setOn (false);
		e->setCurrentFont (varFont);
	}	
	filenameHint="";

	// Restore position of toolbars
	restoreState (settings.value("/satellite/noteeditor/state",0).toByteArray());

	// Save settings in vymrc
	settings.setValue("/mainwindow/printerName",printer->printerName());
}


TextEditor::~TextEditor()
{
    if (printer) delete printer;
	// Save Settings
	settings.setValue( "/satellite/noteeditor/geometry/size", size() );
	settings.setValue( "/satellite/noteeditor/geometry/pos", pos() );
	settings.setValue ("/satellite/noteeditor/state",saveState(0));
	
	settings.setValue( "/satellite/noteeditor/showWithMain",showwithmain);

	QString s;
	if (actionSettingsFonthintDefault->isOn() )
		s="fixed";
	else	
		s="variable";
	settings.setValue( "/satellite/noteeditor/fonts/fonthintDefault",s );
	settings.setValue("/satellite/noteeditor/fonts/varFont", varFont.toString() );
	settings.setValue("/satellite/noteeditor/fonts/fixedFont", fixedFont.toString() );


}

bool TextEditor::isEmpty()
{
	if (e->toPlainText().length()>0)
		return false;
	else
		return true;
}

void TextEditor::setShowWithMain(bool v)
{
	showwithmain=v;
}

bool TextEditor::showWithMain()
{
	return showwithmain;
}


void TextEditor::setFontHint (const QString &fh)
{
	if (fh=="fixed")
		actionFormatUseFixedFont->setOn (true);
	else
		actionFormatUseFixedFont->setOn (false);
}


QString TextEditor::getFontHint()
{
	if (actionFormatUseFixedFont->isOn())
		return "fixed";
	else	
		return "var";
}

QString TextEditor::getFontHintDefault()
{
	if (actionSettingsFonthintDefault->isOn())
		return "fixed";
	else	
		return "var";
}

void TextEditor::setFilename(const QString &fn)
{
	if (state==filledEditor)
	{
		if (fn.isEmpty() )
		{
			filename="";
			statusBar()->message( tr("No filename available for this note.","Statusbar message"), statusbarTime );
		}	
		else
		{
			filename=fn;
			statusBar()->message( tr(QString( "Current filename is %1" ).arg( filename ),"Statusbar message"), statusbarTime );
		}	
	}	
}

QString TextEditor::getFilename()
{
	return filename;
}

void TextEditor::setFilenameHint(const QString &fnh)
{
	filenameHint=fnh;
}

QString TextEditor::getFilenameHint()
{
	return filenameHint;
}

bool TextEditor::findText(const QString &t, const QTextDocument::FindFlags &flags)
{
	if (e->find (t,flags))
		return true;
	else	
		return false;
}

void TextEditor::setupFileActions()
{
    QToolBar *tb = addToolBar ( tr("Note Actions") );
	tb->setObjectName ("noteEditorFileActions");
    QMenu *fileMenu = menuBar()->addMenu( tr( "&Note","Menubar" ));

    QAction *a;
    a = new QAction( QPixmap( iconPath+"fileopen.png"), tr( "&Import..." ),this);
	a->setStatusTip (tr( "Import","Status tip for Note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_O );
    connect( a, SIGNAL( activated() ), this, SLOT( textLoad() ) );
	tb->addAction (a);
	fileMenu->addAction (a);
	actionFileLoad=a;

    fileMenu->addSeparator();
    a = new QAction( QPixmap(iconPath+"filesave.png" ), tr( "&Export..." ),this);
	a->setStatusTip (tr( "Export Note (HTML)","Status tip for Note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_S );
    connect( a, SIGNAL( activated() ), this, SLOT( textSave() ) );
	tb->addAction (a);
	fileMenu->addAction (a);
	actionFileSave=a;
	
    a = new QAction(  QPixmap(), tr( "Export &As... (HTML)" ), this);
	a->setStatusTip (tr( "Export Note As (HTML) ","Status tip for Note Menu"  ));
    connect( a, SIGNAL( activated() ), this, SLOT( textSaveAs() ) );
	fileMenu->addAction (a);
	actionFileSaveAs=a;

    a = new QAction(QPixmap(), tr( "Export &As...(ASCII)" ), this);
	a->setStatusTip ( tr( "Export Note As (ASCII) ","Status tip for note menu" ) );
	a->setShortcut(Qt::ALT + Qt::Key_X );
    connect( a, SIGNAL( activated() ), this, SLOT( textExportAsASCII() ) );
	fileMenu->addAction (a);
	actionFileSaveAs=a;

    fileMenu->addSeparator();
    a = new QAction( QPixmap(iconPath+"fileprint.png" ), tr( "&Print..." ),this);
	a->setStatusTip (tr( "Print Note","Status tip for note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_P );
    connect( a, SIGNAL( activated() ), this, SLOT( textPrint() ) );
	tb->addAction (a);
	fileMenu->addAction (a);
	actionFilePrint=a;
}

void TextEditor::setupEditActions()
{
    QToolBar *tb = addToolBar ( tr( "Edit Actions" ));
	tb->setObjectName ("noteEditorEditActions");
    QMenu *editMenu = menuBar()->addMenu ( tr( "&Edit" ));

    QAction *a;
    a = new QAction(QPixmap(iconPath+"undo.png"), tr( "&Undo" ), this );
	a->setStatusTip ( tr( "Undo","Status tip for note menu" ) );
	a->setShortcut(Qt::CTRL + Qt::Key_Z );
    connect( a, SIGNAL( activated() ), e, SLOT( undo() ) );
	editMenu->addAction (a);
	tb->addAction (a);
	actionEditUndo=a;
	
    a = new QAction(QPixmap(iconPath+"redo.png" ), tr( "&Redo" ),this); 
	a->setStatusTip ( tr( "Redo","Status tip for note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_Y );
    connect( a, SIGNAL( activated() ), e, SLOT( redo() ) );
	editMenu->addAction (a);
	tb->addAction (a);
	actionEditRedo=a;

    editMenu->addSeparator();
    a = new QAction(QPixmap(), tr( "Select and copy &all" ),this); 
	a->setStatusTip ( tr( "Select and copy all","Status tip for note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_A );
    connect( a, SIGNAL( activated() ), this, SLOT( editCopyAll() ) );
	editMenu->addAction (a);

    editMenu->addSeparator();
    a = new QAction(QPixmap(iconPath+"editcopy.png" ), tr( "&Copy" ),this);
	a->setStatusTip ( tr( "Copy","Status tip for note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_C );
    connect( a, SIGNAL( activated() ), e, SLOT( copy() ) );
	editMenu->addAction (a);
	tb->addAction (a);
	actionEditCopy=a;
	
    a = new QAction(QPixmap(iconPath+"editcut.png" ), tr( "Cu&t" ),this);
	a->setStatusTip ( tr( "Cut","Status tip for note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_X );
    connect( a, SIGNAL( activated() ), e, SLOT( cut() ) );
	editMenu->addAction (a);
	tb->addAction (a);
	actionEditCut=a;

    a = new QAction(QPixmap(iconPath+"editpaste.png" ), tr( "&Paste" ),this);
	a->setStatusTip ( tr( "Paste","Status tip for note menu" ) );
	a->setShortcut( Qt::CTRL + Qt::Key_V );
    connect( a, SIGNAL( activated() ), e, SLOT( paste() ) );
	editMenu->addAction (a);
	tb->addAction (a);
	actionEditPaste=a;
	
    a = new QAction( QPixmap( iconPath+"edittrash.png"), tr( "&Delete All" ), this);
	a->setStatusTip (tr( "Delete all","Status tip for note menu" ) );
    connect( a, SIGNAL( activated() ), e, SLOT( clear() ) );
	editMenu->addAction (a);
	tb->addAction (a);
	actionEditDeleteAll=a;

}

void TextEditor::setupFormatActions()
{
    QToolBar *tb = addToolBar ( tr("Format Actions" ));
	tb->setObjectName ("noteEditorFormatActions");
    QMenu *formatMenu = menuBar()->addMenu ( tr( "F&ormat" ));

    QAction *a;

    a = new QAction( QPixmap(iconPath+"formatfixedfont.png"), tr( "&Font hint" ), Qt::ALT + Qt::Key_I, this, "fontHint" );
	a->setStatusTip (tr( "Toggle font hint for the whole text","Status tip for note menu" ) );
	a->setToggleAction (true);
	a->setOn (settings.value("/noteeditor/fonts/useFixedByDefault",false).toBool() );
    connect( a, SIGNAL( activated() ), this, SLOT( toggleFonthint() ) );
	formatMenu->addAction (a);
	tb->addAction (a);
	actionFormatUseFixedFont=a;

    comboFont = new QComboBox;
	tb->addWidget (comboFont);
    QFontDatabase fontDB;
    comboFont->insertStringList( fontDB.families() );
    connect( comboFont, SIGNAL( activated( const QString & ) ),
	     this, SLOT( textFamily( const QString & ) ) );
    comboFont->addItem( QApplication::font().family() );
    comboSize = new QComboBox;
	tb->addWidget (comboSize);
	QList<int> sizes=fontDB.standardSizes();
	QList<int>::iterator i = sizes.begin();
	while (i != sizes.end()) 
	{
		++i; // increment i before using it
		comboSize->insertItem ( QString::number(*i));
	}	
    connect( comboSize, SIGNAL( activated( const QString & ) ),
	     this, SLOT( textSize( const QString & ) ) );
    comboSize->addItem ( QString::number( QApplication::font().pointSize() ) );

    formatMenu->addSeparator();

    QPixmap pix( 16, 16 );
    pix.fill( e->color());
    a = new QAction( pix, tr( "&Color..." ), this);
	formatMenu->addAction (a);
	tb->addAction (a);
    connect( a, SIGNAL( activated() ), this, SLOT( textColor() ) );
    actionTextColor=a;

    a = new QAction( QPixmap (iconPath+"text_bold.png"), tr( "&Bold" ), this);
	a->setShortcut(Qt::CTRL + Qt::Key_B );
    connect( a, SIGNAL( activated() ), this, SLOT( textBold() ) );
	tb->addAction (a);
	formatMenu->addAction (a);
    a->setToggleAction( true );
    actionTextBold=a;
	
    a = new QAction( QPixmap(iconPath+"text_italic.png"), tr( "&Italic" ),  this);
	a->setShortcut(Qt::CTRL + Qt::Key_I);
    connect( a, SIGNAL( activated() ), this, SLOT( textItalic() ) );
	tb->addAction (a);
	formatMenu->addAction (a);
    a->setToggleAction( true );
    actionTextItalic=a;
	
    a = new QAction( QPixmap (iconPath+"text_under.png"), tr( "&Underline" ), this);
	a->setShortcut(Qt::CTRL + Qt::Key_U );
    connect( a, SIGNAL( activated() ), this, SLOT( textUnderline() ) );
	tb->addAction (a);
	formatMenu->addAction (a);
    a->setToggleAction( true );
    actionTextUnderline=a;
    formatMenu->addSeparator();

    QActionGroup *grp2 = new QActionGroup( this );
    grp2->setExclusive(true);
    a = new QAction( QPixmap (iconPath+"text_sub.png"), tr( "Subs&cript" ),grp2 );
	a->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_B );
    a->setToggleAction( true );
	tb->addAction (a);
	formatMenu->addAction (a);
    connect(a, SIGNAL(activated()), this, SLOT(textVAlign()));
    actionAlignSubScript=a;

    a = new QAction( QPixmap (iconPath+"text_super.png"), tr( "Su&perscript" ),grp2  );
	a->setShortcut( Qt::CTRL + Qt::SHIFT + Qt::Key_P );
    a->setToggleAction( true );
	tb->addAction (a);
	formatMenu->addAction (a);
    connect(a, SIGNAL(activated()), this, SLOT(textVAlign()));
    actionAlignSuperScript=a;
    QActionGroup *grp = new QActionGroup( this );
    connect( grp, SIGNAL( selected( QAction* ) ), this, SLOT( textAlign( QAction* ) ) );

    formatMenu->addSeparator();

    a = new QAction( QPixmap (iconPath+"text_left.png"), tr( "&Left" ),grp );
	a->setShortcut( Qt::CTRL+Qt::Key_L );
    a->setToggleAction( true );
	tb->addAction (a);
	formatMenu->addAction (a);
    actionAlignLeft=a;
    a = new QAction( QPixmap (iconPath+"text_center.png"), tr( "C&enter" ),grp);
    a->setShortcut(  Qt::CTRL + Qt::Key_E);
    a->setToggleAction( true );
	tb->addAction (a);
	formatMenu->addAction (a);
    actionAlignCenter=a;
    a = new QAction( QPixmap (iconPath+"text_right.png" ), tr( "&Right" ), grp);
	a->setShortcut(Qt::CTRL + Qt::Key_R );
    a->setToggleAction( true );
	tb->addAction (a);
	formatMenu->addAction (a);
    actionAlignRight=a;
    a = new QAction( QPixmap ( iconPath+"text_block.png"), tr( "&Justify" ), grp );
	a->setShortcut(Qt::CTRL + Qt::Key_J );
    a->setToggleAction( true );
	tb->addAction (a);
	formatMenu->addAction (a);
    actionAlignJustify=a;
}

void TextEditor::setupSettingsActions()
{
    QMenu *settingsMenu = menuBar()->addMenu ( tr( "&Settings" ));

    QAction *a;
    a = new QAction(tr( "Set &fixed font" ), this);
	a->setStatusTip ( tr( "Set fixed font","Status tip for note menu" ));
    connect( a, SIGNAL( activated() ), this, SLOT( setFixedFont() ) );
	settingsMenu->addAction (a);
	actionSettingsFixedFont=a;

    a = new QAction(tr( "Set &variable font" ), this);
	a->setStatusTip ( tr( "Set variable font","Status tip for note menu" ) );
    connect( a, SIGNAL( activated() ), this, SLOT( setVarFont() ) );
	settingsMenu->addAction (a);
	actionSettingsVarFont=a;

    a = new QAction(tr( "&fixed font is default" ),  this);
	a->setStatusTip (tr( "Used fixed font by default","Status tip for note menu" ) );
	a->setToggleAction (true);
	// set state later in constructor...
	settingsMenu->addAction (a);
	actionSettingsFonthintDefault=a;
}

void TextEditor::textLoad()
{
	if (state!=inactiveEditor)
	{
		if (!isEmpty()) 
		{
			QMessageBox mb( vymName + " - " +tr("Note Editor"),
				"Loading will overwrite the existing note",
				QMessageBox::Warning,
				QMessageBox::Yes | QMessageBox::Default,
				QMessageBox::Cancel,
				0 );
			mb.setButtonText( QMessageBox::Yes, "Load note" );
			switch( mb.exec() ) {
				case QMessageBox::Cancel:
					return;
					break;
			}
		} 
		// Load note
		QFileDialog *fd=new QFileDialog( this);
		QStringList types;
		types<< "VYM notes (*.html)" <<
			"ASCII texts (*.txt)" <<
			"All filed (*)";
		fd->setFilters (types);
		fd->setDirectory (QDir().current());
		fd->show();
		QString fn;
		if ( fd->exec() == QDialog::Accepted )
			fn = fd->selectedFile();

		if ( !fn.isEmpty() )
		{
			QFile f( fn );
			if ( !f.open( QIODevice::ReadOnly ) )
			return;

			QTextStream ts( &f );
			setText( ts.read() );
			editorChanged();
		}
	}
}

void TextEditor::closeEvent( QCloseEvent* ce )
{
    ce->accept();	// TextEditor can be reopened with show()
	showwithmain=false;
	hide();
	emit (windowClosed() );
    return;
}

QString TextEditor::getText()
{
	if (e->toPlainText().isEmpty())
		return "";
	else	
		return e->text();
}

void TextEditor::editorChanged()
{
	if (isEmpty())
		state=emptyEditor;
	else
		state=filledEditor;

		if (state==emptyEditor)
			setState (emptyEditor);
		else
			setState (filledEditor);
	// SLOT is LinkableMapObj, which will update systemFlag
	if (!blockChangedSignal) emit (textHasChanged() );
}


void TextEditor::setText(QString t)
{
	blockChangedSignal=true;
	e->setReadOnly(false);
	e->setText(t);
	enableActions();
	blockChangedSignal=false;
}

void TextEditor::setInactive()
{
	state=inactiveEditor;
	setText("");
	setState (inactiveEditor);
	e->setReadOnly (true);

	disableActions();
}

void TextEditor::editCopyAll()
{
	e->selectAll();
	e->copy();
}

void TextEditor::textSaveAs()
{
    QString fn = QFileDialog::getSaveFileName( QString::null, "VYM Note (HTML) (*.html);;All files (*)",
					       this,"export note dialog",tr("Export Note to single file") );

    if ( !fn.isEmpty() ) 
	{
		QFile file (fn);
		if (file.exists())
		{
			QMessageBox mb( vymName,
				tr("The file %1\nexists already.\nDo you want to overwrite it?","dialog 'save note as'").arg(fn),
			QMessageBox::Warning,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::Cancel | QMessageBox::Escape,
			Qt::NoButton );
			mb.setButtonText( QMessageBox::Yes, tr("Overwrite") );
			mb.setButtonText( QMessageBox::No, tr("Cancel"));
			switch( mb.exec() ) {
				case QMessageBox::Yes:
					// save 
					filename = fn;
					textSave();
					return;
				case QMessageBox::Cancel:
					// do nothing
					break;
			}
		} else
		{
			filename = fn;
			textSave();
			return;
		}			
    }
	statusBar()->message(tr( "Couldn't export note ","dialog 'save note as'") + fn, statusbarTime );
}


void TextEditor::textSave()
{
    if ( filename.isEmpty() ) 
	{
		textSaveAs();
		return;
    }

    QString text = e->text();
    QFile f( filename );
    if ( !f.open( QIODevice::WriteOnly ) ) 
	{
		statusBar()->message( QString("Could not write to %1").arg(filename),
					  statusbarTime );
		return;
    }

    QTextStream t( &f );
    t << text;
    f.close();

    e->setModified( FALSE );

    statusBar()->message( QString( "Note exported as %1" ).arg( filename ), statusbarTime );
}

void TextEditor::textExportAsASCII()
{
	QString text = NoteObj (e->text()).getNoteASCII();
	QString fn,s;
	if (!filenameHint.isEmpty())
	{
		if (!filenameHint.contains (".txt"))
			s=filenameHint+".txt";
		else	
			s=filenameHint;
	} else	
		s=QString::null;
	fn = QFileDialog::getSaveFileName( s, "VYM Note (ASCII) (*.txt);;All files (*)", this,"export note dialog",tr("Export Note to single file (ASCII)") );
	int ret=-1;

    if ( !fn.isEmpty() ) 
	{
		QFile file (fn);
		if (file.exists())
		{
			QMessageBox mb( vymName,
				tr("The file %1\nexists already.\nDo you want to overwrite it?","dialog 'save note as'").arg(fn),
			QMessageBox::Warning,
			QMessageBox::Yes | QMessageBox::Default,
			QMessageBox::Cancel | QMessageBox::Escape,
			Qt::NoButton );
			mb.setButtonText( QMessageBox::Yes, tr("Overwrite") );
			mb.setButtonText( QMessageBox::No, tr("Cancel"));
			ret=mb.exec();
		}	
		if (ret==QMessageBox::Cancel)
			return;
			
		// save 
		if ( !file.open( QIODevice::WriteOnly ) ) 
			statusBar()->message( QString("Could not write to %1").arg(filename),
						  statusbarTime );
		else
		{
			QTextStream t( &file );
			t << text;
			file.close();

			statusBar()->message( QString( "Note exported as %1" ).arg( fn ), statusbarTime );
		}
    }
}


void TextEditor::textPrint()
{

    QTextDocument *document = e->document();
    QPrinter printer;

    QPrintDialog *dialog = new QPrintDialog(&printer, this);
    dialog->setWindowTitle(tr("Print Note"));
    if (dialog->exec() != QDialog::Accepted)
        return;

    document->print(&printer);
}

void TextEditor::textEditUndo()
{
}

void TextEditor::toggleFonthint()
{
	setUpdatesEnabled (false);
	e->selectAll ();
	if (!actionFormatUseFixedFont->isOn() ) 
		e->setCurrentFont (varFont);
	else	
		e->setCurrentFont (fixedFont);
	e->selectAll ();
	setUpdatesEnabled (true);
	repaint();
}

void TextEditor::setFixedFont()
{
	bool ok;
	QFont font =QFontDialog::getFont(
                    &ok, fixedFont, this );
    if ( ok ) 
        // font is set to the font the user selected
		fixedFont=font;
}

void TextEditor::setVarFont()
{
	bool ok;
	QFont font =QFontDialog::getFont(
                    &ok, varFont, this );
    if ( ok ) 
        // font is set to the font the user selected
		varFont=font;
}

void TextEditor::textBold()
{
    e->setBold( actionTextBold->isOn() );
}

void TextEditor::textUnderline()
{
    e->setUnderline( actionTextUnderline->isOn() );
}

void TextEditor::textItalic()
{
    e->setItalic( actionTextItalic->isOn() );
}

void TextEditor::textFamily( const QString &f )
{
    e->setFamily( f );
}

void TextEditor::textSize( const QString &p )
{
    e->setPointSize( p.toInt() );
}


void TextEditor::textColor()
{
    QColor col = QColorDialog::getColor( e->color(), this );
    if ( !col.isValid() ) return;
    e->setColor( col );
    QPixmap pix( 16, 16 );
    pix.fill( Qt::black );
    actionTextColor->setIconSet( pix );
}

void TextEditor::textAlign( QAction *a )
{
    if ( a == actionAlignLeft )
		e->setAlignment( Qt::AlignLeft );
    else if ( a == actionAlignCenter )
		e->setAlignment( Qt::AlignHCenter );
    else if ( a == actionAlignRight )
		e->setAlignment( Qt::AlignRight );
    else if ( a == actionAlignJustify )
		e->setAlignment( Qt::AlignJustify );
}

void TextEditor::textVAlign()
{
	QTextCharFormat format;

    if ( sender() == actionAlignSuperScript && actionAlignSuperScript->isOn()) {
		format.setVerticalAlignment(QTextCharFormat::AlignSuperScript);
    } else if (sender() == actionAlignSubScript && actionAlignSubScript->isOn()) {
		format.setVerticalAlignment(QTextCharFormat::AlignSubScript);
    } else {
		format.setVerticalAlignment(QTextCharFormat::AlignNormal);
    }
	e->mergeCurrentCharFormat(format);
}


void TextEditor::fontChanged( const QFont &f )
{
	int i=comboFont->findText(f.family());
    if (i>=0) comboFont->setCurrentIndex (i);
	i=comboSize->findText(QString::number(f.pointSize()));
	if (i>=0) comboSize->setCurrentIndex(i);
    actionTextBold->setOn( f.bold() );
    actionTextItalic->setOn( f.italic() );
    actionTextUnderline->setOn( f.underline() );
}

void TextEditor::colorChanged( const QColor &c )
{
    QPixmap pix( 16, 16 );
    pix.fill( c );
    actionTextColor->setIconSet( pix );
}

void TextEditor::formatChanged( const QTextCharFormat &f )
{
	fontChanged(f.font());
    colorChanged(f.foreground().color());
    alignmentChanged(e->alignment());
	verticalAlignmentChanged (f.verticalAlignment());
}

void TextEditor::alignmentChanged( int a )
{
    if ( ( a == Qt::AlignLeft ) || ( a & Qt::AlignLeft ))
		actionAlignLeft->setOn( true );
    else if ( ( a & Qt::AlignHCenter ) )
		actionAlignCenter->setOn( true );
    else if ( ( a & Qt::AlignRight ) )
		actionAlignRight->setOn( true );
    else if ( ( a & Qt::AlignJustify ) )
		actionAlignJustify->setOn( true );
}

void TextEditor::verticalAlignmentChanged(QTextCharFormat::VerticalAlignment a)
{
	actionAlignSubScript->setOn (false);
	actionAlignSuperScript->setOn (false);
	switch (a)
	{
		case QTextCharFormat::AlignSuperScript: 
			actionAlignSuperScript->setOn (true);
			break;
		case QTextCharFormat::AlignSubScript:
			actionAlignSubScript->setOn (true);
			break;
		default: ;	
	}
}



void TextEditor::enableActions()
{
	actionFileLoad->setEnabled(true);
	actionFileSave->setEnabled(true);
	actionFileSaveAs->setEnabled(true);
	actionFilePrint->setEnabled(true);
	actionEditUndo->setEnabled(true);
	actionEditRedo->setEnabled(true);
	actionEditCopy->setEnabled(true);
	actionEditCut->setEnabled(true);
	actionEditPaste->setEnabled(true);
	actionEditDeleteAll->setEnabled(true);
	actionFormatUseFixedFont->setEnabled(true);
}

void TextEditor::disableActions()
{
	actionFileLoad->setEnabled(false);
	actionFileSave->setEnabled(false);
	actionFileSaveAs->setEnabled(false);
	actionFilePrint->setEnabled(false);
	actionEditUndo->setEnabled(false);
	actionEditRedo->setEnabled(false);
	actionEditCopy->setEnabled(false);
	actionEditCut->setEnabled(false);
	actionEditPaste->setEnabled(false);
	actionEditDeleteAll->setEnabled(false);
	actionFormatUseFixedFont->setEnabled(false);
}

void TextEditor::setState (EditorState s)
{
	
	QPalette p=palette();
	QColor c;
	switch (s)
	{
		case emptyEditor:    c=QColor (150,150,150); break;
		case filledEditor:   c=QColor (255,255,255); break;
		case inactiveEditor: c=QColor (0,0,0);
	}
    p.setColor(QPalette::Active, static_cast<QPalette::ColorRole>(9), c);
    p.setColor(QPalette::Inactive, static_cast<QPalette::ColorRole>(9), c);
    e->setPalette(p);
}


