/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/

#include <iostream>
#include <fstream>
#include <string>
#include <map>
#include <set>
#include <list>
/* -- */
#include <otf.h>
/* -- */
#include "trace/values/Values.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Entitys.hpp"
#include "trace/Trace.hpp"
/* -- */
#include "parser/ParserDefinitionOTF.hpp"
/* -- */
using namespace std;

map<int, Process >       ParserDefinitionOTF::_process;
map<int, ProcessGroup >  ParserDefinitionOTF::_process_group;
map<int, Function >      ParserDefinitionOTF::_function;
map<int, FunctionGroup > ParserDefinitionOTF::_function_group;
map<int, Counter >       ParserDefinitionOTF::_counter;
map<int, CounterGroup >  ParserDefinitionOTF::_counter_group;
map<int, FileSource >    ParserDefinitionOTF::_file_source;
map<int, FileLine >      ParserDefinitionOTF::_file_line;
uint32_t                 ParserDefinitionOTF::_ticks_per_second;

ParserDefinitionOTF::ParserDefinitionOTF() {
    _handlers = OTF_HandlerArray_open();

    // Creation of the empty process
    Process temp_proc = {"0", 0, 0};
    ParserDefinitionOTF::_process[0] = temp_proc;

    // Creation of the empty functionGroup
    FunctionGroup temp_funcgroup = {"no", 0};
    ParserDefinitionOTF::_function_group[0] = temp_funcgroup;

    // Creation of the empty counterGroup
    CounterGroup temp_countergroup = {"no", 0};
    ParserDefinitionOTF::_counter_group[0] = temp_countergroup;

}

ParserDefinitionOTF::~ParserDefinitionOTF() {
    OTF_HandlerArray_close(_handlers);

    // \todo check if we have to delete it?
    //delete ParserDefinitionOTF::_process_group[0]._procs;

    ParserDefinitionOTF::_process.erase(ParserDefinitionOTF::_process.begin(),
                                        ParserDefinitionOTF::_process.end());
    ParserDefinitionOTF::_process_group.erase(ParserDefinitionOTF::_process_group.begin(), ParserDefinitionOTF::_process_group.end());
    ParserDefinitionOTF::_function.erase(ParserDefinitionOTF::_function.begin(),
                                         ParserDefinitionOTF::_function.end());
    ParserDefinitionOTF::_function_group.erase(ParserDefinitionOTF::_function_group.begin(),
                                         ParserDefinitionOTF::_function_group.end());
    ParserDefinitionOTF::_counter.erase(ParserDefinitionOTF::_counter.begin(),
                                        ParserDefinitionOTF::_counter.end());
    ParserDefinitionOTF::_counter_group.erase(ParserDefinitionOTF::_counter_group.begin(),
                                              ParserDefinitionOTF::_counter_group.end());
}

void ParserDefinitionOTF::set_handlers(Trace *t) {
    SET_HANDLER(_handlers, handler_timer_resolution, t, OTF_DEFTIMERRESOLUTION_RECORD);
    SET_HANDLER(_handlers, handler_process, t, OTF_DEFPROCESS_RECORD);
    SET_HANDLER(_handlers, handler_process_group, t, OTF_DEFPROCESSGROUP_RECORD);
    SET_HANDLER(_handlers, handler_function, t, OTF_DEFFUNCTION_RECORD);
    SET_HANDLER(_handlers, handler_function_group, t, OTF_DEFFUNCTIONGROUP_RECORD);
    SET_HANDLER(_handlers, handler_counter, t, OTF_DEFCOUNTER_RECORD);
    SET_HANDLER(_handlers, handler_counter_group, t, OTF_DEFCOUNTERGROUP_RECORD);
    SET_HANDLER(_handlers, handler_file_source_line, t, OTF_DEFSCL_RECORD);
    SET_HANDLER(_handlers, handler_file_source, t, OTF_DEFSCLFILE_RECORD);
}

int ParserDefinitionOTF::handler_timer_resolution (void *, uint32_t, uint32_t ticks_per_sec) {
    ParserDefinitionOTF::_ticks_per_second = ticks_per_sec;
    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_process (void *, uint32_t stream, uint32_t proc_id, const char *name, uint32_t parent) {

    Process temp = {name, parent, stream};
    ParserDefinitionOTF::_process[proc_id] = temp;

    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_process_group (void *, uint32_t stream, uint32_t procGroup, const char *name, uint32_t numberOfProcs, const uint32_t *procs) {

    ProcessGroup temp = {name, numberOfProcs, stream, (uint32_t *)procs};
    ParserDefinitionOTF::_process_group[procGroup] = temp;

    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_function(void *, uint32_t stream, uint32_t func_id, const char *name, uint32_t funcGroup, uint32_t file_id) {

    Function temp = {name, stream, funcGroup, file_id, false};
    ParserDefinitionOTF::_function[func_id] = temp;

    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_function_group(void *, uint32_t stream, uint32_t func_group_id, const char *name) {
    FunctionGroup temp = {name, stream};
    ParserDefinitionOTF::_function_group[func_group_id] = temp;
    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_counter(void *, uint32_t stream, uint32_t counter_id, const char* name, uint32_t properties, uint32_t countergroup, const char* unit) {
    Counter temp = {name, stream, properties, countergroup, unit};
    ParserDefinitionOTF::_counter[counter_id] = temp;
    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_counter_group(void *, uint32_t stream, uint32_t counter_group_id, const char *name) {
    CounterGroup temp = {name, stream};
    ParserDefinitionOTF::_counter_group[counter_group_id] = temp;
    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_file_source(void *, uint32_t stream, uint32_t file_id, const char* name) {
    FileSource temp = {name, stream};
    ParserDefinitionOTF::_file_source[file_id] = temp;
    return OTF_RETURN_OK;
}

int ParserDefinitionOTF::handler_file_source_line(void *, uint32_t stream, uint32_t line_id, uint32_t file_id, uint32_t line_number) {
    FileLine temp = {file_id, line_number, stream};
    ParserDefinitionOTF::_file_line[line_id] = temp;
    return OTF_RETURN_OK;
}

void ParserDefinitionOTF::read_definitions(OTF_Reader *reader) {
    OTF_Reader_readDefinitions(reader, _handlers);
}

Function &ParserDefinitionOTF::get_function_by_id(const int id) {
    return ParserDefinitionOTF::_function[id];
}

FunctionGroup ParserDefinitionOTF::get_function_group_by_id(const int id) {
    return ParserDefinitionOTF::_function_group[id];
}

Counter ParserDefinitionOTF::get_counter_by_id(const int id) {
    return ParserDefinitionOTF::_counter[id];
}

CounterGroup ParserDefinitionOTF::get_counter_group_by_id(const int id) {
    return ParserDefinitionOTF::_counter_group[id];
}

Process ParserDefinitionOTF::get_process_by_id(const int id) {
    return ParserDefinitionOTF::_process[id];
}

ProcessGroup ParserDefinitionOTF::get_processgroup_by_id(const int id) {
    return ParserDefinitionOTF::_process_group[id];
}

ProcessGroup ParserDefinitionOTF::get_processgroup_by_process(const unsigned int process_id) {
    for (map<int, ProcessGroup>::const_iterator it = _process_group.begin() ; it != _process_group.end() ; ++ it) {
        ProcessGroup proc_group_temp = (*it).second;
        for(unsigned int i = 0 ; i < proc_group_temp._numberOfProcs ; i ++) {
            if(proc_group_temp._procs[i] == process_id) {
                return proc_group_temp;
            }
        }
    }
    return ParserDefinitionOTF::_process_group[0];
}



FileSource ParserDefinitionOTF::get_filesource_by_id(const int id) {
    return ParserDefinitionOTF::_file_source[id];
}

FileLine ParserDefinitionOTF::get_fileline_by_id(const int id) {
    return ParserDefinitionOTF::_file_line[id];
}

uint32_t ParserDefinitionOTF::get_ticks_per_second() {
    return ParserDefinitionOTF::_ticks_per_second;
}

void ParserDefinitionOTF::create_container_types(Trace *t) {
    // We add in the processGroup '0' all the process that don't have a processGroup
    set <int> all_process;
    // We add all the process
    for (map<int, Process>::const_iterator it = _process.begin() ; it != _process.end() ; ++ it) {
        all_process.insert((*it).first);
    }

    // We remove process which have a parent
    for (map<int, ProcessGroup>::const_iterator it = _process_group.begin() ; it != _process_group.end() ; ++ it) {
        ProcessGroup proc_group_temp = (*it).second;
        for(unsigned int i = 0 ; i < proc_group_temp._numberOfProcs ; i ++) {
            all_process.erase(proc_group_temp._procs[i]);
        }
    }
    // Creation of the processGroup 0
    const int size = all_process.size();
    ProcessGroup temp;
    temp._name = "no";
    temp._stream = 0;
    temp._numberOfProcs = size;
    temp._procs = new uint32_t[size+1];
    int i = 0;
    for (set<int>::const_iterator it = all_process.begin() ; it != all_process.end() ; ++ it, ++ i) {
        temp._procs[i] = *it;
    }
    ParserDefinitionOTF::_process_group[0] = temp;

    // Create the container types
    for (map<int, ProcessGroup>::const_iterator it = _process_group.begin() ; it != _process_group.end() ; ++ it) {
        ProcessGroup proc_group_temp = (*it).second;
        Name name_temp(proc_group_temp._name, "");
        map<std::string, Value *> extra_fields;
        t->define_container_type(name_temp, NULL, extra_fields);
    }
}


// Debug purposes
void ParserDefinitionOTF::print_definitions() {
    cout << "Process:" << endl;
    for (map<int, Process>::const_iterator it = _process.begin() ; it != _process.end() ; ++ it) {
        cout << "#" << (*it).first;
        Process temp = (*it).second;
        cout << " name: '" << temp._name << "' parent:'" << temp._parent << "' stream: '" << temp._stream << "'" << endl;
    }

    cout << "ProcessGroup:" << endl;
    for (map<int, ProcessGroup>::const_iterator it = _process_group.begin() ; it != _process_group.end() ; ++ it) {
        cout << "#" << (*it).first;
        ProcessGroup temp = (*it).second;
        cout << " name: '" << temp._name << "' stream: '" << temp._stream << "'" << endl << " Procs: ";
        for(unsigned int i = 0 ; i < temp._numberOfProcs ; i ++) {
            cout << temp._procs[i] << " ";
        }
        cout << endl;
    }

    cout << "Functions:" << endl;
    for (map<int, Function>::const_iterator it = _function.begin() ; it != _function.end() ; ++ it) {
        cout << "#" << (*it).first;
        Function temp = (*it).second;
        cout << " name: '" << temp._name << "' stream: '" << temp._stream << "' parent:'" << temp._func_group << "' file_source:'" << temp._file_source << "'" << endl;
    }

    cout << "Function groups:" << endl;
    for (map<int, FunctionGroup>::const_iterator it = _function_group.begin() ; it != _function_group.end() ; ++ it) {
        cout << "#" << (*it).first;
        FunctionGroup temp = (*it).second;
        cout << " name: '" << temp._name << "' stream: '" << temp._stream << "'" << endl;
    }

    cout << "Counters:" << endl;
    for (map<int, Counter>::const_iterator it = _counter.begin() ; it != _counter.end() ; ++ it) {
        cout << "#" << (*it).first;
        Counter temp = (*it).second;
        cout << " name: '" << temp._name << "' stream: '" << temp._stream << "'" << "' unit: '" << temp._unit << "'" << endl;
    }

    cout << "Counter groups:" << endl;
    for (map<int, CounterGroup>::const_iterator it = _counter_group.begin() ; it != _counter_group.end() ; ++ it) {
        cout << "#" << (*it).first;
        CounterGroup temp = (*it).second;
        cout << " name: '" << temp._name << "' stream: '" << temp._stream << "'" << endl;
    }

}
