/* -*- C++ -*-
 
  This file is part of ViPEC
  Copyright (C) 1991-2001 Johan Rossouw (jrossouw@alcatel.altech.co.za)
 
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
 
  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 
*/

#include <Block2Port.h>

#include <Strings.h>
#include <Setup.h>
#include <Utils.h>
#include <Exception.h>
#include <VectorFont.h>
#include <MainWindow.h>

#include "../images/block2port.xpm"

#include <qpoint.h>

QPixmap* Block2Port::pixmap_ = 0;

//-----------------------------------------------------------------
Block2Port::Block2Port( const QPoint& center )
    : Component( center ),
    schematicData_(0),
    fileData_(0)
{}

//-----------------------------------------------------------------
Block2Port::~Block2Port()
{}

//-----------------------------------------------------------------
const QString& Block2Port::getCatagory() const
  {
    return Strings::BlockCatagory;
  }

//-----------------------------------------------------------------
const QString& Block2Port::getName() const
  {
    return Strings::Block2PortName;
  }

//-----------------------------------------------------------------
const QPixmap& Block2Port::getPixmap() const
  {
    if (!pixmap_)
      {
        pixmap_ = new QPixmap(block2port_xpm);
      }
    return *pixmap_;
  }

//-----------------------------------------------------------------
void Block2Port::initComponent()
{
  boundingRect_ = QRect(-32, -24, 64, 48);
  addNode(-32,-8 );
  addNode( 32,-8 );
  addAttribute( Strings::AttrBlockname, "NAME" );
}

//-----------------------------------------------------------------
void Block2Port::drawSymbol(QPainter* p)
{
  p->drawRect(-16,-24,32,32);
  p->drawLine(-32,-8,-16,-8);
  p->drawLine(16,-8,32,-8);
  p->moveTo(0,8);
  p->lineTo(0,24);
  p->moveTo(-2,22);
  p->lineTo(2,22);
  p->moveTo(-4,20);
  p->lineTo(4,20);
  p->moveTo(-6,18);
  p->lineTo(6,18);
  p->save();
  p->setPen( Qt::black );
  QPoint point(-14,-4);
  VectorFont::instance()->draw('1', p, point );
  point.setX(8);
  VectorFont::instance()->draw('2', p, point );
  p->restore();
}

//-----------------------------------------------------------------
bool Block2Port::initSweep()
{
  QString name = name_;
  Attribute* attr = findAttribute( "NAME" );
  ASSERT( attr );
  name_ = attr->value_;
  bool changed = ( name == name_ );
  //If data is from file, trash it. We do not check for file
  //updates at the moment
  if ( fileData_ )
    {
      changed = TRUE;
      fileData_ = 0;
    }
  schematicData_ = 0;
  return changed;
}

//-----------------------------------------------------------------
void Block2Port::addToAdmittanceMatrix( TReal freq, Matrix* yn )
{
  if ( (schematicData_ == 0) && (fileData_ == 0))
    {
      Schematic* schematic = 0;
      schematic = MainWindow::instance()->getSchematic( name_ );
      if ( schematic )
        {
          if ( !schematic->isSolved() )
            {
              Vector& frequencies = Setup::instance()->getFrequencyVector();
              schematic->sweep( frequencies );
            }
          if ( schematic->getNumberOfPorts() != 2)
            {
              throw Exception::NumberOfPortsDoesNotMatch();
            }
          schematicData_ = &(schematic->getYData());
        }
      else
        {
          //Not a schematic, so try to find a file with this name
          fileData_ = MainWindow::instance()->getFileData( name_ );
          if ( fileData_ == 0 )
            {
              throw Exception::NoSuchBlock();
            }
        }
    }
  ASSERT( (schematicData_ != 0) || (fileData_ != 0) );
  //Find data in set
  Matrix data;
  if ( schematicData_ != 0)
    {
      Utils::findParametersInData( freq, *schematicData_, data );
    }
  else
    {
      Utils::findParametersInData( freq, *fileData_, data );
    }
  //Add to nodal matrix
  uint nodes [4];
  nodes[0] = node(0)->getNodeNumber();
  nodes[1] = 0;
  nodes[2] = node(1)->getNodeNumber();
  nodes[3] = 0;
  yn->insertTwoPort(data, nodes);
}
