#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <GL/gl.h>
#include <GL/glu.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <gtkgl/gtkglarea.h>

#include "../include/string.h"

#include "guiutils.h"

#include "printwin.h"
#include "printwincb.h"

#include "vmautils.h"

#ifdef MEMWATCH
# include "memwatch.h"
#endif


#include "images/icon_print_48x48.xpm"
#include "images/icon_print_20x20.xpm"
#include "images/icon_cancel_20x20.xpm"


gint PrintWinGetSourceType(print_win_struct *pw, const gchar *name);
gint PrintWinGetOrientation(print_win_struct *pw);
gint PrintWinGetVisual(print_win_struct *pw);
void PrintWinGetPaperSizeFromCode(
	print_win_struct *pw, gint paper_size_code,
	gint *width, gint *height
);
void PrintWinGetPaperSize(
	print_win_struct *pw,
	gint *width, gint *height
);
void PrintWinGetPToWCoeff(
	print_win_struct *pw,
	gdouble *x_coeff, gdouble *y_coeff
);

void PrintWinPreviewSetSize(
	print_win_struct *pw,
	gint paper_wp, gint paper_hp
);

void PrintWinPreviewDraw(print_win_struct *pw);
gint PrintWinPreviewTextureLoadGreyScale(
	print_win_struct *pw,
	gint width, gint height,
	const guint8 *data_ptr
);
gint PrintWinPreviewTextureLoadColor(
	print_win_struct *pw,
	gint width, gint height,
	const guint8 *data_ptr
);
void PrintWinPreviewTextureUnload(print_win_struct *pw);

void PrintWinDoTranslate(
	print_win_struct *pw,
	gint dx, gint dy
);
void PrintWinDoScale(
	print_win_struct *pw,
	gint dx, gint dy
);

print_win_struct *PrintWinNew(gpointer core_ptr);
gint PrintWinPreviewEnableContext(print_win_struct *pw);
void PrintWinSetCallbacks(
	print_win_struct *pw,
	gpointer client_data,
	void (*print_cb)(
		gpointer, gpointer, print_win_parms_struct *
	),
	void (*cancel_cb)(
		gpointer, gpointer
	),
	void (*preview_request_cb)(
		gpointer, gpointer, gint, gint, gint, gint
	)
);
void PrintWinSetSourceTypeNames(
	print_win_struct *pw, gchar **name, gint total_names
);
void PrintWinSetBusy(print_win_struct *pw);
void PrintWinSetReady(print_win_struct *pw);
void PrintWinMap(print_win_struct *pw);
void PrintWinMapValues(
	print_win_struct *pw, print_win_parms_struct *parms
);
void PrintWinUnmap(print_win_struct *pw);
void PrintWinReset(print_win_struct *pw, gbool need_unmap);
void PrintWinDelete(print_win_struct *pw);


static gint gl_attributes_list[] = {
	GDK_GL_RGBA,
	GDK_GL_DOUBLEBUFFER,
	GDK_GL_DEPTH_SIZE,      1,
	GDK_GL_NONE
};


#define PRINT_WIN_DEF_WIDTH	640
#define PRINT_WIN_DEF_HEIGHT	480


/* Length of preview glarea widget's longest dimension, note that
 * this really sets the preview_fixed widget. There are other widgets
 * within the preview_fixed (like the frame) that would make the
 * preview_glarea a bit smaller than this value.
 */
#define PRINT_WIN_DEF_PREVIEW_LENGTH	200

/* Default paper size in pixels. */
#define PRINT_WIN_DEF_PAPER_WIDTHP	850	/* 8.5 ppi */
#define PRINT_WIN_DEF_PAPER_HEIGHTP	1100	/* 11 ppi */

/* Default output minimum size in pixels. */
#define PRINT_WIN_OUTPUT_MIN_WIDTHP	2
#define PRINT_WIN_OUTPUT_MIN_HEIGHTP	2


/* Tooltips. */
#define PRINT_WIN_TT_SOURCE_TYPE	"\
Specifies the origin of the source data to be printed"

#define PRINT_WIN_TT_PRINT_COMMAND	"\
Specifies the command to run the print client, any occurance of \
%f will be replaced with the tempory postscript output file name"

#define PRINT_WIN_TT_PPI_X		"\
Specifies the pixels per inch (PPI) along the width of print,\
 this value must match the value set on your printer driver"

#define PRINT_WIN_TT_PPI_Y		"\
Specifies the pixels per inch (PPI) along the height of print,\
 this value must match the value set on your printer driver"

#define PRINT_WIN_TT_PREVIEW		"\
Holding Button1 while dragging adjusts offset, holding SHIFT+Button1\
 (or Button2) while dragging adjusts scale"


/*
 *	Returns the currently selected source type matching the given
 *	name from the print window's source_type_combo list.
 *
 *	If name is NULL, then the name will be retrieved from the
 *	source_type_combo's entry widget.
 *
 *	Can return -1 on error.
 */
gint PrintWinGetSourceType(print_win_struct *pw, const gchar *name)
{
	gint i;
	GtkWidget *w;
	GtkCombo *combo;
	GtkEntry *entry;
	GList *glist;


	if(pw == NULL)
	    return(-1);

	w = pw->source_type_combo;
	if(w == NULL)
	    return(-1);

	combo = GTK_COMBO(w);
	entry = GTK_ENTRY(combo->entry);

	if(name == NULL)
	    name = (const gchar *)gtk_entry_get_text(entry);
	if(name == NULL)
	    return(-1);

	glist = GUIComboGetList(GTK_WIDGET(combo));

	for(i = 0; glist != NULL; i++)
	{
	    if(glist->data != NULL)
	    {
		if(!strcmp((char *)glist->data, name))
		    return(i);
	    }

	    glist = glist->next;
	}

	return(-1);
}

/*
 *	Returns the orientation value of the printer window.
 *
 *	Returns 0 for `portrait' or 1 for `landscape'.
 */
gint PrintWinGetOrientation(print_win_struct *pw)
{
	GtkWidget *w;

	if(pw == NULL)
	    return(0);

	w = pw->orient_landscape_radio;
	if(w == NULL)
	    return(0);

	if(GTK_TOGGLE_BUTTON(w)->active)
	    return(1);
	else
	    return(0);
}

/*
 *	Returns the visual (color or greyscale).
 */
gint PrintWinGetVisual(print_win_struct *pw)
{
	GtkWidget *w;
	
	if(pw == NULL)
	    return(PRINT_WIN_VISUAL_GREYSCALE);

	w = pw->print_color_radio;
	if(w == NULL)
	    return(PRINT_WIN_VISUAL_GREYSCALE);

	if(GTK_TOGGLE_BUTTON(w)->active)
	    return(PRINT_WIN_VISUAL_COLOR);
	else
	    return(PRINT_WIN_VISUAL_GREYSCALE);
}

/*
 *	Returns the paper size (in units of pixels) with respect to the
 *	given paper size code (one of PRINT_WIN_PAPER_SIZE_*).
 */
void PrintWinGetPaperSizeFromCode(
	print_win_struct *pw, gint paper_size_code,
	gint *width, gint *height
)
{
	gdouble ppi = 100.0;


	if(width != NULL)
	    (*width) = PRINT_WIN_DEF_PAPER_WIDTHP;
	if(height != NULL)
	    (*height) = PRINT_WIN_DEF_PAPER_HEIGHTP;


	if(pw == NULL)
	    return;

	switch(paper_size_code)
	{
	  case PRINT_WIN_PAPER_SIZE_LETTER:
	    if(width != NULL)
		(*width) = (gint)(8.5 * ppi);
	    if(height != NULL)
		(*height) = (gint)(11.0 * ppi);
	    break;

	  case PRINT_WIN_PAPER_SIZE_LEGAL:
	    if(width != NULL)
		(*width) = (gint)(8.5 * ppi);
	    if(height != NULL)
		(*height) = (gint)(14.0 * ppi);
	    break;

	  case PRINT_WIN_PAPER_SIZE_EXECUTIVE:
	    if(width != NULL)
		(*width) = (gint)(7.5 * ppi);
	    if(height != NULL)
		(*height) = (gint)(10.0 * ppi);
	    break;

	  case PRINT_WIN_PAPER_SIZE_A4:
	    /* 210 mm */
	    if(width != NULL)
		(*width) = (gint)(8.2677165 * ppi);
	    /* 297 mm */
	    if(height != NULL)
		(*height) = (gint)(11.692913 * ppi);
	    break;
	}

	return;
}

/*
 *	Returns the paper width and height in pixels.
 */
void PrintWinGetPaperSize(
	print_win_struct *pw,
	gint *width, gint *height
)
{
	gint paper_size_code = PRINT_WIN_PAPER_SIZE_LETTER;
	GtkWidget *w;

	if(width != NULL)
	    (*width) = PRINT_WIN_DEF_PAPER_WIDTHP;
	if(height != NULL)
	    (*height) = PRINT_WIN_DEF_PAPER_HEIGHTP;

	if(pw == NULL)
	    return;

	/* Letter size? */
	w = pw->paper_size_letter_radio;
	if(w != NULL)
	{
	    if(GTK_TOGGLE_BUTTON(w)->active)
		paper_size_code = PRINT_WIN_PAPER_SIZE_LETTER;
	}
	/* Legal size? */
	w = pw->paper_size_legal_radio;
	if(w != NULL)
	{
	    if(GTK_TOGGLE_BUTTON(w)->active)
		paper_size_code = PRINT_WIN_PAPER_SIZE_LEGAL;
	}
	/* Executive size? */
	w = pw->paper_size_executive_radio;
	if(w != NULL)
	{
	    if(GTK_TOGGLE_BUTTON(w)->active)
		paper_size_code = PRINT_WIN_PAPER_SIZE_EXECUTIVE;
	}
	/* A4 size? */
	w = pw->paper_size_a4_radio;
	if(w != NULL)
	{
	    if(GTK_TOGGLE_BUTTON(w)->active)
		paper_size_code = PRINT_WIN_PAPER_SIZE_A4;
	}

	/* Now that we have the paper size code, match it up with
	 * the correct actual paper size in pixels.
	 */
	PrintWinGetPaperSizeFromCode(
	    pw, paper_size_code,
	    width, height
	);

	return;
}

/*
 *	Calculates the conversion coefficients based on the given
 *	print window's values for pixels to preview widget window
 *	coordinate units.
 */
void PrintWinGetPToWCoeff(
	print_win_struct *pw, 
	gdouble *x_coeff, gdouble *y_coeff  
)
{
	gint ww, wh;
	gint paper_wp, paper_hp;
	GtkWidget *w;


	if(x_coeff != NULL)
	    (*x_coeff) = 0.0;
	if(y_coeff != NULL)
	    (*y_coeff) = 0.0;

	if(pw == NULL)
	    return;

	/* Get size of paper. */
	PrintWinGetPaperSize(
	    pw, &paper_wp, &paper_hp
	);

	w = pw->preview_glarea;
	if(w == NULL)
	    return;

	/* Get size of preview glarea widget in window coordinates. */
	ww = w->allocation.width;
	wh = w->allocation.height;

	/* Calculate pixels to window coordinates conversion coefficient. */
	if((paper_hp > 0) && (x_coeff != NULL))
	    (*x_coeff) = (gdouble)ww / (gdouble)paper_wp;
	if((paper_wp > 0) && (y_coeff != NULL))
	    (*y_coeff) = (gdouble)wh / (gdouble)paper_hp;

	return;
}


/*
 *	Updates the preview glarea widget size on the given print window
 *	based on the given paper size in pixels.
 */
void PrintWinPreviewSetSize(
	print_win_struct *pw,
	gint paper_wp, gint paper_hp
)
{
	GtkWidget *w;
	gdouble aspect;


	if(pw == NULL)
	    return;
	if(!pw->initialized)
	    return;

	if(paper_wp < 2)
	    paper_wp = 2;
	if(paper_hp < 2)
	    paper_hp = 2;


	w = pw->preview_glarea;
	if(w != NULL)
	{
	    gint new_w, new_h;

	    /* Check which dimension is longer. */
	    if(paper_wp > paper_hp)
	    {
		/* Width is longer dimension. */

		aspect = (gdouble)paper_hp / (gdouble)paper_wp;
		new_w = PRINT_WIN_DEF_PREVIEW_LENGTH;
		new_h = PRINT_WIN_DEF_PREVIEW_LENGTH * aspect;
	    }
	    else
	    {
		/* Height is longer or equal dimension. */

		aspect = (gdouble)paper_wp / (gdouble)paper_hp;
		new_w = PRINT_WIN_DEF_PREVIEW_LENGTH * aspect;
		new_h = PRINT_WIN_DEF_PREVIEW_LENGTH;
	    }

	    /* Set new size. */
	    gtk_widget_set_usize(w, new_w, new_h);
	    gtk_widget_queue_resize(pw->toplevel);
	}

	return;
}

/*
 *	Redraws print window's the preview glarea widget.
 *
 *	The preview glarea widget must be realized, it will be out into
 *	GL context by this function.
 */
void PrintWinPreviewDraw(print_win_struct *pw)
{
	gint orientation;
	gint output_offset_xp, output_offset_yp, output_wp, output_hp;
	gint paper_wp, paper_hp;
	gdouble ptow_x_coeff = 0.0, ptow_y_coeff = 0.0;
	GtkWidget *w;
	GtkGLArea *glarea;
	gint ww, wh;


	if(pw == NULL)
	    return;

	if(!pw->initialized || !pw->preview_realized)
	    return;

	/* Get preview glarea widget. */
	w = pw->preview_glarea;
	if(w == NULL)
	    return;

	glarea = GTK_GL_AREA(w);

	/* Get orientation. */
	orientation = PrintWinGetOrientation(pw);

	/* We'll be using two types of units here, window coordinates
	 * (the actual size of the glarea widget) and pixels (the
	 * `world space unit').
	 */

	/* Get size of glarea widget in window coordinates. */
	ww = (gint)w->allocation.width;
	wh = (gint)w->allocation.height;
	if((ww <= 0) || (wh <= 0))
	    return;

	/* The size of the glarea widget in window coordinates represents
	 * the size of the paper in pixels.
	 */

	/* Get output offset and size in pixels. */
	output_offset_xp = pw->output_offset_x;
	output_offset_yp = pw->output_offset_y;
	output_wp = pw->output_width;
	output_hp = pw->output_height;

	/* Get paper size in pixels. */
	PrintWinGetPaperSize(
	    pw, &paper_wp, &paper_hp
	);

	/* Calculate pixel to window coefficient. */
	PrintWinGetPToWCoeff(pw, &ptow_x_coeff, &ptow_y_coeff);


	/* Put preview glarea widget into GL context. */
	if(PrintWinPreviewEnableContext(pw))
	    return;

	/* Set up GL for simple 2D drawing in window coordinates. */
	glViewport(0, 0, ww, wh);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();

	/* Go to 2D and make units the same as the size of the window. */
	gluOrtho2D(0, ww, wh, 0);
	glMatrixMode(GL_MODELVIEW);

	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	/* Update states. */
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	glDisable(GL_LIGHT0);
	glDisable(GL_TEXTURE_1D);
	glDisable(GL_COLOR_MATERIAL);
	glDisable(GL_TEXTURE_3D);

	/* Clear background. */
	glClearColor(1.0, 1.0, 1.0, 0.0);
	glClearDepth(1.0);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);


	/* Preview texture valid? */
	if(pw->preview_texture > 0)
	{
	    gint x, y, width, height;
	    GLuint tex_num = pw->preview_texture;

	    glEnable(GL_TEXTURE_2D);
	    glBindTexture(GL_TEXTURE_2D, tex_num);
	    glColor4f(1.0, 1.0, 1.0, 1.0);

	    /* Calculate bounds, x and y offset are always upper left
	     * corner oriented.
	     */
	    x = output_offset_xp * ptow_x_coeff;
	    y = output_offset_yp * ptow_y_coeff;

	    /* Calculate size bounds by current orientation. */
	    if(orientation)
	    {
		/* Landscape. */
		width = output_hp * ptow_x_coeff;
		height = output_wp * ptow_y_coeff;

		glBegin(GL_QUADS);
		{
		    glTexCoord2i(1, 1);
		    glVertex2i(x, y);
		    glTexCoord2i(1, 0);
		    glVertex2i(x + width, y);
		    glTexCoord2i(0, 0);
		    glVertex2i(x + width, y + height);
		    glTexCoord2i(0, 1);
		    glVertex2i(x, y + height);
		}
		glEnd();
	    }
	    else
	    {
		/* Portrait. */

		width = output_wp * ptow_x_coeff;
		height = output_hp * ptow_y_coeff;

		glBegin(GL_QUADS);
		{
		    glTexCoord2i(0, 1);  
		    glVertex2i(x, y);
		    glTexCoord2i(1, 1);
		    glVertex2i(x + width, y);
		    glTexCoord2i(1, 0);
		    glVertex2i(x + width, y + height);
		    glTexCoord2i(0, 0);
		    glVertex2i(x, y + height);
		}
		glEnd();
	    }
	}


	/* Draw black outline. */
	if(1)
	{
	    gint x, y, width, height;

	    /* Calculate bounds, x and y offset are always upper left
	     * corner oriented.
	     */
	    x = output_offset_xp * ptow_x_coeff;
	    y = output_offset_yp * ptow_y_coeff;

	    /* Calculate size bounds by current orientation. */
	    if(orientation)
	    {
		width = output_hp * ptow_x_coeff;
		height = output_wp * ptow_y_coeff;
	    }
	    else
	    {
		width = output_wp * ptow_x_coeff;
		height = output_hp * ptow_y_coeff;
	    }


	    glDisable(GL_TEXTURE_2D);
	    glColor4f(0.0, 0.0, 0.0, 1.0);

	    glBegin(GL_LINE_LOOP);
	    {
		glVertex2i(x, y);
		glVertex2i(x + width, y);
		glVertex2i(x + width, y + height);
		glVertex2i(x, y + height);
	    }
	    glEnd();
	}

	/* Make glarea rendered buffer active. */
	gtk_gl_area_swapbuffers(glarea);

	/* Check for GL errors. */
	if(1)
	{
	    GLenum error_code = glGetError();
	    if(error_code != GL_NO_ERROR)
		VMAReportGLError(NULL, (int)error_code);
	}

	return;
}

/*
 *      Loads the given data as a texture on the print window's preview
 *      widget. If a texture is already loaded, then it will be unloaded.
 *
 *      The data type for the given data_ptr must be in unsigned 8 bit
 *      GL_LUMINANCE format. If data_ptr is NULL then this is the equvilent
 *      of calling PrintWinPreviewTextureUnload().
 *
 *      Returns non-zero on error.
 */
gint PrintWinPreviewTextureLoadGreyScale(
	print_win_struct *pw,
	gint width, gint height,
	const guint8 *data_ptr
)
{
	GLint status;
	GLvoid *new_data_ptr;
	GLsizei new_width = 256, new_height = 256, bytes_per_pixel = 1;
	GLuint gl_texture_id;


	if(pw == NULL)
	    return(-1);

	/* Put preview glarea widget into GL context and unload
	 * current texture if any.
	 */
	PrintWinPreviewTextureUnload(pw);

	if(data_ptr == NULL)
	    return(-1);

	/* Allocate local data buffer for a rescaled version of the
	 * given data_ptr.
	 */
	new_data_ptr = (GLvoid *)malloc(
	    new_width * new_height * bytes_per_pixel
	);
	if(new_data_ptr == NULL)
	{
	    /* Memory allocation error. */
	    return(-1);
	}

	/* Scale image data. */ 
	status = gluScaleImage(
	    GL_LUMINANCE,
	    (GLsizei)width, (GLsizei)height,
	    GL_UNSIGNED_BYTE,
	    (const void *)data_ptr,
	    new_width, new_height,
	    GL_UNSIGNED_BYTE,
	    new_data_ptr
	);
	if(status != 0)
	{
	    fprintf(
		stderr,
 "PrintWinPreviewTextureLoadGreyScale(): gluScaleImage(): Error: %s\n",
		(const char *)gluErrorString(status)
	    );
	}

	/* Generate a new texture ID. */
	glGenTextures(1, &gl_texture_id);
	if(gl_texture_id == 0)
	{
	    fprintf(
		stderr,
 "PrintWinPreviewTextureLoadGreyScale(): Error generating texture.\n"
	    );

	    free(new_data_ptr);
	    new_data_ptr = NULL;

	    return(-1);
	}

	/* Record new texture. */
	pw->preview_texture = gl_texture_id;

	/* Actually generate texture and set texture into GL
	 * context.
	 */
	glBindTexture(GL_TEXTURE_2D, gl_texture_id);

	/* Set parameters of selected texture. */
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(
	    GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST
	);
	glTexParameteri(
	    GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST
	);
	glTexParameterf(
	    GL_TEXTURE_2D, GL_TEXTURE_PRIORITY, 1.0
	);

	glTexImage2D(
	    GL_TEXTURE_2D,      /* GL_TEXTURE_2D or GL_PROXY_TEXTURE_2D. */
	    0,                  /* Level. */
	    GL_LUMINANCE,	/* Internal format. */
	    new_width,          /* Width. */
	    new_height,         /* Height (same as width). */
	    0,                  /* Border. */
	    GL_LUMINANCE,       /* Image data format. */
	    GL_UNSIGNED_BYTE,   /* Image data type. */ 
	    /* Pointer to image data. */
	    (const GLvoid *)new_data_ptr
	);

	glFlush();

	/* Deallocate new data, it is no longer needed. */
	free(new_data_ptr);
	new_data_ptr = NULL;

	return(0);
}

/*
 *	Loads the given data as a texture on the print window's preview
 *	widget. If a texture is already loaded, then it will be unloaded.
 *
 *	The data type for the given data_ptr must be in unsigned 8 bit
 *	GL_RGBA format. If data_ptr is NULL then this is the equvilent
 *	of calling PrintWinPreviewTextureUnload().
 *
 *	Returns non-zero on error.
 */
gint PrintWinPreviewTextureLoadColor(
	print_win_struct *pw,
	gint width, gint height,
	const guint8 *data_ptr
)
{
	GLint status;
	GLvoid *new_data_ptr;
	GLsizei new_width = 256, new_height = 256, bytes_per_pixel = 4;
	GLuint gl_texture_id;


	if(pw == NULL)
	    return(-1);

	/* Put preview glarea widget into GL context and unload
	 * current texture if any.
	 */
	PrintWinPreviewTextureUnload(pw);

	if(data_ptr == NULL)
	    return(-1);

	/* Allocate local data buffer for a rescaled version of the
	 * given data_ptr.
	 */
	new_data_ptr = (GLvoid *)malloc(
	    new_width * new_height * bytes_per_pixel
	);
	if(new_data_ptr == NULL)
	{
	    /* Memory allocation error. */
	    return(-1);
	}

	/* Scale image data. */
	status = gluScaleImage(
	    GL_RGBA,
	    (GLsizei)width, (GLsizei)height,
	    GL_UNSIGNED_BYTE,
	    (const void *)data_ptr,
	    new_width, new_height,
	    GL_UNSIGNED_BYTE,
	    new_data_ptr
	);
	if(status != 0)
	{
	    fprintf(
		stderr,
 "PrintWinPreviewTextureLoadColor(): gluScaleImage(): Error: %s\n",
		(const char *)gluErrorString(status)
	    );
	}

	/* Generate a new texture ID. */
	glGenTextures(1, &gl_texture_id);
	if(gl_texture_id == 0)
	{
	    fprintf(
		stderr,
 "PrintWinPreviewTextureLoadColor(): Error generating texture.\n"
	    );

	    free(new_data_ptr);
	    new_data_ptr = NULL;

	    return(-1);
	}

	/* Record new texture. */
	pw->preview_texture = gl_texture_id;

	/* Actually generate texture and set texture into GL
	 * context.
	 */
	glBindTexture(GL_TEXTURE_2D, gl_texture_id);

	/* Set parameters of selected texture. */
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	glTexParameteri(
	    GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST
	);
	glTexParameteri(
	    GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST
	);
	glTexParameterf(
	    GL_TEXTURE_2D, GL_TEXTURE_PRIORITY, 1.0       
	);

	glTexImage2D(
	    GL_TEXTURE_2D,	/* GL_TEXTURE_2D or GL_PROXY_TEXTURE_2D. */
	    0,			/* Level. */
	    GL_RGBA,		/* Internal format. */
	    new_width,		/* Width. */
	    new_height,		/* Height (same as width). */
	    0,			/* Border. */
	    GL_RGBA,		/* Image data format. */
	    GL_UNSIGNED_BYTE,	/* Image data type. */
	    /* Pointer to image data. */
	    (const GLvoid *)new_data_ptr
	);

	glFlush();

	/* Deallocate new data, it is no longer needed. */
	free(new_data_ptr);
	new_data_ptr = NULL;

	return(0);
}

/*
 *	Puts print window's preview glarea into GL context and deallocates
 *	its GL texture.
 */
void PrintWinPreviewTextureUnload(print_win_struct *pw)
{
	if(pw == NULL)
	    return;

	if(!pw->initialized)
	    return;

	/* Put preview glarea into GL context and begin destroying
	 * its GL resources.
	 */
	PrintWinPreviewEnableContext(pw);

	/* Destroy preview texture. */
	if(pw->preview_texture > 0)
	{
	    glDeleteLists(pw->preview_texture, 1);
	    pw->preview_texture = 0;
	    glFlush();
	}
}


/*
 *	Adjust output offset based on given coordinate change in
 *	window coordinates.
 *
 *	Preview will be redrawn.
 */
void PrintWinDoTranslate(
	print_win_struct *pw,
	gint dx, gint dy
)
{
	gdouble ptow_x_coeff, ptow_y_coeff;


	if(pw == NULL)
	    return;

	PrintWinGetPToWCoeff(
	    pw, &ptow_x_coeff, &ptow_y_coeff
	);

	if(ptow_x_coeff > 0.0)
	    pw->output_offset_x += (gint)((gdouble)dx / ptow_x_coeff);
	if(ptow_y_coeff > 0.0)
	    pw->output_offset_y += (gint)((gdouble)dy / ptow_y_coeff);

	PrintWinPreviewDraw(pw);

	return;
}

/*
 *      Adjust output size based on given coordinate change in
 *      window coordinates.
 *
 *	Preview will be redrawn.
 */
void PrintWinDoScale(
	print_win_struct *pw,
	gint dx, gint dy
)
{
	gint orientation;
	gdouble ptow_x_coeff, ptow_y_coeff;


	if(pw == NULL)
	    return;

	orientation = PrintWinGetOrientation(pw);
	PrintWinGetPToWCoeff(
	    pw, &ptow_x_coeff, &ptow_y_coeff
	);

	/* Flip dx and dy if orientation is set to landscape. */
	if(orientation)
	{
	    gint t = dx;
	    dx = dy;
	    dy = t;
	}

	if(ptow_x_coeff > 0.0)
	    pw->output_width += (gint)((gdouble)dx / ptow_x_coeff);
	if(ptow_y_coeff > 0.0)
	    pw->output_height += (gint)((gdouble)dy / ptow_y_coeff);

	/* Sanitize size. */
	if(pw->output_width < PRINT_WIN_OUTPUT_MIN_WIDTHP)
	    pw->output_width = PRINT_WIN_OUTPUT_MIN_WIDTHP;
	if(pw->output_height < PRINT_WIN_OUTPUT_MIN_HEIGHTP)
	    pw->output_height = PRINT_WIN_OUTPUT_MIN_HEIGHTP;


	PrintWinPreviewDraw(pw);

	return;
} 


/*
 *	Creates a new print window.
 */
print_win_struct *PrintWinNew(gpointer core_ptr)
{
	gint	bw = (100 + (2 * 3)),
		bh = (30 + (2 * 3));
	gint	border_major = 5,
		border_minor = 2;
	GSList *gslist;
	GtkAccelGroup *accelgrp;
	gpointer entry, combo_rtn, browse_btn;
	GtkCombo *combo;
	GtkWidget	*w, *menu, *parent, *parent2, *parent3, *parent4,
			*parent5, *main_vbox;
	print_win_struct *pw = (print_win_struct *)calloc(
	    1, sizeof(print_win_struct)
	);
	if(pw == NULL)
	    return(NULL);


	/* Reset values. */
	pw->initialized = TRUE;
	pw->map_state = FALSE;
	pw->preview_realized = FALSE;

	pw->flags = 0;

	/* Load cursors. */
	pw->translate_cur = gdk_cursor_new(GDK_FLEUR);
	pw->scale_cur = gdk_cursor_new(GDK_SIZING);
	pw->busy_cur = gdk_cursor_new(GDK_WATCH);

	/* Keyboard accelerator group. */
	pw->accelgrp = accelgrp = gtk_accel_group_new();

	/* Toplevel. */
	pw->toplevel = parent = w = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_press_event",
	    GTK_SIGNAL_FUNC(PrintWinEventCB),
	    (gpointer)pw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "key_release_event",
	    GTK_SIGNAL_FUNC(PrintWinEventCB),
	    (gpointer)pw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "configure_event",
	    GTK_SIGNAL_FUNC(PrintWinEventCB),
	    (gpointer)pw  
	);
	gtk_widget_add_events(
	    w,
	    GDK_KEY_PRESS_MASK | GDK_KEY_RELEASE_MASK | GDK_STRUCTURE_MASK
	);
	gtk_widget_set_usize(
	    w, PRINT_WIN_DEF_WIDTH, PRINT_WIN_DEF_HEIGHT
	);
	gtk_widget_realize(w);
	GUISetWMIcon(w->window, (u_int8_t **)icon_print_48x48_xpm);
	gtk_window_set_policy(GTK_WINDOW(w), TRUE, TRUE, FALSE);
	if(!GTK_WIDGET_NO_WINDOW(w))
	{
	    GdkGeometry geometry;

	    geometry.min_width = 100;
	    geometry.min_height = 70;

	    geometry.base_width = 0;
	    geometry.base_height = 0;

	    geometry.width_inc = 1;
	    geometry.height_inc = 1;

/*
	    geometry.min_aspect = 1.3;
	    geometry.max_aspect = 1.3;
 */
	    gdk_window_set_geometry_hints(
		w->window,
		&geometry,
		GDK_HINT_MIN_SIZE |
		GDK_HINT_BASE_SIZE |
		/* GDK_HINT_ASPECT | */
		GDK_HINT_RESIZE_INC
	    );
	}
	gtk_signal_connect(
	    GTK_OBJECT(w), "delete_event",
	    GTK_SIGNAL_FUNC(PrintWinCloseCB),
	    (gpointer)pw
	);
	gtk_signal_connect(
	    GTK_OBJECT(w), "destroy",
	    GTK_SIGNAL_FUNC(PrintWinDestroyCB),
	    (gpointer)pw
	);
	gtk_window_set_title(GTK_WINDOW(w), "Print");
	gtk_accel_group_attach(accelgrp, GTK_OBJECT(w));
	gtk_container_border_width(GTK_CONTAINER(w), 0);


	/* Main vbox. */
	main_vbox = w = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(parent), w);
	gtk_widget_show(w);
	parent = w;

	/* Vbox to hold frames. */
	w = gtk_vbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent = w;

	/* Source data frame. */
	w = gtk_frame_new("Source Data");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Source type combo. */
	w = (GtkWidget *)GUIComboCreate(
	    NULL,               /* No label. */
	    "Default",          /* Initial value. */
	    NULL,               /* Initial glist. */
	    16,                 /* Max items, 16 should be enough. */
	    &combo_rtn,    
	    (void *)pw,         /* Client data. */
	    NULL,               /* Don't need callbacks. */
	    NULL
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	pw->source_type_combo = w = (GtkWidget *)combo_rtn;
	combo = GTK_COMBO(w);
	gtk_entry_set_editable(GTK_ENTRY(combo->entry), FALSE);
	gtk_combo_set_case_sensitive(combo, TRUE);
	gtk_signal_connect(
	    GTK_OBJECT((GtkEditable *)combo->entry), "insert_text",
	    GTK_SIGNAL_FUNC(PrintWinSourceTypeTextInsertCB),
	    (gpointer)pw   
	);
	GUISetWidgetTip(
	    combo->entry,
	    PRINT_WIN_TT_SOURCE_TYPE
	);

	/* `Print to' frame. */
	w = gtk_frame_new("Print To");
	gtk_box_pack_start(GTK_BOX(parent), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent2 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent2), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent2 = w;


	/* Hbox for print to printer widgets. */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Print to printer radio button. */
	gslist = NULL;
	pw->print_to_printer_radio = w =
	    gtk_radio_button_new_with_label(gslist, "Printer");
	gtk_widget_set_usize(w, 75, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinPrintToToggleCB),                
	    (gpointer)pw
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* Print command prompt. */
	w = (GtkWidget *)GUIPromptBar(
	    NULL, "Print Command:",
	    NULL, &entry
	);
	pw->print_command_entry = (GtkWidget *)entry;
	GUISetWidgetTip(
	    entry,
	    PRINT_WIN_TT_PRINT_COMMAND
	);
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);


	/* Hbox for print to file widgets. */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Print to file radio button. */
	pw->print_to_file_radio = w =
	    gtk_radio_button_new_with_label(gslist, "File");
	gtk_widget_set_usize(w, 75, -1);
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinPrintToToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);

	/* Print file prompt. */
	w = (GtkWidget *)GUIPromptBarWithBrowse(
	    NULL, "Name:",
	    NULL, &entry, &browse_btn,
	    pw, PrintWinPrintFileBrowseCB
	);
	pw->print_file_entry = (GtkWidget *)entry;
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_widget_show(w);



	/* Hbox to hold print parameters and preview widgets. */
	w = gtk_hbox_new(FALSE, border_major);
	gtk_box_pack_start(GTK_BOX(parent), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent2 = w;


	/* Left column vbox to hold print parameter widgets. */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent3 = w;


	/* Print orientation frame. */
	w = gtk_frame_new("Orientation");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN); 
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent4 = w;

	/* Portrait radio button. */
	gslist = NULL;
	w = gtk_radio_button_new_with_label(gslist, "Portrait");
	pw->orient_portrait_radio = w;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinOrientationToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* Landscape radio button. */  
	w = gtk_radio_button_new_with_label(gslist, "Landscape");
	pw->orient_landscape_radio = w;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinOrientationToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);


	/* Visual frame. */
	w = gtk_frame_new("Visual");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent4 = w;

	/* GreyScale button. */
	gslist = NULL;
	w = gtk_radio_button_new_with_label(gslist, "GreyScale");
	pw->print_greyscale_radio = w;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinVisualToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* Color button. */
	w = gtk_radio_button_new_with_label(gslist, "Color");
	pw->print_color_radio = w;
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinVisualToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);


	/* Resolution, pixels per inch (PPI) frame. */
	w = gtk_frame_new("Resolution (PPI)");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent4 = w;

	/* PPI X prompt. */   
	w = (GtkWidget *)GUIPromptBar(
	    NULL, "X:",
	    NULL, &entry   
	);
	pw->ppi_x_entry = (GtkWidget *)entry;
	GUISetWidgetTip(
	    entry,
	    PRINT_WIN_TT_PPI_X
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_set_usize((GtkWidget *)entry, 80, -1);
	gtk_widget_show(w);

	/* PPI Y prompt. */
	w = (GtkWidget *)GUIPromptBar(
	    NULL, "Y:",
	    NULL, &entry
	);
	pw->ppi_y_entry = (GtkWidget *)entry;
	GUISetWidgetTip(
	    entry,
	    PRINT_WIN_TT_PPI_Y
	);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_set_usize((GtkWidget *)entry, 80, -1);
	gtk_widget_show(w);


	/* Paper size frame. */
	w = gtk_frame_new("Paper Size");
	gtk_box_pack_start(GTK_BOX(parent3), w, FALSE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Letter size button. */
	gslist = NULL;
	w = gtk_radio_button_new_with_label(gslist,
	    "Letter (8.5\" x 11\")"
	);
	pw->paper_size_letter_radio = w;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_set_usize(w, 180, -1);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinPaperSizeToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* Legal size button. */
	w = gtk_radio_button_new_with_label(gslist,
	    "Legal (8.5\" x 14\")"
	);
	pw->paper_size_legal_radio = w;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_set_usize(w, 180, -1);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinPaperSizeToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, FALSE, FALSE, 0);
	gtk_widget_show(w);
	parent5 = w;

	/* Executive size button. */
	w = gtk_radio_button_new_with_label(gslist,
	    "Executive (7.5\" x 10\")"
	);
	pw->paper_size_executive_radio = w;
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_widget_set_usize(w, 180, -1);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinPaperSizeToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);
	gslist = gtk_radio_button_group(GTK_RADIO_BUTTON(w));

	/* A4 size button. */
	w = gtk_radio_button_new_with_label(gslist,
	    "A4 (210mm x 297mm)"
	);
	pw->paper_size_a4_radio = w;
	gtk_widget_set_usize(w, 180, -1);
	gtk_box_pack_start(GTK_BOX(parent5), w, FALSE, FALSE, 0);
	gtk_signal_connect(
	    GTK_OBJECT(w), "toggled",
	    GTK_SIGNAL_FUNC(PrintWinPaperSizeToggleCB),
	    (gpointer)pw
	);
	gtk_widget_show(w);





	/* Right column vbox to hold preview. */
	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, TRUE, 0);
	gtk_widget_show(w);
	parent3 = w;

	/* Preview frame. */
	w = gtk_frame_new("Preview");
	gtk_box_pack_start(GTK_BOX(parent3), w, TRUE, TRUE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_ETCHED_IN);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_hbox_new(FALSE, border_minor);
	gtk_container_add(GTK_CONTAINER(parent4), w);
	gtk_container_border_width(GTK_CONTAINER(w), border_major);
	gtk_widget_show(w);
	parent4 = w;

	w = gtk_vbox_new(FALSE, border_minor);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_widget_show(w);
	parent4 = w;

	/* Frame widget to hold preview glarea. */
	w = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX(parent4), w, TRUE, FALSE, 0);
	gtk_frame_set_shadow_type(GTK_FRAME(w), GTK_SHADOW_IN);
	gtk_widget_show(w);
	parent4 = w;

	/* Preview glarea widget. */
	pw->preview_glarea = w = gtk_gl_area_new(gl_attributes_list);
	if(w != NULL)
	{
/* 	    GtkGLArea *glarea = GTK_GL_AREA(w); */

	    gtk_widget_add_events(
		w,
		GDK_EXPOSURE_MASK | GDK_KEY_PRESS_MASK |
		GDK_KEY_RELEASE_MASK | GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK | GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK |
		GDK_ENTER_NOTIFY_MASK | GDK_LEAVE_NOTIFY_MASK
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_press_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "key_release_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_press_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "button_release_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "motion_notify_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "expose_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_signal_connect(
		GTK_OBJECT(w), "configure_event",
		GTK_SIGNAL_FUNC(PrintWinPreviewEventCB),
		(gpointer)pw
	    );
	    gtk_container_add(GTK_CONTAINER(parent4), w);
	    GUISetWidgetTip(
		w,
		PRINT_WIN_TT_PREVIEW
	    );
	    gtk_widget_show(w);
	}


	/* Preview right-click menu. */
	pw->preview_menu = menu = GUIMenuCreate();
	if(menu != NULL)
	{
	    u_int8_t **icon;
	    const gchar *label;
	    gint accel_key;
	    guint accel_mods;
	    GtkWidget *fw;
	    gpointer accel_group = NULL;
	    gpointer mclient_data = (gpointer)pw;
	    void (*func_cb)(GtkWidget *, gpointer) = NULL;

#define DO_ADD_MENU_ITEM_LABEL	\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_LABEL, accel_group, \
  icon, label, accel_key, accel_mods, (void **)&fw, \
  mclient_data, func_cb \
 ); \
}
#define DO_ADD_MENU_SEP		\
{ \
 w = GUIMenuItemCreate( \
  menu, GUI_MENU_ITEM_TYPE_SEPARATOR, NULL, \
  NULL, NULL, 0, 0, NULL, \
  NULL, NULL \
 ); \
}

	    icon = NULL;
	    label = "Center";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = PrintWinPreviewCenterCB;
	    DO_ADD_MENU_ITEM_LABEL
/*          pw->center_mi = w; */

	    icon = NULL;
	    label = "Original Size";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = PrintWinPreviewOriginalSizeCB;
	    DO_ADD_MENU_ITEM_LABEL
/*	    pw->actual_size_mi = w; */

	    icon = NULL;
	    label = "Size To Fit";
	    accel_key = 0;
	    accel_mods = 0;
	    func_cb = PrintWinPreviewSizeToFitCB;
	    DO_ADD_MENU_ITEM_LABEL
/*          pw->size_to_fit_mi = w; */

#undef DO_ADD_MENU_ITEM_LABEL
#undef DO_ADD_MENU_SEP
	}

	/* Separator. */
	w = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, 0);
	gtk_widget_show(w);


	/* Hbox to hold buttons. */
	w = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), w, FALSE, FALSE, border_major);
	gtk_widget_show(w);
	parent2 = w;

	/* Print button. */
	pw->print_btn = w = GUIButtonPixmapLabelH(
	    (u_int8_t **)icon_print_20x20_xpm, "Print", NULL
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_signal_connect(   
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PrintWinPrintButtonCB),
	    (gpointer)pw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Return, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_3270_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_KP_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_accel_group_add(
	    accelgrp, GDK_ISO_Enter, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w);

	/* Cancel button. */
	pw->cancel_btn = w = GUIButtonPixmapLabelH(
	    (u_int8_t **)icon_cancel_20x20_xpm, "Cancel", NULL
	);
	gtk_box_pack_start(GTK_BOX(parent2), w, TRUE, FALSE, 0);
	gtk_widget_set_usize(w, bw, bh);
	GTK_WIDGET_SET_FLAGS(w, GTK_CAN_DEFAULT);
	gtk_signal_connect(
	    GTK_OBJECT(w), "clicked",
	    GTK_SIGNAL_FUNC(PrintWinCancelButtonCB),
	    (gpointer)pw
	);
	gtk_accel_group_add(
	    accelgrp, GDK_Escape, 0, GTK_ACCEL_VISIBLE,
	    GTK_OBJECT(w), "clicked"
	);
	gtk_widget_show(w);


	/* Set callbacks. */
	pw->client_data = NULL;
	pw->print_cb = NULL;
	pw->cancel_cb = NULL;


	/* Reset values. */
	PrintWinReset(pw, FALSE);

	return(pw);
}

/*
 *      Makes the glarea preview widget on the print window into GL
 *	context.
 *
 *      Returns 0 on success or -1 on error.
 */
gint PrintWinPreviewEnableContext(print_win_struct *pw)
{
	GtkWidget *w;
	gint status;


	if(pw == NULL)
	    return(-1);

	if(!pw->initialized || !pw->preview_realized)
	    return(-1);

	w = pw->preview_glarea;
	if(w != NULL)
	{
	    status = gtk_gl_area_make_current(GTK_GL_AREA(w));
	    if(status)
		return(0);
	    else
		return(-1);
	}

	return(-1);
}


/*
 *	Sets callback client data and function pointers for the given
 *	print window.
 */
void PrintWinSetCallbacks(
	print_win_struct *pw,
	gpointer client_data,
	void (*print_cb)(
		gpointer, gpointer, print_win_parms_struct *
	),
	void (*cancel_cb)(
		gpointer, gpointer
	),
	void (*preview_request_cb)(
		gpointer, gpointer, gint, gint, gint, gint
	)
)
{
	if(pw == NULL)
	    return;

	if(!pw->initialized)
	    return;

	pw->client_data = client_data;
	pw->print_cb = print_cb;
	pw->cancel_cb = cancel_cb;
	pw->preview_request_cb = preview_request_cb;

	return;
}

/*
 *	Replaces the print window's list of source type names with
 *	the given list.
 *
 *	The given list will not be modified.
 */
void PrintWinSetSourceTypeNames(
	print_win_struct *pw, gchar **name, gint total_names
)
{
	gint i;
	GtkWidget *w;
	GtkCombo *combo;
	GList *glist;
	const gchar *cstrptr;


	if(pw == NULL)
	    return;

	w = pw->source_type_combo;
	if(w == NULL)
	    return;

	combo = GTK_COMBO(w);

	/* Create glist of items. */
	glist = NULL;
	for(i = 0; i < total_names; i++)
	{
	    cstrptr = (const char *)name[i];
	    if(cstrptr == NULL)
		cstrptr = "(null)";

	    glist = g_list_append(glist, (gpointer)strdup(cstrptr));
	}

	/* Set new list on combo, the given glist will be
	 * deallocated by this function.
	 */
	GUIComboSetList((void *)w, (void *)glist);
	glist = NULL;

	return;
}


/*
 *	Maps the print window.
 */
void PrintWinMap(print_win_struct *pw)
{
	GtkWidget *w;

	if(pw == NULL)
	    return;

	if(pw->initialized && !pw->map_state)
	{
	    w = pw->print_btn;
	    if(w != NULL)
	    {
		gtk_widget_grab_focus(w);
		gtk_widget_grab_default(w);
	    }

	    w = pw->toplevel;
	    if(w != NULL)
		gtk_widget_show(w);

	    pw->map_state = TRUE;

	    /* Request preview on map. */
	    if(pw->preview_request_cb != NULL)
	    {
		gint source_type = PrintWinGetSourceType(pw, NULL);
		gint visual = PrintWinGetVisual(pw);

		pw->preview_request_cb(
		    (gpointer)pw,
		    pw->client_data,
		    source_type,
		    visual,
		    pw->output_width, pw->output_height
		);
		PrintWinPreviewDraw(pw);
	    }
	}
}

/*
 *	Maps the print window and sets the given values.
 *
 *	If the print window is already mapped then only the values will
 *	be updated.
 */
void PrintWinMapValues(
	print_win_struct *pw, print_win_parms_struct *parms
)
{
	guint flags;
	GtkWidget *w;


	if((pw == NULL) || (parms == NULL))
	    return;

	if(!pw->initialized)
	    return;


	flags = parms->flags;

	/* Print to. */
	if(flags & PRINT_WIN_PARM_FLAG_PRINT_TO)
	{
	    w = pw->print_to_printer_radio;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = pw->print_to_file_radio;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;

	    switch(parms->print_to)
	    {
	      case PRINT_WIN_PRINT_TO_PRINTER:
		w = pw->print_to_printer_radio;
		if(w != NULL)
		{
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		    gtk_signal_emit_by_name(
			GTK_OBJECT(w), "toggled"
		    );
		}
		break;

	      case PRINT_WIN_PRINT_TO_FILE:
		w = pw->print_to_file_radio;
		if(w != NULL)
		{
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		    gtk_signal_emit_by_name(
			GTK_OBJECT(w), "toggled"
		    );
		}
		break;
	    }
	}

	/* Print command. */
	if(flags & PRINT_WIN_PARM_FLAG_PRINT_COMMAND)
	{
	    if(parms->print_command == NULL)
	    {
	        w = pw->print_command_entry;
	        if(w != NULL)
		    gtk_entry_set_text(GTK_ENTRY(w), "");
	    }
	    else
	    {
		w = pw->print_command_entry;
		if(w != NULL)
		    gtk_entry_set_text(GTK_ENTRY(w), parms->print_command);
	    }
	}

	/* Print file. */
	if(flags & PRINT_WIN_PARM_FLAG_PRINT_FILE)
	{
	    if(parms->print_file == NULL)
	    {
		w = pw->print_file_entry;
		if(w != NULL)
		    gtk_entry_set_text(GTK_ENTRY(w), "");
	    }
	    else
	    {
		w = pw->print_file_entry;
		if(w != NULL)
		    gtk_entry_set_text(GTK_ENTRY(w), parms->print_file);
	    }
	}

	/* Source type. */
	if(flags & PRINT_WIN_PARM_FLAG_SOURCE_TYPE)
	{
	    w = pw->source_type_combo;
	    if(w != NULL)
	    {
		GtkEntry *entry = GTK_ENTRY(GTK_COMBO(w)->entry);
		GList *glist = GUIComboGetList((void *)w);
		if((glist != NULL) && (entry != NULL))
		{
		    gint i = 0, n = parms->source_type;

		    while(glist != NULL)
		    {
			if(i == n)
			{
			    if(glist->data != NULL)
				gtk_entry_set_text(entry, (const gchar *)glist->data);
			    break;
			}

			glist = glist->next;
			i++;
		    }
		}
	    }
	}

	/* Orientation. */
	if(flags & PRINT_WIN_PARM_FLAG_ORIENTATION)
	{
	    if(parms->orientation)
	    {
		w = pw->orient_portrait_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = FALSE;
		w = pw->orient_landscape_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
	    }
	    else
	    {
		w = pw->orient_portrait_radio; 
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		w = pw->orient_landscape_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    }
	}

	/* Visual. */
	if(flags & PRINT_WIN_PARM_FLAG_VISUAL)
	{
	    if(parms->visual == PRINT_WIN_VISUAL_COLOR)
	    {
	        /* Color. */
		w = pw->print_greyscale_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = FALSE;
		w = pw->print_color_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
	    }
	    else
	    {
	        /* Greyscale. */
		w = pw->print_greyscale_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		w = pw->print_color_radio;     
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    }
	}


	/* Pixels per inch (PPI). */
	if(flags & PRINT_WIN_PARM_FLAG_PPI_X)
	{
	    char num_str[80];

	    sprintf(num_str, "%.0f", parms->ppi_x);

	    w = pw->ppi_x_entry;
	    if(w != NULL)
		gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}
	if(flags & PRINT_WIN_PARM_FLAG_PPI_Y)
	{
	    char num_str[80];

	    sprintf(num_str, "%.0f", parms->ppi_y);

	    w = pw->ppi_y_entry;
	    if(w != NULL)
		gtk_entry_set_text(GTK_ENTRY(w), num_str);
	}

	/* Paper size. */
	if(flags & PRINT_WIN_PARM_FLAG_PAPER_SIZE)
	{
	    w = pw->paper_size_letter_radio;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = pw->paper_size_legal_radio;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = pw->paper_size_executive_radio;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    w = pw->paper_size_a4_radio;
	    if(w != NULL)
		GTK_TOGGLE_BUTTON(w)->active = FALSE;
	    switch(parms->paper_size)
	    {
	      case PRINT_WIN_PAPER_SIZE_A4:
		w = pw->paper_size_a4_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
 	        break;

	      case PRINT_WIN_PAPER_SIZE_EXECUTIVE:
		w = pw->paper_size_executive_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		break;

	      case PRINT_WIN_PAPER_SIZE_LEGAL:
		w = pw->paper_size_legal_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		break;

	      default:	/* PRINT_WIN_PAPER_SIZE_LETTER */
		w = pw->paper_size_letter_radio;
		if(w != NULL)
		    GTK_TOGGLE_BUTTON(w)->active = TRUE;
		break;
	    }
	}

	/* Output offset. */
	if(flags & PRINT_WIN_PARM_FLAG_OUTPUT_OFFSET_X)
	{
	    pw->output_offset_x = parms->output_offset_xp;
	}
	if(flags & PRINT_WIN_PARM_FLAG_OUTPUT_OFFSET_Y)
	{
	    pw->output_offset_y = parms->output_offset_yp; 
	}

	/* Output size. */
	if(flags & PRINT_WIN_PARM_FLAG_OUTPUT_WIDTH)
	{
	    pw->output_width = parms->output_widthp;
	}
	if(flags & PRINT_WIN_PARM_FLAG_OUTPUT_HEIGHT)
	{
	    pw->output_height = parms->output_heightp;
	}


	/* Map print window. */
	PrintWinMap(pw);
}

/*
 *	Unmaps the print window.
 */
void PrintWinUnmap(print_win_struct *pw)
{
	GtkWidget *w;

	if(pw == NULL)
	    return;

	if(pw->initialized && pw->map_state)
	{
	    w = pw->toplevel;
	    if(w != NULL)
		gtk_widget_hide(w);

	    pw->map_state = FALSE;
	}
}

/*
 *	Deallocates any loaded data and clears all values on the given
 *	print window's widgets.
 */
void PrintWinReset(print_win_struct *pw, gbool need_unmap)
{
	if(pw == NULL)
	    return;

	if(!pw->initialized)
	    return;

	/* Unload preview GL texture. */
	PrintWinPreviewTextureUnload(pw);

	/* Reset flags. */
	pw->flags = 0;

	/* Reset output offset and size. */
	pw->output_offset_x = 0;
	pw->output_offset_y = 0;
	pw->output_width = 0;
	pw->output_height = 0;

	/* Reset preview glarea widget size. */
 	PrintWinPreviewSetSize(
	    pw, PRINT_WIN_DEF_PAPER_WIDTHP, PRINT_WIN_DEF_PAPER_HEIGHTP
	);

/* Reset other widgets here. */

	/* Need to unmap? */
	if(need_unmap)
	    PrintWinUnmap(pw);
}

/*
 *	Sets print window as busy.
 */
void PrintWinSetBusy(print_win_struct *pw)
{
	GtkWidget *w;
	GdkCursor *cur;

	if(pw == NULL)
	    return;

	w = pw->toplevel;
	if(w == NULL)
	    return;

	cur = pw->busy_cur;
	if(cur == NULL)
	    return;

	if(GTK_WIDGET_NO_WINDOW(w))
	    return;

	gdk_window_set_cursor(w->window, cur);
	gdk_flush();
}

/*
 *	Sets print window as ready.
 */
void PrintWinSetReady(print_win_struct *pw)
{
	GtkWidget *w;

	if(pw == NULL)
	    return;

	w = pw->toplevel;
	if(w == NULL)
	    return;

	if(GTK_WIDGET_NO_WINDOW(w))
	    return;

	gdk_window_set_cursor(w->window, NULL);
	gdk_flush();
}



/*
 *	Deletes the given print window.
 */
void PrintWinDelete(print_win_struct *pw)
{
	GtkWidget **w;
	GdkCursor **cur;

	if(pw == NULL)
	    return;

	if(pw->initialized)
	{
#define DO_DESTROY_CURSOR       \
{ \
 if((*cur) != NULL) \
 { \
  GdkCursor *tc = *cur; \
  (*cur) = NULL; \
  gdk_cursor_destroy(tc); \
 } \
}

#define DO_DESTROY_WIDGET       \
{ \
 if((*w) != NULL) \
 { \
  GtkWidget *tmp_w = *w; \
  (*w) = NULL; \
  gtk_widget_destroy(tmp_w); \
 } \
}

	    /* Deallocate and clear all data in print window's widgets. */
	    PrintWinReset(pw, pw->map_state);


	    /* Begin destroying widgets. */

	    w = &pw->preview_menu;
	    DO_DESTROY_WIDGET

	    w = &pw->source_type_combo;
	    DO_DESTROY_WIDGET

	    w = &pw->print_command_entry;
	    DO_DESTROY_WIDGET

	    w = &pw->print_file_entry;
	    DO_DESTROY_WIDGET


	    w = &pw->print_to_printer_radio;
	    DO_DESTROY_WIDGET

	    w = &pw->print_to_file_radio;
	    DO_DESTROY_WIDGET


	    w = &pw->preview_glarea;
	    DO_DESTROY_WIDGET

	    w = &pw->preview_fixed;
	    DO_DESTROY_WIDGET

	    w = &pw->toplevel;
	    DO_DESTROY_WIDGET


	    cur = &pw->translate_cur;
	    DO_DESTROY_CURSOR
	    cur = &pw->scale_cur;
	    DO_DESTROY_CURSOR
	    cur = &pw->busy_cur;
	    DO_DESTROY_CURSOR

	    if(pw->accelgrp != NULL)
	    {
		gtk_accel_group_unref(pw->accelgrp);
		pw->accelgrp = NULL;
	    }

#undef DO_DESTROY_WIDGET
#undef DO_DESTROY_CURSOR
	}

	/* Deallocate structure itself. */
	free(pw);
}

