/* valavariabledeclarator.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valavariabledeclarator.h>
#include <vala/valaexpression.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valacodevisitor.h>

struct _ValaVariableDeclaratorPrivate {
	ValaExpression* _initializer;
	ValaDataType* _data_type;
};
#define VALA_VARIABLE_DECLARATOR_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_VARIABLE_DECLARATOR, ValaVariableDeclaratorPrivate))
enum  {
	VALA_VARIABLE_DECLARATOR_DUMMY_PROPERTY,
	VALA_VARIABLE_DECLARATOR_INITIALIZER,
	VALA_VARIABLE_DECLARATOR_TYPE_REFERENCE
};
static void vala_variable_declarator_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_variable_declarator_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_variable_declarator_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node);
static void vala_variable_declarator_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type);
static gpointer vala_variable_declarator_parent_class = NULL;
static void vala_variable_declarator_dispose (GObject * obj);


/**
 * Creates a new variable declarator.
 *
 * @param name   name of the variable
 * @param init   optional initializer expression
 * @param source reference to source code
 * @return       newly created variable declarator
 */
ValaVariableDeclarator* vala_variable_declarator_new (const char* name, ValaExpression* initializer, ValaSourceReference* source_reference) {
	ValaVariableDeclarator * self;
	g_return_val_if_fail (initializer == NULL || VALA_IS_EXPRESSION (initializer), NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_VARIABLE_DECLARATOR, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_variable_declarator_set_initializer (self, initializer);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


static void vala_variable_declarator_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_variable_declarator (visitor, self);
}


static void vala_variable_declarator_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (base);
	g_return_if_fail (visitor == NULL || VALA_IS_CODE_VISITOR (visitor));
	if (vala_variable_declarator_get_initializer (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_variable_declarator_get_initializer (self)), visitor);
		vala_code_visitor_visit_end_full_expression (visitor, vala_variable_declarator_get_initializer (self));
	}
	if (vala_variable_declarator_get_type_reference (self) != NULL) {
		vala_code_node_accept (VALA_CODE_NODE (vala_variable_declarator_get_type_reference (self)), visitor);
	}
}


static void vala_variable_declarator_real_replace_expression (ValaCodeNode* base, ValaExpression* old_node, ValaExpression* new_node) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (base);
	g_return_if_fail (old_node == NULL || VALA_IS_EXPRESSION (old_node));
	g_return_if_fail (new_node == NULL || VALA_IS_EXPRESSION (new_node));
	if (vala_variable_declarator_get_initializer (self) == old_node) {
		vala_variable_declarator_set_initializer (self, new_node);
	}
}


static void vala_variable_declarator_real_replace_type (ValaCodeNode* base, ValaDataType* old_type, ValaDataType* new_type) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (base);
	g_return_if_fail (old_type == NULL || VALA_IS_DATA_TYPE (old_type));
	g_return_if_fail (new_type == NULL || VALA_IS_DATA_TYPE (new_type));
	if (vala_variable_declarator_get_type_reference (self) == old_type) {
		vala_variable_declarator_set_type_reference (self, new_type);
	}
}


ValaExpression* vala_variable_declarator_get_initializer (ValaVariableDeclarator* self) {
	g_return_val_if_fail (VALA_IS_VARIABLE_DECLARATOR (self), NULL);
	return self->priv->_initializer;
}


void vala_variable_declarator_set_initializer (ValaVariableDeclarator* self, ValaExpression* value) {
	ValaExpression* _tmp2;
	ValaExpression* _tmp1;
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_initializer = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL))), _tmp2);
	if (self->priv->_initializer != NULL) {
		vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_initializer), VALA_CODE_NODE (self));
	}
}


ValaDataType* vala_variable_declarator_get_type_reference (ValaVariableDeclarator* self) {
	g_return_val_if_fail (VALA_IS_VARIABLE_DECLARATOR (self), NULL);
	return self->priv->_data_type;
}


void vala_variable_declarator_set_type_reference (ValaVariableDeclarator* self, ValaDataType* value) {
	ValaDataType* _tmp2;
	ValaDataType* _tmp1;
	g_return_if_fail (VALA_IS_VARIABLE_DECLARATOR (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_data_type = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL))), _tmp2);
	vala_code_node_set_parent_node (VALA_CODE_NODE (self->priv->_data_type), VALA_CODE_NODE (self));
}


static void vala_variable_declarator_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (object);
	switch (property_id) {
		case VALA_VARIABLE_DECLARATOR_INITIALIZER:
		g_value_set_object (value, vala_variable_declarator_get_initializer (self));
		break;
		case VALA_VARIABLE_DECLARATOR_TYPE_REFERENCE:
		g_value_set_object (value, vala_variable_declarator_get_type_reference (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_variable_declarator_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (object);
	switch (property_id) {
		case VALA_VARIABLE_DECLARATOR_INITIALIZER:
		vala_variable_declarator_set_initializer (self, g_value_get_object (value));
		break;
		case VALA_VARIABLE_DECLARATOR_TYPE_REFERENCE:
		vala_variable_declarator_set_type_reference (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_variable_declarator_class_init (ValaVariableDeclaratorClass * klass) {
	vala_variable_declarator_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaVariableDeclaratorPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_variable_declarator_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_variable_declarator_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_variable_declarator_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_variable_declarator_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_variable_declarator_real_accept_children;
	VALA_CODE_NODE_CLASS (klass)->replace_expression = vala_variable_declarator_real_replace_expression;
	VALA_CODE_NODE_CLASS (klass)->replace_type = vala_variable_declarator_real_replace_type;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_VARIABLE_DECLARATOR_INITIALIZER, g_param_spec_object ("initializer", "initializer", "initializer", VALA_TYPE_EXPRESSION, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_VARIABLE_DECLARATOR_TYPE_REFERENCE, g_param_spec_object ("type-reference", "type-reference", "type-reference", VALA_TYPE_DATA_TYPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_variable_declarator_init (ValaVariableDeclarator * self) {
	self->priv = VALA_VARIABLE_DECLARATOR_GET_PRIVATE (self);
}


static void vala_variable_declarator_dispose (GObject * obj) {
	ValaVariableDeclarator * self;
	self = VALA_VARIABLE_DECLARATOR (obj);
	(self->priv->_initializer == NULL ? NULL : (self->priv->_initializer = (g_object_unref (self->priv->_initializer), NULL)));
	(self->priv->_data_type == NULL ? NULL : (self->priv->_data_type = (g_object_unref (self->priv->_data_type), NULL)));
	G_OBJECT_CLASS (vala_variable_declarator_parent_class)->dispose (obj);
}


GType vala_variable_declarator_get_type (void) {
	static GType vala_variable_declarator_type_id = 0;
	if (G_UNLIKELY (vala_variable_declarator_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaVariableDeclaratorClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_variable_declarator_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaVariableDeclarator), 0, (GInstanceInitFunc) vala_variable_declarator_init };
		vala_variable_declarator_type_id = g_type_register_static (VALA_TYPE_SYMBOL, "ValaVariableDeclarator", &g_define_type_info, 0);
	}
	return vala_variable_declarator_type_id;
}




