/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse m�l :
	BILLARD, non joignable par m�l ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant � visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est r�gi par la licence CeCILL soumise au droit fran�ais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffus�e par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez acc�der � cet en-t�te signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept� les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_data.h"


#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h> 

#include "visu_object.h"
#include "visu_tools.h"
#include "visu_nodes.h"
#include "visu_extension.h"
#include "visu_rendering.h"
#include "openGLFunctions/objectList.h"
#include "openGLFunctions/text.h"
#include "extraFunctions/dataNode.h"
#include "coreTools/toolMatrix.h"

/**
 * SECTION:visudata
 * @short_description: Give methods to store and manage data from
 * input file(s).
 * @include: visu_nodes.h
 * 
 * <para>The main goal of V_Sim is to draw lists of elements. For
 * example, when used to render atoms, a box that contains 24 silicon
 * atoms and 46 germanium atoms is a box with two elements (silicon
 * and germanium) where the silicon element has 24 nodes and the
 * germanium element has 46 nodes. This module gives then methods to
 * create nodes (see #VisuElement to create and managed
 * elements).</para>
 *
 * <para>A node is characterised by its position in the space and its
 * orientation. When rendered, a specific translation can be
 * applied. It has also one number that is its position in the read
 * input file and one boolean to control if it is rendered on
 * screen. One can associate whatever properties using
 * visuDataSet_nodeProperty(). These properties can be retrieved with
 * visuDataGet_nodeProperty(). But before using these methods, the
 * needed space must be allocated using
 * visuDataAdd_nodeProperty(). The stored properties must be pointers
 * on something.</para>
 *
 * <para>All nodes are stored in a structure called #VisuData. V_Sim
 * uses one #VisuData per input file(s). A pointer %currentVisuData
 * points to the current rendered #VisuData. This structure contains a
 * list of pointers on all the #VisuElement used in this file. It also
 * contains the list of all #VisuNodes of this file (or these
 * files).</para>
 */


struct FileDescription_struct
{
  /* Identity. */
  int kind;

  /* The name of a file. */
  gchar *name;

  /* The format of the file.
     This format can be null, if the file has not been
     parsed yet and the format is unknown.
     If the file file is not parsed but the format is
     set, then it is just an indication and may not
     be the right format. */
  FileFormat *format;
};

enum {
  BOX_SIZE_CHANGED_SIGNAL,
  ELEMENT_RENDERED_CHANGED_SIGNAL,
  NODE_RENDERED_CHANGED_SIGNAL,
  NODE_POSITION_CHANGED_SIGNAL,
  NODE_ASK_FOR_SHOW_HIDE_SIGNAL,
  NODE_POPULATION_INCREASE_SIGNAL,
  NODE_POPULATION_DECREASE_SIGNAL,
  NODE_POPULATION_DEFINED_SIGNAL,
  OPENGL_THETA_PHI_OMEGA_CHANGED_SIGNAL,
  OPENGL_XS_YS_CHANGED_SIGNAL,
  OPENGL_GROSS_CHANGED_SIGNAL,
  OPENGL_PERSP_CHANGED_SIGNAL,
  OPENGL_NEAR_FAR_CHANGED_SIGNAL,
  OPENGL_WIDTH_HEIGHT_CHANGED_SIGNAL,
  OPENGL_FACETTES_CHANGED_SIGNAL,
  VISU_DATA_FREED_SIGNAL,
  LAST_SIGNAL
};

/* Define a key for a DataNode value. */
#define COORDINATES_ID     "visuData_coordinates"

struct _VisuDataClass
{
  GObjectClass parent;

  int identifierAllNodes;
};

/* Local variables. */
static DataNode *dataNodeCoord;

struct VisuData_private_struct
{
  gboolean dispose_has_run;

  /*************/
  /* The nodes */
  /*************/
  VisuNodeArray *nodeArray;

  /********************/
  /* Files attributes */
  /********************/
  /* Files that correspond to that VisuData.
     They are identified by a kind which is an integer.
     The list is pointers to FileDescription_struct. */
  GList *files;
  /* Commentary associated to the rendered data, this commentary can
     be different for each set of nodes. */
  gchar** commentary;
  /* The number of set of nodes in one file, and the currently loaded one. */
  int nSets, iSet;

  /********************/
  /* Box   attributes */
  /********************/
  /* This is the geometry of the box. Array position 1 to 6
     correspond to xx, xy, yy, zx, zy and zz. */
  double boxGeometry[6];
  /* This is the unitary vector corresponding to boxGeometry. */
  double boxNorm[6];
  /* This is the matrix that transform cartesian coordinates to
     coordinates in the box geometry. Use visuDataConvert_XYZtoBoxCoordinates()
     to access this matrix. */
  double fromXYZtoBoxD0[3][3];
  /* This is the matrix that transform box coordinates to
     cartesian coordinates. Use visuDataConvert_boxCoordinatestoXYZ()
     to access this matrix. */
  double fromBoxtoXYZD0[3][3];
  
  /* Translation applied to all nodes when rendered. */
  gboolean translationApply;
  float translation[3];
  /* The extension from the box. */
  float extension[3];

  /********************/
  /* Misc. attributes */
  /********************/
  /* This list contains pointers on source id, to be removed when the object
     is finalized. */
  GList *timeoutList;
  /* This function is used to scale the nodes before drawing them. */
  VisuDataScalingFunc scaling;

  /* This pointer is used to gain access to the window the #VisuData may
     be attached to. */
  GenericRenderingWindow attachingWindow;

  /* This object is used to describe the geometry of the OpenGL
     context. */
  OpenGLView *attachedView;
};


static void visu_data_dispose   (GObject* obj);
static void visu_data_finalize  (GObject* obj);

/* Local callbacks. */
static void onAskForShowHideSignal(VisuData *visuData, gboolean *redraw, gpointer data);
static void visuData_rebuildList(VisuData *dataObj);

/* Local routines. */
static gboolean setCoordFromString(DataNode *dataNode, VisuData *dataObj,
				   VisuNode *node, gchar* labelIn,
				   gchar **labelOut, gboolean *modify);
static gchar* getCoordFromString(DataNode *dataNode, VisuData *dataObj,
				 VisuNode *node);
static int* shrinkNodeList(VisuData *data, int coord, float valueTo);
static int* extendNodeList(VisuData *data, int coord, float valueTo);
static void applyBox(VisuData *data, float refLength, float margin);
static float getAllNodeExtens(VisuData *dataObj);
static float defaultScaling(VisuData *data, VisuNode *node);


static guint visu_data_signals[LAST_SIGNAL] = { 0 };
static GList* allObjects;

G_DEFINE_TYPE(VisuData, visu_data, G_TYPE_OBJECT)

static void visu_data_class_init(VisuDataClass *klass)
{
  GType paramPointer[1] = {G_TYPE_POINTER};
  char *nameI18n    = _("All elements");
  char *description = _("Draw all the nodes.");
  OpenGLExtension *extensionData;

  DBG_fprintf(stderr, "Visu data : creating the class of the object.\n");
  DBG_fprintf(stderr, "                - adding new signals ;\n");
  /**
   * VisuData::BoxSizeChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when the box size is changed (because of box
   * duplication for instance).
   *
   * Since: 3.4
   */
  visu_data_signals[BOX_SIZE_CHANGED_SIGNAL] =
    g_signal_newv("BoxSizeChanged",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::ElementRenderedChanged:
   * @dataObj: the object which received the signal ;
   * @element: the #VisuElement which has changed.
   *
   * Gets emitted when one of the rendered #VisuElement is made
   * visible or hidden.
   *
   * Since: 3.2
   */
  visu_data_signals[ELEMENT_RENDERED_CHANGED_SIGNAL] =
    g_signal_newv("ElementRenderedChanged",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodeRenderedChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when one or more nodes have changed of
   * visibility. Some may have appeared, some may have disappeared.
   *
   * Since: 3.2
   */
  visu_data_signals[NODE_RENDERED_CHANGED_SIGNAL] =
    g_signal_newv("NodeRenderedChanged",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::NodePositionChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when one or more nodes have moved, because of
   * translations or because the user has moved them manually.
   *
   * Since: 3.2
   */
  visu_data_signals[NODE_POSITION_CHANGED_SIGNAL] =
    g_signal_newv ("NodePositionChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::NodePopulationIncrease:
   * @dataObj: the object which received the signal ;
   * @ids: an array of #VisuNode ids.
   *
   * Gets emitted when the number of nodes has changed,
   * increasing. @ids contains all new ids and is -1 terminated.
   *
   * Since: 3.4
   */
  visu_data_signals[NODE_POPULATION_INCREASE_SIGNAL] =
    g_signal_newv ("NodePopulationIncrease",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodePopulationDecrease:
   * @dataObj: the object which received the signal ;
   * @ids: an array of #VisuNode ids.
   *
   * Gets emitted when the number of nodes has changed,
   * decreasing. @ids contains all removed ids and is -1 terminated.
   * When emitted, nodes have already been removed, so no external
   * routines should keep pointers on these nodes.
   *
   * Since: 3.4
   */
  visu_data_signals[NODE_POPULATION_DECREASE_SIGNAL] =
    g_signal_newv ("NodePopulationDecrease",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodePopulationDefined:
   * @dataObj: the object which received the signal ;
   * @nodes: a #VisuNodeArray object or NULL.
   *
   * Gets emitted when the population of nodes is created or
   * destroyed. It is possible then to associate new #VisuNodeProperty
   * for instance.
   *
   * Since: 3.5
   */
  visu_data_signals[NODE_POPULATION_DEFINED_SIGNAL] =
    g_signal_newv ("NodePopulationDefined",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::NodeAskForShowHide:
   * @dataObj: the object which received the signal ;
   * @redraw: a location on a boolean.
   *
   * Gets emitted when external modules should recompute their masking
   * effect on nodes. Location pointed by @redraw must be set to TRUE
   * if the visibility of at least one node is changed.
   *
   * Since: 3.2
   */
  visu_data_signals[NODE_ASK_FOR_SHOW_HIDE_SIGNAL] =
    g_signal_newv ("NodeAskForShowHide",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLThetaPhiOmega:
   * @dataObj: the object which received the signal ;
   * @view: the new #OpenGLView.
   *
   * Gets emitted when the camera angles have been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_THETA_PHI_OMEGA_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLThetaPhiOmega", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLXsYs:
   * @dataObj: the object which received the signal ;
   * @view: the new #OpenGLView.
   *
   * Gets emitted when the camera position has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_XS_YS_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLXsYs", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLGross:
   * @dataObj: the object which received the signal ;
   * @view: the new #OpenGLView.
   *
   * Gets emitted when the camera zoom factor has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_GROSS_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLGross", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLPersp:
   * @dataObj: the object which received the signal ;
   * @view: the new #OpenGLView.
   *
   * Gets emitted when the camera perspective factor has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_PERSP_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLPersp", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLNearFar:
   * @dataObj: the object which received the signal ;
   * @view: the new #OpenGLView.
   *
   * Gets emitted when the viewing field has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLNearFar", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLWidthHeight:
   * @dataObj: the object which received the signal ;
   * @view: the new #OpenGLView.
   *
   * Gets emitted when the viewing frame has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_WIDTH_HEIGHT_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLWidthHeight", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuData::OpenGLFacetteChanged:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when precision of the drawn object has been changed.
   *
   * Since: 3.2
   */
  visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL] =
    g_signal_newv ("OpenGLFacetteChanged", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE, 0, NULL);

  /**
   * VisuData::objectFreed:
   * @dataObj: the object which received the signal ;
   *
   * Gets emitted when the object is been destroyed. All external
   * objects having a reference on this #VisuData should clean it.
   *
   * Since: 3.3
   */
  visu_data_signals[VISU_DATA_FREED_SIGNAL] =
    g_signal_newv ("objectFreed", G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL, g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE, 0, NULL);

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose = visu_data_dispose;
  G_OBJECT_CLASS(klass)->finalize = visu_data_finalize;

  /* Initialise internal variables. */
  /* Get an OpenGL identifier to store all the materials. */
  klass->identifierAllNodes     = openGLObjectList_new(NMAX_TP + 1);
  extensionData = OpenGLExtension_new(EXT_VISU_DATA_ID, nameI18n, description,
				      klass->identifierAllNodes,
				      visuData_rebuildList);
  OpenGLExtensionSet_priority(extensionData, OPENGL_EXTENSION_PRIORITY_FIRST);
  OpenGLExtensionSet_sensitiveToRenderingMode(extensionData, TRUE);
  extensionData->used = 1;
  OpenGLExtensionRegister(extensionData);

  allObjects = (GList*)0;

  /* Register a new NodeData. */
  dataNodeCoord = nodeDataNew_withCallbacks(COORDINATES_ID,
					    setCoordFromString, getCoordFromString);
  nodeDataSet_label(dataNodeCoord, _("Coord. (x, y, z)"));
  nodeDataSet_editable(dataNodeCoord, TRUE);
}

static void visu_data_init(VisuData *obj)
{
  int i;

  DBG_fprintf(stderr, "Visu data : initializing a new object (%p).\n",
	      (gpointer)obj);
  
  /* Connect a method to make all nodes rendered attribute to TRUE
     when the NodeAskForShowHide signal is emited. */
  g_signal_connect(G_OBJECT(obj), "NodeAskForShowHide",
		   G_CALLBACK(onAskForShowHideSignal), (gpointer)0);

  obj->privateDt = g_malloc(sizeof(VisuData_private));
  obj->privateDt->dispose_has_run = FALSE;

  /* Public data. */
  obj->ntype= 0;
  obj->fromVisuElementToInt = (GHashTable*)0;
  obj->fromIntToVisuElement = (VisuElement**)0;
  obj->setColor = (setColorFunc)0;

  /* Private data. */
  obj->privateDt->nodeArray        = (VisuNodeArray*)0;
  obj->privateDt->files            = (GList*)0;
  obj->privateDt->commentary       = g_malloc(sizeof(gchar*) * 2);
  obj->privateDt->commentary[0]    = g_strdup("");
  obj->privateDt->commentary[1]    = (gchar*)0;
  obj->privateDt->translationApply = FALSE;
  for (i = 0; i < 3; i++)
    {
      obj->privateDt->translation[i] = 0.;
      obj->privateDt->extension[i]   = 0.;
    }
  for (i = 0; i < 6; i++)
    obj->privateDt->boxGeometry[i] = 0.;
  obj->privateDt->nSets            = 1;
  obj->privateDt->iSet             = -1;
  obj->privateDt->timeoutList      = (GList*)0;  
  obj->privateDt->attachingWindow  = (GenericRenderingWindow)0;
  obj->privateDt->attachedView     = (OpenGLView*)0;
  obj->privateDt->scaling          = defaultScaling;

  /* Initial allocations. */
  obj->fromVisuElementToInt = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						     NULL, g_free);
  g_return_if_fail(obj->fromVisuElementToInt);

  nodeDataSet_used(dataNodeCoord, obj, 3);

  /* Add object from allObjects list. */
  allObjects = g_list_append(allObjects, (gpointer)obj);

  DBG_fprintf(stderr, "Visu Data: emit the 'DataNew' signal.\n");
  g_signal_emit(VISU_INSTANCE, VISU_SIGNALS[DATANEW_SIGNAL],
		0 /* details */, G_OBJECT(obj), NULL);
  DBG_fprintf(stderr, "Visu Data: emission OK of the 'DataNew' signal.\n");
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void visu_data_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Visu data : dispose object %p.\n", (gpointer)obj);

  if (VISU_DATA(obj)->privateDt->dispose_has_run)
    return;

#if DEBUG == 1
  g_mem_profile();
#endif

  VISU_DATA(obj)->privateDt->dispose_has_run = TRUE;

  DBG_fprintf(stderr, "Visu Data : emit a 'objectFreed' signal.\n");
  g_signal_emit(obj, visu_data_signals[VISU_DATA_FREED_SIGNAL],
		0, NULL);

  nodeDataSet_used(dataNodeCoord, VISU_DATA(obj), 0);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visu_data_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void visu_data_finalize(GObject* obj)
{
  VisuData *data;
  GList *lst;

  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Visu data : finalize object %p.\n", (gpointer)obj);

  /* Remove object from allObjects list. */
  allObjects = g_list_remove(allObjects, (gpointer)obj);

  data = VISU_DATA(obj);

  /* Free public elements. */
  if (data->fromVisuElementToInt)
    g_hash_table_destroy(data->fromVisuElementToInt);
  if (data->fromIntToVisuElement)
    g_free(data->fromIntToVisuElement);

  /* Free privateDts elements. */
  if (data->privateDt)
    {
      DBG_fprintf(stderr, "Visu data: free private data.\n");
      if (data->privateDt->nodeArray)
	visuNodeFree_nodes(data->privateDt->nodeArray);
      if (data->privateDt->files)
	{
	  lst = data->privateDt->files;
	  while (lst)
	    {
	      g_free(((struct FileDescription_struct*)lst->data)->name);
	      g_free(lst->data);
	      lst = g_list_next(lst);
	    }
	  g_list_free(data->privateDt->files);
	}
      if (data->privateDt->commentary)
	g_strfreev(data->privateDt->commentary);
      if (data->privateDt->timeoutList)
	{
	  lst = data->privateDt->timeoutList;
	  while (lst)
	    {
	      g_source_remove(*(guint*)lst->data);
	      g_free(lst->data);
	      lst = g_list_next(lst);
	    }
	  g_list_free(data->privateDt->timeoutList);
	}
      if (data->privateDt->attachedView)
	OpenGLViewFree(data->privateDt->attachedView);
      g_free(data->privateDt);
    }
  /* The free is called by g_type_free_instance... */
/*   g_free(data); */

  /* Chain up to the parent class */
  DBG_fprintf(stderr, "Visu data: chain to parent.\n");
  G_OBJECT_CLASS(visu_data_parent_class)->finalize(obj);

  DBG_fprintf(stderr, "Visu data: freeing ... OK.\n");

#if DEBUG == 1
  g_mem_profile();
#endif
}


/**
 * visuDataNew:
 *
 * This creates an empty #VisuData object.
 *
 * Returns: a newly created #VisuData object (its ref count is set to 1).
 */
VisuData* visuDataNew()
{
  VisuData *data;

#if DEBUG == 1
  g_mem_profile();
#endif

  data = VISU_DATA(g_object_new(VISU_DATA_TYPE, NULL));

  if (!data)
    return (VisuData*)0;

  data->privateDt->attachedView = OpenGLViewNew();

#if DEBUG == 1
  g_mem_profile();
#endif

  return data;
}

VisuData* visuDataNew_withSize(guint w, guint h)
{
  VisuData *data;

#if DEBUG == 1
  g_mem_profile();
#endif

  data = VISU_DATA(g_object_new(VISU_DATA_TYPE, NULL));

  if (!data)
    return (VisuData*)0;

  data->privateDt->attachedView = OpenGLViewNew();
  data->privateDt->attachedView->window->width = w;
  data->privateDt->attachedView->window->height = h;

#if DEBUG == 1
  g_mem_profile();
#endif

  return data;
}

VisuData* visuDataNew_withOpenGLView(OpenGLView *view)
{
  VisuData *data;
  
  g_return_val_if_fail(view, (VisuData*)0);

#if DEBUG == 1
  g_mem_profile();
#endif

  data = VISU_DATA(g_object_new(VISU_DATA_TYPE, NULL));

  if (!data)
    return (VisuData*)0;

  data->privateDt->attachedView = OpenGLViewCopy(view);

#if DEBUG == 1
  g_mem_profile();
#endif

  return data;
}


static gboolean setCoordFromString(DataNode *dataNode, VisuData *dataObj,
				   VisuNode *node, gchar* labelIn,
				   gchar **labelOut, gboolean *modify)
{
  int res, ln, i;
  gchar **datas;
  gboolean error;
  float valueFloat;

  g_return_val_if_fail(IS_DATA_NODE_TYPE(dataNode) &&
		       IS_VISU_DATA_TYPE(dataObj) && node, FALSE);
  g_return_val_if_fail(labelIn && labelOut && modify, FALSE);

  /* Parse the given labelIn.
     remove first and last parenthesis. */
  if (labelIn[0] == '(')
    labelIn += 1;
  ln = strlen(labelIn);
  if (labelIn[ln - 1] == ')')
    labelIn[ln - 1] = '\0';
  datas = g_strsplit(labelIn, ";", 3);
  *modify = FALSE;
  for (i = 0; datas[i]; i++)
    {
      error = FALSE;
      res = sscanf(datas[i], "%f", &valueFloat);
      if (res != 1)
	error = TRUE;
      else
	if (node->xyz[i] != valueFloat)
	  {
	    node->xyz[i] = valueFloat;
	    *modify = TRUE;
	  }
      if (error)
	{
	  *labelOut = getCoordFromString(dataNode, dataObj, node);
	  g_strfreev(datas);
	  return FALSE;
	}
    }
  if (i != 3)
    error = TRUE;
  else
    error = FALSE;

  *labelOut = getCoordFromString(dataNode, dataObj, node);
  g_strfreev(datas);

  if (*modify)
    {
      visuData_createNodes(dataObj, dataObj->fromIntToVisuElement[node->posElement]);
      visuDataEmit_nodePositionChanged(dataObj);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }

  return !error;
}
static gchar* getCoordFromString(DataNode *dataNode, VisuData *dataObj,
				 VisuNode *node)
{
  GString *str;
  gchar *value;

  g_return_val_if_fail(IS_DATA_NODE_TYPE(dataNode) &&
		       IS_VISU_DATA_TYPE(dataObj) && node, (gchar*)0);

  /* Check if the given property has an association with the given VisuData. */
  DBG_fprintf(stderr, "Visu Data: get label for node coordinates.\n");

  /* Set the format attribute. */
  str = g_string_new("");
  g_string_append_printf(str, "( %g ; %g ; %g )",
			 node->xyz[0], node->xyz[1], node->xyz[2]);
  value = str->str;
  g_string_free(str, FALSE);

  DBG_fprintf(stderr, "Visu Data: get values '%s'.\n", value);
  return value;
}
static void onAskForShowHideSignal(VisuData *visuData, gboolean *redraw,
				   gpointer data _U_)
{
  unsigned int i, j;

  DBG_fprintf(stderr, "Visu Data : caught the 'NodeAskForShowHide' signal,"
	      " setting all node rendered attribute to TRUE.\n");

  for (i = 0; i < visuData->privateDt->nodeArray->ntype; i++)
    for(j = 0; j < visuData->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
      *redraw = visuNodeSet_visibility
	(&visuData->privateDt->nodeArray->nodes[i][j], TRUE) || *redraw;

  DBG_fprintf(stderr, "             - returned redraw value : %d.\n", *redraw);
}




/***************/
/* OpenGL Part */
/***************/
void visuData_createNodes(VisuData *data, VisuElement *ele)
{
  visuRenderingCreate_nodeFunc createNode;
  VisuNode *node;
  int *nEle;
  unsigned int j;
  int mat;

  g_return_if_fail(data && ele);

  nEle = (int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)ele);
  g_return_if_fail(nEle);
  createNode = visuRenderingClassGet_currentNodeFunc();
  g_return_if_fail(createNode);

  if (!ele->materialIsUpToDate)
    visuElement_createMaterial(ele);

  glNewList(VISU_DATA_GET_CLASS(data)->identifierAllNodes + *nEle + 1, GL_COMPILE);
  if (ele->rendered)
    {
      /* Set the material by default if available. */
      mat = visuElementGet_identifierMaterial(ele);
      if (!mat)
	g_warning("Can't find the material identifier for the"
		  " element '%s'.\n", ele->name);
      else
	glCallList(mat);
      DBG_fprintf(stderr, "Visu Data: creating glObjectList of nodes for '%s' - %d.\n",
		  ele->name, mat);
      for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[*nEle]; j++)
	{
	  node = data->privateDt->nodeArray->nodes[*nEle] + j;
	  if (node->rendered)
	    {
	      glLoadName((GLuint)node->number);
	      createNode(data, node, ele);
	    }
	}
    }
  else
    DBG_fprintf(stderr, "Visu Data: skipping glObjectList of nodes for '%s' - %d.\n",
		ele->name, ele->openGLIdentifier);
  glEndList();
}

void visuData_createNode(VisuData *data, VisuNode *node)
{
  visuRenderingCreate_nodeFunc createNode;
  int mat;
  VisuElement *ele;

  createNode = visuRenderingClassGet_currentNodeFunc();
  g_return_if_fail(createNode);

  ele = data->fromIntToVisuElement[node->posElement];
  g_return_if_fail(ele);
  /* Set the material by default if available. */
  mat = visuElementGet_identifierMaterial(ele);
  if (!mat)
    g_warning("Can't find the material identifier for the"
	      " element '%s'.\n", ele->name);
  else
    glCallList(mat);

  glLoadName((GLuint)node->number);
  createNode(data, node, ele);
}

void visuData_createAllNodes(VisuData *data)
{
  unsigned int i;
#if DEBUG == 1
  GTimer *timer;
  gulong fractionTimer;
#endif

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  DBG_fprintf(stderr, "Visu Data: 'visuData_createAllNodes' called.\n");

#if DEBUG == 1
  timer = g_timer_new();
  g_timer_start(timer);
#endif

  DBG_fprintf(stderr, "Visu Data: loop on elements.\n");
  for (i = 0; i < data->ntype; i++)
    visuData_createNodes(data, data->fromIntToVisuElement[i]);
  DBG_fprintf(stderr, "Visu Data: OK.\n");

  glNewList(VISU_DATA_GET_CLASS(data)->identifierAllNodes, GL_COMPILE);
  glLineWidth(1);
  for (i = 0; i < data->ntype; i++)
    {
      DBG_fprintf(stderr, "Visu Data: add list %d.\n",
		  VISU_DATA_GET_CLASS(data)->identifierAllNodes + i + 1);
/*       if (currentVisuData->fromIntToVisuElement[i]->rendered) */
      glCallList(VISU_DATA_GET_CLASS(data)->identifierAllNodes + i + 1);
    }
  glEndList();

#if DEBUG == 1
  g_timer_stop(timer);
  fprintf(stderr, "Visu Data: lists built in %g micro-s.\n", g_timer_elapsed(timer, &fractionTimer)/1e-6);
  g_timer_destroy(timer);
#endif
}
void visuData_createAllElements(VisuData *data)
{
  unsigned int i;
  int id;
  visuRenderingCreate_elementFunc createElement;

  createElement = visuRenderingClassGet_currentElementFunc();
  g_return_if_fail(createElement);

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  DBG_fprintf(stderr, "Visu Data: create OpenGl elements for"
	      " all VisuElement used in given VisuData %p.\n", (gpointer)data);
  for (i = 0; i < data->ntype; i++)
    {
      DBG_fprintf(stderr, "Visu Data: element '%s' %p.\n",
		  data->fromIntToVisuElement[i]->name,
                  (gpointer)data->fromIntToVisuElement[i]);
      id = createElement(data, data->fromIntToVisuElement[i]);
      data->fromIntToVisuElement[i]->openGLIdentifier = id;
    }
  DBG_fprintf(stderr, "Visu Data: creation done for all elements.\n");
}

int visuDataGet_objectList(VisuData *data)
{
  return VISU_DATA_GET_CLASS(data)->identifierAllNodes;
}

static void visuData_rebuildList(VisuData *dataObj)
{
  unsigned int i;

  DBG_fprintf(stderr, "Visu Data : rebuilding object list for visuData %p.\n", (gpointer)dataObj);
  g_return_if_fail(dataObj);

  for (i = 0; i < dataObj->ntype; i++)
    visuElement_createMaterial(dataObj->fromIntToVisuElement[i]);
  visuData_createAllElements(dataObj);
  visuData_createAllNodes(dataObj);
}

int visuDataSet_population(VisuData *data, unsigned int nbOfTypes,
			   unsigned int* nNodesPerElement,
			   VisuElement **visuElementUsed)
{
  unsigned int i;
  int *pos;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  g_return_val_if_fail(nbOfTypes > 0 && nNodesPerElement, 0);

  DBG_fprintf(stderr, "Visu Data: populating VisuData %p", (gpointer)data);

  data->fromIntToVisuElement = g_malloc(sizeof(VisuElement*) * nbOfTypes);
  data->ntype                = nbOfTypes;
  
  for(i = 0; i < data->ntype; i++)
    {
      pos = g_malloc(sizeof(int));
      *pos = i;
      g_hash_table_insert(data->fromVisuElementToInt,
			  (gpointer)visuElementUsed[i], (gpointer)pos);
      data->fromIntToVisuElement[i] = visuElementUsed[i];
    }
  DBG_fprintf(stderr, " ... OK\n");

  data->privateDt->nodeArray = visuNodeNew_nodes((unsigned int)nbOfTypes,
						 (unsigned int*)nNodesPerElement);
  g_return_val_if_fail(data->privateDt->nodeArray, 0);

  DBG_fprintf(stderr, "VisuData: emit NodePopulationDefined.\n");
  g_signal_emit(data, visu_data_signals[NODE_POPULATION_DEFINED_SIGNAL],
		0, data->privateDt->nodeArray, NULL);
  DBG_fprintf(stderr, "VisuData: emission done (NodePopulationDefined).\n");

  return 1;
}
void visuDataFree_population(VisuData *data)
{
  if (!data)
    return;

  DBG_fprintf(stderr, "Visu Data: freeing the population of VisuData %p ...\n", (gpointer)data);

  if (data->fromVisuElementToInt)
    g_hash_table_destroy(data->fromVisuElementToInt);
  data->fromVisuElementToInt = g_hash_table_new_full(g_direct_hash, g_direct_equal,
						     NULL, g_free);
  if (data->fromIntToVisuElement)
    g_free(data->fromIntToVisuElement);
  data->fromIntToVisuElement = (VisuElement**)0;

  if (data->privateDt->nodeArray)
    visuNodeFree_nodes(data->privateDt->nodeArray);
  data->privateDt->nodeArray = (VisuNodeArray*)0;

  data->privateDt->iSet = -1;
  g_object_set_data(G_OBJECT(data), "unit", (gpointer)0);
  data->privateDt->extension[0]   = 0.f;
  data->privateDt->extension[1]   = 0.f;
  data->privateDt->extension[2]   = 0.f;
  data->privateDt->translation[0] = 0.f;
  data->privateDt->translation[1] = 0.f;
  data->privateDt->translation[2] = 0.f;

  DBG_fprintf(stderr, "Visu Data: freeing ... OK.\n");

  DBG_fprintf(stderr, "VisuData: emit NodePopulationDefined.\n");
  g_signal_emit(data, visu_data_signals[NODE_POPULATION_DEFINED_SIGNAL],
		0, data->privateDt->nodeArray, NULL);
  DBG_fprintf(stderr, "VisuData: emission done (NodePopulationDefined).\n");
}

void visuDataAdd_file(VisuData *data, gchar* file, int kind, FileFormat *format)
{
  struct FileDescription_struct *dt;

  g_return_if_fail(data && file);

  dt = g_malloc(sizeof(struct FileDescription_struct));
  dt->kind = kind;
  dt->name = g_strdup(file);
  dt->format = format;
  data->privateDt->files = g_list_prepend(data->privateDt->files, (gpointer)dt);
  DBG_fprintf(stderr, "Visu Data : adding '%s' filename (key: %d) to the VisuData %p.\n",
	      file, kind, (gpointer)data);
}
void visuDataRemove_allFiles(VisuData *data)
{
  GList *lst;

  g_return_if_fail(data);

  lst = data->privateDt->files;
  while (lst)
    {
      g_free(((struct FileDescription_struct*)lst->data)->name);
      g_free(lst->data);
      lst = g_list_next(lst);
    }
  g_list_free(data->privateDt->files);
  data->privateDt->files = (GList*)0;
  visuDataSet_nSet(data, 1);
}
gchar* visuDataGet_file(VisuData *data, int kind, FileFormat **format)
{
  GList *lst;

  g_return_val_if_fail(data, (gchar*)0);

  DBG_fprintf(stderr, "Visu Data : retrieve filename with key %d from the VisuData %p.\n",
	      kind, (gpointer)data);
  if (format)
    *format = (FileFormat*)0;
  lst = data->privateDt->files;
  while (lst)
    {
      if (((struct FileDescription_struct*)lst->data)->kind == kind)
	{
	  if (format)
	    *format = ((struct FileDescription_struct*)lst->data)->format;
	  return ((struct FileDescription_struct*)lst->data)->name;
	}
      lst = g_list_next(lst);
    }
  return (gchar*)0;
}
void visuDataSet_fileCommentary(VisuData *data, gchar* commentary, gint iSet)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) &&
		   iSet >= 0 && iSet < data->privateDt->nSets);
  
  g_free(data->privateDt->commentary[iSet]);
  data->privateDt->commentary[iSet] = g_strdup(commentary);
}
gchar* visuDataGet_fileCommentary(VisuData *data, gint iSet)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) &&
		       iSet >= 0 && iSet < data->privateDt->nSets, (gchar*)0);
  
  return data->privateDt->commentary[iSet];
}
void visuDataSet_nSet(VisuData *data, int nSet)
{
  int i;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && nSet > 0);

  data->privateDt->nSets = nSet;
  if (data->privateDt->commentary)
    g_strfreev(data->privateDt->commentary);
  data->privateDt->commentary = g_malloc(sizeof(gchar*) * (nSet + 1));
  for (i = 0; i < nSet; i++)
    data->privateDt->commentary[i] = g_strdup("");
  data->privateDt->commentary[nSet] = (gchar*)0;
}
int visuDataGet_nSet(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 1);

  return data->privateDt->nSets;
}
void visuDataSet_setId(VisuData *data, int iSet)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));
  DBG_fprintf(stderr, "Visu Data: set id to %d (%d).\n",
	      iSet, data->privateDt->nSets);

  g_return_if_fail(iSet >= 0 && iSet < data->privateDt->nSets);

  data->privateDt->iSet = iSet;
}
int visuDataGet_setId(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), -1);

  DBG_fprintf(stderr, "Visu Data: get current set id %d.\n", data->privateDt->iSet);
  return data->privateDt->iSet;
}

gboolean visuData_compareElements(VisuData *data1, VisuData *data2)
{
  unsigned int i;

  g_return_val_if_fail(data1 && data2, FALSE);

  DBG_fprintf(stderr, "Visu Data : comparing composition of '%p' and '%p'.\n",
	      (gpointer)data1, (gpointer)data2);
  if (data1 == data2)
    return FALSE;

  if (data1->ntype != data2->ntype)
    return FALSE;

  for (i = 0; i< data1->ntype; i++)
    if (!g_hash_table_lookup(data2->fromVisuElementToInt,
			     (gpointer)data1->fromIntToVisuElement[i]))
      return FALSE;

  for (i = 0; i< data2->ntype; i++)
    if (!g_hash_table_lookup(data1->fromVisuElementToInt,
			     (gpointer)data2->fromIntToVisuElement[i]))
      return FALSE;

  return TRUE;
}
void visuDataSet_changeElementFlag(VisuData *data, gboolean changeElement)
{
  gboolean *val;

  g_return_if_fail(data);

  /* Test if already exists */
  val = (gboolean*)g_object_get_data(G_OBJECT(data), "changeElementListFlag");
  if (!val)
    {
      val = g_malloc(sizeof(gboolean));
      g_object_set_data_full(G_OBJECT(data), "changeElementListFlag",
			     val, g_free);
    }
  *val = changeElement;
}
gboolean visuDataGet_changeElementFlag(VisuData *data)
{
  gboolean *val;

  g_return_val_if_fail(data, FALSE);

  val = (gboolean*)g_object_get_data(G_OBJECT(data), "changeElementListFlag");
  if (val)
    return *val;
  else
    return FALSE;
}


/*************************/
/* The geometry routines */
/*************************/
float* visuDataGet_XYZtranslation(VisuData* data)
{
  float *trans;
  int i;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (float*)0);

  trans = g_malloc(sizeof(float) * 3);
  for (i = 0; i < 3; i++)
    trans[i] = data->privateDt->translation[i];
  return trans;
}
int visuDataSet_XYZtranslation(VisuData* data, float xyz[3])
{
  int i, res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && xyz, 0);

  res = 0;
  for (i = 0; i < 3; i++)
    {
      if (data->privateDt->translation[i] != xyz[i])
	{
	  data->privateDt->translation[i] = xyz[i];
	  res = 1;
	}
    }
  DBG_fprintf(stderr, "Visu Data : set translation to : %f %f %f\n",
	      data->privateDt->translation[0], data->privateDt->translation[1],
	      data->privateDt->translation[2]);

  return res;
}

void visuDataGet_boxMatrix(VisuData *data, float matrix[3][3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && matrix);

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      matrix[i][j] = (float)data->privateDt->fromBoxtoXYZD0[i][j];
}
void visuDataGet_boxMatrixD0(VisuData *data, double matrix[3][3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && matrix);

  for (i = 0; i < 3; i++)
    for (j = 0; j < 3; j++)
      matrix[i][j] = data->privateDt->fromBoxtoXYZD0[i][j];
}
void visuDataGet_boxVertices(VisuData *data, float vertices[8][3],
			     gboolean withExtension)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  openGLViewGet_boxVertices(data->privateDt->attachedView, vertices, withExtension);
}
void visuDataConvert_XYZtoBoxCoordinates(VisuData *data, float boxCoord[3], float xyz[3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && boxCoord && xyz);

  for (i = 0; i < 3; i++)
    {
      boxCoord[i] = 0.;
      for (j = 0; j < 3; j++)
	boxCoord[i] += (float)data->privateDt->fromXYZtoBoxD0[i][j] * xyz[j];
    }
}

void visuDataConvert_boxCoordinatestoXYZ(VisuData *data, float xyz[3], float boxCoord[3])
{
  int i, j;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && boxCoord && xyz);

  for (i = 0; i < 3; i++)
    {
      xyz[i] = 0.;
      for (j = 0; j < 3; j++)
	xyz[i] += (float)data->privateDt->fromBoxtoXYZD0[i][j] * boxCoord[j];
    }
}

OpenGLBoxBoundaries visuDataGet_boundaryConditions(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), BOX_PERIODIC);

  return data->privateDt->attachedView->box->bc;
}
void visuDataSet_boxGeometry(VisuData *data, double geometry[6],
			     OpenGLBoxBoundaries bc)
{
  int i;
  double normY, normZ;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && geometry);

  openGLViewSet_boundaryConditions(data->privateDt->attachedView, bc);
  for (i = 0; i < 6; i++)
    data->privateDt->boxGeometry[i] = geometry[i];
  normY = sqrt(geometry[VISU_DATA_BOX_DYX] * geometry[VISU_DATA_BOX_DYX] +
	       geometry[VISU_DATA_BOX_DYY] * geometry[VISU_DATA_BOX_DYY]);
  normZ = sqrt(geometry[VISU_DATA_BOX_DZX] * geometry[VISU_DATA_BOX_DZX] +
	       geometry[VISU_DATA_BOX_DZY] * geometry[VISU_DATA_BOX_DZY] +
	       geometry[VISU_DATA_BOX_DZZ] * geometry[VISU_DATA_BOX_DZZ]);
  data->privateDt->boxNorm[0] = 1.;
  for (i = 1; i < 3; i++)
    data->privateDt->boxNorm[i] = geometry[i] / normY;
  for (i = 3; i < 6; i++)
    data->privateDt->boxNorm[i] = geometry[i] / normZ;

  DBG_fprintf(stderr, "Visu Data: the bounding box is set to:\n %f %f %f\n %f %f %f\n",
	      geometry[0], geometry[1], geometry[2], geometry[3],
	      geometry[4], geometry[5]);

  /* Create the transformation matrix. */
  data->privateDt->fromXYZtoBoxD0[0][0] =
    1. / data->privateDt->boxGeometry[VISU_DATA_BOX_DXX];
  data->privateDt->fromXYZtoBoxD0[0][1] =
    - data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYY];
  data->privateDt->fromXYZtoBoxD0[0][2] =
    - (data->privateDt->boxGeometry[VISU_DATA_BOX_DZX] /
       data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] -
       data->privateDt->boxGeometry[VISU_DATA_BOX_DYX] *
       data->privateDt->boxGeometry[VISU_DATA_BOX_DZY] / 
       data->privateDt->boxGeometry[VISU_DATA_BOX_DXX] / 
       data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] ) /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];
  data->privateDt->fromXYZtoBoxD0[1][0] = 0.;
  data->privateDt->fromXYZtoBoxD0[1][1] =
    1. / data->privateDt->boxGeometry[VISU_DATA_BOX_DYY];
  data->privateDt->fromXYZtoBoxD0[1][2] =
    - data->privateDt->boxGeometry[VISU_DATA_BOX_DZY] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYY] /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];
  data->privateDt->fromXYZtoBoxD0[2][0] = 0.;
  data->privateDt->fromXYZtoBoxD0[2][1] = 0.;
  data->privateDt->fromXYZtoBoxD0[2][2] = 1. /
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];

  data->privateDt->fromBoxtoXYZD0[0][0] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DXX];
  data->privateDt->fromBoxtoXYZD0[0][1] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYX];
  data->privateDt->fromBoxtoXYZD0[0][2] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZX];
  data->privateDt->fromBoxtoXYZD0[1][0] = 0.;
  data->privateDt->fromBoxtoXYZD0[1][1] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DYY];
  data->privateDt->fromBoxtoXYZD0[1][2] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZY];
  data->privateDt->fromBoxtoXYZD0[2][0] = 0.;
  data->privateDt->fromBoxtoXYZD0[2][1] = 0.;
  data->privateDt->fromBoxtoXYZD0[2][2] =
    data->privateDt->boxGeometry[VISU_DATA_BOX_DZZ];

  DBG_fprintf(stderr, "Visu Data: set box geometry done.\n");
}
void visuDataApply_boxGeometry(VisuData *data, float refLength)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  DBG_fprintf(stderr, "Visu Data: set box ref length %g.\n", refLength);
  /* Set the box definition and raise change signals. */
  applyBox(data, refLength, visuDataGet_allElementExtens(data) +
	   getAllNodeExtens(data));
}
float visuDataGet_boxGeometry(VisuData *data, int vector)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && vector >= 0 && vector < 6, -1);

  return (float)data->privateDt->boxGeometry[vector];
}
gboolean visuDataGet_nodeBoxFromNumber(VisuData *data, guint nodeId, int nodeBox[3])
{
  float xred[3];
  float xcart[3];
  VisuNodeArray *visuNodeArray = visuDataGet_nodeArray(data);

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) &&
		       nodeId < visuNodeArray->idCounter, FALSE);
  visuDataGet_nodePosition(data, visuDataGet_nodeFromNumber(data,nodeId), xcart);
  visuDataConvert_XYZtoBoxCoordinates(data, xred, xcart);
  nodeBox[0] = floor(xred[0]);
  nodeBox[1] = floor(xred[1]);
  nodeBox[2] = floor(xred[2]);
  DBG_fprintf(stderr, "Visu Data: nodeBox found for atom %d : %d %d %d.\n", nodeId, nodeBox[0], nodeBox[1], nodeBox[2]);
  return TRUE;
}
gboolean visuDataGet_nodeBoxFromCoord(VisuData *data, float xcart[3], int nodeBox[3])
{
  float xred[3];
  visuDataConvert_XYZtoBoxCoordinates(data, xred, xcart);
  nodeBox[0] = floor(xred[0]);
  nodeBox[1] = floor(xred[1]);
  nodeBox[2] = floor(xred[2]);
  DBG_fprintf(stderr, "Visu Data: nodeBox found for atom at %f %f %f : %d %d %d.\n",
		xcart[0], xcart[1], xcart[2], nodeBox[0], nodeBox[1], nodeBox[2]);
  return TRUE;
}
VisuUnits visuDataGet_unit(VisuData *data)
{
  gpointer unit_;

  unit_ = g_object_get_data(G_OBJECT(data), "unit");
  if (unit_)
    return GPOINTER_TO_INT(unit_);
  else
    return unit_undefined;
}
gboolean visuDataSet_unit(VisuData *data, VisuUnits unit)
{
  VisuUnits unit_;
  double fact;
  double box[6];
  VisuDataIter iter;

  unit_ = visuDataGet_unit(data);
  DBG_fprintf(stderr, "Visu Data: set unit to %d (%d).\n", unit, unit_);
  if (unit_ == unit)
    return FALSE;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  g_object_set_data(G_OBJECT(data), "unit", GINT_TO_POINTER(unit));
  if (unit_ == unit_undefined || unit == unit_undefined)
    return TRUE;

  fact = (double)visuToolsGet_unitValueInMeter(unit_) /
    visuToolsGet_unitValueInMeter(unit);
  DBG_fprintf(stderr, "Visu Data: multiplying factor is %g.\n", fact);

  /* We do an homothety on the box. */
  box[0] = data->privateDt->boxGeometry[0] * fact;
  box[1] = data->privateDt->boxGeometry[1] * fact;
  box[2] = data->privateDt->boxGeometry[2] * fact;
  box[3] = data->privateDt->boxGeometry[3] * fact;
  box[4] = data->privateDt->boxGeometry[4] * fact;
  box[5] = data->privateDt->boxGeometry[5] * fact;
  visuDataSet_boxGeometry(data, box, data->privateDt->attachedView->box->bc);

  /* We do an homothety on the nodes. */
  data->privateDt->translation[0] *= fact;
  data->privateDt->translation[1] *= fact;
  data->privateDt->translation[2] *= fact;
  visuDataIter_new(data, &iter);
  for( visuDataIter_start(data, &iter); iter.node;
       visuDataIter_next(data, &iter))
    {
      iter.node->xyz[0] *= fact;
      iter.node->xyz[1] *= fact;
      iter.node->xyz[2] *= fact;
      iter.node->translation[0] *= fact;
      iter.node->translation[1] *= fact;
      iter.node->translation[2] *= fact;
    }

  /* We raise the signals. */
  visuDataApply_boxGeometry(data, 0.f);
  visuDataEmit_nodePositionChanged(data);

  return TRUE;
}
gboolean visuData_constrainedElementInTheBox(VisuData *data, VisuElement *element)
{
  int *indexEle, i;
  unsigned int j, k;
  gboolean moved, changed;
  float cartCoord[3], boxCoord[3], bounds[3], size[3];
  VisuNode *node;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && element, 0);

  indexEle = (int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)element);
  g_return_val_if_fail(indexEle, 0);

  i = *indexEle;

  if (!data->fromIntToVisuElement[i]->rendered)
    return 0;

  DBG_fprintf(stderr, "Visu Data : Checking for nodes of element '%s' to be in the box.\n",
	      element->name);
  changed = FALSE;
  bounds[0] = ceil(data->privateDt->extension[0]);
  bounds[1] = ceil(data->privateDt->extension[1]);
  bounds[2] = ceil(data->privateDt->extension[2]);
  size[0]   = 1. + 2. * bounds[0];
  size[1]   = 1. + 2. * bounds[1];
  size[2]   = 1. + 2. * bounds[2];
  for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
    {
      node = data->privateDt->nodeArray->nodes[i] + j;
      cartCoord[0] = node->xyz[0] + data->privateDt->translation[0] +
	node->translation[0];
      cartCoord[1] = node->xyz[1] + data->privateDt->translation[1] +
	node->translation[1];
      cartCoord[2] = node->xyz[2] + data->privateDt->translation[2] +
	node->translation[2];
      visuDataConvert_XYZtoBoxCoordinates(data, boxCoord, cartCoord);
      moved = FALSE;
      for (k = 0; k < 3; k++)
	{
	  while (boxCoord[k] < - bounds[k])
	    {
	      moved = TRUE;
	      boxCoord[k] += size[k];
	    }
	  while (boxCoord[k] >= 1. + bounds[k])
	    {
	      moved = TRUE;
	      boxCoord[k] -= size[k];
	    }
	}
      if (moved)
	{
	  changed = TRUE;
	  visuDataConvert_boxCoordinatestoXYZ(data, cartCoord, boxCoord);
	  node->translation[0] = cartCoord[0] - node->xyz[0] -
	    data->privateDt->translation[0];
	  node->translation[1] = cartCoord[1] - node->xyz[1] -
	    data->privateDt->translation[1];
	  node->translation[2] = cartCoord[2] - node->xyz[2] -
	    data->privateDt->translation[2];
	  DBG_fprintf(stderr, "Visu Data: move atom %d to: %gx%gx%g | %gx%gx%g.\n",
		      node->number, cartCoord[0], cartCoord[1], cartCoord[2],
		      boxCoord[0], boxCoord[1], boxCoord[2]);
	}
    }
  data->privateDt->translationApply = TRUE;
  return changed;
}
gboolean visuData_constrainedInTheBox(VisuData *data)
{
  unsigned int i;
  gboolean changed;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);

  changed = FALSE;
  for (i = 0; i < data->ntype; i++)
    changed = visuData_constrainedElementInTheBox(data, data->fromIntToVisuElement[i]) || changed;
  return changed;
}
gboolean visuData_constrainedFree(VisuData *data)
{
  unsigned int i, j;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);

  for (i = 0; i < data->privateDt->nodeArray->ntype; i++)
    for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
      {
	data->privateDt->nodeArray->nodes[i][j].translation[0] = 0.;
	data->privateDt->nodeArray->nodes[i][j].translation[1] = 0.;
	data->privateDt->nodeArray->nodes[i][j].translation[2] = 0.;
      }
  data->privateDt->translationApply = FALSE;
  return TRUE;
}
gboolean visuDataGet_translationStatus(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  
  return data->privateDt->translationApply;
}

void visuDataSet_tightBox(VisuData *data)
{
  double xMin, yMin, zMin, xMax, yMax, zMax;
  double boxGeometry[6];
  float translation[3];
  unsigned int i, j;
  
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  /* Store the coordinates */
  xMin = 1e5;
  yMin = 1e5;
  zMin = 1e5;
  xMax = -1e5;
  yMax = -1e5;
  zMax = -1e5;
  
  for (i = 0; i < data->privateDt->nodeArray->ntype; i++)
    for(j = 0; j < data->privateDt->nodeArray->numberOfStoredNodes[i]; j++)
      {
	xMin = MIN(xMin, data->privateDt->nodeArray->nodes[i][j].xyz[0]);
	yMin = MIN(yMin, data->privateDt->nodeArray->nodes[i][j].xyz[1]);
	zMin = MIN(zMin, data->privateDt->nodeArray->nodes[i][j].xyz[2]);
	xMax = MAX(xMax, data->privateDt->nodeArray->nodes[i][j].xyz[0]);
	yMax = MAX(yMax, data->privateDt->nodeArray->nodes[i][j].xyz[1]);
	zMax = MAX(zMax, data->privateDt->nodeArray->nodes[i][j].xyz[2]);
      }

  DBG_fprintf(stderr, "Visu Data: the elements are in [%f, %f]x[%f, %f]x[%f, %f].\n",
	      xMin, xMax, yMin, yMax, zMin, zMax);
  boxGeometry[0] = xMax - xMin;
  boxGeometry[1] = 0.;
  boxGeometry[2] = yMax - yMin;
  boxGeometry[3] = 0.;
  boxGeometry[4] = 0.;
  boxGeometry[5] = zMax - zMin;
  visuDataSet_boxGeometry(data, boxGeometry, BOX_FREE);

  translation[0] = -xMin;
  translation[1] = -yMin;
  translation[2] = -zMin;
  visuDataSet_XYZtranslation(data, translation);
}

static void applyBox(VisuData *data, float refLength, float margin)
{
  float box[6];

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  /* We update the extension of the OpenGL box. */
  DBG_fprintf(stderr, "Visu Data: change box and ref length"
	      " (%f - %f) and emit signal.\n", refLength, margin);
  box[0] = (float)data->privateDt->boxGeometry[0];
  box[1] = (float)data->privateDt->boxGeometry[1];
  box[2] = (float)data->privateDt->boxGeometry[2];
  box[3] = (float)data->privateDt->boxGeometry[3];
  box[4] = (float)data->privateDt->boxGeometry[4];
  box[5] = (float)data->privateDt->boxGeometry[5];
  openGLViewSet_box(data->privateDt->attachedView, box,
		    data->privateDt->extension, margin);
  if (refLength >= 0.f)
    openGLViewSet_refLength(data->privateDt->attachedView, refLength);
  g_signal_emit(data, visu_data_signals[BOX_SIZE_CHANGED_SIGNAL],
		0 , NULL);
  DBG_fprintf(stderr, "VisuData: emission done (BoxSizeChanged).\n");

  DBG_fprintf(stderr, "VisuData: emit OpenGLNearFar.\n");
  g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		0, data->privateDt->attachedView, NULL);
  DBG_fprintf(stderr, "VisuData: emission done (OpenGLNearFar).\n");

  DBG_fprintf(stderr, "VisuData: call 'visuData_createAllElements'.\n");
  visuData_createAllElements(data);
  DBG_fprintf(stderr, "VisuData: call 'visuData_createAllElements' done.\n");
}
gboolean visuData_replicate(VisuData *data, float extension[3], gboolean *rebuild)
{
  int i;
  gboolean rebuild_, changed;
  /* Will store the index of removed or added nodes, always finish by -1.
     The two first values are specific and used to store:
     - the current number of index ;
     - the allocated size of the array (not including these two
       specific values and the -1 at the end. */
  int *index;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);
  g_return_val_if_fail(extension[0] >= 0. &&
		       extension[1] >= 0. &&
		       extension[2] >= 0., FALSE);

  /* Keep only three digits for the extension to avoid rounding
     troubles. */
  extension[0] = (float)((int)(extension[0] * 1000)) / 1000;
  extension[1] = (float)((int)(extension[1] * 1000)) / 1000;
  extension[2] = (float)((int)(extension[2] * 1000)) / 1000;
  DBG_fprintf(stderr, "Visu Data: modify extension (%g, %g, %g).\n",
	      extension[0], extension[1], extension[2]);

  rebuild_ = changed = FALSE;
  for (i = 0; i < 3; i++)
    {
      if (data->privateDt->extension[i] > extension[i])
	{
	  index = shrinkNodeList(data, i, extension[i]);
	  if (index)
	    {
	      /* We finish the list with -1. */
	      index[index[0] + 2] = -1;
	      visuDataRemove_nodes(data, index + 2);
	      rebuild_ = TRUE;
	      g_free(index);
	    }
	  changed = TRUE;
	}
      else if (data->privateDt->extension[i] < extension[i])
	{
	  index = extendNodeList(data, i, extension[i]);
	  if (index)
	    {
	      /* We finish the list with -1. */
	      index[index[0] + 2] = -1;
	      DBG_fprintf(stderr, "Visu Data: emit a "
			  "'NodePopulationIncrease' signal.\n");
	      g_signal_emit(G_OBJECT(data),
			    visu_data_signals[NODE_POPULATION_INCREASE_SIGNAL],
			    0, (gpointer)index, NULL);
	      rebuild_ = TRUE;
	      g_free(index);
	    }
	  changed = TRUE;
	}
      data->privateDt->extension[i] = extension[i];
    }
  if (changed)
    applyBox(data, -1.f, -1.f);
  if (rebuild)
    *rebuild = rebuild_;
  if (DEBUG)
    visuNodeTrace_property(data->privateDt->nodeArray, "originalId");
  return changed;
}
gboolean visuData_restore(VisuData *data)
{
  gboolean rebuild;
  int *nodeNumbers;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  data->privateDt->extension[0] = 0.f;
  data->privateDt->extension[1] = 0.f;
  data->privateDt->extension[2] = 0.f;
  applyBox(data, -1.f, -1.f);

  nodeNumbers = (int*)0;
  rebuild = visuNodeRemove_allDuplicateNodes(data->privateDt->nodeArray,
					     &nodeNumbers);

  if (rebuild)
    {
      DBG_fprintf(stderr, "Visu Data: emit a 'NodePopulationDecrease' signal.\n");
      g_signal_emit(G_OBJECT(data),
		    visu_data_signals[NODE_POPULATION_DECREASE_SIGNAL],
		    0, (gpointer)nodeNumbers, NULL);
      g_free(nodeNumbers);
      return TRUE;
    }
  return FALSE;
}
static int* reallocIndexList(int *index, int size)
{
  int currentSize;
  
  if (index)
    currentSize = index[1];
  else
    currentSize = 0;

  index = g_realloc(index, sizeof(int) * (currentSize + size + 3));
  index[0] = currentSize;
  index[1] = currentSize + size;

  DBG_fprintf(stderr, "Visu Data: realloc internal index list (%d %d).\n",
	      index[0], index[1]);
  return index;
}
static int* addIndexList(int *index, int value, int size)
{
  if (!index || index[0] == index[1])
    index = reallocIndexList(index, size);
  index[index[0] + 2]  = value;
  index[0]            += 1;
  return index;
}
static int* shrinkNodeList(VisuData *data, int coord, float valueTo)
{
  float cartCoord[3], boxCoord[3];
  int *index;
  VisuDataIter iter;

  g_return_val_if_fail(coord == 0 || coord == 1 || coord == 2, FALSE);
  g_return_val_if_fail(valueTo < data->privateDt->extension[coord], FALSE);

  DBG_fprintf(stderr, "Visu Data: shrink from (%g, %g, %g).\n",
	      data->privateDt->extension[0], data->privateDt->extension[1],
	      data->privateDt->extension[2]);
  index = (int*)0;
  visuDataIter_new(data, &iter);
  for (visuDataIter_start(data, &iter); iter.node;
       visuDataIter_next(data, &iter))
    {
      cartCoord[0] = iter.node->xyz[0] + data->privateDt->translation[0] +
	iter.node->translation[0];
      cartCoord[1] = iter.node->xyz[1] + data->privateDt->translation[1] +
	iter.node->translation[1];
      cartCoord[2] = iter.node->xyz[2] + data->privateDt->translation[2] +
	iter.node->translation[2];
      visuDataConvert_XYZtoBoxCoordinates(data, boxCoord, cartCoord);
      if ((boxCoord[coord] + valueTo < 0.f ||
	   boxCoord[coord] - 1.f - valueTo >= -1e-6) &&
	  visuNodeGet_original(data->privateDt->nodeArray, iter.node->number) >= 0)
	/* We remove the element. */
	index = addIndexList(index, iter.node->number,
			     data->privateDt->nodeArray->nbOfAllStoredNodes);
      DBG_fprintf(stderr, "Visu Data: test shrink for %d: %d %15.12fx%15.12fx%15.12f.\n",
		  iter.node->number, (index)?index[0]:0,
		  boxCoord[0], boxCoord[1], boxCoord[2]);
    }
  return index;
}
static int* extendNodeList(VisuData *data, int coord, float valueTo)
{
  int k, id;
  unsigned nbInit, nb;
  VisuNode *newNode;
  float cartCoord[3], boxCoord[3], ratio;
  int *index;
  VisuDataIter iter;

  g_return_val_if_fail(coord == 0 || coord == 1 || coord == 2, FALSE);
  g_return_val_if_fail(valueTo > data->privateDt->extension[coord], FALSE);

  DBG_fprintf(stderr, "Visu Data: expand in %d direction to %g.\n",
	      coord, valueTo);
  DBG_fprintf(stderr, " | k runs in [%d %d[ ]%d %d].\n", (int)floor(-valueTo),
	      -(int)data->privateDt->extension[coord],
	      (int)data->privateDt->extension[coord],
	      (int)ceil(valueTo));
  DBG_fprintf(stderr, " | keeps new ele in [%g %g] [%g %g].\n", -valueTo,
	      -data->privateDt->extension[coord],
	      data->privateDt->extension[coord] + 1.f,
	      valueTo + 1.f);
  index = (int*)0;

  /* We estimate the number of data to be added and we call a realloc of
     this amount now to avoid to much small reallocations.
     The slab of box to be extend is 2*(valueTo-extension[coord]).
     So the volume in box coordinates is the same value and since
     the volume in box coordinates is product(1+2*extension),
     the ratio of new space is the fraction.
     So we realloc all elements on this ratio. */
  ratio = (2.f * (valueTo - data->privateDt->extension[coord])) /
    (1.f + 2.f * data->privateDt->extension[coord]);
  visuDataIter_new(data, &iter);
  for (visuDataIter_start(data, &iter); iter.element;
       visuDataIter_nextElement(data, &iter))
    {
      nb = (int)ceil((float)iter.nStoredNodes[iter.iElement] * ratio);
      if (iter.nStoredNodes[iter.iElement] + nb >
	  data->privateDt->nodeArray->numberOfNodes[iter.iElement])
	visuNodeAllocate_newNodes(data->privateDt->nodeArray, iter.iElement, nb);
    }
  
  /* All node with an id higher than nbInit are considered as new
     nodes. */
  nbInit = data->privateDt->nodeArray->idCounter;
  for (visuDataIter_startNumber(data, &iter); iter.node;
       visuDataIter_nextNodeNumber(data, &iter))
    {
      /* Do not duplicate the new nodes. */
      if (iter.node->number >= nbInit)
	continue;

      cartCoord[0] = iter.node->xyz[0] + data->privateDt->translation[0] +
	iter.node->translation[0];
      cartCoord[1] = iter.node->xyz[1] + data->privateDt->translation[1] +
	iter.node->translation[1];
      cartCoord[2] = iter.node->xyz[2] + data->privateDt->translation[2] +
	iter.node->translation[2];
      visuDataConvert_XYZtoBoxCoordinates(data, boxCoord, cartCoord);
      for (k = (int)floor(-valueTo); k < (int)ceil(valueTo) + 1; k++)
	{
	  if (k >= -(int)data->privateDt->extension[coord] &&
	      k <   (int)data->privateDt->extension[coord] + 1)
	    continue;
	  boxCoord[coord] += (float)k;
	  if ((boxCoord[coord] >= -valueTo &&
	       boxCoord[coord] < -data->privateDt->extension[coord] ) ||
	      (boxCoord[coord] < valueTo + 1.f  &&
	       boxCoord[coord] >= data->privateDt->extension[coord] + 1.f))
	    {
	      DBG_fprintf(stderr, "Visu Data: replicating node %d, (%d)"
			  " (%15.12fx%15.12fx%15.12f).\n", iter.node->number, coord,
			  boxCoord[0], boxCoord[1], boxCoord[2]);
	      /* We save the current node id, because the pointer may be
		 relocated by the visuNodeGet_copyNode() call. */
	      id = iter.node->number;
	      /* We create and add a new element. */
	      newNode = visuNodeGet_copyNode(data->privateDt->nodeArray, iter.node);
	      index = addIndexList(index, newNode->number,
				   data->privateDt->nodeArray->nbOfAllStoredNodes);
	      visuDataConvert_boxCoordinatestoXYZ(data, newNode->xyz, boxCoord);
	      newNode->xyz[0] -= data->privateDt->translation[0] +
		newNode->translation[0];
	      newNode->xyz[1] -= data->privateDt->translation[1] +
		newNode->translation[1];
	      newNode->xyz[2] -= data->privateDt->translation[2] +
		newNode->translation[2];
	      /* We reset the iter.node pointer. */
	      iter.node = data->privateDt->nodeArray->nodeTable[id];
	    }
	  boxCoord[coord] -= (float)k;
	}
    }
  return index;
}
void visuDataGet_extension(VisuData *dataObj, float extension[3])
{
  g_return_if_fail(IS_VISU_DATA_TYPE(dataObj));

  extension[0] = dataObj->privateDt->extension[0];
  extension[1] = dataObj->privateDt->extension[1];
  extension[2] = dataObj->privateDt->extension[2];
}
float visuDataGet_allElementExtens(VisuData *dataObj)
{
  float ext;
  guint i;
  VisuRendering *method;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0.f);
  
  /* Compute the max size of a drawn node. */
  method = visuRenderingClassGet_current();
  ext = 0.f;
  for (i = 0; i < dataObj->ntype; i++)
    ext = MAX(ext, visuRenderingGet_sizeOfElement
	      (method, dataObj->fromIntToVisuElement[i]));
  return ext;
}
static float getAllNodeExtens(VisuData *dataObj)
{
  guint i, j;
  VisuNodeArray *array;
  float xyz[2][3], t[3], lg[2], coord[3];

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0.f);

  t[0] = (float)(dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DXX] +
		 dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DYX] +
		 dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DZX]);
  t[1] = (float)(dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DYY] +
		 dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DZY]);
  t[2] = (float)(dataObj->privateDt->boxGeometry[VISU_DATA_BOX_DZZ]);
  xyz[0][0] = xyz[0][1] = xyz[0][2] = 0.f;
  xyz[1][0] = xyz[1][1] = xyz[1][2] = 0.f;

  array = dataObj->privateDt->nodeArray;
  if (!array)
    return 0.f;
  for (i = 0; i < array->ntype; i++)
    for (j = 0; j < array->numberOfStoredNodes[i]; j++)
      {
	visuDataGet_nodePosition(dataObj, array->nodes[i] + j, coord);
	xyz[0][0] = MIN(xyz[0][0], coord[0]);
	xyz[0][1] = MIN(xyz[0][1], coord[1]);
	xyz[0][2] = MIN(xyz[0][2], coord[2]);

	xyz[1][0] = MAX(xyz[1][0], coord[0] - t[0]);
	xyz[1][1] = MAX(xyz[1][1], coord[1] - t[1]);
	xyz[1][2] = MAX(xyz[1][2], coord[2] - t[2]);
      }
  /* Compute the longest vector out of the box. */
  lg[0] = sqrt(xyz[0][0] * xyz[0][0] + 
	       xyz[0][1] * xyz[0][1] + 
	       xyz[0][2] * xyz[0][2]);
  lg[1] = sqrt(xyz[1][0] * xyz[1][0] + 
	       xyz[1][1] * xyz[1][1] + 
	       xyz[1][2] * xyz[1][2]);
  DBG_fprintf(stderr, "VisuData: vectors outside of the box %g %g.\n", lg[0], lg[1]);
  return MAX(lg[0], lg[1]);
}
VisuNodeInfo* visuDataGet_distanceList(VisuData *data, guint nodeId, float *minVal)
{
  VisuNodeInfo *infos;
  int nNodes;
  VisuDataIter iter;
  VisuNode *nodeRef;
  float xyz[3], xyzRef[3], min;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuNodeInfo*)0);

  DBG_fprintf(stderr, "Visu Data: get distance list for node %d.\n", nodeId);
  nodeRef = visuDataGet_nodeFromNumber(data, nodeId);
  g_return_val_if_fail(nodeRef, (VisuNodeInfo*)0);

  nNodes = data->privateDt->nodeArray->nbOfAllStoredNodes;
  infos = g_malloc(sizeof(VisuNodeInfo) * nNodes);
  
  visuDataGet_nodePosition(data, nodeRef, xyzRef);

  min = G_MAXFLOAT;
  nNodes = 0;
  visuDataIter_new(data, &iter);
  for (visuDataIter_start(data, &iter); iter.node;
       visuDataIter_nextVisible(data, &iter))
    {
      infos[nNodes].id = iter.node->number;
      visuDataGet_nodePosition(data, iter.node, xyz);
      infos[nNodes].dist =
	(xyz[0] - xyzRef[0]) * (xyz[0] - xyzRef[0]) +
	(xyz[1] - xyzRef[1]) * (xyz[1] - xyzRef[1]) +
	(xyz[2] - xyzRef[2]) * (xyz[2] - xyzRef[2]);
      if (infos[nNodes].dist > 0.0001f)
	{
	  min = MIN(min, infos[nNodes].dist);
	  nNodes += 1;
	}
    }
  infos[nNodes].id = nodeId;

  DBG_fprintf(stderr, " | min value = %g.\n", min);
  if (minVal)
    *minVal = min;

  return infos;
}
gboolean visuDataSet_newBasis(VisuData *data, guint nO, guint nA, guint nB, guint nC)
{
  VisuNode *orig, *nodeA, *nodeB, *nodeC;
  double mat_[3][3];
  float inv[3][3], matA[3][3], vect[3], xred[3], O[3];
  double box[6];
  float vectEps[3], deltaEps[3];
  VisuDataIter iter;
  int *rmNodes, i;
#define EPS 1.e-5

  orig = visuDataGet_nodeFromNumber(data, nO);
  nodeA = visuDataGet_nodeFromNumber(data, nA);
  nodeB = visuDataGet_nodeFromNumber(data, nB);
  nodeC = visuDataGet_nodeFromNumber(data, nC);
  g_return_val_if_fail(orig && nodeA && nodeB && nodeC, FALSE);

  O[0] = orig->xyz[0] + orig->translation[0];
  O[1] = orig->xyz[1] + orig->translation[1];
  O[2] = orig->xyz[2] + orig->translation[2];

  matA[0][0] = nodeA->xyz[0] + nodeA->translation[0] - O[0];
  matA[1][0] = nodeA->xyz[1] + nodeA->translation[1] - O[1];
  matA[2][0] = nodeA->xyz[2] + nodeA->translation[2] - O[2];
  matA[0][1] = nodeB->xyz[0] + nodeB->translation[0] - O[0];
  matA[1][1] = nodeB->xyz[1] + nodeB->translation[1] - O[1];
  matA[2][1] = nodeB->xyz[2] + nodeB->translation[2] - O[2];
  matA[0][2] = nodeC->xyz[0] + nodeC->translation[0] - O[0];
  matA[1][2] = nodeC->xyz[1] + nodeC->translation[1] - O[1];
  matA[2][2] = nodeC->xyz[2] + nodeC->translation[2] - O[2];

  mat_[0][0] = (double)matA[0][0];
  mat_[1][0] = (double)matA[0][1];
  mat_[2][0] = (double)matA[0][2];
  mat_[0][1] = (double)matA[1][0];
  mat_[1][1] = (double)matA[1][1];
  mat_[2][1] = (double)matA[1][2];
  mat_[0][2] = (double)matA[2][0];
  mat_[1][2] = (double)matA[2][1];
  mat_[2][2] = (double)matA[2][2];

  DBG_fprintf(stderr, "Visu Data: basis matrice:\n");
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      matA[0][0], matA[0][1], matA[0][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      matA[1][0], matA[1][1], matA[1][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      matA[2][0], matA[2][1], matA[2][2]);

  if (!matrix_invert(inv, matA))
    return FALSE;

  DBG_fprintf(stderr, "Visu Data: transformation matrice:\n");
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      inv[0][0], inv[0][1], inv[0][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      inv[1][0], inv[1][1], inv[1][2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      inv[2][0], inv[2][1], inv[2][2]);

  if (!matrix_reducePrimitiveVectors(box, mat_))
    return FALSE;
  
  DBG_fprintf(stderr, "Visu Data: new box:\n");
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      box[0], box[1], box[2]);
  DBG_fprintf(stderr, "  (%10.5f  %10.5f  %10.5f)\n",
	      box[3], box[4], box[5]);

  visuDataSet_boxGeometry(data, box, data->privateDt->attachedView->box->bc);

  /* We need to move all the atoms of (eps, eps, eps) in the new box
     to avoid rounding problems. */
  xred[0] = 1.f;
  xred[1] = 1.f;
  xred[2] = 1.f;
  matrix_productVector(vect, matA, xred);
  vectEps[0] = (vect[0] >= 0.f)?EPS:-EPS;
  vectEps[1] = (vect[1] >= 0.f)?EPS:-EPS;
  vectEps[2] = (vect[2] >= 0.f)?EPS:-EPS;
  matrix_productVector(xred, inv, vectEps);
  visuDataConvert_boxCoordinatestoXYZ(data, deltaEps, xred);
  DBG_fprintf(stderr, "Visu Data: applied epsilon (%10.5f  %10.5f  %10.5f)\n",
	      vectEps[0], vectEps[1], vectEps[2]);

  /* Transform each atomic coordinates using this matrice. */
  DBG_fprintf(stderr, "Visu Data: reset the coordinates for all nodes.\n");
  visuDataIter_new(data, &iter);
  rmNodes = g_malloc(sizeof(int) * iter.nAllStoredNodes);
  i = 0;
  for (visuDataIter_start(data, &iter); iter.node;
       visuDataIter_next(data, &iter))
    {
      vect[0] = iter.node->xyz[0] + iter.node->translation[0] - O[0] + vectEps[0];
      vect[1] = iter.node->xyz[1] + iter.node->translation[1] - O[1] + vectEps[1];
      vect[2] = iter.node->xyz[2] + iter.node->translation[2] - O[2] + vectEps[2];
      matrix_productVector(xred, inv, vect);
      if (xred[0] < 0.f || xred[0] >= 1.f ||
	  xred[1] < 0.f || xred[1] >= 1.f ||
	  xred[2] < 0.f || xred[2] >= 1.f)
	{
	  rmNodes[i] = (int)iter.node->number;
	  DBG_fprintf(stderr, " | %d  (%6.1f %6.1f %6.1f)"
		      " %10.5f %10.5f %10.5f -> removed\n",
		      iter.node->number, vect[0], vect[1], vect[2],
		      xred[0], xred[1], xred[2]);
	  i+= 1;
	}
      else
	{
	  visuDataConvert_boxCoordinatestoXYZ(data, iter.node->xyz, xred);
	  iter.node->xyz[0] -= deltaEps[0];
	  iter.node->xyz[1] -= deltaEps[1];
	  iter.node->xyz[2] -= deltaEps[2];
	  iter.node->translation[0] = 0.f;
	  iter.node->translation[1] = 0.f;
	  iter.node->translation[2] = 0.f;
	  visuNodeSet_original(data->privateDt->nodeArray, iter.node->number);
	  DBG_fprintf(stderr, " | %d  (%6.1f %6.1f %6.1f)"
		      " %10.5f %10.5f %10.5f -> %10.5f %10.5f %10.5f\n",
		      iter.node->number, vect[0], vect[1], vect[2],
		      xred[0], xred[1], xred[2], iter.node->xyz[0],
		      iter.node->xyz[1], iter.node->xyz[2]);
	}
    }
  rmNodes[i] = -1;

  visuDataRemove_nodes(data, rmNodes);

  /* Remove possible translation. */
  data->privateDt->translation[0] = 0.f;
  data->privateDt->translation[1] = 0.f;
  data->privateDt->translation[2] = 0.f;
  data->privateDt->translationApply = FALSE;
  visuData_createAllNodes(data);

  g_free(rmNodes);

  /* Remove possible extension. */
  data->privateDt->extension[0] = 0.f;
  data->privateDt->extension[1] = 0.f;
  data->privateDt->extension[2] = 0.f;
  visuDataApply_boxGeometry(data, -1.f);

  visuDataEmit_nodePositionChanged(data);

  g_idle_add(visuObjectRedraw, GINT_TO_POINTER(TRUE));

  return TRUE;
}


/*****************************/
/* The node related routines */
/*****************************/
void visuDataRemove_nodes(VisuData *data, int *nodeNumbers)
{
  visuNodeRemove_nodes(data->privateDt->nodeArray, nodeNumbers);

  DBG_fprintf(stderr, "Visu Data: emit a 'NodePopulationDecrease' signal.\n");
  g_signal_emit(G_OBJECT(data), visu_data_signals[NODE_POPULATION_DECREASE_SIGNAL],
		0, (gpointer)nodeNumbers, NULL);
}
VisuNode* visuDataAdd_nodeFromIndex(VisuData *data, unsigned int position,
				    float xyz[3], gboolean emitSignal)
{
  VisuNode *node;
  int index[4];

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && position < data->ntype,
		       (VisuNode*)0);

  node = visuNodeGet_newNode(data->privateDt->nodeArray, position);
  g_return_val_if_fail(node, (VisuNode*)0);

  visuNodeNew_values(node, xyz);

  if (emitSignal)
    {
      index[0] = 1;
      index[1] = 2;
      index[2] = node->number;
      index[3] = -1;
      DBG_fprintf(stderr, "Visu Data: emit a "
		  "'NodePopulationIncrease' signal.\n");
      g_signal_emit(G_OBJECT(data),
		    visu_data_signals[NODE_POPULATION_INCREASE_SIGNAL],
		    0, (gpointer)(&index), NULL);
    }
  return node;
}
VisuNode* visuDataAdd_nodeFromElement(VisuData *data, VisuElement *ele,
				      float xyz[3], gboolean emitSignal)
{
  unsigned int *pos;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && ele, (VisuNode*)0);

  pos = (unsigned int*)g_hash_table_lookup(data->fromVisuElementToInt, (gpointer)ele);
  g_return_val_if_fail(pos && *pos < data->ntype, (VisuNode*)0);

  return visuDataAdd_nodeFromIndex(data, *pos, xyz, emitSignal);
}
void visuDataGet_nodePosition(VisuData *data, VisuNode *node, float coord[3])
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && node && coord);

  coord[0] = node->xyz[0] + node->translation[0] + data->privateDt->translation[0];
  coord[1] = node->xyz[1] + node->translation[1] + data->privateDt->translation[1];
  coord[2] = node->xyz[2] + node->translation[2] + data->privateDt->translation[2];
}
void visuDataGet_reducedNodePosition(VisuData *data, VisuNode *node, float red[3])
{
  float coord[3];

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && node && red);

  coord[0] = node->xyz[0] + node->translation[0] + data->privateDt->translation[0];
  coord[1] = node->xyz[1] + node->translation[1] + data->privateDt->translation[1];
  coord[2] = node->xyz[2] + node->translation[2] + data->privateDt->translation[2];
  red[0] =
    (float)data->privateDt->fromXYZtoBoxD0[0][0] * coord[0] +
    (float)data->privateDt->fromXYZtoBoxD0[0][1] * coord[1] +
    (float)data->privateDt->fromXYZtoBoxD0[0][2] * coord[2];
  red[1] =
    (float)data->privateDt->fromXYZtoBoxD0[1][0] * coord[0] +
    (float)data->privateDt->fromXYZtoBoxD0[1][1] * coord[1] +
    (float)data->privateDt->fromXYZtoBoxD0[1][2] * coord[2];
  red[2] =
    (float)data->privateDt->fromXYZtoBoxD0[2][0] * coord[0] +
    (float)data->privateDt->fromXYZtoBoxD0[2][1] * coord[1] +
    (float)data->privateDt->fromXYZtoBoxD0[2][2] * coord[2];
}
VisuNode* visuDataGet_nodeFromNumber(VisuData *data, unsigned int number)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuNode*)0);

  DBG_fprintf(stderr, "Visu Data: get VisuNode from number %d.\n", number);
  if (data->privateDt->nodeArray && number < data->privateDt->nodeArray->idCounter)
    return data->privateDt->nodeArray->nodeTable[number];
  else
    return (VisuNode*)0;
}


/*********************/
/* The emit routines */
/*********************/
void visuDataEmit_askForShowHideNodes(VisuData *data, gboolean *redraw)
{
  g_return_if_fail(data && redraw);

  *redraw = FALSE;
  DBG_fprintf(stderr, "Visu Data: %p emit the 'NodeAskForShowHide' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[NODE_ASK_FOR_SHOW_HIDE_SIGNAL],
		0 , (gpointer)redraw, NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (NodeAskForShowHide).\n");
}
void visuDataEmit_nodePositionChanged(VisuData *data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu Data: %p emit the 'NodePositionChanged' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[NODE_POSITION_CHANGED_SIGNAL],
		0 , NULL);
  DBG_fprintf(stderr, "Visu Data: emission done (NodePositionChanged).\n");
}
void visuDataEmit_elementRenderedChange(VisuData *data, VisuElement *element)
{
  g_return_if_fail(data && element);

  DBG_fprintf(stderr, "Visu Data: %p emit the 'ElementRenderedChanged'"
	      " signal for element '%s'.\n", (gpointer)data, element->name);
  g_signal_emit(data, visu_data_signals[ELEMENT_RENDERED_CHANGED_SIGNAL],
		0 , (gpointer)element, NULL);
}
void visuDataEmit_nodeRenderedChange(VisuData *data)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu Data: %p emit the 'NodeRenderedChanged' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[NODE_RENDERED_CHANGED_SIGNAL],
		0 , NULL);
}
void visuDataEmit_facettesChanged(VisuData *data)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));
  
  DBG_fprintf(stderr, "Visu Data: %p emit the 'OpenGLFacetteChanged' signal.\n", (gpointer)data);
  g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		0 , NULL);
}



/****************/
/* The timeouts */
/****************/
guint visuDataAdd_timeout(VisuData *data, guint time, GSourceFunc func, gpointer user_data)
{
  guint *id;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data) && func, (guint)0);

  id = g_malloc(sizeof(guint));
  *id = g_timeout_add(time, func, user_data);
  data->privateDt->timeoutList = g_list_prepend(data->privateDt->timeoutList, (gpointer)id);
  DBG_fprintf(stderr, "Visu data : create a new timeout callback : %d.\n", *id);
  return *id;
}
gboolean visuDataRemove_timeout(VisuData *data, guint timeoutId)
{
  GList *lst;
  gboolean found;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  DBG_fprintf(stderr, "Visu data : trying to remove a timeout callback (%d) ... ", timeoutId);
  found = FALSE;
  lst = data->privateDt->timeoutList;
  while(lst)
    {
      if (*(guint*)lst->data == timeoutId)
	{
	  found = g_source_remove(timeoutId);
	  data->privateDt->timeoutList =
	    g_list_delete_link(data->privateDt->timeoutList, lst);
	  DBG_fprintf(stderr, "OK (%d).\n", found);
	  return found;
	}
      lst = g_list_next(lst);
    }
  DBG_fprintf(stderr, "not found.\n");
  return found;
}


/***********************/
/* The view properties */
/***********************/
OpenGLView* visuDataGet_openGLView(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (OpenGLView*)0);

  return data->privateDt->attachedView;
}
int visuDataSet_angleOfView(VisuData *data, float valueTheta,
			    float valuePhi, float valueOmega, int mask)
{
  int res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  
  res = openGLViewSet_thetaPhiOmega(data->privateDt->attachedView, valueTheta,
				    valuePhi, valueOmega, mask);
  if (res)
    g_signal_emit(data, visu_data_signals[OPENGL_THETA_PHI_OMEGA_CHANGED_SIGNAL],
		  0, data->privateDt->attachedView, NULL);
    
  return res;
}
int visuDataSet_positionOfView(VisuData *data, float valueX, float valueY, int mask)
{
  int res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  
  res = openGLViewSet_XsYs(data->privateDt->attachedView, valueX,
			   valueY, mask);
  if (res)
    {
      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_XS_YS_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
    }
    
  return res;
}
int visuDataSet_zoomOfView(VisuData *data, float value)
{
  int res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  
  res = openGLViewSet_gross(data->privateDt->attachedView, value);
  if (res)
    {
      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_GROSS_CHANGED_SIGNAL],
		    0 , data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		    0 , NULL);
      visuData_createAllElements(data);
    }
    
  return res;
}
int visuDataSet_perspectiveOfView(VisuData *data, float value)
{
  int res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  
  res = openGLViewSet_persp(data->privateDt->attachedView, value);
  if (res)
    {
      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_PERSP_CHANGED_SIGNAL],
		    0 , data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		    0 , NULL);
      visuData_createAllElements(data);
    }
    
  return res;
}
int visuDataSet_sizeOfView(VisuData *data, guint width, guint height)
{
  int res;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0);
  
  res = OpenGLViewSet_windowSize(data->privateDt->attachedView, width, height);
  if (res)
    {
      g_signal_emit(data, visu_data_signals[OPENGL_NEAR_FAR_CHANGED_SIGNAL],
		    0, data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_WIDTH_HEIGHT_CHANGED_SIGNAL],
		    0 , data->privateDt->attachedView, NULL);
      g_signal_emit(data, visu_data_signals[OPENGL_FACETTES_CHANGED_SIGNAL],
		    0 , NULL);
      visuData_createAllElements(data);
    }
    
  return res;
}



/*****************/
/* Miscellaneous */
/*****************/
GList* visuDataGet_allObjects()
{
  DBG_fprintf(stderr, "Visu Data : get the allObjects list.\n");
  return allObjects;
}
GenericRenderingWindow visuDataGet_renderingWindow(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (GenericRenderingWindow)0);

  return data->privateDt->attachingWindow;
}
void visuDataSet_renderingWindow(VisuData *data, GenericRenderingWindow window)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  data->privateDt->attachingWindow = window;
}
/**
 * visuDataTrace:
 * @data: the #VisuData to trace.
 *
 * This method gives informations about the argument.
 */
void visuDataTrace(VisuData *data)
{
  VisuElement *ele;
  unsigned int i;

  if (!data)
    return;

  fprintf(stderr, "Visu Data : data %d.\n", GPOINTER_TO_INT(data));
  fprintf(stderr, " | nb VisuElements : %d\n", data->ntype);
  for (i = 0; i < data->ntype; i++)
    {
      ele = data->fromIntToVisuElement[i];
      fprintf(stderr, " | VisuElements '%s' (%p), %d allocated VisuNodes"
	      " and %d stored.\n", data->fromIntToVisuElement[i]->name,
	      (gpointer)data->fromIntToVisuElement[i],
	      data->privateDt->nodeArray->numberOfNodes[i],
	      data->privateDt->nodeArray->numberOfStoredNodes[i]);
    }
}
void visuDataSet_ColorFunc(VisuData *data, setColorFunc func)
{
  g_return_if_fail(data);

  DBG_fprintf(stderr, "Visu Data : set the color method to %d.\n",
	      GPOINTER_TO_INT(func));
  data->setColor = func;
}
void visuDataSet_nodeScalingFunc(VisuData *data, VisuDataScalingFunc scaling)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  if (scaling)
    data->privateDt->scaling = scaling;
  else
    data->privateDt->scaling = defaultScaling;
}
float visuDataGet_nodeScalingFactor(VisuData *data, VisuNode *node)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), 0.f);

  return data->privateDt->scaling(data, node);
}
static float defaultScaling(VisuData *data _U_, VisuNode *node _U_)
{
  return 1.f;
}



/***********************/
/* The node properties */
/***********************/
VisuNodeArray* visuDataGet_nodeArray(VisuData *data)
{
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), (VisuNodeArray*)0);
  return data->privateDt->nodeArray;
}


/****************/
/* The iterator */
/****************/
void visuDataIter_new(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && data->privateDt->nodeArray && iter);

  iter->data            = data;
  iter->idMax           = data->privateDt->nodeArray->idCounter - 1;
  iter->nAllStoredNodes = data->privateDt->nodeArray->nbOfAllStoredNodes;
  iter->nElements       = data->privateDt->nodeArray->ntype;
  iter->nStoredNodes    = data->privateDt->nodeArray->numberOfStoredNodes;
  iter->iElement        = -1;
  iter->node            = (VisuNode*)0;
  iter->element         = (VisuElement*)0;
}
void visuDataIter_start(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);

  iter->iElement = -1;
  iter->node     = (VisuNode*)0;
  iter->element  = (VisuElement*)0;
  if (data->ntype == 0)
    return;

  iter->iElement = 0;
  iter->element  = data->fromIntToVisuElement[0];
  /* We look for an element with stored nodes. */
  while (data->privateDt->nodeArray->numberOfStoredNodes[iter->iElement] == 0)
    {
      iter->iElement += 1;
      if (iter->iElement >= data->privateDt->nodeArray->ntype)
	{
	  /* We found nothing. */
	  iter->iElement = -1;
	  iter->element  = (VisuElement*)0;
	  return;
	}
      iter->element   = data->fromIntToVisuElement[iter->iElement];
    }

  iter->node     = data->privateDt->nodeArray->nodes[iter->iElement];
}
void visuDataIter_startNumber(VisuData *data, VisuDataIter *iter)
{
  guint i;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);

  iter->iElement = -1;
  iter->node     = (VisuNode*)0;
  iter->element  = (VisuElement*)0;
  if (data->ntype == 0)
    return;

  i = 0;
  iter->node = (VisuNode*)0;
  do
    {
      iter->node = visuDataGet_nodeFromNumber(data, i);
      i += 1;
    }
  while (!iter->node && i < data->privateDt->nodeArray->nodeTableSize);
  if (!iter->node)
    return;
  iter->iElement = iter->node->posElement;
  iter->element  = data->fromIntToVisuElement[iter->iElement];
}
void visuDataIter_startVisible(VisuData *data, VisuDataIter *iter)
{
  visuDataIter_start(data, iter);
  if (iter->node && iter->node->rendered && iter->element->rendered)
    /* Ok, first is good. */
    return;

  /* First was not visible, we go next. */
  visuDataIter_nextVisible(data, iter);
}
void visuDataIter_restartNode(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->iElement < data->ntype);

  iter->node = data->privateDt->nodeArray->nodes[iter->iElement];
}
void visuDataIter_next(VisuData *data, VisuDataIter *iter)
{
  unsigned int iNode;
  VisuNodeArray *array;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->node && iter->iElement == iter->node->posElement);
  
  array = data->privateDt->nodeArray;
  iNode = iter->node->posNode + 1;
  if (iNode < array->numberOfStoredNodes[iter->iElement])
    iter->node = array->nodes[iter->iElement] + iNode;
  else
    {
      iter->iElement += 1;
      if (iter->iElement >= array->ntype)
	{
	  iter->iElement = -1;
	  iter->node     = (VisuNode*)0;
	  iter->element  = (VisuElement*)0;
	}
      else
	{
	  iter->node     = array->nodes[iter->iElement];
	  iter->element  = data->fromIntToVisuElement[iter->iElement];
	}
    }
}
void visuDataIter_nextVisible(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);

  /* Get the next node, and test if it is rendered. */
  visuDataIter_next(data, iter);
  if (!iter->node || (iter->element->rendered && iter->node->rendered))
    return;

  /* From the current node, we go next to find one that is rendred. */
  for (; iter->element; visuDataIter_nextElement(data, iter))
    if (iter->element->rendered)
      for (; iter->node; visuDataIter_nextNode(data, iter))
	if (iter->node->rendered)
	  return;
}
void visuDataIter_nextNode(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->node);

  if (iter->node->posNode + 1 <
      data->privateDt->nodeArray->numberOfStoredNodes[iter->node->posElement])
    iter->node = iter->node + 1;
  else
    iter->node = (VisuNode*)0;
}
void visuDataIter_nextNodeNumber(VisuData *data, VisuDataIter *iter)
{
  unsigned int i;

  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->node);

  for (i = iter->node->number + 1;
       !(iter->node = visuDataGet_nodeFromNumber(data, i)) &&
	 (i < data->privateDt->nodeArray->idCounter) ; i++);

  if (iter->node)
    {
      iter->iElement = iter->node->posElement;
      iter->element  = data->fromIntToVisuElement[iter->iElement];
    }
  else
    iter->element = (VisuElement*)0;
}
void visuDataIter_nextElement(VisuData *data, VisuDataIter *iter)
{
  g_return_if_fail(IS_VISU_DATA_TYPE(data) && iter && data == iter->data);
  g_return_if_fail(iter->iElement < data->ntype);

  do
    iter->iElement += 1;
  while(iter->iElement < data->privateDt->nodeArray->ntype &&
	iter->nStoredNodes[iter->iElement] == 0);

  if (iter->iElement == data->privateDt->nodeArray->ntype)
    {
      iter->iElement = -1;
      iter->node     = (VisuNode*)0;
      iter->element  = (VisuElement*)0;
    }
  else
    {
      iter->node     = data->privateDt->nodeArray->nodes[iter->iElement];
      iter->element  = data->fromIntToVisuElement[iter->iElement];
    }
}
