/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include <glib.h>
#include <glib/gstdio.h>

#include "renderingAtomic.h"
#include "renderingAtomic_ascii.h"
#include "renderingAtomic_d3.h"
#include "atomic_xyz.h"
#include <visu_basic.h>
#include <visu_elements.h>
#include <visu_configFile.h>
#include <renderingBackend/visu_windowInterface.h>
#include <openGLFunctions/view.h>
#include <openGLFunctions/objectList.h>
#include <opengl.h>
#include <coreTools/toolConfigFile.h>
#include <coreTools/toolElements.h>

#include <GL/gl.h>
#include <GL/glu.h> 

#include <math.h>


/* The OpenGL identifier to store the glObjectLists that
   describe the spheres. */
/* int identifierSpheres; */

/* The icosahedron drawing. */
#define X .525731112119133606 
#define Z .850650808352039932
static GLfloat vdata[12][3] = {    
   {X, 0.0, -Z}, {-X, 0.0, -Z}, {X, 0.0, Z}, {-X, 0.0, Z},    
   {0.0, -Z, -X}, {0.0, -Z, X}, {0.0, Z, -X}, {0.0, Z, X},    
   {-Z, -X, 0.0}, {Z, -X, 0.0}, {-Z, X, 0.0}, {Z, X, 0.0} 
};
static GLuint tindices[20][3] = { 
   {0,4,1}, {0,9,4}, {9,5,4}, {4,5,8}, {4,8,1},    
   {8,10,1}, {8,3,10}, {5,3,8}, {5,2,3}, {2,7,3},    
   {7,10,3}, {7,6,10}, {7,11,6}, {11,0,6}, {0,1,6}, 
   {6,1,10}, {9,0,11}, {9,11,2}, {9,2,5}, {7,2,11} };

#define RADIUS_DEFAULT 1.
#define SHAPE_DEFAULT shapeSphere
#define RATIO_DEFAULT 2.
#define PHI_DEFAULT  0.
#define THETA_DEFAULT 90.
/* Dealing with the radius resource. */
#define FLAG_RESOURCE_RADIUS_SHAPE "atomic_radius_shape"
#define DESC_RESOURCE_RADIUS_SHAPE "The radius of the element and its shape, a real > 0. & [Sphere Cube Elipsoid Point]"
static gboolean readAtomicRadiusShape(gchar **lines, int nbLines, int position,
			      VisuData *dataObj, GError **error);
/* These functions write all the element list to export there associated resources. */
static void exportAtomicRadiusShape(GString *data, VisuData* dataObj);
#define FLAG_PARAMETER_SHAPE "atomic_sphere_method"
#define DESC_PARAMETER_SHAPE "The sphere drawing method, [GluSphere Icosahedron]"
static gboolean readAtomicShape(gchar **lines, int nbLines, int position,
				VisuData *dataObj, GError **error);
static void exportAtomicShape(GString *data, VisuData* dataObj);
enum
  {
    sphere_glu,
    sphere_icosahedron,
    sphere_nb
  };
static guint sphereMethod = sphere_icosahedron;
const char* sphereName[sphere_nb + 1] = {"GluSphere", "Icosahedron", (const char*)0};

const char* shapeName[nbAtomicShapes + 1] = {"Sphere", "Cube", "Elipsoid", "Point", "Torus", (const char*)0};
const char* shapeNameI18n[nbAtomicShapes + 1];
struct atomicResources_struct
{
  /* One of this structure is associated to each element. */

  /* The radius of the element. */
  float radius;
  /* The ratio long axis, short axis when the shape is an elipsoid. */
  float ratio;
  /* The angles positioning of the long axis when the shape is
     an elipsoid. */
  float phi, theta;

  /* The shape used. */
  int shape;

  /* An id of the list associated with the form. */
  int openGLIdentifier;
};


static VisuRendering *atom;
/* To build a list of rendering loading methods. */
/* This list contains all the method than can load an atomic file. */
static GList *allAtomicLoadingMethods;
static VisuRendering* atomic = (VisuRendering*)0;

static struct atomicResources_struct *getRadiusAndShape(VisuElement *ele);

void renderingAtomicInit()
{
  char *name = _("Atom visualisation");
  char *descr = _("It draws spheres at specified positions to represent atoms."
		  " The radius of the sphere can vary.");
  gchar *iconPath;
  GList *formatList;
  VisuConfigFileEntry *resourceEntry;
  GList *tmpLst;
  RenderingFormatLoad *meth;

  DBG_fprintf(stderr,"Initialising the atomic rendering method...\n");
  formatList = (GList*)0;
  allAtomicLoadingMethods = (GList*)0;

  meth = atomicD3Init();
  if (meth)
    allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  meth = atomicAsciiInit();
  if (meth)
    allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  meth = atomicXyzInit();
  if (meth)
    allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  allAtomicLoadingMethods = g_list_sort(allAtomicLoadingMethods,
					visuRenderingFormatCompare_priority);
  tmpLst = allAtomicLoadingMethods;
  while (tmpLst)
    {
      formatList = g_list_append(formatList, ((RenderingFormatLoad*)tmpLst->data)->fmt);
      tmpLst = g_list_next(tmpLst);
    }

  atom = visuRendering_new(VISU_RENDERING_ATOMIC, name, descr,
			   1, renderingAtomicLoad,
			   renderingAtomic_createShape,
			   renderingAtomic_positionShape,
			   renderingAtomicGet_radius);
  visuRenderingSet_fileType(atom, formatList, 0, _("Position files"));
  iconPath = g_build_filename(V_SIM_PIXMAPS_DIR, "stock-atomic.png", NULL);
  visuRenderingSet_icon(atom, iconPath);
  g_free(iconPath);

  /* Dealing with config files. */
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_RADIUS_SHAPE,
					  DESC_RESOURCE_RADIUS_SHAPE,
					  1, readAtomicRadiusShape);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportAtomicRadiusShape);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_SHAPE,
					  DESC_PARAMETER_SHAPE,
					  1, readAtomicShape);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportAtomicShape);
					  

  /* Initialise the OpenGl part. */
/*   identifierSpheres = openGLObjectList_new(NMAX_TP); */

  /* Set the general pointer to retrieve easily this method. */
  atomic = atom;

  shapeNameI18n[0] = _("Sphere");
  shapeNameI18n[1] = _("Cube");
  shapeNameI18n[2] = _("Elipsoid");
  shapeNameI18n[3] = _("Point");
  shapeNameI18n[4] = _("Torus");
  shapeNameI18n[5] = (const char*)0;
}

gboolean renderingAtomicLoad(VisuData *data, FileFormat *format _U_,
			     int nSet, GError **error)
{
  gchar *file;
  gboolean loadOk;
  GList *tmpLst;
  FileFormat *fmt;
  RenderingFormatLoad *load;
#if GLIB_MINOR_VERSION > 5
  struct stat buf;
#endif

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  if (!data)
    return 0;

  file = visuDataGet_file(data, 0, &fmt);
  if (!file)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("No file name available."));
      return FALSE;
    }
  
  if (!g_file_test(file, G_FILE_TEST_IS_REGULAR))
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("The specified file is"
			     " not a regular file."));
      return FALSE;
    }
#if GLIB_MINOR_VERSION > 5
  if (!g_stat(file, &buf) && buf.st_size == 0)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("The specified file is"
			     " an empty file."));
      return FALSE;
    }
#endif
  
  loadOk = FALSE;
  tmpLst = allAtomicLoadingMethods;
  while (tmpLst && !loadOk)
    {
      /* Each load may set error even if the format is not recognise
	 and loadOK is FALSE, then we need to free the error. */
      if (*error)
	g_error_free(*error);
      *error = (GError*)0;
      load = (RenderingFormatLoad*)tmpLst->data;
      if (!fmt || load->fmt == fmt)
	{
	  DBG_fprintf(stderr,"Rendering Atomic: testing '%s' with format: %s.\n",
		      file, load->name);
	  loadOk = load->load(data, file, load->fmt, nSet, error);
	  DBG_fprintf(stderr,"Rendering Atomic: with format '%s' returns %d.\n",
		      load->name, loadOk);
	}
      if (*error && (*error)->domain == G_FILE_ERROR)
	return FALSE;
      
      if (fmt && load->fmt == fmt)
	tmpLst = (GList*)0;
      else
	tmpLst = g_list_next(tmpLst);
    }

  if (!loadOk)
    {
      if (*error)
	g_error_free(*error);
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_FILE,
			   _("Impossible to load this file, unrecognised format.\n"));
      return FALSE;
    }
  else if (*error)
    return FALSE;

  return TRUE;
}

static struct atomicResources_struct *getRadiusAndShape(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, (struct atomicResources_struct*)0);

  str = (struct atomicResources_struct*)visuElementGet_property
    (ele, "radiusAndShape");
  if (!str)
    {
      str = g_malloc(sizeof(struct atomicResources_struct));
      if (!toolElementsGet_elementFromSymbol((int*)0, &str->radius, ele->name))
	str->radius = RADIUS_DEFAULT;
      str->shape  = SHAPE_DEFAULT;
      str->ratio  = RATIO_DEFAULT;
      str->phi    = PHI_DEFAULT;
      str->theta  = THETA_DEFAULT;
      str->openGLIdentifier = openGLObjectList_new(1);

      visuElementSet_property(ele, "radiusAndShape", (gpointer)str);
    }
  return str;
}

float renderingAtomicGet_radius(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, RADIUS_DEFAULT);

  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, -1.f);
  return str->radius;
}
float renderingAtomicGet_radiusDefault()
{
  return RADIUS_DEFAULT;
}
gboolean renderingAtomicSet_radius(VisuElement* ele, float value)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele && value > 0.f, FALSE);
  
  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  if (value == str->radius)
    return FALSE;
  else
    str->radius = value;

  visuRenderingEmit_elementSize(atom, value);

  /* Ask for reDraw if needed. */
  return TRUE;
}

int renderingAtomicGet_shape(VisuElement *ele)
{
  struct atomicResources_struct *str;

  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  return str->shape;
}
const char* renderingAtomicGet_shapeName(int shape)
{
  if (shape < 0 || shape >= nbAtomicShapes)
    return (char*)0;
  else
    return shapeName[shape];
}
const char* renderingAtomicGet_shapeNameDefault()
{
  return shapeName[SHAPE_DEFAULT];
}
int renderingAtomicGet_shapeDefault()
{
  return SHAPE_DEFAULT;
}
int renderingAtomicSet_shape(VisuElement *ele, int shape)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele && shape >= 0 && shape < nbAtomicShapes, 0);
  
  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  if (shape == str->shape)
    return 0;
  else
    str->shape = shape;

  return 1;
}

int renderingAtomicSet_shapeFromName(VisuElement *ele, char* shape)
{
  int res, i;

  res = -1;
  for (i = 0; shapeName[i] && res < 0; i++)
    {
      if (!strcmp(shapeName[i], shape))
	res = i;
    }
  if (res < 0)
    {
      g_warning("Unknown shape name in the call of renderingAtomicSet_shapeFromName.");
      return 0;
    }
  else
    return renderingAtomicSet_shape(ele, res);
}
const char** renderingAtomicGet_allShapes()
{
  return shapeName;
}
const char** renderingAtomicGet_allShapesI18n()
{
  return shapeNameI18n;
}
gboolean renderingAtomicSet_elipsoidParameters(VisuElement *ele, float ratio,
					       float phi, float theta)
{
  struct atomicResources_struct *str;
  gboolean refresh;

  g_return_val_if_fail(ele && (ratio >= 1.), 0);
  
  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  refresh = FALSE;
  if (ratio != str->ratio)
    {
      str->ratio = ratio;
      refresh = TRUE;
    }
  if (phi != str->phi)
    {
      str->phi = phi;
      refresh = TRUE;
    }
  if (theta != str->theta)
    {
      str->theta = theta;
      refresh = TRUE;
    }

  return refresh && (str->shape == shapeElipsoid || str->shape == shapeTorus);
}
gboolean renderingAtomicSet_elipsoidRatio(VisuElement *ele, float ratio)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele && (ratio >= 1.), FALSE);
  
  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  if (ratio != str->ratio)
    {
      str->ratio = ratio;
      return (str->shape == shapeElipsoid || str->shape == shapeTorus);
    }
  return FALSE;
}
gboolean renderingAtomicSet_elipsoidPhi(VisuElement *ele, float phi)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, FALSE);
  
  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  if (phi != str->phi)
    {
      str->phi = phi;
      return (str->shape == shapeElipsoid || str->shape == shapeTorus);
    }
  return FALSE;
}
gboolean renderingAtomicSet_elipsoidTheta(VisuElement *ele, float theta)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, FALSE);
  
  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, FALSE);

  if (theta != str->theta)
    {
      str->theta = theta;
      return (str->shape == shapeElipsoid || str->shape == shapeTorus);
    }
  return FALSE;
}
float renderingAtomicGet_elipsoidRatio(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, RATIO_DEFAULT);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (str)
    return str->ratio;
  else
    return RATIO_DEFAULT;
}
float renderingAtomicGet_elipsoidPhi(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, PHI_DEFAULT);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (str)
    return str->phi;
  else
    return PHI_DEFAULT;
}
float renderingAtomicGet_elipsoidTheta(VisuElement *ele)
{
  struct atomicResources_struct *str;

  g_return_val_if_fail(ele, THETA_DEFAULT);

  str = (struct atomicResources_struct*)visuElementGet_property(ele, "radiusAndShape");
  if (str)
    return str->theta;
  else
    return THETA_DEFAULT;
}


static gboolean readAtomicRadiusShape(gchar **lines, int nbLines, int position,
				      VisuData *dataObj _U_, GError **error)
{
  VisuElement* ele;
  int shapeNum, i, token;
  float radius;
  gchar **shape;
  gchar **tokens;

  g_return_val_if_fail(nbLines == 1, FALSE);

  /* Tokenize the line of values. */
  tokens = g_strsplit_set(g_strchug(lines[0]), " \n", MAX_LINE_LENGTH);
  token = 0;

  /* Get the two elements. */
  if (!configFileRead_elementFromTokens(tokens, &token, &ele, 1, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }

  /* Read 1 int. */
  if (!configFileRead_floatFromTokens(tokens, &token, &radius, 1, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  radius = CLAMP(radius, 0., G_MAXFLOAT);

  /* Read 1 string. */
  if (!configFileRead_stringFromTokens(tokens, &token, &shape, 1, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  shapeNum = -1;
  for (i = 0; shapeName[i] && shapeNum < 0; i++)
    {
      if (!strcmp(shapeName[i], shape[0]))
	shapeNum = i;
    }
  g_free(shape);
  if (shapeNum < 0)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: the shape '%s' is unknown.\n"),
			   position, shape[0]);
      return FALSE;
    }
  renderingAtomicSet_radius(ele, radius);
  renderingAtomicSet_shape(ele, shapeNum);

  return TRUE;
}

/* These functions write all the element list to export there associated resources. */
static void exportAtomicRadiusShape(GString *data, VisuData *dataObj)
{
  GList *pos, *eleList;
  unsigned int i;
  struct atomicResources_struct *str;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_RADIUS_SHAPE);
  /* We create a list of elements, or get the whole list. */
  eleList = (GList*)0;
  if (dataObj)
    {
      for (i = 0; i < dataObj->ntype; i++)
	eleList = g_list_prepend(eleList, (gpointer)dataObj->fromIntToVisuElement[i]);
      pos = eleList;
    }
  else
    pos = visuElementGet_allElements();
  while(pos)
    {
      str = (struct atomicResources_struct*)
	visuElementGet_property((VisuElement*)pos->data, "radiusAndShape");
      if (str)
	{
	  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_RADIUS_SHAPE);
	  g_string_append_printf(data, "    %s %10.3f %s\n",
				 ((VisuElement*)pos->data)->name,
				 str->radius, shapeName[str->shape]);
	}
      pos = g_list_next(pos);
    }
  g_string_append_printf(data, "\n");
  if (eleList)
    g_list_free(eleList);
}

static gboolean readAtomicShape(gchar **lines, int nbLines, int position,
				VisuData *dataObj _U_, GError **error)
{
  gchar **vals;
  guint i;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_string(lines[0], position, &vals, 1, FALSE, error))
    return FALSE;

  for (i = 0; i < sphere_nb; i++)
    if (!strcmp(vals[0], sphereName[i]))
      {
	sphereMethod = i;
	break;
      }
  if (i == sphere_nb)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: the sphere method '%s' is unknown.\n"),
			   position, vals[0]);
      g_strfreev(vals);
      return FALSE;
    }
  g_strfreev(vals);
  return TRUE;
}
static void exportAtomicShape(GString *data, VisuData* dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_SHAPE);
  g_string_append_printf(data, "%s: %s\n\n", FLAG_PARAMETER_SHAPE,
			 sphereName[sphereMethod]);
}


/***************/
/* OpenGL part */
/***************/

void renderingAtomic_positionShape(VisuData *visuData, VisuNode *node, VisuElement* ele)
{
  float rgba[4];
  float xyz[3];
  float scale;

  visuDataGet_nodePosition(visuData, node, xyz);
  scale = visuDataGet_nodeScalingFactor(visuData, node);

  glPushMatrix();
  glTranslated(xyz[0], xyz[1], xyz[2]);
  if (visuData->setColor)
    {
      visuData->setColor(visuData, rgba, ele, node);
      openGLSet_color(ele->material, rgba);
    }
  glScalef(scale, scale, scale);
  glCallList(ele->openGLIdentifier);
  glPopMatrix();
}
static void drawtriangle(float *v1, float *v2, float *v3) 
{ 
   glBegin(GL_TRIANGLES); 
      glNormal3fv(v1); glVertex3fv(v1);    
      glNormal3fv(v2); glVertex3fv(v2);    
      glNormal3fv(v3); glVertex3fv(v3);    
   glEnd(); 
}
static void normalize(float v[3]) {    
   GLfloat d = sqrt(v[0]*v[0]+v[1]*v[1]+v[2]*v[2]); 

   g_return_if_fail(d > 0.);
   v[0] /= d; v[1] /= d; v[2] /= d; 
}
static void subdivide(float *v1, float *v2, float *v3, int depth) 
{ 
  GLfloat v12[3], v23[3], v31[3];    
  GLint i;

  if (depth == 0)
    {
      drawtriangle(v1, v2, v3);
      return;
    }
  for (i = 0; i < 3; i++)
    { 
      v12[i] = v1[i]+v2[i]; 
      v23[i] = v2[i]+v3[i];     
      v31[i] = v3[i]+v1[i];    
    } 
  normalize(v12);    
  normalize(v23); 
  normalize(v31); 
  subdivide(v1, v12, v31, depth - 1);    
  subdivide(v2, v23, v12, depth - 1);    
  subdivide(v3, v31, v23, depth - 1);    
  subdivide(v12, v23, v31, depth - 1); 
}
int renderingAtomic_createShape(VisuData *visuData, VisuElement* ele)
{
  struct atomicResources_struct *str;
  OpenGLView *view;
  int nlat, i, nfac;
  GLUquadricObj *obj;

  obj = gluNewQuadric();

  g_return_val_if_fail(visuData && ele, -1);

  str = getRadiusAndShape(ele);
  g_return_val_if_fail(str, -1);

  view = visuDataGet_openGLView(visuData);
  nlat = OpenGLViewGet_numberOfFacettes(view, str->radius);

  DBG_fprintf(stderr, "Rendering Atomic: creating '%s' for %s (%d - OpenGL id"
	      " %d - fac %d)\n", shapeName[str->shape],
	      ele->name, ele->typeNumber, str->openGLIdentifier,
	      nlat);
  if (nlat < 0)
    return -1;
   
  glNewList(str->openGLIdentifier, GL_COMPILE);
  switch (str->shape)
    {
    case shapeSphere:
      DBG_fprintf(stderr, " | use sphere method %d\n", sphereMethod);
      if (sphereMethod == sphere_glu)
	gluSphere(obj, (double)str->radius, nlat, nlat);
      else if (sphereMethod == sphere_icosahedron)
	{
	  nfac = (int)(log((float)(nlat + 2) / 4.f) / log(2.f));
	  DBG_fprintf(stderr, " | glusphere vs. dodecahedron %dx%d\n",
		      nlat * nlat, 20 * (int)pow(4, nfac));
	  glPushMatrix();
	  glScalef(str->radius, str->radius, str->radius);
	  glBegin(GL_TRIANGLES);    
	  for (i = 0; i < 20; i++)
	    subdivide(&vdata[tindices[i][0]][0],       
		      &vdata[tindices[i][1]][0],       
		      &vdata[tindices[i][2]][0], nfac);
	  glEnd();
	  glPopMatrix();
	}
      else
	g_warning("Wrong sphere method.");
      break;
    case shapeElipsoid:
      glPushMatrix();
      glRotatef(str->phi, 0., 0., 1.);
      glRotatef(str->theta, 0., 1., 0.);
      glScalef(1.0, 1.0, str->ratio);
      gluSphere(obj, (double)str->radius, nlat, nlat);
      glPopMatrix();
      break;
    case shapePoint:
      glPushMatrix();
      glPointSize(MAX(1, (int)(str->radius * view->camera->gross * 5.)));
      glBegin(GL_POINTS);
      glVertex3f(0., 0., 0.);
      glEnd();
      glPopMatrix();
      break;
    case shapeCube:
      glBegin(GL_QUADS);

      glNormal3f(0., 0., 1.);
      glVertex3f(str->radius / 2., str->radius / 2., str->radius / 2.);
      glVertex3f(-str->radius / 2., str->radius / 2., str->radius / 2.);
      glVertex3f(-str->radius / 2., -str->radius / 2., str->radius / 2.);
      glVertex3f(str->radius / 2., -str->radius / 2., str->radius / 2.);

      glNormal3f(0., 0., -1.);
      glVertex3f(str->radius / 2.,str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,-str->radius / 2.);

      glNormal3f(1., 0., 0.);
      glVertex3f(str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,str->radius / 2.,-str->radius / 2.);

      glNormal3f(-1., 0., 0.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,str->radius / 2.);

      glNormal3f(0., 1., 0.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,-str->radius / 2.);
      glVertex3f(-str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,str->radius / 2.,str->radius / 2.);
      glVertex3f(str->radius / 2.,str->radius / 2.,-str->radius / 2.);

      glNormal3f(0., -1., 0.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,-str->radius / 2.);
      glVertex3f(str->radius / 2.,-str->radius / 2.,str->radius / 2.);
      glVertex3f(-str->radius / 2.,-str->radius / 2.,str->radius / 2.);

      glEnd();
      break;
    case shapeTorus:
      
      glPushMatrix();
      glRotatef(str->phi, 0., 0., 1.);
      glRotatef(str->theta, 0., 1., 0.);
      openGLObjectListDraw_torus(obj, 0, str->radius, str->ratio, nlat, nlat, FALSE);
      glPopMatrix();
      
      break;
	
    }
  glEndList();

  gluDeleteQuadric(obj);
   
/*   ele->openGLIdentifier = identifierSpheres + ele->typeNumber; */
  DBG_fprintf(stderr, " | shape created %d\n", str->openGLIdentifier);
  return str->openGLIdentifier;
}

void renderingAtomicAdd_loadMethod(RenderingFormatLoad* meth)
{
  g_return_if_fail(meth && meth->load);
  g_return_if_fail(atomic);

  DBG_fprintf(stderr, "Rendering Atomic: adding a new loading method '%s'.\n",
	      meth->name);
  allAtomicLoadingMethods = g_list_prepend(allAtomicLoadingMethods, meth);
  allAtomicLoadingMethods = g_list_sort(allAtomicLoadingMethods,
					visuRenderingFormatCompare_priority);
  if (meth->fmt)
    visuRenderingAdd_fileFormat(atomic, meth->fmt, 0);
}
