/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "panelBox.h"

#include <support.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_data.h>
#include <visu_basic.h>
#include <extensions/box.h>
#include <coreTools/toolColor.h>
#include <extraGtkFunctions/gtk_stippleComboBoxWidget.h>
#include <extraGtkFunctions/gtk_lineObjectWidget.h>
#include <extraFunctions/geometry.h>

/* Local objects. */
static GtkWidget *panelBox;
static GtkWidget *lineBox;
static GtkWidget *stippleExpandBox;
static GtkWidget *vboxPeriodic;
static GtkWidget *checkAllowTranslations;
static GtkWidget *spinTransXYZ[3];
static GtkWidget *checkAllowExpand;
static GtkWidget *spinExpandXYZ[3];
static GtkWidget *imagePeriodic;
static GtkWidget *labelPeriodic;
static gulong transId;
static gboolean transStatus = FALSE;
static GtkWidget *comboUnit;
static GtkWidget *checkDiff;

/* Local variables. */
static gboolean disableCallbacks;
static gboolean widgetsNotBuilt;

/* Local routines. */
static GtkWidget *createInteriorBox();
static void updateValues();
static void updatePeriodic(VisuData *dataObj);
static void applyTranslation(float transBox[3], gboolean setFree);
static void applyExpansion(float expansion[3]);

/* Local callbacks. */
static void onBoxEnter(ToolPanel *toolPanel, gpointer data);
static void onBoxChecked(LineObject *line, gboolean used, gpointer data);
static void onRGBValueChanged(LineObject *line, float *rgb, gpointer data);
static void onLineWidthChanged(LineObject *line, gint width, gpointer user_data);
static void onVisuDataChanged(GObject *obj, VisuData* visuData, gpointer data);
static void onVisuDataReady(GObject *obj, VisuData* visuData, gpointer data);
static void onTranslationChanged(GtkSpinButton* button, gpointer data);
static void onTranslationChecked(GtkToggleButton* button, gpointer data);
static void onExpandChanged(GtkSpinButton* button, gpointer data);
static void onExpandChecked(GtkToggleButton* button, gpointer data);
static void onElementRenderChanged(VisuData *dataObj,
				   VisuElement *element, gpointer data);
static void onNodePositionChanged(VisuData *dataObj, gpointer data);
static void onStippleChanged(LineObject *line, gint stipple, gpointer data);
static void onStippleExpandChanged(StippleComboBox *combo, gint stipple, gpointer data);
static void onUnitChanged(GtkComboBox *combo, gpointer data);
static void onDiffChanged(GtkToggleButton *toggle, gpointer data);

ToolPanel* panelBox_init()
{
  int i;
  GtkObject *adj;

  panelBox = toolPanelNew_withIconFromPath("Panel_box", _("Box and symmetries"),
					   _("Box"), "stock-box_20.png");
  if (!panelBox)
    return (ToolPanel*)0;

  toolPanelSet_dockable(TOOL_PANEL(panelBox), TRUE);

  /* Create the widgets that are needed even without the GTK interface be buit. */
  imagePeriodic = gtk_image_new_from_stock(GTK_STOCK_DIALOG_WARNING,
					   GTK_ICON_SIZE_BUTTON);
  labelPeriodic = gtk_label_new(_("<span size=\"smaller\"><i>non periodic"
			  " file format</i></span>"));
  vboxPeriodic = gtk_vbox_new(FALSE, 0);
  checkAllowTranslations =
    gtk_check_button_new_with_mnemonic(_("Translations"));
  checkAllowExpand =
    gtk_check_button_new_with_mnemonic(_("Expand nodes"));
  for (i = 0; i < 3; i++)
    {
      /* For the translation. */
      adj = gtk_adjustment_new(0, -1, 1, 0.05, 0.10, 0);
      spinTransXYZ[i] = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 2);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spinTransXYZ[i]), TRUE);
      /* For the expansion. */
      adj = gtk_adjustment_new(0, 0, 5, 0.05, 0.10, 0);
      spinExpandXYZ[i] = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1, 2);
      gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(spinExpandXYZ[i]), TRUE);
    }
  checkDiff =
    gtk_check_button_new_with_mnemonic(_("Visualise the node _displacements"));
  comboUnit = gtk_combo_box_new_text();

  /* Create the callbacks of all the sensitive widgets. */
  g_signal_connect(G_OBJECT(panelBox), "page-entered",
		   G_CALLBACK(onBoxEnter), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataLoaded",
		   G_CALLBACK(onVisuDataChanged), (gpointer)0);
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onVisuDataReady), (gpointer)0);

  /* Private parameters. */
  disableCallbacks = FALSE;
  widgetsNotBuilt  = TRUE;

  return TOOL_PANEL(panelBox);
}

static GtkWidget *createInteriorBox()
{
  GtkWidget *vbox, *hbox;
  GtkWidget *label, *align;
  int i;
#define X_LABEL _("dx:")
#define Y_LABEL _("dy:")
#define Z_LABEL _("dz:")
  char *xyz[3];
  const gchar **units;
#if GTK_MINOR_VERSION < 12
  GtkTooltips *tooltips;

  tooltips = gtk_tooltips_new ();
#endif

  vbox = gtk_vbox_new(FALSE, 0);

  /*********************/
  /* The Bounding box. */
  /*********************/
  lineBox = lineObjectNew(_("Bounding box"));
  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_container_add(GTK_CONTAINER(align), lineBox);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(lineBox), "use-changed",
		   G_CALLBACK(onBoxChecked), (gpointer)0);
  g_signal_connect(G_OBJECT(lineBox), "width-changed",
		   G_CALLBACK(onLineWidthChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineBox), "color-changed",
		   G_CALLBACK(onRGBValueChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(lineBox), "stipple-changed",
		   G_CALLBACK(onStippleChanged), (gpointer)0);

  /**************************/
  /* The periodicity stuff. */
  /**************************/
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("<b>Periodic operations</b>"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

  gtk_box_pack_start(GTK_BOX(hbox), imagePeriodic, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(hbox), labelPeriodic, FALSE, FALSE, 2);
  gtk_label_set_use_markup(GTK_LABEL(labelPeriodic), TRUE);
  gtk_label_set_line_wrap(GTK_LABEL(labelPeriodic), TRUE);

  align = gtk_alignment_new(0.5, 0., 1, 1);
  gtk_alignment_set_padding(GTK_ALIGNMENT(align), 0, 15, 0, 0);
  gtk_container_add(GTK_CONTAINER(align), vboxPeriodic);
  gtk_box_pack_start(GTK_BOX(vbox), align, FALSE, FALSE, 0);
  /* The translations. */
  gtk_widget_set_tooltip_text(checkAllowTranslations,
			      _("Translations are given in box coordinates and nodes are "
				"automatically translated back into the bounding box."));
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), checkAllowTranslations, FALSE, FALSE, 0);
  transId = g_signal_connect(G_OBJECT(checkAllowTranslations), "toggled",
			     G_CALLBACK(onTranslationChecked), (gpointer)0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);

  xyz[0] = X_LABEL;
  xyz[1] = Y_LABEL;
  xyz[2] = Z_LABEL;
  for (i = 0; i < 3; i++)
    {
      label = gtk_label_new(xyz[i]);
      gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 3);
      gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

      g_signal_connect(G_OBJECT(spinTransXYZ[i]), "value-changed",
		       G_CALLBACK(onTranslationChanged), (gpointer)0);
      gtk_box_pack_start(GTK_BOX(hbox), spinTransXYZ[i], FALSE, FALSE, 0);
    }

  /* The replication. */
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), hbox, FALSE, FALSE, 0);
  gtk_widget_set_tooltip_text(checkAllowExpand,
			      _("The size of the expansion is given in box coordinates."
				" Nodes are automatically translated back into the new"
				" defined area. The drawn bounding box is kept to the"
				" original size."));
  gtk_box_pack_start(GTK_BOX(hbox), checkAllowExpand, TRUE, TRUE, 0);
  g_signal_connect(G_OBJECT(checkAllowExpand), "toggled",
		   G_CALLBACK(onExpandChecked), (gpointer)0);
  /* The stipple for expand. */
  label = gtk_label_new(_("line:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);
  /* The stipple pattern. */
  stippleExpandBox = stippleComboBox_new();
  gtk_box_pack_start(GTK_BOX(hbox), stippleExpandBox, FALSE, FALSE, 0);
  g_signal_connect(G_OBJECT(stippleExpandBox), "stipple-selected",
		   G_CALLBACK(onStippleExpandChanged), (gpointer)0);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vboxPeriodic), hbox, FALSE, FALSE, 0);
  gtk_container_set_border_width(GTK_CONTAINER(hbox), 3);

  xyz[0] = X_LABEL;
  xyz[1] = Y_LABEL;
  xyz[2] = Z_LABEL;
  for (i = 0; i < 3; i++)
    {
      label = gtk_label_new(xyz[i]);
      gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 3);
      gtk_misc_set_alignment(GTK_MISC(label), 1, 0.5);

      g_signal_connect(G_OBJECT(spinExpandXYZ[i]), "value-changed",
		       G_CALLBACK(onExpandChanged), (gpointer)0);
      gtk_box_pack_start(GTK_BOX(hbox), spinExpandXYZ[i], FALSE, FALSE, 0);
    }

  /***********************/
  /* The geometry stuff. */
  /***********************/
  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("<b>Geometry</b>"));
  gtk_widget_set_name(label, "label_head");
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.5);
  gtk_label_set_use_markup(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 2);

  hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
  label = gtk_label_new(_("Set the unit of the file:"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  gtk_widget_set_sensitive(comboUnit, FALSE);
  units = visuToolsGet_unitNames();
  for (i = 0; units[i]; i++)
    gtk_combo_box_append_text(GTK_COMBO_BOX(comboUnit), units[i]);
  gtk_combo_box_set_active(GTK_COMBO_BOX(comboUnit), 0);
  g_signal_connect(G_OBJECT(comboUnit), "changed",
		   G_CALLBACK(onUnitChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(hbox), comboUnit, FALSE, FALSE, 0);

  gtk_widget_set_tooltip_text(checkDiff,
			      _("When a new file is loaded, draw arrows on  nodes that"
				" represent their displacements with respect to their"
				" previous positions."));
  g_signal_connect(G_OBJECT(checkDiff), "toggled",
		   G_CALLBACK(onDiffChanged), (gpointer)0);
  gtk_box_pack_start(GTK_BOX(vbox), checkDiff, FALSE, FALSE, 0);

  gtk_widget_show_all(vbox);
  return vbox;
}
static void updateValues()
{
  int i;
  float *transXYZ, transBox[3], expand[3];
  VisuData *dataObj;
  gboolean set;
  guint16 stipple;

  disableCallbacks = TRUE;
  DBG_fprintf(stderr, "Panel Box: update inside values.\n");

  lineObjectSet_use(LINE_OBJECT(lineBox), boxGet_isOn());
  lineObjectSet_width(LINE_OBJECT(lineBox), boxGet_lineWidth());
  lineObjectSet_color(LINE_OBJECT(lineBox), boxGet_RGBValues());
  lineObjectSet_stipple(LINE_OBJECT(lineBox), boxGet_lineStipple());

  stipple = boxGet_expandStipple();
  if (!stippleComboBoxSet_selectionByStipple(STIPPLE_COMBOX(stippleExpandBox), stipple))
    {
      stippleComboBoxAdd_pattern(STIPPLE_COMBOX(stippleExpandBox), stipple);
      stippleComboBoxSet_selectionByStipple(STIPPLE_COMBOX(stippleExpandBox), stipple);
    }
  
  /* Values dependent on a visuData. */
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  DBG_fprintf(stderr, "Panel Box: update values for object %p.\n",
	      (gpointer)dataObj);
  if (dataObj)
    {
      if (visuDataGet_boundaryConditions(dataObj) != BOX_FREE)
	{
	  transXYZ = visuDataGet_XYZtranslation(dataObj);
	  visuDataConvert_XYZtoBoxCoordinates(dataObj, transBox, transXYZ);
	  set = visuDataGet_translationStatus(dataObj);
	  if (set)
	    for (i = 0; i < 3; i++)
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinTransXYZ[i]),
					(double)transBox[i]);
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowTranslations),
				       set);
	  g_free(transXYZ);
	  visuDataGet_extension(dataObj, expand);
	  set = (expand[0] != 0. || expand[1] != 0. || expand[2] != 0.);
	  if (set)
	    for (i = 0; i < 3; i++)
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]),
					(double)expand[i]);
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowExpand), set);
	}
      gtk_combo_box_set_active(GTK_COMBO_BOX(comboUnit),
			       visuDataGet_unit(dataObj));
      gtk_widget_set_sensitive(comboUnit, TRUE);
    }
  else
    {
      for (i = 0; i < 3; i++)
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinTransXYZ[i]), 0.);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowTranslations), FALSE);
      for (i = 0; i < 3; i++)
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]), 0.);
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkAllowExpand), FALSE);
      gtk_widget_set_sensitive(comboUnit, FALSE);
    }
  disableCallbacks = FALSE;
  
  /* Set the sensitivity of the translation. */
  set = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowExpand));
  gtk_widget_set_sensitive(checkAllowTranslations, !set);
  gtk_widget_set_sensitive(spinTransXYZ[0], !set);
  gtk_widget_set_sensitive(spinTransXYZ[1], !set);
  gtk_widget_set_sensitive(spinTransXYZ[2], !set);
}
static void updatePeriodic(VisuData *dataObj)
{
  gboolean periodic;

  periodic = (dataObj && visuDataGet_boundaryConditions(dataObj) != BOX_FREE);
  gtk_widget_set_sensitive(vboxPeriodic, periodic);
  if (!periodic && dataObj)
    {
      gtk_widget_show(imagePeriodic);
      gtk_widget_show(labelPeriodic);
    }
  else
    {
      gtk_widget_hide(imagePeriodic);
      gtk_widget_hide(labelPeriodic);
    }
}
static void applyTranslation(float transBox[3], gboolean setFree)
{
  VisuData *data;
  float cartCoord[3];
  gboolean rebuild;

  data = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  if (!data)
    return;

  visuDataConvert_boxCoordinatestoXYZ(data, cartCoord, transBox);
  /*   fprintf(stderr, "%f %f %f\n", cartCoord[0], cartCoord[1], cartCoord[2]); */
  rebuild = (gboolean)visuDataSet_XYZtranslation(data, cartCoord);
  if (setFree)
    rebuild = visuData_constrainedFree(data) || rebuild;
  else
    rebuild = visuData_constrainedInTheBox(data) || rebuild;

  if (rebuild)
    {
      visuDataEmit_nodePositionChanged(data);
      visuData_createAllNodes(data);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
}
static void applyExpansion(float expansion[3])
{
  VisuData *data;
  gboolean rebuild, redraw;

  data = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  if (!data)
    return;

  redraw = visuData_replicate(data, expansion, &rebuild);
  if (rebuild)
    visuData_createAllNodes(data);
  if (redraw)
    g_idle_add(visuObjectRedraw, (gpointer)0);
}


/*************/
/* Callbacks */
/*************/
static void onBoxEnter(ToolPanel *toolPanel _U_, gpointer data _U_)
{
  if (widgetsNotBuilt)
    {
      DBG_fprintf(stderr, "Panel Box: first build on enter.\n");
      gtk_container_add(GTK_CONTAINER(panelBox), createInteriorBox());
      widgetsNotBuilt = FALSE;
    }
  updateValues();
  updatePeriodic(toolPanelGet_visuData(TOOL_PANEL(panelBox)));
}
static void boxRedraw()
{
  VisuData *dataObj;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  if (!dataObj) 
    return;
  boxDraw(dataObj);
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onBoxChecked(LineObject *line _U_, gboolean used, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  boxSet_isOn(used);
  boxRedraw();
}
static void onRGBValueChanged(LineObject *line _U_, float *rgb, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (boxSet_RGBValues(rgb, MASK_RGB_ALL))
    boxRedraw();
}
static void onLineWidthChanged(LineObject *line _U_, gint width, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (boxSet_lineWidth((float)width))
    boxRedraw();
}
static void onStippleChanged(LineObject *line _U_, gint stipple, gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (boxSet_lineStipple(stipple))
    boxRedraw();
}
static void onStippleExpandChanged(StippleComboBox *combo _U_, gint stipple,
				   gpointer data _U_)
{
  if (disableCallbacks)
    return;

  if (boxSet_expandStipple(stipple))
    boxRedraw();
}
static void onVisuDataChanged(GObject *obj _U_, VisuData* visuData, gpointer data _U_)
{
  float cartCoord[3];
  float transBox[3];
  int i;
  VisuData *dataRef;
  
  DBG_fprintf(stderr, "Panel Box: Catch 'dataLoaded' signal, update.\n");

  /* Update the sensitivity of the periodic stuff. */
  updatePeriodic(visuData);

  if (!visuData)
    return;

  /* Apply the periodic stuffs. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)) &&
      visuDataGet_boundaryConditions(visuData) != BOX_FREE)
    {
      DBG_fprintf(stderr, "Panel Box: Catch 'dataLoaded' signal,"
		  " applying current translations.\n");
      for (i = 0; i < 3; i++)
	transBox[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinTransXYZ[i]));
      visuDataConvert_boxCoordinatestoXYZ(visuData, cartCoord, transBox);
      visuDataSet_XYZtranslation(visuData, cartCoord);
      visuData_constrainedInTheBox(visuData);
    }

  /* Make a geometry diff. */
  dataRef = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDiff)) &&
      dataRef)
    {
      geometryDiff(dataRef, visuData);
      geometryRebuild_list(visuData);
    }

  g_signal_connect(G_OBJECT(visuData), "ElementRenderedChanged",
		   G_CALLBACK(onElementRenderChanged), (gpointer)0);
  g_signal_connect(G_OBJECT(visuData), "NodePositionChanged",
		   G_CALLBACK(onNodePositionChanged), (gpointer)0);
}
static void onVisuDataReady(GObject *obj _U_, VisuData* visuData, gpointer data _U_)
{
  float expand[3];
  int i, val;

  if (!visuData)
    return;

  /* Update the units if necessary. */
  if (visuBasicGet_preferedUnit(visuData) == unit_undefined)
    {
      val = gtk_combo_box_get_active(GTK_COMBO_BOX(comboUnit));
      if (val > 0 && visuDataGet_unit(visuData) != unit_undefined)
	visuDataSet_unit(visuData, (VisuUnits)val);
    }

  if (visuDataGet_boundaryConditions(visuData) == BOX_FREE)
    return;

  /* Apply the periodic stuffs. */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowExpand)))
    {
      DBG_fprintf(stderr, "Panel Box: Catch 'dataReadyForRendering' signal,"
		  " applying current expension.\n");
      for (i = 0; i < 3; i++)
	expand[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]));
      visuData_replicate(visuData, expand, (gboolean*)0);
    }
}
static void onTranslationChanged(GtkSpinButton* button _U_, gpointer data _U_)
{
  float transBox[3];
  int i;

  if (disableCallbacks)
    return;

  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
    return;

  for (i = 0; i < 3; i++)
    transBox[i] =
      (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinTransXYZ[i]));
  applyTranslation(transBox, FALSE);
}
static void onTranslationChecked(GtkToggleButton* button, gpointer data _U_)
{
  float transBox[3];
  float transZero[3] = { 0., 0., 0.};
  int i;

  if (disableCallbacks)
    return;

  if (gtk_toggle_button_get_active(button))
    {
      for (i = 0; i < 3; i++)
	transBox[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinTransXYZ[i]));
      applyTranslation(transBox, FALSE);
    }
  else
    applyTranslation(transZero, TRUE);
}
static void onElementRenderChanged(VisuData *dataObj,
				   VisuElement *element, gpointer data _U_)
{
  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
    return;

  if (element->rendered)
    {
      DBG_fprintf(stderr, "Panel Box: caught 'ElementRenderedChanged' applying"
		  " translation for element '%s'.\n", element->name);
      if (visuData_constrainedElementInTheBox(dataObj, element))
	visuDataEmit_nodePositionChanged(dataObj);
    }
  else
    DBG_fprintf(stderr, "Panel Box: caught 'ElementRenderedChanged' but"
		" do not apply translation since element '%s' is masked.\n",
		element->name);
}
static void onExpandChanged(GtkSpinButton* button _U_, gpointer data _U_)
{
  float expansion[3];
  int i;

  if (disableCallbacks)
    return;

  if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowExpand)))
    return;

  for (i = 0; i < 3; i++)
    expansion[i] =
      (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]));
  applyExpansion(expansion);
}
static void onExpandChecked(GtkToggleButton* button, gpointer data _U_)
{
  float expansion[3];
  float transBox[3], cartCoord[3];
  float transZero[3] = { 0.f, 0.f, 0.f};
  int i;
  gboolean set;
  VisuData *dataObj;
  gboolean rebuildTrans, rebuildExp, redraw;

  if (disableCallbacks)
    return;

  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  if (!dataObj)
    return;

  redraw = rebuildTrans = FALSE;
  if (gtk_toggle_button_get_active(button))
    {
      for (i = 0; i < 3; i++)
	expansion[i] =
	  (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinExpandXYZ[i]));

      /* Before applying expansion we put everything into the box
	 if it is not there yet. */
      if (!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkAllowTranslations)))
	{
	  g_signal_handler_block(G_OBJECT(checkAllowTranslations), transId);
	  gtk_toggle_button_set_active
	    (GTK_TOGGLE_BUTTON(checkAllowTranslations), TRUE);
	  g_signal_handler_unblock(G_OBJECT(checkAllowTranslations), transId);
	  for (i = 0; i < 3; i++)
	    transBox[i] =
	      (float)gtk_spin_button_get_value(GTK_SPIN_BUTTON(spinTransXYZ[i]));
	  visuDataConvert_boxCoordinatestoXYZ(dataObj, cartCoord, transBox);
	  rebuildTrans = (gboolean)visuDataSet_XYZtranslation(dataObj, cartCoord);
	  rebuildTrans = visuData_constrainedInTheBox(dataObj) || rebuildTrans;
	  if (rebuildTrans)
	    visuDataEmit_nodePositionChanged(dataObj);
	  transStatus = FALSE;
	}
      else
	  transStatus = TRUE;

      redraw = visuData_replicate(dataObj, expansion, &rebuildExp);
    }
  else
    {
      redraw = rebuildExp = visuData_restore(dataObj);

      /* We release the translation if not checked. */
      if (!transStatus)
	{
	  g_signal_handler_block(G_OBJECT(checkAllowTranslations), transId);
	  gtk_toggle_button_set_active
	    (GTK_TOGGLE_BUTTON(checkAllowTranslations), FALSE);
	  g_signal_handler_unblock(G_OBJECT(checkAllowTranslations), transId);
	  rebuildTrans = (gboolean)visuDataSet_XYZtranslation(dataObj, transZero);
	  rebuildTrans = visuData_constrainedFree(dataObj) || rebuildTrans;
	  if (rebuildTrans)
	    visuDataEmit_nodePositionChanged(dataObj);
	}
    }
  if (rebuildTrans || rebuildExp)
    visuData_createAllNodes(dataObj);
  if (rebuildTrans || redraw)
    g_idle_add(visuObjectRedraw, (gpointer)0);

  /* Set the sensitivity of the translation. */
  set = gtk_toggle_button_get_active(button);
  gtk_widget_set_sensitive(checkAllowTranslations, !set);
  gtk_widget_set_sensitive(spinTransXYZ[0], !set);
  gtk_widget_set_sensitive(spinTransXYZ[1], !set);
  gtk_widget_set_sensitive(spinTransXYZ[2], !set);
}
static void onUnitChanged(GtkComboBox *combo, gpointer data _U_)
{
  gint val;
  VisuUnits unit;
  VisuData *dataObj;

  val = gtk_combo_box_get_active(combo);
  g_return_if_fail(val >= 0);

  unit = (VisuUnits)val;
  
  dataObj = toolPanelGet_visuData(TOOL_PANEL(panelBox));
  g_return_if_fail(dataObj);

  if (visuDataSet_unit(dataObj, unit))
    {
      visuData_createAllNodes(dataObj);
      g_idle_add(visuObjectRedraw, (gpointer)0);
    }
}
static void onDiffChanged(GtkToggleButton *toggle, gpointer data _U_)
{
  geometrySet_active(gtk_toggle_button_get_active(toggle));
  g_idle_add(visuObjectRedraw, (gpointer)0);
}
static void onNodePositionChanged(VisuData *dataObj, gpointer data _U_)
{
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(checkDiff)))
    geometryRebuild_list(dataObj);
}
