/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#ifndef UTOUCH_QML_DEVICE_H_
#define UTOUCH_QML_DEVICE_H_

#include <QDeclarativeItem>

#include <geis/geis.h>

/**
 * A property type used for providing information about an axis of a touch
 * device.
 */

class AxisInfo : public QObject {
  Q_OBJECT

  /**
   * This property holds the minimum value of the axis.
   */
  Q_PROPERTY(float min READ min CONSTANT)

  /**
   * This property holds the maximum value of the axis.
   */
  Q_PROPERTY(float max READ max CONSTANT)

  /**
   * This property holds the resolution of the axis in units per millimeter.
   *
   * When the resolution of the device is unknown, the value of this property is
   * set to 0.
   */
  Q_PROPERTY(float resolution READ resolution CONSTANT)

 public:
  AxisInfo(float min, float max, float resolution, QObject* parent = 0)
    : QObject(parent),
      min_(min),
      max_(max),
      resolution_(resolution) {}

  float min() const { return min_; }
  float max() const { return max_; }
  float resolution() const { return resolution_; }

 private:
  const float min_;
  const float max_;
  const float resolution_;

  Q_DISABLE_COPY(AxisInfo)
};

/**
 * An item that provides information about a gesture input device.
 */

class Device : public QDeclarativeItem {
  Q_OBJECT
  Q_ENUMS(DeviceType)

  /**
   * This property holds the ID of the touch.
   */
  Q_PROPERTY(int id READ id CONSTANT)

  /**
   * This property holds the type of the device.
   */
  Q_PROPERTY(DeviceType type READ device_type CONSTANT)

  /**
   * This property holds information on the X axis
   */
  Q_PROPERTY(AxisInfo* x READ x CONSTANT)

  /**
   * This property holds information on the Y axis
   */
  Q_PROPERTY(AxisInfo* y READ y CONSTANT)

  /**
   * This property holds a variant map of attributes of the touch.
   *
   * All uTouch touch attributes are available in this property. The keys may
   * be referenced in the geis.h header file.
   */
  Q_PROPERTY(const QVariantMap attributes READ attributes CONSTANT)

 public:
  Device(GeisDevice device, QDeclarativeItem *parent = 0);
  ~Device();

  /**
   * An enumeration of classes of multitouch input devices.
   */
  enum DeviceType {
    /**
     * A touchscreens or similar direct input devices.
     *
     * Individual touches are mapped to the screen. Touch locations are
     * provided in screen coordinates.
     */
    TouchScreen,

    /**
     * A touchpads and similar indirect input devices.
     *
     * Individual touches are directed to the location of the associated
     * cursor. Single touch drags that are not accepted as gesture events
     * move the cursor. Touch locations are provided in device coordinates.
     */
    TouchPad,

    /**
     * An independent input devices.
     *
     * Individual touches are directed to the location of the associated
     * cursor. Cursor movement is completely independent of touch input.
     * Touch locations are provided in device coordinates.
     */
    Independent,

    /**
     * An unknown input device.
     *
     * No information is known about how the device interacts with the cursor
     * or how locations are provided.
     */
    Unknown
  };

  int id() const { return id_; }
  DeviceType device_type() const { return device_type_; }
  AxisInfo* x() { return x_; }
  AxisInfo* y() { return y_; }
  const QVariantMap attributes() const { return attributes_; }

 private:
  GeisDevice device_;
  int id_;
  DeviceType device_type_;
  AxisInfo* x_;
  AxisInfo* y_;
  QVariantMap attributes_;

  Q_DISABLE_COPY(Device)
};

QML_DECLARE_TYPE(Device)

#endif  // UTOUCH_QML_DEVICE_H_
