/*
 * Copyright (C) 2013 Kylin Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Weihua Zhang <fly2high@126.com>
 * Authored by liuxing <liuxingcs@yeah.net>
 * Modified Information: 
 * Rewrite perform_search to search China music from music.baidu.com
 */

/*
 * Copyright (C) 2011 Canonical Ltd
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authored by Alex Launi <alex.launi@canonical.com>
 *
 */
using GLib;
using Config;

namespace Unity.MusicLens
{

static Application? app = null;
static MusicBaiduScopeProxy? daemon = null;


public class MusicBaiduScopeProxy : SimpleScope
{
    private MusicBaiduCollection collection;
    private Unity.MusicPreview? music_preview;

    public MusicBaiduScopeProxy ()
    {
        base ();

        scope = new Unity.Scope ("/com/canonical/unity/scope/musicbaidu");
        scope.search_in_global = true;
        scope.activate_uri.connect (activate);
        scope.preview_uri.connect (preview);

        base.initialize ();

        collection = new MusicBaiduCollection ();

        try
        {
            scope.export ();
        }
        catch (GLib.IOError e)
        {
            stdout.printf (e.message);
        }

    }

    protected override int num_results_without_search { get { return 100;
                                                            }
                                                      }
    protected override int num_results_global_search { get { return 20;
                                                           }
                                                     }
    protected override int num_results_lens_search { get { return 50;
                                                         }
                                                   }


    public Unity.ActivationResponse activate (string uri)
    {
        /* launch the music baidu streaming client */
        try
        {

            if(uri.has_prefix ("song://"))
            {
                AppInfo.launch_default_for_uri (uri.substring (7), null);//strip off "song://" from the uri
            }
            else if(uri.has_prefix ("album://"))
            {
                AppInfo.launch_default_for_uri (uri.substring (8), null);//strip off "album://" from the uri
            }
        }
        catch (GLib.Error e)
        {
            warning ("Failed to open uri %s. %s", uri, e.message);
        }
        return new Unity.ActivationResponse (Unity.HandledType.HIDE_DASH);
    }

    public Unity.ActivationResponse download_album (string uri)
    {
        return activate (uri);
    }

    public Unity.Preview preview (string uri)
    {
        music_preview = null;
        Album album = null;
        SList<Track> tracks = null;


        if(uri.has_prefix ("album://"))
        {
            collection.get_album_details(uri,  out album, out tracks,null);
        }
        else if(uri.has_prefix ("song://"))
        {
            collection.get_song_details (uri, out album, out tracks,null);
        }

        if (album != null)
        {

            GLib.Icon? icon_file = null;
            int i = 1;


            var artwork_path = album.artwork_path;
            icon_file = new GLib.FileIcon(File.new_for_uri (artwork_path));
            music_preview = new Unity.MusicPreview (album.title, album.artist, icon_file);


            if (tracks != null)
            {
                foreach (Track track in tracks)
                {
                    TrackMetadata tm = new TrackMetadata ();
                    tm.uri = track.uri;
                    tm.track_no = i++; //FIXME: u1ms search doesn't provide track numbers *yet*, this will change soon
                    tm.title = track.title;
                    tm.length = track.duration;
                    music_preview.add_track (tm);
                }
            }

            GLib.Icon? icon = new GLib.FileIcon (File.new_for_path (Config.DATADIR + "/icons/unity-icon-theme/places/svg/service-u1.svg"));
            var download_action = new Unity.PreviewAction ("download_album", _("Download"), icon);

            download_action.activated.connect (download_album);
            music_preview.add_action (download_action);
        }
        return music_preview;
    }

    public override async void perform_search (LensSearch search,
            SearchType search_type,
            owned List<FilterParser> filters,
            int max_results = -1,
            Cancellable? cancellable = null)
    {
        if (is_search_empty (search))
            return;

        try
        {   debug ("model has %u rows before search", search.results_model.get_n_rows ());
            yield collection.search (search,
                                     search_type,
                                     (owned) filters,
                                     max_results,
                                     cancellable);
           debug ("model has %u rows after search", search.results_model.get_n_rows ());
        }
        catch (IOError e)
        {
            warning ("Failed to search for '%s': %s", search.search_string, e.message);
        }
    }

}

/* Check if a given well known DBus is owned.
 * WARNING: This does sync IO!  */
public static bool dbus_name_has_owner (string name)
{
    try
    {
        bool has_owner;
        DBusConnection bus = Bus.get_sync (BusType.SESSION);
        Variant result = bus.call_sync ("org.freedesktop.DBus",
                                        "/org/freedesktop/dbus",
                                        "org.freedesktop.DBus",
                                        "NameHasOwner",
                                        new Variant ("(s)", name),
                                        new VariantType ("(b)"),
                                        DBusCallFlags.NO_AUTO_START,
                                        -1);
        result.get ("(b)", out has_owner);
        return has_owner;
    }
    catch (Error e)
    {
        warning ("Unable to decide whether '%s' is running: %s", name, e.message);
    }
    return false;
}


public static int main (string[] args)
{
    GLib.Environment.set_prgname ("unity-musicbaidu-daemon");

    /* Sort up locale to get translations but also sorting and
     * punctuation right */
    GLib.Intl.textdomain (Config.PACKAGE);
    GLib.Intl.bindtextdomain (Config.PACKAGE, Config.LOCALEDIR);
    GLib.Intl.bind_textdomain_codeset (Config.PACKAGE, "UTF-8");
    GLib.Intl.setlocale(GLib.LocaleCategory.ALL, "");

    /* Workaround for https://bugzilla.gnome.org/show_bug.cgi?id=640714
     * GApplication.register() call owns our DBus name in a sync manner
     * making it race against GDBus' worker thread to export our
     * objects on the bus before/after owning our name and receiving
     * method calls on our objects (which may not yet be up!)*/
    if (dbus_name_has_owner ("com.canonical.Unity.Scope.MusicBaidu"))
    {
        print ("Another instance of the China Music  Daemon " +
               "already appears to be running.\nBailing out.\n");
        return 2;
    }

    /* Now register our DBus objects *before* acquiring the name!
     * See above for reasons */
    daemon = new MusicBaiduScopeProxy ();

    /* Use GApplication directly for single instance app functionality */
    app = new Application ("com.canonical.Unity.Scope.MusicBaidu",
                           ApplicationFlags.IS_SERVICE);
    try
    {
        app.register ();
    }
    catch (Error e)
    {
        /* FIXME: We get this error if another daemon is already running,
         * but it uses a generic error so we can't detect this reliably... */
        print ("Failed to start the China music daemon: %s\n", e.message);
        return 1;
    }

    if (app.get_is_remote ())
    {
        print ("Another instance of the China Music Daemon " +
               "already appears to be running.\nBailing out.\n");
        return 2;
    }

    /* Hold()ing the app makes sure the GApplication doesn't exit */
    app.hold();
    return app.run ();
}

} /* namespace */
