;;  Filename : test-member.scm
;;  About    : unit tests for memq, memv, member
;;
;;  Copyright (C) 2006 YAMAMOTO Kengo <yamaken AT bp.iij4u.or.jp>
;;  Copyright (c) 2007 SigScheme Project <uim AT freedesktop.org>
;;
;;  All rights reserved.
;;
;;  Redistribution and use in source and binary forms, with or without
;;  modification, are permitted provided that the following conditions
;;  are met:
;;
;;  1. Redistributions of source code must retain the above copyright
;;     notice, this list of conditions and the following disclaimer.
;;  2. Redistributions in binary form must reproduce the above copyright
;;     notice, this list of conditions and the following disclaimer in the
;;     documentation and/or other materials provided with the distribution.
;;  3. Neither the name of authors nor the names of its contributors
;;     may be used to endorse or promote products derived from this software
;;     without specific prior written permission.
;;
;;  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS
;;  IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
;;  THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
;;  PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR
;;  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
;;  EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
;;  PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
;;  PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
;;  LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

(load "./test/unittest.scm")

(define tn test-name)

(define elm0 (lambda () #f))
(define elm1 (lambda () #f))
(define elm2 (lambda () #f))
(define elm3 (lambda () #f))
(define nil  '())
(define cdr3 (cons elm3 nil))
(define cdr2 (cons elm2 cdr3))
(define cdr1 (cons elm1 cdr2))
(define cdr0 (cons elm0 cdr1))
(define lst cdr0)

;; Remake char object to avoid constant optimization. If the implementation
;; does not have neither immediate char nor preallocated char objects, (eq? c
;; (char c)) will be false.
(define char
  (lambda (c)
    (integer->char (char->integer c))))

;;
;; memq
;;

(tn "memq")
(assert-false  (tn)       (memq 'a '()))
(assert-false  (tn)       (memq '() '()))
(assert-equal? (tn) '(()) (memq '() '(())))
(tn "memq builtin procedures")
(assert-equal? (tn) (list + - * /) (memq + (list + - * /)))
(assert-equal? (tn) (list - * /)   (memq - (list + - * /)))
(assert-equal? (tn) (list * /)     (memq * (list + - * /)))
(assert-equal? (tn) (list /)       (memq / (list + - * /)))
(assert-false  (tn)                (memq car (list + - * /)))
(tn "memq closures")
(assert-equal? (tn) cdr0 (memq elm0 lst))
(assert-equal? (tn) cdr1 (memq elm1 lst))
(assert-equal? (tn) cdr2 (memq elm2 lst))
(assert-equal? (tn) cdr3 (memq elm3 lst))
(assert-false  (tn)      (memq 'none lst))
(tn "memq strings with non-constant key")
;; These tests assume that (string #\a) is not optimized as constant string.
(assert-false  (tn) (memq (string #\a) '("a" "b" "c" "d")))
(assert-false  (tn) (memq (string #\b) '("a" "b" "c" "d")))
(assert-false  (tn) (memq (string #\c) '("a" "b" "c" "d")))
(assert-false  (tn) (memq (string #\d) '("a" "b" "c" "d")))
(assert-false  (tn) (memq (string #\e) '("a" "b" "c" "d")))
(tn "memq lists with non-constant key")
;; These tests assume that the keys are not optimized as constant object.
(assert-false  (tn) (memq (list (string #\a)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memq (list (string #\b)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memq (list (string #\c)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memq (list (string #\d)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memq (list (string #\e)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn)
               (memq (list (string #\a #\B #\c)
                           (list (string #\d) (list (string #\e))))
                     '(("aBc" ("d" ("E"))) ("aBc" ("d" ("e"))) "f" ("g"))))

(tn "memq improper lists: symbols")
(assert-equal? (tn) '(a b c . d) (memq 'a '(a b c . d)))
(assert-equal? (tn) '(b c . d)   (memq 'b '(a b c . d)))
(assert-equal? (tn) '(c . d)     (memq 'c '(a b c . d)))
(assert-error  (tn) (lambda ()   (memq 'd '(a b c . d))))
(assert-error  (tn) (lambda ()   (memq 'e '(a b c . d))))
(assert-error  (tn) (lambda ()   (memq 'a 'a)))
(assert-error  (tn) (lambda ()   (memq 'b 'a)))
(tn "memq improper lists: strings with non-constant key")
(assert-error  (tn) (lambda () (memq (string #\a) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memq (string #\b) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memq (string #\c) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memq (string #\d) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memq (string #\e) '("a" "b" "c" . "d"))))
(tn "memq improper lists: lists with non-constant key")
(assert-error  (tn) (lambda () (memq (list (string #\a))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memq (list (string #\b))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memq (list (string #\c))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memq (list (string #\d))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memq (list (string #\e))
                                     '(("a") ("b") ("c") . "d"))))

(tn "memq from R5RS examples")
(assert-equal? (tn) '(a b c) (memq 'a '(a b c)))
(assert-equal? (tn) '(b c)   (memq 'b '(a b c)))
(assert-false  (tn)          (memq 'a '(b c d)))
(assert-false  (tn)          (memq (list 'a) '(b (a) c)))

;;
;; memv
;;

(tn "memv")
(assert-false  (tn)       (memv 'a '()))
(assert-false  (tn)       (memv '() '()))
(assert-equal? (tn) '(()) (memv '() '(())))
(tn "memv symbols")
(assert-equal? (tn) '(a b c) (memv 'a '(a b c)))
(assert-equal? (tn) '(b c)   (memv 'b '(a b c)))
(assert-false  (tn)          (memv 'a '(b c d)))
(tn "memv builtin procedures")
(assert-equal? (tn) (list + - * /) (memv + (list + - * /)))
(assert-equal? (tn) (list - * /)   (memv - (list + - * /)))
(assert-equal? (tn) (list * /)     (memv * (list + - * /)))
(assert-equal? (tn) (list /)       (memv / (list + - * /)))
(assert-false  (tn)                (memv car (list + - * /)))
(tn "memv closures")
(assert-equal? (tn) cdr0 (memv elm0 lst))
(assert-equal? (tn) cdr1 (memv elm1 lst))
(assert-equal? (tn) cdr2 (memv elm2 lst))
(assert-equal? (tn) cdr3 (memv elm3 lst))
(assert-false  (tn)      (memv 'none lst))
(tn "memv numbers")
(assert-equal? (tn) '(0 1 2 3) (memv 0 '(0 1 2 3)))
(assert-equal? (tn) '(1 2 3)   (memv 1 '(0 1 2 3)))
(assert-equal? (tn) '(2 3)     (memv 2 '(0 1 2 3)))
(assert-equal? (tn) '(3)       (memv 3 '(0 1 2 3)))
(assert-false  (tn)            (memv 4 '(0 1 2 3)))
(tn "memv chars")
(assert-equal? (tn) '(#\a #\b #\c #\d) (memv #\a '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\b #\c #\d)     (memv #\b '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\c #\d)         (memv #\c '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\d)             (memv #\d '(#\a #\b #\c #\d)))
(assert-false  (tn)                    (memv #\e '(#\a #\b #\c #\d)))
(tn "memv strings with non-constant key")
;; These tests assume that (string #\a) is not optimized as constant string.
(assert-false  (tn) (memv (string #\a) '("a" "b" "c" "d")))
(assert-false  (tn) (memv (string #\b) '("a" "b" "c" "d")))
(assert-false  (tn) (memv (string #\c) '("a" "b" "c" "d")))
(assert-false  (tn) (memv (string #\d) '("a" "b" "c" "d")))
(assert-false  (tn) (memv (string #\e) '("a" "b" "c" "d")))
(tn "memv lists")
(assert-false  (tn) (memv (list 'a) '(b (a) c)))
(tn "memv lists with non-constant key")
;; These tests assume that the keys are not optimized as constant object.
(assert-false  (tn) (memv (list (string #\a)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memv (list (string #\b)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memv (list (string #\c)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memv (list (string #\d)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (memv (list (string #\e)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn)
               (memv (list (string #\a #\B #\c)
                           (list (string #\d) (list (string #\e))))
                     '(("aBc" ("d" ("E"))) ("aBc" ("d" ("e"))) "f" ("g"))))

(tn "memv improper lists: symbols")
(assert-equal? (tn) '(a b c . d) (memv 'a '(a b c . d)))
(assert-equal? (tn) '(b c . d)   (memv 'b '(a b c . d)))
(assert-equal? (tn) '(c . d)     (memv 'c '(a b c . d)))
(assert-error  (tn) (lambda ()   (memv 'd '(a b c . d))))
(assert-error  (tn) (lambda ()   (memv 'e '(a b c . d))))
(assert-error  (tn) (lambda ()   (memv 'a 'a)))
(assert-error  (tn) (lambda ()   (memv 'b 'a)))
(tn "memv improper lists: chars")
(assert-equal? (tn) '(#\a #\b #\c . #\d) (memv #\a '(#\a #\b #\c . #\d)))
(assert-equal? (tn) '(#\b #\c . #\d)     (memv #\b '(#\a #\b #\c . #\d)))
(assert-equal? (tn) '(#\c . #\d)         (memv #\c '(#\a #\b #\c . #\d)))
(assert-error  (tn) (lambda ()           (memv #\d '(#\a #\b #\c . #\d))))
(assert-error  (tn) (lambda ()           (memv #\e '(#\a #\b #\c . #\d))))
(tn "memv improper lists: strings with non-constant key")
(assert-error  (tn) (lambda () (memv (string #\a) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memv (string #\b) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memv (string #\c) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memv (string #\d) '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda () (memv (string #\e) '("a" "b" "c" . "d"))))
(tn "memv improper lists: lists with non-constant key")
(assert-error  (tn) (lambda () (memv (list (string #\a))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memv (list (string #\b))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memv (list (string #\c))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memv (list (string #\d))
                                     '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (memv (list (string #\e))
                                     '(("a") ("b") ("c") . "d"))))

;;
;; member
;;

(tn "member")
(assert-false  (tn)       (member 'a '()))
(assert-false  (tn)       (member '() '()))
(assert-equal? (tn) '(()) (member '() '(())))
(tn "member symbols")
(assert-equal? (tn) '(a b c) (member 'a '(a b c)))
(assert-equal? (tn) '(b c)   (member 'b '(a b c)))
(assert-false  (tn)          (member 'a '(b c d)))
(assert-equal? (tn) '((a) c) (member (list 'a) '(b (a) c)))  ;; R5RS
(tn "member builtin procedures")
(assert-equal? (tn) (list + - * /) (member + (list + - * /)))
(assert-equal? (tn) (list - * /)   (member - (list + - * /)))
(assert-equal? (tn) (list * /)     (member * (list + - * /)))
(assert-equal? (tn) (list /)       (member / (list + - * /)))
(assert-false  (tn)                (member car (list + - * /)))
(tn "member closures")
(assert-equal? (tn) cdr0 (member elm0 lst))
(assert-equal? (tn) cdr1 (member elm1 lst))
(assert-equal? (tn) cdr2 (member elm2 lst))
(assert-equal? (tn) cdr3 (member elm3 lst))
(assert-false  (tn)      (member 'none lst))
(tn "member numbers")
(assert-equal? (tn) '(0 1 2 3) (member 0 '(0 1 2 3)))
(assert-equal? (tn) '(1 2 3)   (member 1 '(0 1 2 3)))
(assert-equal? (tn) '(2 3)     (member 2 '(0 1 2 3)))
(assert-equal? (tn) '(3)       (member 3 '(0 1 2 3)))
(assert-false  (tn)            (member 4 '(0 1 2 3)))
(tn "member chars")
(assert-equal? (tn) '(#\a #\b #\c #\d) (member #\a '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\b #\c #\d)     (member #\b '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\c #\d)         (member #\c '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\d)             (member #\d '(#\a #\b #\c #\d)))
(assert-false  (tn)                    (member #\e '(#\a #\b #\c #\d)))
(tn "member chars with non-constant key")
(assert-equal? (tn) '(#\a #\b #\c #\d) (member (char #\a) '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\b #\c #\d)     (member (char #\b) '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\c #\d)         (member (char #\c) '(#\a #\b #\c #\d)))
(assert-equal? (tn) '(#\d)             (member (char #\d) '(#\a #\b #\c #\d)))
(assert-false  (tn)                    (member (char #\e) '(#\a #\b #\c #\d)))
(tn "member strings")
(assert-equal? (tn) '("a" "b" "c" "d") (member "a" '("a" "b" "c" "d")))
(assert-equal? (tn) '("b" "c" "d")     (member "b" '("a" "b" "c" "d")))
(assert-equal? (tn) '("c" "d")         (member "c" '("a" "b" "c" "d")))
(assert-equal? (tn) '("d")             (member "d" '("a" "b" "c" "d")))
(assert-false  (tn)                    (member "e" '("a" "b" "c" "d")))
(tn "member strings with non-constant key")
;; These tests assume that (string #\a) is not optimized as constant string.
(assert-equal? (tn) '("a" "b" "c" "d") (member (string #\a) '("a" "b" "c" "d")))
(assert-equal? (tn) '("b" "c" "d")    (member (string #\b) '("a" "b" "c" "d")))
(assert-equal? (tn) '("c" "d")        (member (string #\c) '("a" "b" "c" "d")))
(assert-equal? (tn) '("d")            (member (string #\d) '("a" "b" "c" "d")))
(assert-false  (tn)                   (member (string #\e) '("a" "b" "c" "d")))
(tn "member lists")
(assert-equal? (tn)
               '(("a") ("b") ("c") ("d"))
               (member '("a") '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("b") ("c") ("d"))
               (member '("b") '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("c") ("d"))
               (member '("c") '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("d"))
               (member '("d") '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (member '("e") '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("aBc" ("d" ("e"))) "f" ("g"))
               (member '("aBc" ("d" ("e")))
                       '(("aBc" ("d" ("E"))) ("aBc" ("d" ("e"))) "f" ("g"))))
(tn "member lists with non-constant key")
;; These tests assume that the keys are not optimized as constant object.
(assert-equal? (tn)
               '(("a") ("b") ("c") ("d"))
               (member (list (string #\a)) '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("b") ("c") ("d"))
               (member (list (string #\b)) '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("c") ("d"))
               (member (list (string #\c)) '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("d"))
               (member (list (string #\d)) '(("a") ("b") ("c") ("d"))))
(assert-false  (tn) (member (list (string #\e)) '(("a") ("b") ("c") ("d"))))
(assert-equal? (tn)
               '(("aBc" ("d" ("e"))) "f" ("g"))
               (member (list (string #\a #\B #\c)
                             (list (string #\d) (list (string #\e))))
                       '(("aBc" ("d" ("E"))) ("aBc" ("d" ("e"))) "f" ("g"))))

(tn "member improper lists: symbols")
(assert-equal? (tn) '(a b c . d) (member 'a '(a b c . d)))
(assert-equal? (tn) '(b c . d)   (member 'b '(a b c . d)))
(assert-equal? (tn) '(c . d)     (member 'c '(a b c . d)))
(assert-error  (tn) (lambda ()   (member 'd '(a b c . d))))
(assert-error  (tn) (lambda ()   (member 'e '(a b c . d))))
(assert-error  (tn) (lambda ()   (member 'a 'a)))
(assert-error  (tn) (lambda ()   (member 'b 'a)))
(tn "member improper lists: chars")
(assert-equal? (tn) '(#\a #\b #\c . #\d) (member #\a '(#\a #\b #\c . #\d)))
(assert-equal? (tn) '(#\b #\c . #\d)     (member #\b '(#\a #\b #\c . #\d)))
(assert-equal? (tn) '(#\c . #\d)         (member #\c '(#\a #\b #\c . #\d)))
(assert-error  (tn) (lambda ()           (member #\d '(#\a #\b #\c . #\d))))
(assert-error  (tn) (lambda ()           (member #\e '(#\a #\b #\c . #\d))))
(tn "member improper lists: chars with non-constant key")
(assert-equal? (tn) '(#\a #\b #\c . #\d) (member (char #\a)
                                                 '(#\a #\b #\c . #\d)))
(assert-equal? (tn) '(#\b #\c . #\d)     (member (char #\b)
                                                 '(#\a #\b #\c . #\d)))
(assert-equal? (tn) '(#\c . #\d)         (member (char #\c)
                                                 '(#\a #\b #\c . #\d)))
(assert-error  (tn) (lambda ()           (member (char #\d)
                                                 '(#\a #\b #\c . #\d))))
(assert-error  (tn) (lambda ()           (member (char #\e)
                                                 '(#\a #\b #\c . #\d))))
(tn "member improper lists: strings")
(assert-equal? (tn) '("a" "b" "c" . "d") (member "a" '("a" "b" "c" . "d")))
(assert-equal? (tn) '("b" "c" . "d")     (member "b" '("a" "b" "c" . "d")))
(assert-equal? (tn) '("c" . "d")         (member "c" '("a" "b" "c" . "d")))
(assert-error  (tn) (lambda ()           (member "d" '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda ()           (member "e" '("a" "b" "c" . "d"))))
(tn "member improper lists: strings with non-constant key")
(assert-equal? (tn) '("a" "b" "c" . "d") (member (string #\a)
                                                 '("a" "b" "c" . "d")))
(assert-equal? (tn) '("b" "c" . "d")     (member (string #\b)
                                                 '("a" "b" "c" . "d")))
(assert-equal? (tn) '("c" . "d")         (member (string #\c)
                                                 '("a" "b" "c" . "d")))
(assert-error  (tn) (lambda ()           (member (string #\d)
                                                 '("a" "b" "c" . "d"))))
(assert-error  (tn) (lambda ()           (member (string #\e)
                                                 '("a" "b" "c" . "d"))))
(tn "member improper lists: lists with non-constant key")
(assert-equal? (tn)
               '(("a") ("b") ("c") . "d")
               (member (list (string #\a)) '(("a") ("b") ("c") . "d")))
(assert-equal? (tn)
               '(("b") ("c") . "d")
               (member (list (string #\b)) '(("a") ("b") ("c") . "d")))
(assert-equal? (tn)
               '(("c") . "d")
               (member (list (string #\c)) '(("a") ("b") ("c") . "d")))
(assert-error  (tn) (lambda () (member (list (string #\d))
                                       '(("a") ("b") ("c") . "d"))))
(assert-error  (tn) (lambda () (member (list (string #\e))
                                       '(("a") ("b") ("c") . "d"))))


(total-report)
