/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

// ----------------------------------------------------------------------------
#ifdef _WIN32
#include <windows.h>
#include <winsock.h>
// ------------------------------------
// Windows functions
static void	win32_winsock_init (void)
{
	WSADATA WSAData;

	WSAStartup (MAKEWORD (2, 2), &WSAData);
}
static void	win32_winsock_finalize (void)
{
	WSACleanup ();
}
#endif	// _WIN32

// ----------------------------------------------------------------------------
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <stdlib.h>		// exit(), EXIT_SUCCESS, EXIT_FAILURE
#include <signal.h>
// uglib
#include <uglib.h>
#include <UgetCmd.h>

// GStreamer
#ifdef HAVE_GSTREAMER
#include <gst/gst.h>
gboolean	gst_inited	= FALSE;
#endif  // HAVE_GSTREAMER

// I18N
#ifndef GETTEXT_PACKAGE
#define GETTEXT_PACKAGE		"uget"
#endif
#ifndef PACKAGE_VERSION
#define PACKAGE_VERSION		"1.6.2"
#endif
#include <glib/gi18n.h>

// ----------------------------------------------------------------------------
// ug_get_data_dir() defined in uget-cmd.h
const gchar*	ug_get_data_dir (void)
{
#ifdef _WIN32
	static	gchar*	data_dir = NULL;

	if (data_dir == NULL) {
		gchar*		path;
		gunichar2*	path_wcs;
		HMODULE		hmod;

		hmod = GetModuleHandle (NULL);
		// UNICODE
		path_wcs = g_malloc (sizeof (wchar_t) * MAX_PATH);
		GetModuleFileNameW (hmod, path_wcs, MAX_PATH);
		path = g_utf16_to_utf8 (path_wcs, -1, NULL, NULL, NULL);
		g_free (path_wcs);
		data_dir = g_path_get_dirname (path);
		g_free (path);
	}
	return data_dir;
#elif defined (DATADIR)
	return DATADIR;
#else
	return "/usr/share";
#endif  // _WIN32
}

// ----------------------------------------------------------------------------
static	UgetCmd*	ugcmd;

static void term_signal_handler (int sig)
{
	uget_cmd_save (ugcmd);
	ug_ipc_finalize (&ugcmd->ipc);
#if defined (_WIN32)
	win32_winsock_finalize ();
#endif
	exit (0);
}

// ----------------------------------------------------------------------------
int main (int argc, char** argv)
{
	gchar*		string;
	gint		ipc_status;

	// I18N
#ifdef LOCALEDIR
	bindtextdomain (GETTEXT_PACKAGE, LOCALEDIR);
#else
	string = g_build_filename (ug_get_data_dir (), "locale", NULL);
	bindtextdomain (GETTEXT_PACKAGE, string);
	g_free (string);
#endif
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);

	string = ug_arg_find_version (argc, argv);
	if (string) {
#if defined (_WIN32) && defined (_WINDOWS)
		win32_console_init ();
#endif  // _WIN32 && _WINDOWS
		g_print ("Uget " PACKAGE_VERSION " for commandline" "\n");
		return EXIT_SUCCESS;
	}
	string = ug_arg_find_help (argc, argv);
	// initialize for MS-Windows
#if defined (_WIN32)
	win32_winsock_init ();
#endif  // _WIN32

	// GLib: GThread
	if (g_thread_supported () == FALSE)
		g_thread_init (NULL);

	// uglib: options
	ugcmd = g_malloc0 (sizeof (UgetCmd));
	ug_option_init (&ugcmd->option);
	if (string) {
		g_print ("Uget " PACKAGE_VERSION " for commandline" "\n");
		ug_option_help (&ugcmd->option, argv[0], string);
	}

	// GStreamer
#ifdef HAVE_GSTREAMER
	// initialize the GLib thread system using g_thread_init()
	// before any other GLib functions are called.
	gst_inited = gst_init_check (&argc, &argv, NULL);
#endif

	// IPC initialize & check exist Uget program
	ipc_status = ug_ipc_init_with_args (&ugcmd->ipc, argc, argv);
	if (ipc_status < 1) {
		if (ipc_status == -1)
			g_print ("ugtk-gtk: IPC failed.\n");
//		if (ipc_status == 0)
//			g_print ("ugtk-gtk: Server exists.\n");
		ug_ipc_finalize (&ugcmd->ipc);
		goto exit;
	}
	// register Uget classes
	uget_class_init ();
	uget_cmd_class_init ();

	// main program
	signal (SIGTERM, term_signal_handler);
	uget_cmd_run (ugcmd);

	// shutdown IPC and sleep 2 second to wait thread
	ug_ipc_finalize (&ugcmd->ipc);
	g_usleep (2 * 1000000);

exit:
	// finalize for MS-Windows
#if defined (_WIN32)
	win32_winsock_finalize ();
#endif

	return EXIT_SUCCESS;
}

