/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <uget.h>
#include <ug_download_dialog.h>
#include <ug_stdio.h>
#include <ug_item.h>
#include <ug_utils.h>
#include <ug_html.h>
#include <ug_url.h>

#include <glib/gi18n.h>

static gboolean	html_file_to_selector (gchar* filename_utf8, UgSelector* selector);
static gboolean	text_file_to_selector (gchar* filename_utf8, UgSelector* selector);

static void	on_download_dialog_response (GtkDialog *dialog, gint response, UgDownloadDialog* ddialog);
static void on_message_dialog_response  (GtkWidget *dialog, gint response, GtkWidget* parent);
static void on_file_chooser_response (GtkDialog *dialog, gint response, Uget* app);
static void on_file_chooser_show (GtkWindow* widget, gpointer user_data);


void	uget_on_import_html_file (GtkWidget* widget, Uget* app)
{
	GtkWidget*		dialog;
	GtkFileFilter*	filter;
	gchar*			title;

	title = g_strconcat ("Uget - ", _("Import URLs from HTML file"), NULL);
	dialog = gtk_file_chooser_dialog_new (title,
	                                      app->window,
	                                      GTK_FILE_CHOOSER_ACTION_OPEN,
	                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                      GTK_STOCK_OK,     GTK_RESPONSE_OK,
	                                      NULL);
	g_free (title);
	gtk_window_set_destroy_with_parent ((GtkWindow*) dialog, TRUE);
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (filter, "HTML file (*.htm, *.html)");
	gtk_file_filter_add_mime_type (filter, "text/html");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (dialog), filter);
	g_signal_connect (dialog, "response", G_CALLBACK (on_file_chooser_response), app);
	g_signal_connect (dialog, "show",     G_CALLBACK (on_file_chooser_show),     NULL);

	gtk_widget_show (dialog);
}

void	uget_on_import_text_file (GtkWidget* widget, Uget* app)
{
	GtkWidget*		dialog;
	GtkFileFilter*	filter;
	gchar*			title;

	title = g_strconcat ("Uget - ", _("Import URLs from text file"), NULL);
	dialog = gtk_file_chooser_dialog_new (title,
	                                      app->window,
	                                      GTK_FILE_CHOOSER_ACTION_OPEN,
	                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                      GTK_STOCK_OK,     GTK_RESPONSE_OK,
	                                      NULL);
	g_free (title);
	gtk_window_set_destroy_with_parent ((GtkWindow*) dialog, TRUE);
	filter = gtk_file_filter_new ();
	gtk_file_filter_set_name (filter, "Plain text file");
	gtk_file_filter_add_mime_type (filter, "text/plain");
	gtk_file_chooser_add_filter (GTK_FILE_CHOOSER (dialog), filter);
	g_signal_connect (dialog, "response", G_CALLBACK (on_file_chooser_response), app);
	g_signal_connect (dialog, "show",     G_CALLBACK (on_file_chooser_show), NULL);

	gtk_widget_show (dialog);
}

void	uget_import_list_file (Uget* app, gchar* filename_utf8, gboolean quiet)
{
	UgDownloadDialog*	ddialog;
	GtkWidget*			mdialog;	// GtkMessageDialog
	gboolean			import_ok;
	gchar*				title;

	title = g_strconcat ("Uget - ", _("New Download"), NULL);
	ddialog = ug_download_dialog_new_selector (title, app->window);
	g_free (title);

	import_ok = text_file_to_selector (filename_utf8, &ddialog->selector);

	if (import_ok == FALSE) {
		ug_download_dialog_destroy (ddialog);
		title = g_strconcat (_("Can't open file"), " - ", filename_utf8, NULL);
		mdialog = gtk_message_dialog_new ((GtkWindow*) app->window,
		                                  GTK_DIALOG_DESTROY_WITH_PARENT,
		                                  GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
		                                  title);
		g_free (title);
		title = g_strconcat ("Uget - ", _("Error"), NULL);
		gtk_window_set_title ((GtkWindow*) mdialog, title);
		g_free (title);
		g_signal_connect (mdialog, "response", G_CALLBACK (gtk_widget_destroy), NULL);
		gtk_widget_show (mdialog);
		return;
	}
	// show download dialog
	ug_download_dialog_set_category_list (ddialog, app->category_list);
	ug_download_dialog_select_category (ddialog, app->current->queuing);
	ug_download_form_set_folder_list (&ddialog->download, app->folder_list);
	ddialog->user_data = app;
	g_signal_connect (ddialog->self, "response", G_CALLBACK (on_download_dialog_response), ddialog);

	gtk_widget_show ((GtkWidget*) ddialog->self);
	if (quiet)
		gtk_dialog_response (ddialog->self, GTK_RESPONSE_OK);
}

void	filter_callback	(UgHtmlFilter* filter, const char* value, GtkListStore* list_store)
{
	UgItem*			item;
	GtkTreeIter		iter;

	item = g_malloc0 (sizeof (UgItem));
	gtk_list_store_append (list_store, &iter);
	gtk_list_store_set (list_store, &iter, 0, item, -1);
	item->value = g_strdup (value);
	item->mark = TRUE;
}

static gboolean  html_file_to_selector (gchar* filename_utf8, UgSelector* selector)
{
	UgHtmlContext*	context;
	UgHtmlFilter*	filter;

	context	= ug_html_context_new ();
	// <A HREF="Link">
	filter	= ug_html_filter_new ("A", "HREF");
	filter->add_value = (UgHtmlFilterFunc) filter_callback;
	filter->user_data = ug_selector_add_page (selector, _("Link <A>"));
	ug_html_context_add_filter (context, filter);
	ug_html_filter_unref (filter);
	// <IMG SRC="Link">
	filter	= ug_html_filter_new ("IMG", "SRC");
	filter->add_value = (UgHtmlFilterFunc) filter_callback;
	filter->user_data = ug_selector_add_page (selector, _("Image <IMG>"));
	ug_html_context_add_filter (context, filter);
	ug_html_filter_unref (filter);

	ug_html_context_parse_file (context, filename_utf8);
	if (context->base_href)
		gtk_entry_set_text (selector->href_entry, context->base_href);
	ug_html_context_free (context);

	return TRUE;
}

static gboolean  text_file_to_selector (gchar* filename_utf8, UgSelector* selector)
{
	UgUrlPart*		urlpart;
	GtkTreeIter		iter;
	GtkListStore*	item_store;
	UgItem*			item;
	gchar*			buffer;
	FILE*			file;
	guint			line_len;

	file = ug_fopen (filename_utf8, "r");
	if (file == NULL)
		return FALSE;
	ug_selector_hide_href (selector);
	item_store = ug_selector_add_page (selector, _("Import from text file"));
	buffer = g_malloc (4096);
	urlpart = g_malloc (sizeof (UgUrlPart));
	// clear iter for ug_item_store_realloc_next()
	memset (&iter, 0, sizeof (GtkTreeIter));
	while (feof (file) == 0) {
		if (fgets (buffer, 4096, file) == NULL)
			break;
		line_len = ug_str_line_len (buffer, -1, 0);
		ug_url_part (urlpart, buffer, line_len);
		if (urlpart->url_scheme_len == 0)
			continue;
		item = ug_item_store_realloc_next (item_store, &iter);
		item->value = g_strndup (buffer, line_len);
		item->mark = TRUE;
	}
	g_free (urlpart);
	g_free (buffer);
	fclose (file);
	return TRUE;
}

// signal ---------------------------------------------------------------------
static void	on_download_dialog_response (GtkDialog *dialog, gint response, UgDownloadDialog* ddialog)
{
	UgCategory*	category;
	UgDataset*	dataset;
	Uget*		app;

	if (ug_download_dialog_response (ddialog, response) == TRUE)
		return;

	if (response == GTK_RESPONSE_OK) {
		app = ddialog->user_data;
		app->folder_list = ug_download_form_get_folder_list (&ddialog->download, app->folder_list);
		category = ug_download_dialog_get_category (ddialog);
		if (category) {
			dataset  = ug_download_dialog_get_download_list (ddialog);
			ug_category_append (category, dataset);
		}
		gtk_widget_queue_draw ((GtkWidget*) app->category_tree_view);
		gtk_widget_queue_draw ((GtkWidget*) app->total_list_view);
	}

	ug_download_dialog_destroy (ddialog);
}

static void on_message_dialog_response (GtkWidget *dialog, gint response, GtkWidget* parent)
{
	gtk_widget_destroy (dialog);
	gtk_widget_set_sensitive (parent, TRUE);
}

static void on_file_chooser_response (GtkDialog *dialog, gint response, Uget* app)
{
	UgDownloadDialog*	ddialog;
	GtkWidget*			mdialog;	// GtkMessageDialog
	GtkFileChooser*		chooser;
	GtkFileFilter*		filter;
	const gchar*		filter_name;
	gchar*				file_name;
	gchar*				file_name_utf8;
	gboolean			import_ok;
	gchar*				title;

	if (response != GTK_RESPONSE_OK) {
		gtk_widget_destroy ((GtkWidget*) dialog);
		return;
	}
	// create download dialog
	ddialog = ug_download_dialog_new_selector (gtk_window_get_title ((GtkWindow*) dialog), app->window);
	// get selected filename and import it
	chooser = GTK_FILE_CHOOSER (dialog);
	filter = gtk_file_chooser_get_filter (chooser);
	filter_name = gtk_file_filter_get_name (filter);	// can't free
	file_name = gtk_file_chooser_get_filename (chooser);
	file_name_utf8 = g_filename_to_utf8 (file_name, -1, NULL, NULL, NULL);
	// Import by File Type : HTML or Plain text
	if (strncmp (filter_name, "HTML", 4) == 0)
		import_ok = html_file_to_selector (file_name_utf8, &ddialog->selector);
	else
		import_ok = text_file_to_selector (file_name_utf8, &ddialog->selector);
	g_free (file_name);
	g_free (file_name_utf8);

	if (import_ok == FALSE) {
		ug_download_dialog_destroy (ddialog);
		mdialog = gtk_message_dialog_new ((GtkWindow*) dialog,
		                                  GTK_DIALOG_DESTROY_WITH_PARENT,
		                                  GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
		                                  _("Can't open selected file. Permission denied?"));
		title = g_strconcat ("Uget - ", _("Error"), NULL);
		gtk_window_set_title ((GtkWindow*) mdialog, title);
		g_free (title);
		g_signal_connect (mdialog, "response", G_CALLBACK (on_message_dialog_response), dialog);
		gtk_widget_show (mdialog);
		gtk_widget_set_sensitive ((GtkWidget*) dialog, FALSE);
		return;
	}
	gtk_widget_destroy (GTK_WIDGET(dialog));
	// show download dialog
	ug_download_dialog_set_category_list (ddialog, app->category_list);
	ug_download_dialog_select_category (ddialog, app->current->queuing);
	ug_download_form_set_folder_list (&ddialog->download, app->folder_list);
	ddialog->user_data = app;
	g_signal_connect (ddialog->self, "response", G_CALLBACK (on_download_dialog_response), ddialog);
	gtk_widget_show ((GtkWidget*) ddialog->self);
}

static void on_file_chooser_show (GtkWindow* widget, gpointer user_data)
{
	// resize File Chooser Dialog
//	gtk_window_resize (widget, 600, 400);
}

