/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <uget.h>
#include <ug_data_app.h>
#include <ug_list_view.h>
#include <ug_stdio.h>

#include <glib/gi18n.h>

typedef struct	UgExportDialog_			UgExportDialog;

struct UgExportDialog_
{
	GtkDialog*		self;

	GtkTreeView*	category_view;
	GtkWidget*		file_chooser;
	GtkWidget*		entry;

	gpointer	user_data;
};

static void on_export_dialog_response  (GtkWidget* widget, gint response, UgExportDialog* dialog);
static void on_message_dialog_response (GtkWidget* widget, gint response, UgExportDialog* dialog);
static void on_choose_button_click (GtkWidget* button, UgExportDialog* dialog);
static void on_entry_changed (GtkEditable *editable, UgExportDialog* dialog);

static UgExportDialog*	ug_export_dialog_new (GtkWindow* parent, GtkTreeStore* category_tree);

void	uget_on_export_text_file (GtkWidget* widget, Uget* app)
{
	UgExportDialog*	dialog;
	GtkTreePath*	path;

	dialog = ug_export_dialog_new (app->window, app->category_tree);
	gtk_tree_view_get_cursor (app->category_tree_view, &path, NULL);
	if (path) {
		gtk_tree_view_set_cursor (dialog->category_view, path, NULL, FALSE);
		gtk_tree_path_free (path);
	}
	gtk_widget_show ((GtkWidget*) dialog->self);
}

static UgExportDialog*  ug_export_dialog_new (GtkWindow* parent, GtkTreeStore* category_tree)
{
	UgExportDialog*		dialog;
	GtkBox*				vbox;
	GtkBox*				hbox;
	GtkWidget*			widget;
	GtkWidget*			scrolled;
	GtkTreeView*		category_view;
	GtkTreeViewColumn*	column;
	gchar*				string;

	dialog = g_malloc0 (sizeof (UgExportDialog));
	string = g_strconcat ("Uget - ", _("Export URLs to file"), NULL);
	dialog->self = (GtkDialog*) gtk_dialog_new_with_buttons (string, parent,
	         (GTK_DIALOG_NO_SEPARATOR | GTK_DIALOG_DESTROY_WITH_PARENT),
	                  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                  GTK_STOCK_OK,     GTK_RESPONSE_OK,
	                  NULL);
	g_free (string);
	gtk_dialog_set_response_sensitive (dialog->self, GTK_RESPONSE_OK, FALSE);
	gtk_dialog_set_default_response (dialog->self, GTK_RESPONSE_OK);
//	gtk_window_resize ((GtkWindow*) dialog->self, 480, 300);
	gtk_window_set_resizable ((GtkWindow*) dialog->self, FALSE);
	g_signal_connect (dialog->self, "response", G_CALLBACK (on_export_dialog_response), dialog);

	vbox = (GtkBox*) dialog->self->vbox;
	gtk_widget_set_size_request ((GtkWidget*) vbox, 480, 290);
	gtk_box_pack_start (vbox, gtk_label_new (_("Select a category or it's child to export")), FALSE, FALSE, 3);
	gtk_box_pack_start (vbox, gtk_hseparator_new (), FALSE, TRUE, 3);

	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, TRUE, TRUE, 1);

	// left side --------------------------------
	category_view = ug_category_view_new ();
	gtk_tree_view_set_model (category_view, (GtkTreeModel*) category_tree);
	gtk_tree_view_expand_all (category_view);
	column = gtk_tree_view_get_column (category_view, UG_CATEGORY_COLUMN_QUANTITY);
	gtk_tree_view_column_set_visible (column, FALSE);
	gtk_widget_set_size_request ((GtkWidget*) category_view, 165, 100);
	// scrolled window
	scrolled = gtk_scrolled_window_new (NULL, NULL);
	gtk_widget_show (scrolled);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scrolled), GTK_SHADOW_IN);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
	                                GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_container_add (GTK_CONTAINER (scrolled), GTK_WIDGET(category_view));
	gtk_box_pack_start (hbox, scrolled, FALSE, TRUE, 1);
	dialog->category_view = category_view;

	// right side -------------------------------
	vbox = (GtkBox*) gtk_vbox_new (FALSE, 2);
	gtk_box_pack_start (hbox, (GtkWidget*) vbox, TRUE, TRUE, 1);
	hbox = (GtkBox*) gtk_hbox_new (FALSE, 2);
	gtk_box_pack_start (vbox, (GtkWidget*) hbox, FALSE, FALSE, 2);
	string = g_strconcat (_("Export to"), ":", NULL);
	widget = gtk_label_new (string);
	g_free (string);
	gtk_box_pack_start (hbox, widget, FALSE, FALSE, 2);
	widget = gtk_button_new_with_mnemonic (_("Choose _file..."));;
	g_signal_connect (widget, "clicked", G_CALLBACK (on_choose_button_click), dialog);
	gtk_box_pack_start (hbox, widget, FALSE, FALSE, 2);

	widget = gtk_entry_new ();
	gtk_box_pack_start (vbox, widget, FALSE, TRUE, 2);
	g_signal_connect (GTK_EDITABLE (widget), "changed", G_CALLBACK (on_entry_changed), dialog);
	dialog->entry = widget;

	widget = gtk_hseparator_new ();
	gtk_box_pack_start (vbox, widget, FALSE, TRUE, 2);
	widget = gtk_label_new (_("This will export URLs to text file."));
	gtk_box_pack_start (vbox, widget, FALSE, TRUE, 2);

	gtk_widget_show_all (dialog->self->vbox);
	return dialog;
}

// signal ---------------------------------------------------------------------
static void on_export_dialog_response (GtkWidget* widget, gint response, UgExportDialog* dialog)
{
	UgCategory*		category;
	UgDataset*		dataset;
	GtkWidget*		mdialog;
	GtkTreeModel*	model;
	GtkTreePath*	path;
	GtkTreeIter		iter;
	const gchar*	filename;
	FILE*			file;
	gboolean		valid;
	gchar*			title;

	if (response == GTK_RESPONSE_OK) {
		gtk_tree_view_get_cursor (dialog->category_view, &path, NULL);
		if (path == NULL)
			goto exit;
		model = gtk_tree_view_get_model (dialog->category_view);
		gtk_tree_model_get_iter (model, &iter, path);
		gtk_tree_path_free (path);
		gtk_tree_model_get (model, &iter, 0, &category, -1);
		filename = gtk_entry_get_text (GTK_ENTRY (dialog->entry));
		file = ug_fopen (filename, "w");
		if (file == NULL) {
			mdialog = gtk_message_dialog_new ((GtkWindow*) dialog->self,
	                                 GTK_DIALOG_DESTROY_WITH_PARENT,
	                                 GTK_MESSAGE_ERROR, GTK_BUTTONS_OK,
	                                  _("Can't open selected file. Permission denied?"));
			title = g_strconcat ("Uget - ", _("Error"), NULL);
			gtk_window_set_title ((GtkWindow*) mdialog, title);
			g_free (title);
			g_signal_connect (mdialog, "response", G_CALLBACK (on_message_dialog_response), dialog);
			gtk_widget_show (mdialog);
			gtk_widget_set_sensitive ((GtkWidget*) dialog->self, FALSE);
			return;
		}
		model = GTK_TREE_MODEL (category->download_store);
		valid = gtk_tree_model_get_iter_first (model, &iter);
		while (valid) {
			gtk_tree_model_get (model, &iter, 0, &dataset, -1);
			valid = gtk_tree_model_iter_next (model, &iter);
			if (UG_DATASET_COMMON (dataset)->url == NULL)
				continue;
			fputs (UG_DATASET_COMMON (dataset)->url, file);
			fputc ('\n', file);
		}
		fclose (file);
	}

exit:
	gtk_widget_destroy (widget);
	g_free (dialog);
}

static void on_message_dialog_response (GtkWidget* widget, gint response, UgExportDialog* dialog)
{
	gtk_widget_set_sensitive ((GtkWidget*) dialog->self, TRUE);
	gtk_widget_destroy (widget);
}

static void on_file_chooser_response (GtkDialog* chooser, gint response, UgExportDialog* dialog)
{
	gchar*	file;
	gchar*	path;

	if (response == GTK_RESPONSE_OK ) {
		file = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
		path = g_filename_to_utf8 (file, -1, NULL, NULL, NULL);
		gtk_entry_set_text (GTK_ENTRY (dialog->entry), path);
		g_free (path);
		g_free (file);
	}
	gtk_widget_destroy (GTK_WIDGET (chooser));
	dialog->file_chooser = NULL;
}

static void on_file_chooser_show (GtkWindow* window, gpointer user_data)
{
	// resize File Chooser Dialog
//	gtk_window_resize (window, 600, 400);
}

static void on_choose_button_click (GtkWidget* button, UgExportDialog* dialog)
{
	GtkWidget*		chooser;
	const gchar*	filename;

	if (dialog->file_chooser)
		return;

	chooser = gtk_file_chooser_dialog_new (_("Export to"),
	                                      (GtkWindow*) dialog->self,
	                                      GTK_FILE_CHOOSER_ACTION_SAVE,
	                                      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
	                                      GTK_STOCK_OK,     GTK_RESPONSE_OK,
	                                      NULL);
//	gtk_window_set_transient_for ((GtkWindow*) chooser, (GtkWindow*) dialog->self);
	gtk_window_set_destroy_with_parent ((GtkWindow*) chooser, TRUE);
	dialog->file_chooser = chooser;

	filename = gtk_entry_get_text ((GtkEntry*) dialog->entry);
	gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (chooser), filename);

	g_signal_connect (chooser, "response", G_CALLBACK (on_file_chooser_response), dialog);
	g_signal_connect (chooser, "show",     G_CALLBACK (on_file_chooser_show),     NULL);

	if (gtk_window_get_modal ((GtkWindow*) dialog->self))
		gtk_dialog_run ((GtkDialog*) chooser);
	else {
		gtk_window_set_modal ((GtkWindow*) chooser, FALSE);
		gtk_widget_show (chooser);
	}
}

static void on_entry_changed (GtkEditable *editable, UgExportDialog* dialog)
{
	const gchar*	text;

	text = gtk_entry_get_text ((GtkEntry*) dialog->entry);
	gtk_dialog_set_response_sensitive (dialog->self, GTK_RESPONSE_OK, (*text) ? TRUE : FALSE);
}

