/***************************************************************************
 *   Copyright (C) 2008 by Alexey Balakin                                  *
 *   mathgl.abalakin@gmail.com                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <QTime>
#include <QMenu>
#include <QPrinter>
#include <QTimer>
#include <QScrollArea>
#include <QPainter>
#include <QPrintDialog>
#include <QCloseEvent>
#include <QToolBar>
#include <QSpinBox>
#include <QMainWindow>
#include <QMenuBar>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QTableWidget>
#include <QPushButton>
#include <QInputDialog>
#include <QLineEdit>
#include <QMessageBox>
#include <QMdiArea>
#include <mgl/mgl_parse.h>
#include "animparam.h"
#include "graphwindow.h"
#include "qmglcanvas.h"
#include "infodialog.h"
#include "tablewindow.h"
extern bool mglAutoSave;
extern mglParse parser;
int animDelay=500;
//-----------------------------------------------------------------------------
//
//		GraphWindow
//
//-----------------------------------------------------------------------------
GraphWindow::GraphWindow(QWidget *wp) : QMdiSubWindow(wp)
{
	gifOn = jpgOn = false;	setWindowTitle("Canvas");
	animDialog = new AnimParam(this);	animPos = -1;
	printer = new QPrinter;
	timer = new QTimer(this);
	connect(timer, SIGNAL(timeout()), this, SLOT(nextSlide()));
	connect(animDialog, SIGNAL(putText(const QString &)), this, SLOT(animText(const QString &)));

	sv = new QScrollArea(this);	setWidget(sv);
	mgl = new QMGLCanvas(this);	sv->setWidget(mgl);
}
//-----------------------------------------------------------------------------
GraphWindow::~GraphWindow()	{	delete printer;	}
//-----------------------------------------------------------------------------
void GraphWindow::animText(const QString &txt)	{	animPutText(txt);	}
//-----------------------------------------------------------------------------
void GraphWindow::printPlot()
{
	QPrintDialog printDlg(printer, this);
	if (printDlg.exec() == QDialog::Accepted)
	{
		statusBarMessage(tr("Printing..."));
		QPainter p;
		if(!p.begin(printer))	return;	// paint on printer
		QRect r = p.viewport();
		int w = r.width(), h = r.height(), h1;
		h1 = int(w/mgl->getRatio());
		if(h1<h)	h = h1;	else	w = int(h*mgl->getRatio());
		mglGraphZB gr(w, h);
		if(w*h > 240000)	gr.BaseLineWidth = sqrt(w*h/2.4e5);
		mgl->execute(&gr);

		uchar *grBuf=0;
		QPixmap pic;
		convertFromGraph(pic, &gr, &grBuf);
		p.drawPixmap(0,0,pic);
		delete []grBuf;
		statusBarMessage(tr("Printing completed"));
	}
	else
		statusBarMessage(tr("Printing aborted"));
}
//-----------------------------------------------------------------------------
void GraphWindow::closeEvent(QCloseEvent* ce)	{	ce->ignore();	}
//-----------------------------------------------------------------------------
void GraphWindow::pressF5()
{
	if(mglAutoSave)	save();
	showMaximized();	raise();
	QTime t;	t.start();
	mgl->execute();
	statusBarMessage(QString(tr("Drawing time %1 ms")).arg(t.elapsed()*1e-3));
}
//-----------------------------------------------------------------------------
void GraphWindow::pressF9()
{
	int l=animParam.length(), i;
	wchar_t *str = new wchar_t[l+2];
	animPos = 0;
	QString cur = animParam.section('\n',animPos,animPos);
	for(i=0;i<l;i++)	str[i] = (cur[i]).unicode();
	str[i] = 0;
	parser.AddParam(0,str);
	delete []str;

	mgl->reload();
}
//-----------------------------------------------------------------------------
void GraphWindow::animStart(bool st)
{
	if(animParam.isEmpty())
	{
		if(animDialog->exec())
		{
			animParam = animDialog->getResult();
			gifOn = animDialog->gifOn;
			jpgOn = animDialog->jpgOn;
		}
		else	return;
	}
	if(st)
	{
		timer->start(animDelay);
		if(gifOn)	mgl->startGIF(animDelay);
		showMaximized();	raise();
	}
	else
	{
		timer->stop();
		if(gifOn)	mgl->closeGIF();
	}
}
//-----------------------------------------------------------------------------
void GraphWindow::nextSlide()
{
	if(animParam.isEmpty())
	{
		if(animDialog->exec())
		{
			animParam = animDialog->getResult();
			gifOn = animDialog->gifOn;
			jpgOn = animDialog->jpgOn;
		}
		else	return;
	}
//	showMaximized();	raise();
	int l=animParam.length(), n=animParam.count('\n') + (animParam[l-1]=='\n' ? 0:1), i;
	wchar_t *str = new wchar_t[l+2];
	animPos = (animPos+1)%n;
	QString cur = animParam.section('\n',animPos,animPos);
	for(i=0;i<l;i++)	str[i] = (cur[i]).unicode();
	str[i] = 0;
	parser.AddParam(0,str);
	delete []str;
	if(mgl->graph->GetNumFrame() >= n)
		mgl->execute();
	else
	{
		mgl->graph->NewFrame();
		mgl->execute();
		mgl->graph->EndFrame();
		if(jpgOn)	mgl->graph->WriteFrame();
		QString s;	s.sprintf("%d - %d of %d",mgl->graph->GetNumFrame(),animPos,n);
		statusBarMessage(QString(tr("Frame %1 of %2")).arg(animPos).arg(n));
	}
}
//-----------------------------------------------------------------------------
void GraphWindow::prevSlide()
{
	if(animParam.isEmpty())
	{
		if(animDialog->exec())
		{
			animParam = animDialog->getResult();
			gifOn = animDialog->gifOn;
			jpgOn = animDialog->jpgOn;
		}
		else	return;
	}
//	showMaximized();	raise();
	int l=animParam.length(), n=animParam.count('\n') + (animParam[l-1]=='\n' ? 0:1), i;
	wchar_t *str = new wchar_t[l+2];
	animPos = (animPos-1+n)%n;
	QString cur = animParam.section('\n',animPos,animPos);
	for(i=0;i<l;i++)	str[i] = (cur[i]).unicode();
	str[i] = 0;
	parser.AddParam(0,str);
	delete []str;
	mgl->execute();
}
//-----------------------------------------------------------------------------
void GraphWindow::animSetup()
{
	if(animDialog->exec())
	{
		animParam = animDialog->getResult();
		gifOn = animDialog->gifOn;
		jpgOn = animDialog->jpgOn;
		animPos = -1;
	}
}
//-----------------------------------------------------------------------------
void GraphWindow::adjust()
{
	mgl->setSize(sv->width()-5, sv->height()-5);
	showMaximized();	raise();
}
//-----------------------------------------------------------------------------
void GraphWindow::setMGLFont(const QString &path)	{	mgl->setMGLFont(path);	}
//-----------------------------------------------------------------------------
void GraphWindow::animParseText(const QString &txt)
{
	int i, n = txt.count('\n')+1;
	QString s, all;
	for(i=0;i<n;i++)
	{
		s = txt.section('\n',i,i);
		if(s[0]=='#' && s[1]=='#' && s[2]=='a' && (s[3]==' ' || s[3]=='\t'))
			all = all + s.mid(4) + "\n";
	}
	if(!all.isEmpty())
	{
		animDialog->setResult(all);
		animParam = all;
	}
}
//-----------------------------------------------------------------------------
#include "xpm/wire.xpm"
void GraphWindow::makeMenu(QMainWindow *wnd)
{
	QAction *a;
	QMenu *o, *oo;
	QToolBar *bb;

	menu = new QMenu(this);
	popup = new QMenu(this);	mgl->setPopup(popup);
	// graphics menu
	{
	bb = new QToolBar(tr("Graphics"),this);
	wnd->addToolBar(Qt::TopToolBarArea, bb);
	o = wnd->menuBar()->addMenu(tr("&Graphics"));
	a = new QAction(QPixmap(":/xpm/alpha.png"), tr("&Alpha"), this);
	a->setShortcut(Qt::CTRL+Qt::Key_T);	a->setCheckable(true);
	connect(a, SIGNAL(toggled(bool)), mgl, SLOT(setAlpha(bool)));
	connect(mgl, SIGNAL(alphaChanged(bool)), a, SLOT(setOn(bool)));
	a->setToolTip(tr("Switch on/off transparency for the graphics (Ctrl+T)."));
	o->addAction(a);		bb->addAction(a);
	a = new QAction(QPixmap(":/xpm/weather-clear.png"), tr("&Light"), this);
	a->setShortcut(Qt::CTRL+Qt::Key_L);	a->setCheckable(true);
	connect(a, SIGNAL(toggled(bool)), mgl, SLOT(setLight(bool)));
	connect(mgl, SIGNAL(lightChanged(bool)), a, SLOT(setOn(bool)));
	a->setToolTip(tr("Switch on/off lightning for the graphics (Ctrl+L)."));
	o->addAction(a);		bb->addAction(a);
	a = new QAction(QPixmap(wire_xpm), tr("&Grid"), this);
	a->setShortcut(Qt::CTRL+Qt::Key_G);	a->setCheckable(true);
	connect(a, SIGNAL(toggled(bool)), mgl, SLOT(setGrid(bool)));
	a->setToolTip(tr("Switch on/off grid drawing for absolute coordinates (Ctrl+G)."));
	o->addAction(a);		bb->addAction(a);

	a = new QAction(QPixmap(":/xpm/object-rotate-right.png"), tr("&Rotate by mouse"), this);
	a->setCheckable(true);
	connect(a, SIGNAL(toggled(bool)), mgl, SLOT(setRotate(bool)));
	connect(mgl, SIGNAL(rotateChanged(bool)), a, SLOT(setOn(bool)));
	a->setToolTip(tr("Switch on/off mouse handling of the graphics\n(rotation, shifting, zooming and perspective)."));
	bb->addAction(a);
	a = new QAction(QPixmap(":/xpm/zoom-fit-best.png"), tr("&Zoom by mouse"), this);
	a->setCheckable(true);
	connect(a, SIGNAL(toggled(bool)), mgl, SLOT(setZoom(bool)));
	connect(mgl, SIGNAL(zoomChanged(bool)), a, SLOT(setOn(bool)));
	a->setToolTip(tr("Switch on/off mouse zoom of selected region."));
	bb->addAction(a);
	o->addSeparator();
	a = new QAction(QPixmap(":/xpm/zoom-original.png"), tr("Res&tore"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(restore()));
	a->setToolTip(tr("Restore default graphics rotation, zoom and perspective (Ctrl+Space)."));
	a->setShortcut(Qt::CTRL+Qt::Key_Space);
	o->addAction(a);	bb->addAction(a);	popup->addAction(a);
	bb->addSeparator();
	a = new QAction(QPixmap(":/xpm/view-refresh.png"), tr("Re&draw"), this);
	connect(a, SIGNAL(activated()), this, SLOT(pressF5()));
	a->setToolTip(tr("Execute script and redraw graphics (F5)."));
	a->setShortcut(Qt::Key_F5);
	o->addAction(a);	bb->addAction(a);	popup->addAction(a);
	a = new QAction(tr("&Adjust size"), this);
	connect(a, SIGNAL(activated()), this, SLOT(adjust()));
	a->setToolTip(tr("Change canvas size to fill whole region (F6)."));
	a->setShortcut(Qt::Key_F6);		o->addAction(a);
	a = new QAction(tr("Re&load"), this);
	connect(a, SIGNAL(activated()), this, SLOT(pressF9()));
	a->setToolTip(tr("Restore status for 'once' command and reload data (F9)."));
	a->setShortcut(Qt::Key_F9);	o->addAction(a);	popup->addAction(a);
	a = new QAction(QPixmap(":/xpm/process-stop.png"), tr("&Stop"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(stop()));
	a->setToolTip(tr("Stop script execution (F7)."));
	a->setShortcut(Qt::Key_F7);	o->addAction(a);		bb->addAction(a);
	a = new QAction(QPixmap(":/xpm/edit-copy.png"), tr("&Copy plot"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(copy()));
	a->setToolTip(tr("Copy graphics to clipboard (Ctrl+Shift+C)."));
	a->setShortcut(Qt::CTRL+Qt::SHIFT+Qt::Key_C);
	o->addAction(a);		bb->addAction(a);	popup->addAction(a);
	a = new QAction(QPixmap(":/xpm/edit-copy.png"), tr("&Copy click coor."), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(copyClickCoor()));
	a->setToolTip(tr("Copy coordinates of last mouse click to clipboard."));
	o->addAction(a);		bb->addAction(a);	popup->addAction(a);
	bb->addSeparator();
	tet = new QSpinBox(this);	tet->setWrapping(true);
	bb->addWidget(tet);	tet->setRange(-180, 180);	tet->setSingleStep(10);
	connect(tet, SIGNAL(valueChanged(int)), mgl, SLOT(setTet(int)));
	connect(mgl, SIGNAL(tetChanged(int)), tet, SLOT(setValue(int)));
	tet->setToolTip(tr("Set value of \\theta angle."));
	bb->addSeparator();
	phi = new QSpinBox(this);	phi->setWrapping(true);
	bb->addWidget(phi);	phi->setRange(-180, 180);	phi->setSingleStep(10);
	connect(phi, SIGNAL(valueChanged(int)), mgl, SLOT(setPhi(int)));
	connect(mgl, SIGNAL(phiChanged(int)), phi, SLOT(setValue(int)));
	phi->setToolTip(tr("Set value of \\phi angle."));
//	bb->addSeparator();

	oo = new QMenu(tr("&Export as ..."),this);
	oo->addAction(tr("PNG"), mgl, SLOT(exportPNG()),Qt::META+Qt::Key_P);
	oo->addAction(tr("solid PNG"), mgl, SLOT(exportPNGs()),Qt::META+Qt::Key_F);
	oo->addAction(tr("JPEG"), mgl, SLOT(exportJPG()),Qt::META+Qt::Key_J);
	oo->addAction(tr("GIF"), mgl, SLOT(exportGIF()),Qt::META+Qt::Key_G);
	oo->addAction(tr("bitmap EPS"), mgl, SLOT(exportBPS()));
	oo->addAction(tr("vector EPS"), mgl, SLOT(exportEPS()),Qt::META+Qt::Key_E);
	oo->addAction(tr("SVG"), mgl, SLOT(exportSVG()),Qt::META+Qt::Key_S);
	oo->addAction(tr("C++"), mgl, SLOT(exportCPP()));
	oo->addAction(tr("IDTF"), mgl, SLOT(exportIDTF()));
	o->addMenu(oo);
	popup->addMenu(oo);

	}
	// zooming menu
	{
	oo = o->addMenu(tr("Zoom/move"));
	bb = new QToolBar(tr("Zoom graphics"),this);
	wnd->addToolBar(Qt::RightToolBarArea, bb);
	a = new QAction(QPixmap(":/xpm/go-previous.png"), tr("Move &left"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(shiftLeft()));
	a->setToolTip(tr("Move graphics left by 1/3 of its width."));
	bb->addAction(a);		oo->addAction(a);
	a = new QAction(QPixmap(":/xpm/go-up.png"), tr("Move &up"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(shiftUp()));
	a->setToolTip(tr("Move graphics up by 1/3 of its height."));
	bb->addAction(a);		oo->addAction(a);
	a = new QAction(QPixmap(":/xpm/zoom-in.png"), tr("Zoom &in"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(zoomIn()));
	a->setToolTip(tr("Zoom in graphics."));
	bb->addAction(a);		oo->addAction(a);
	a = new QAction(QPixmap(":/xpm/zoom-out.png"), tr("Zoom &out"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(zoomOut()));
	a->setToolTip(tr("Zoom out graphics."));
	bb->addAction(a);		oo->addAction(a);
	a = new QAction(QPixmap(":/xpm/go-down.png"), tr("Move &down"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(shiftDown()));
	a->setToolTip(tr("Move graphics up down 1/3 of its height."));
	bb->addAction(a);		oo->addAction(a);
	a = new QAction(QPixmap(":/xpm/go-next.png"), tr("Move &right"), this);
	connect(a, SIGNAL(activated()), mgl, SLOT(shiftRight()));
	a->setToolTip(tr("Move graphics right by 1/3 of its width."));
	bb->addAction(a);		oo->addAction(a);
	}
	// animation menu
	{
	oo = wnd->menuBar()->addMenu(tr("&Animation"));
	bb = new QToolBar(tr("Animation"),this);
	wnd->addToolBar(Qt::RightToolBarArea, bb);
	a = new QAction(QPixmap(":/xpm/media-seek-forward.png"), tr("&Next slide"), this);
	connect(a, SIGNAL(activated()), this, SLOT(nextSlide()));
	a->setToolTip(tr("Show next slide (Meta+Right)."));
	a->setShortcut(Qt::META+Qt::Key_Right);	oo->addAction(a);	bb->addAction(a);
	a = new QAction(QPixmap(":/xpm/media-seek-backward.png"), tr("&Prev slide"), this);
	connect(a, SIGNAL(activated()), this, SLOT(prevSlide()));
	a->setToolTip(tr("Show previous slide (Meta+Left)."));
	a->setShortcut(Qt::META+Qt::Key_Left);	oo->addAction(a);	bb->addAction(a);
	a = new QAction(QPixmap(":/xpm/film-b.png"), tr("&Slideshow"), this);
	a->setCheckable(true);
	connect(a, SIGNAL(toggled(bool)), this, SLOT(animStart(bool)));
	a->setToolTip(tr("Run slideshow (Ctrl+F5). If no parameter specified\nthen the dialog with slideshow options will appear."));
	a->setShortcut(Qt::CTRL+Qt::Key_F5);	oo->addAction(a);		bb->addAction(a);
	oo->addAction(tr("Se&tup show"), this, SLOT(animSetup()), Qt::CTRL+Qt::Key_W);
	}
}
//-----------------------------------------------------------------------------
//
//		MemoryWindow
//
//-----------------------------------------------------------------------------
#include "xpm/table.xpm"
#include "xpm/preview.xpm"
MemoryWindow::MemoryWindow(QWidget *wp) : QMdiSubWindow(wp)
{
	QWidget *p;
	QHBoxLayout *h;
	QVBoxLayout *v;
	QPushButton *b;

	infoDlg = new InfoDialog(this);
	infoDlg->setModal(true);	infoDlg->allowRefresh=false;

	p = new QWidget(this);
	v = new QVBoxLayout(p);	h = new QHBoxLayout();	v->addLayout(h);
	b = new QPushButton(QPixmap(":/xpm/document-new.png"), tr("New"), p);	h->addWidget(b);
	connect(b, SIGNAL(clicked()), this, SLOT(newTable()));
	b = new QPushButton(QPixmap(table_xpm), tr("Edit"), p);	h->addWidget(b);
	connect(b, SIGNAL(clicked()), this, SLOT(editData()));
	b = new QPushButton(QPixmap(":/xpm/edit-delete.png"), tr("Delete"), p);	h->addWidget(b);
	connect(b, SIGNAL(clicked()), this, SLOT(delData()));
	b = new QPushButton(QPixmap(preview_xpm), tr("Info"), p);	h->addWidget(b);
	connect(b, SIGNAL(clicked()), this, SLOT(infoData()));
	b = new QPushButton(QPixmap(":/xpm/view-refresh.png"), tr("Update"), p);	h->addWidget(b);
	connect(b, SIGNAL(clicked()), this, SLOT(refresh()));
	h->addStretch(1);
	b = new QPushButton(QPixmap(":/xpm/edit-delete.png"), tr("Delete All"), p);	h->addWidget(b);
	connect(b, SIGNAL(clicked()), this, SLOT(delAllData()));

	colSort = 0;
	tab = new QTableWidget(p);	v->addWidget(tab);	tab->setColumnCount(3);
	QStringList sl;	sl<<tr("Name")<<tr("Sizes")<<tr("Memory");
	tab->setHorizontalHeaderLabels(sl);
	connect(tab, SIGNAL(cellClicked(int,int)), this, SLOT(tableClicked(int,int)));
	connect(tab, SIGNAL(cellDoubleClicked(int,int)), this, SLOT(tableDClicked(int,int)));

	setWidget(p);	setWindowTitle(tr("Memory"));
}
//-----------------------------------------------------------------------------
MemoryWindow::~MemoryWindow()	{}
//-----------------------------------------------------------------------------
void MemoryWindow::closeEvent(QCloseEvent* ce)	{	ce->ignore();	}
//-----------------------------------------------------------------------------
void MemoryWindow::tableClicked(int, int col)
{	colSort = col;	tab->sortItems(col);	}
//-----------------------------------------------------------------------------
void MemoryWindow::tableDClicked(int row, int)	{	editData(row);	}
//-----------------------------------------------------------------------------
TableWindow *MemoryWindow::newWindow(mglVar *v)
{
	TableWindow *t = new TableWindow(infoDlg);
	if(v)	t->setVar(v);
	mdiArea()->addSubWindow(t);
	t->setWindowState(Qt::WindowMaximized);
	return t;
}
//-----------------------------------------------------------------------------
void MemoryWindow::newTable()
{
	bool ok;
	QString name = QInputDialog::getText(this, tr("UDAV - New variable"),
				tr("Enter name for new variable"), QLineEdit::Normal, "", &ok);
	if(!ok || name.isEmpty())	return;
	mglVar *v = parser.AddVar(name.toAscii());
	TableWindow *t;
	if(v->o)	t = (TableWindow *)v->o;
	else		t = newWindow(v);
	t->showMaximized();	t->activateWindow();
	refresh();
}
//-----------------------------------------------------------------------------
void MemoryWindow::editData(int n)
{
	if(tab->rowCount()<1)	return;
	if(n<0)	n = tab->currentRow();
	if(n<0)	n = 0;
	mglVar *v = parser.FindVar(tab->item(n,0)->text().toAscii());
	if(!v)	return;
	TableWindow *t;
	if(v->o)	t = (TableWindow *)v->o;
	else		t = newWindow(v);
	t->showMaximized();	t->activateWindow();
}
//-----------------------------------------------------------------------------
void MemoryWindow::delData()
{
	if(tab->rowCount()<1)	return;
	int	n = tab->currentRow();
	if(n<0)	n = 0;
	mglVar *v = parser.FindVar(tab->item(n,0)->text().toAscii());
	if(!v)	return;
	if(v->o)	((TableWindow *)v->o)->close();
	parser.DeleteVar(v);
	refresh();
}
//-----------------------------------------------------------------------------
void MemoryWindow::delAllData()
{
	if(QMessageBox::information(this, tr("UDAV - delete all data"),
			tr("Do you want to delete all data?"), QMessageBox::No,
			QMessageBox::Yes)!=QMessageBox::Yes)	return;
	while(parser.DataList)	parser.DeleteVar(parser.DataList);
	refresh();
}
//-----------------------------------------------------------------------------
void MemoryWindow::infoData()
{
	if(tab->rowCount()<1)	return;
	int	n = tab->currentRow();
	if(n<0)	n = 0;
	mglVar *v = parser.FindVar(tab->item(n,0)->text().toAscii());
	if(!v)	return;
	infoDlg->setVar(v);
	QString s = QString::fromWCharArray(v->s);
	infoDlg->setWindowTitle(s + tr(" - UDAV preview"));
	infoDlg->refresh();
	infoDlg->show();
}
//-----------------------------------------------------------------------------
void MemoryWindow::refresh()
{
	mglVar *v = parser.DataList;
	int n = 0;
	while(v)	{	v = v->next;	n++;	}
	tab->setRowCount(n);
	v = parser.DataList;	n = 0;
	QString s;
	QTableWidgetItem *it;
	Qt::ItemFlags flags=Qt::ItemIsSelectable|Qt::ItemIsEnabled;
	while(v)
	{
		register unsigned i,m=wcslen(v->s);
		QChar *ss = new QChar[m+1];
		for(i=0;i<m;i++)	ss[i] = v->s[i];
		s = QString(ss, m);	delete []ss;
		it = new QTableWidgetItem(s);
		tab->setItem(n,0,it);	it->setFlags(flags);
		s.sprintf("%ld * %ld * %ld", v->d.nx, v->d.ny, v->d.nz);
		it = new QTableWidgetItem(s);
		tab->setItem(n,1,it);	it->setFlags(flags);
		it->setTextAlignment(Qt::AlignHCenter|Qt::AlignVCenter);
		s.sprintf("%12ld", v->d.nx*v->d.ny*v->d.nz*sizeof(float));
		it = new QTableWidgetItem(s);
		tab->setItem(n,2,it);	it->setFlags(flags);
		it->setTextAlignment(Qt::AlignRight|Qt::AlignVCenter);
		if(v->o)	((TableWindow *)v->o)->refresh();
		v = v->next;	n++;
	}
	tab->sortItems(colSort);
}
//-----------------------------------------------------------------------------
