# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Volumes management."""

import cgi
import json
import urllib2

from oauth import oauth

from ubuntuone.filestorageapi.logger import setup_logging


HMAC_SHA1 = oauth.OAuthSignatureMethod_HMAC_SHA1()
FILES_API_BASE = u'https://edge.one.ubuntu.com/api/file_storage/v1'

USER_INFO = u'/~/'
LIST_VOLUMES = u'/volumes/'

ROOT_TYPE = u'root'
UDF_TYPE = u'udf'

logger = setup_logging('volumes')


class Request(object):
    """A base request."""

    def __init__(self, credentials):
        self._credentials = credentials

    def call(self, http_method, path):
        """Perform a request as per 'http_method'.

        Return the parsed JSON result. 'path' should not be quoted.

        """
        url = (FILES_API_BASE + path).encode('utf-8')
        logger.debug('Request.call: unquoted url is %r', url)

        scheme, netloc, path, params, query, fragment = \
            urllib2.urlparse.urlparse(url)
        path = urllib2.quote(path)

        data = (scheme, netloc, path, params, query, fragment)
        url = urllib2.urlparse.urlunparse(data)
        logger.info('Request.call: quoted url is %r', url)

        consumer = oauth.OAuthConsumer(self._credentials['consumer_key'],
                                       self._credentials['consumer_secret'])
        token = oauth.OAuthToken(self._credentials['token'],
                                 self._credentials['token_secret'])
        parameters = dict(cgi.parse_qsl(query))
        logger.debug('Request.call: parameters for OAuthRequest are %r',
                     parameters)
        get_request = oauth.OAuthRequest.from_consumer_and_token
        oauth_req = get_request(oauth_consumer=consumer, token=token,
                                http_method=http_method, http_url=url,
                                parameters=parameters)
        oauth_req.sign_request(HMAC_SHA1, consumer, token)
        request = urllib2.Request(url, headers=oauth_req.to_header())
        response = urllib2.urlopen(request)
        logger.info('Request.call: response.code is %r', response.code)

        result = None
        try:
            result = json.load(response)
        except ValueError:
            logger.exception('Could not parse JSON.')

        logger.debug('Request.call: result is %r', result)
        return result

    def get(self, path):
        """Perform a GET request."""
        return self.call(http_method='GET', path=path)

    def put(self, path):
        """Perform a PUT request."""
        return self.call(http_method='PUT', path=path)


class VolumeError(Exception):
    """General error when managing a Volume."""


class InvalidVolumeDictError(VolumeError):
    """The given dictionary is not valid to build a new Volume instance."""


class Volume(object):
    """An Ubuntu One volume abstraction."""

    def __init__(self):
        self.type = None
        self.generation = None
        self.key = None
        self.path = None
        self.when_created = None

    @classmethod
    def valid_type(cls, volume_type):
        """Return True if 'volume_type' is a valid volume type."""
        return volume_type in (ROOT_TYPE, UDF_TYPE)

    @classmethod
    def from_dict(cls, credentials, items):
        """Build a new Volume instance setting attributes from 'items'.

        'items' should be a dictionary with a valid 'type' value. See
        'valid_type' for the definition of a valid type.

        """
        vol_type = items.pop('type', None)
        if not cls.valid_type(vol_type):
            raise InvalidVolumeDictError(items)

        # Access to a protected member _from_dict of a client class
        # pylint: disable=W0212

        result = None
        if vol_type == ROOT_TYPE:
            result = Root._from_dict(credentials, items)
        elif vol_type == UDF_TYPE:
            result = UserDefinedFolder._from_dict(credentials, items)

        return result

    @classmethod
    def _from_dict(cls, credentials, items):
        """Build a new Volume instance setting attributes from 'items'."""
        result = cls(credentials)
        for key, value in items.iteritems():
            setattr(result, key, value)

        return result

    def __eq__(self, other):
        """Is this instance equal to 'other'?"""
        result = self.type == other.type and \
                 self.generation == other.generation and \
                 self.key == other.key and \
                 self.path == other.path and \
                 self.when_created == other.when_created

        return result

    def __repr__(self):
        """String representation of this instance."""
        result = '<%s: %s>' % (self.__class__.__name__, self.path)
        return result

    __str__ = __repr__


class Root(Volume, Request):
    """The Ubuntu One root volume abstraction."""

    def __init__(self, credentials):
        Volume.__init__(self)
        Request.__init__(self, credentials)
        self.type = ROOT_TYPE


class UserDefinedFolder(Volume, Request):
    """An Ubuntu One UserDefinedFolder abstraction."""

    def __init__(self, credentials):
        Volume.__init__(self)
        Request.__init__(self, credentials)
        self.type = UDF_TYPE


class Volumes(Request):
    """The Ubuntu One volumes abstraction."""

    root = '/volumes/'

    def get_children(self):
        """List all volumes for the user."""
        result = self.get(path=self.root)
        result = [Volume.from_dict(self._credentials, res) for res in result]
        return result


if __name__ == '__main__':

    # Invalid name "loop", "cmt"
    # pylint: disable=C0103
    # Generator 'find_credentials' has no 'addCallback' member
    # pylint: disable=E1101

    import gobject
    from ubuntuone.platform.linux.credentials import CredentialsManagementTool

    loop = gobject.MainLoop()

    def get_requests(credentials):
        """Get some testing info."""
        req = Request(credentials)
        print req.get('/~/')
        print req.get('/~/?include_children=true')
        print req.get('/volumes/')
        print req.get('/volumes/Ubuntu One')
        loop.quit()

    def get_volumes(credentials):
        """Get this user volumes' list."""
        vol = Volumes(credentials)
        print vol.get_children()
        loop.quit()

    cmt = CredentialsManagementTool()
    d = cmt.find_credentials()
    d.addCallback(get_volumes)

    loop.run()
