# -*- coding: utf-8 -*-

# Authors: Natalia B Bidart <natalia.bidart@canonical.com>
#
# Copyright 2011 Canonical Ltd.
#
# This program is free software: you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 3, as published
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranties of
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program.  If not, see <http://www.gnu.org/licenses/>.

"""The test suite for the volumes management."""

import urllib2

from ubuntuone.devtools.handlers import MementoHandler
from ubuntuone.devtools.testcase import BaseTestCase

from ubuntuone.filestorageapi import volumes


FAKED_JSON_RESULT = []
FAKED_TOKEN = {
    'name': 'Test me please', 'consumer_key': 'faked_consumer_key',
    'consumer_secret': 'faked_consumer_secret',
    'token': 'faked_token', 'token_secret': 'faked_token_secret',
}
FAKED_ROOT = {
    u'generation': 722,
    u'key': u'LmBcVBfCQDSHefXWUoS3Ow',
    u'path': u'~/Ubuntu One',
    u'type': u'root',
    u'when_created': u'2010-07-17T16:09:17Z',
}
FAKED_UDF = {
    u'generation': 4983,
    u'key': u'HesodD0oRq6ZmdX0jen0YA',
    u'path': u'~/Pictures',
    u'type': u'udf',
    u'when_created': u'2010-07-15T17:57:41Z',
}

FAKED_VOLUMES_LIST = [
    FAKED_ROOT, FAKED_UDF,
    {u'generation': 3761,
     u'key': u'LbJi9aFRTBmWnLtc7XU8YQ',
     u'path': u'~/Documents/foo',
     u'type': u'udf',
     u'when_created': u'2010-03-15T16:25:40Z'},
    {u'generation': 3,
     u'key': u'9cxrDYWgQEanL71CxBU4yw',
     u'path': u'~/udf0',
     u'type': u'udf',
     u'when_created': u'2010-12-13T20:15:03Z'},
    {u'generation': 0,
     u'key': u'pPaS2EQ0RySj8FLvpEoMOA',
     u'path': u'~/udf1',
     u'type': u'udf',
     u'when_created': u'2010-12-13T20:15:06Z'},
]

# Access to a protected member of a client class
# pylint: disable=W0212


class FakedUrllib2(object):
    """Replace the urllib2 module."""

    response = urllib2.addinfourl(fp=file('/dev/null'), headers=None,
                                  url=None, code=200)

    def __init__(self):
        self._url = None
        self._headers = None
        self.urlopen = lambda request: self.response
        self.quote = urllib2.quote
        self.urlparse = urllib2.urlparse

    # Invalid name "Request"
    # pylint: disable=C0103

    def Request(self, url, headers):
        """Fake a Request."""
        self._url = url
        self._headers = headers


class FakedOAuthRequest(object):
    """Replace the OAuthRequest class."""

    params = {}

    def __init__(self):
        self.sign_request = lambda *args, **kwargs: None
        self.to_header = lambda *args, **kwargs: {}

    def from_consumer_and_token(oauth_consumer, **kwargs):
        """Fake the method storing the params for check."""
        FakedOAuthRequest.params.update(kwargs)
        return FakedOAuthRequest()
    from_consumer_and_token = staticmethod(from_consumer_and_token)


class BasicTestCase(BaseTestCase):
    """Test suite for the Volumes classes."""

    # self.klass is not callable
    # pylint: disable=E1102

    klass = None

    def setUp(self):
        super(BasicTestCase, self).setUp()
        self.json_loaded = None
        self.json_result = FAKED_JSON_RESULT
        self.patch(volumes.json, 'load', self.load_json)
        self.fake_urllib2 = FakedUrllib2()
        self.patch(volumes, 'urllib2', self.fake_urllib2)

        self.memento = MementoHandler()
        volumes.logger.addHandler(self.memento)

        if self.klass:
            self.obj = self.klass(credentials=FAKED_TOKEN)

    def load_json(self, data):
        """Simulate a json parser."""
        self.json_loaded = data
        return self.json_result

    def build_header(self, url, http_method='GET'):
        """Build an Oauth header for comparison."""
        consumer = volumes.oauth.OAuthConsumer(FAKED_TOKEN['consumer_key'],
                                               FAKED_TOKEN['consumer_secret'])
        token = volumes.oauth.OAuthToken(FAKED_TOKEN['token'],
                                         FAKED_TOKEN['token_secret'])
        get_request = volumes.oauth.OAuthRequest.from_consumer_and_token
        oauth_req = get_request(oauth_consumer=consumer, token=token,
                                http_method=http_method, http_url=url)
        oauth_req.sign_request(volumes.HMAC_SHA1, consumer, token)
        return oauth_req.to_header()

    def dictify_header(self, header):
        """Convert an OAuth header into a dict."""
        result = {}
        fields = header.split(', ')
        for field in fields:
            key, value = field.split('=')
            result[key] = value.strip('"')

        return result

    def assert_header_equal(self, expected, actual):
        """Is 'expected' equals to 'actual'?"""
        expected = self.dictify_header(expected['Authorization'])
        actual = self.dictify_header(actual['Authorization'])
        for header in (expected, actual):
            header.pop('oauth_nonce')
            header.pop('oauth_timestamp')
            header.pop('oauth_signature')

        self.assertEqual(expected, actual)


class RequestTestCase(BasicTestCase):
    """Test suite for the Request class."""

    klass = volumes.Volumes
    http_method = 'GET'

    def test_token(self):
        """The given tokens are stored."""
        self.assertEqual(self.obj._credentials, FAKED_TOKEN)

    def test_call(self):
        """Calling 'get' triggers an OAuth signed GET request."""
        path = '/test/'
        self.obj.get(path=path)

        url = volumes.FILES_API_BASE + path
        self.assertEqual(self.fake_urllib2._url, url)

        expected = self.build_header(url, http_method=self.http_method)
        self.assert_header_equal(self.fake_urllib2._headers, expected)

    def test_quotes_path(self):
        """Calling 'get' triggers an OAuth signed GET request."""
        path = u'/test me more, sí!/'
        self.obj.get(path=path)

        url = volumes.FILES_API_BASE + urllib2.quote(path.encode('utf8'))
        self.assertEqual(self.fake_urllib2._url, url)

    def test_adds_parameters_to_oauth_request(self):
        """The query string from the path is used in the oauth request."""
        self.patch(volumes.oauth, 'OAuthRequest', FakedOAuthRequest)

        path = '/test/?foo=bar'
        self.obj.get(path=path)

        url = volumes.FILES_API_BASE + path
        self.assertEqual(self.fake_urllib2._url, url)
        self.assertIn('parameters', FakedOAuthRequest.params)
        self.assertEqual(FakedOAuthRequest.params['parameters'],
                         {'foo': 'bar'})

    def test_process(self):
        """JSON response is processed and returned."""
        result = self.obj.get(path='test')

        self.assertEqual(self.json_loaded, FakedUrllib2.response)
        self.assertEqual(result, FAKED_JSON_RESULT)

    def test_invalid_json(self):
        """JSON failure is handled and logged. None is returned."""

        def value_error(*a, **kw):
            """Simulate a JSON parse error."""
            raise ValueError()

        self.patch(volumes.json, 'load', value_error)
        result = self.obj.get(path='fail')

        self.assertEqual(result, None)
        self.assertTrue(self.memento.check_exception(ValueError))


class VolumesTestCase(BasicTestCase):
    """Test suite for the Volumes class."""

    klass = volumes.Volumes

    def test_token(self):
        """The given tokens are stored."""
        self.assertEqual(self.obj._credentials, FAKED_TOKEN)

    def test_from_dict_without_type(self):
        """InvalidVolumeDictError is raised if no type is given."""
        self.assertRaises(volumes.InvalidVolumeDictError,
                          volumes.Volume.from_dict, None, {})

    def test_from_dict_invalid_type(self):
        """InvalidVolumeDictError is raised if invalid type is given."""
        self.assertRaises(volumes.InvalidVolumeDictError,
                          volumes.Volume.from_dict, None, {'type': 'foo'})

    def test_get_children(self):
        """The children can be listed."""
        self.obj.get_children()

        expected = volumes.FILES_API_BASE + volumes.LIST_VOLUMES
        self.assertEqual(self.fake_urllib2._url, expected)
        expected = self.build_header(self.fake_urllib2._url, http_method='GET')
        self.assert_header_equal(self.fake_urllib2._headers, expected)

    def test_get_children_result(self):
        """The children can be listed."""
        self.json_result = [d.copy() for d in FAKED_VOLUMES_LIST]
        result = self.obj.get_children()

        expected = [volumes.Volume.from_dict(FAKED_TOKEN, d.copy())
                    for d in FAKED_VOLUMES_LIST]
        self.assertEqual(result, expected)


class RootTestCase(BasicTestCase):
    """Test suite for the Root class."""

    klass = volumes.Root

    def test_token(self):
        """The given tokens are stored."""
        self.assertEqual(self.obj._credentials, FAKED_TOKEN)

    def test_from_dict(self):
        """A Root volume can be build from a dict."""
        volume = volumes.Volume.from_dict(FAKED_TOKEN, FAKED_ROOT.copy())

        self.assertEqual(volume.type, volumes.ROOT_TYPE)
        self.assertEqual(volume._credentials, FAKED_TOKEN)
        for key, val in FAKED_ROOT.iteritems():
            self.assertTrue(hasattr(volume, key))
            self.assertEqual(getattr(volume, key), val)

    def test_type(self):
        """The type is 'root'."""
        self.assertEqual(self.obj.type, volumes.ROOT_TYPE)


class UserDefinedFolderTestCase(BasicTestCase):
    """Test suite for the UserDefinedFolder class."""

    klass = volumes.UserDefinedFolder

    def test_token(self):
        """The given tokens are stored."""
        self.assertEqual(self.obj._credentials, FAKED_TOKEN)

    def test_from_dict(self):
        """A Root volume can be build from a dict."""
        volume = volumes.Volume.from_dict(FAKED_TOKEN, FAKED_UDF.copy())

        self.assertEqual(volume.type, volumes.UDF_TYPE)
        self.assertEqual(volume._credentials, FAKED_TOKEN)
        for key, val in FAKED_UDF.iteritems():
            self.assertTrue(hasattr(volume, key))
            self.assertEqual(getattr(volume, key), val)

    def test_type(self):
        """The type is 'udf'."""
        self.assertEqual(self.obj.type, volumes.UDF_TYPE)
