/*
    Tucnak - VHF contest log
    Copyright (C) 2002-2006  Ladislav Vaiz <ok1zia@nagano.cz>
    and Even Nermerson (ftdi_new and ftdi_free) and other
    authors of libftdi

    This program is free software; you can redistribute it and/or                                                        
    modify it under the terms of the GNU General Public License                                                          
    version 2 as published by the Free Software Foundation.

*/

#include "header.h"
unsigned short ftdi_checksum(char *eeprom, int eeprom_size);

#ifdef HAVE_LIBFTDI

int davac4_init(struct cwdaemon *cwda){
    cwda->ftdi = NULL;
    davac4_open(cwda, 1);
    return 0;
}

int davac4_open(struct cwdaemon *cwda, int verbose){
    int ret;
    
    if (cwda->ftdi != NULL) return 0;

    cwda->ftdi = ftdi_new();
    if (!cwda->ftdi){
        if (verbose) log_addf("Can't init ftdi library");
        return 1;
    }
    ret=ftdi_usb_open(cwda->ftdi, cfg->cwda_vid, cfg->cwda_pid);
    if (ret){
        if (verbose) log_addf("Can't open ftdi device %04x:%04x, error=%d %s", cfg->cwda_vid, cfg->cwda_pid, ret, cwda->ftdi->error_str);
        davac4_free(cwda);
        return ret;
    }
    ret=ftdi_set_baudrate(cwda->ftdi, 115200);
    if (ret){
        if (verbose) log_addf("Can't set baudrate for davac4, error=%d %s", ret, cwda->ftdi->error_str);
        davac4_free(cwda);
        return ret;
    }
    //cwda->ftdi->bitbang_mode = 0x01;
    //ret=ftdi_enable_bitbang(cwda->ftdi, 0xff);
    ret=ftdi_set_bitmode(cwda->ftdi, 0xff, BITMODE_BITBANG);
    if (ret){
        if (verbose) log_addf("Can't enable bitbang, error=%d %s", ret, cwda->ftdi->error_str);
        davac4_free(cwda);
        return ret;
    }
    cwda->ftdi_state=0xff;

    /*log_addf("FTDI type %d\n", cwda->ftdi->type);    */
    
    return 0;
}

int davac4_free(struct cwdaemon *cwda){
    if (!cwda) return 0;

    if (cwda->ftdi!=NULL){
        ftdi_free(cwda->ftdi);
        cwda->ftdi=NULL;
    }
    return 0;
}


int davac4_reset(struct cwdaemon *cwda){
    davac4_ptt(cwda, 0);
    davac4_cw(cwda, 0);
    davac4_ssbway(cwda, 0);
    return 0;
}

int davac4_cw(struct cwdaemon *cwda, int onoff){
    int ret;
    
    if (cwda->ftdi == NULL) return 1;

    if (onoff)
        cwda->ftdi_state &= ~0x10;
    else
        cwda->ftdi_state |= 0x10;
    
    ret=ftdi_write_data(cwda->ftdi, &cwda->ftdi_state, 1);
    if (ret!=1){
        log_addf("Can't write to davac4, error=%d %s", ret, cwda->ftdi->error_str); 
        davac4_free(cwda);
        return ret;
    }
    return 0;
}

int davac4_ptt(struct cwdaemon *cwda, int onoff){
    int ret;
    
    if (cwda->ftdi == NULL){
        ret=davac4_open(cwda, 0);
        if (ret) return ret;
    }
    if (onoff)
        cwda->ftdi_state &= ~0x04;
    else
        cwda->ftdi_state |= 0x04;
    ret=ftdi_write_data(cwda->ftdi, &cwda->ftdi_state, 1);
/*    char s[2];
    if (onoff)
        s[0] = 0x00;
    else
        s[0] = 0xff;
    ret=ftdi_write_data(cwda->ftdi, s, 1);*/
    
    if (ret!=1){
        log_addf("Can't write to davac4, error=%d %s", ret, cwda->ftdi->error_str); 
        davac4_free(cwda);
        return ret;
    }
    return 0;
}

/* 0=microphone, 1=soundcard */
int davac4_ssbway(struct cwdaemon *cwda, int onoff){
    int ret;
    
    if (cwda->ftdi == NULL){
        ret=davac4_open(cwda, 0);
        if (ret) return ret;
    }
    if (onoff)
        cwda->ftdi_state |= 0x01;   /* soundcard */
    else
        cwda->ftdi_state &= ~0x01;  /* microphone */
    
    ret=ftdi_write_data(cwda->ftdi, &cwda->ftdi_state, 1);
    if (ret!=1){
        log_addf("Can't write to davac4, error=%d %s", ret, cwda->ftdi->error_str); 
        davac4_free(cwda);
        return ret;
    }
    return 0;
}

int davac4_monitor(struct cwdaemon *cwda, int onoff){
    int ret;
    
    if (cwda->ftdi == NULL){
        ret=davac4_open(cwda, 0);
        if (ret) return ret;
    }
    if (onoff)
        cwda->ftdi_state &= ~0x08;
    else
        cwda->ftdi_state |= 0x08;
    
    ret=ftdi_write_data(cwda->ftdi, &cwda->ftdi_state, 1);
    if (ret!=1){
        log_addf("Can't write to davac4, error=%d %s", ret, cwda->ftdi->error_str); 
        davac4_free(cwda);
        return ret;
    }
    return 0;
}

#ifndef HAVE_FTDI_NEW

/* compatibility with libftdi < 0.11 */
/* code from libftdi 0.11 */

struct ftdi_context *ftdi_new()
{
    struct ftdi_context * ftdi = (struct ftdi_context *)malloc(sizeof(struct ftdi_context));

    if (ftdi == NULL) {
        return NULL;
    }

    if (ftdi_init(ftdi) != 0) {
        free(ftdi);
        return NULL;
    }

    return ftdi;
}

void ftdi_free(struct ftdi_context *ftdi)
{
    ftdi_deinit(ftdi);
    free(ftdi);
}

#endif

void usb_info(void){
    int ret;
    struct ftdi_context ftdi;
    struct ftdi_device_list *list, *dev;
    int ids[] = {0x0403, 0x6001, 
                 0xa600, 0xE110,
                 0xa600, 0xE111,
                 0xa600, 0xE112,
                 0};
    int idi;
    char manufacturer[128], description[128], serial[128];
    unsigned char eeprom_buf[128];
    
    printf("\n  usb_info:\n");
    ret = ftdi_init(&ftdi); 
    if (ret < 0) {
        printf("error calling ftdi_init, error=%d %s", ret, ftdi.error_str);
        return;
    }
    for (idi = 0; ids[idi] != 0; idi += 2){
        ret = ftdi_usb_find_all(&ftdi, &list, ids[idi], ids[idi+1]);
        if (ret < 0) {
            printf("error calling ftdi_usb_find_all, error=%d %s", ret, ftdi.error_str);
            return;
        }
        for (dev = list; dev != NULL; dev = dev->next){
            printf("%04x:%04x  ", dev->dev->descriptor.idVendor, dev->dev->descriptor.idProduct);
            ret = ftdi_usb_get_strings(&ftdi, dev->dev, manufacturer, 128, description, 128, serial, 128);
            if (ret < 0){
                printf("Can't get strings, error=%d %s", ret, ftdi.error_str);
            }else{
                printf("%s %s ", manufacturer, description);
                if (strlen(serial)>0 && serial[0]!='?')
                    printf("%s  ", serial);
            }
            printf("\n");
            
            ret = ftdi_usb_open_dev(&ftdi, dev->dev);
            if (ret < 0){
                printf("Can't open device, error=%d %s", ret, ftdi.error_str);
            }else{
                ret = ftdi_read_eeprom(&ftdi, eeprom_buf);
                if (ret < 0){
                    printf("Can't open device, error=%d %s", ret, ftdi.error_str);
                }else{
                    int i;
                    unsigned short chksum_in_eeprom, chksum_computed;
                    for (i=0; i<128; i++){
                        printf("%02x ", (unsigned char)eeprom_buf[i]);
                        if (i%16==15) printf("\n");
                        
                    }
                    chksum_in_eeprom = eeprom_buf[0x7e] + 256 * eeprom_buf[0x7f];
                    chksum_computed = ftdi_checksum(eeprom_buf, 0x80);
                    printf("Checksum eeprom=%04x computed=%04x ", chksum_in_eeprom, chksum_computed);
                }
                ftdi_usb_close(&ftdi);
            }
            printf("\n");
        }
    }
    ftdi_list_free(&list);
    ftdi_deinit(&ftdi);
}

// taken from libftdi
// calculate checksum
unsigned short ftdi_checksum(char *eeprom, int eeprom_size){
    int i;
    unsigned short value, checksum = 0xAAAA;

    for (i = 0; i < eeprom_size/2-1; i++) {
        value = eeprom[i*2];
        value += eeprom[(i*2)+1] << 8;

        checksum = value^checksum;
        checksum = (checksum << 1) | (checksum >> 15);
    }
    return checksum;
}
#endif
