# Copyright (C) 2006,2007 Daiki Ueno <ueno@unixuser.org>

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

require 'cairo'

module Treil
  class CairoWriter < Writer
    def do_write(context, rect)
      rect.each do |r|
        if @config.rect_visible?(r)
          do_write_rect(context, r, @config.rect_fill?(r))
        end
      end
      rect.each do |r|
        if @config.name_visible?(r)
          do_write_name(context, r)
        end
      end
      context.show_page
    end

    def do_write_rect(context, rect, fill = false)
      if fill
        context.set_source_rgba(*@config.background_color(rect))
        context.rectangle(rect.x, rect.y, rect.width, rect.height).fill
        context.stroke
      end
      context.set_source_rgb(*@config.border_color(rect))
      context.set_line_width(@config.border_width(rect))
      context.rectangle(rect.x, rect.y, rect.width, rect.height)
      context.stroke
    end

    def do_write_name(context, rect)
      face = @config.name_font_face(rect)
      if face
        context.select_font_face(face['family'],
                                 face['slant'] || Cairo::FONT_SLANT_NORMAL,
                                 face['weight'] || Cairo::FONT_WEIGHT_NORMAL)
      end
      context.set_source_rgba(@config.name_color(rect))
      context.set_font_size(@config.name_font_size(rect))
      text_extents = context.text_extents(rect.tree.name)
      if text_extents.width > rect.width
        context.set_font_size(@config.name_font_size(rect) * rect.width / (1.0 * text_extents.width))
        text_extents = context.text_extents(rect.tree.name)
      end
      context.move_to((rect.x + rect.width / 2) - text_extents.width / 2 -
                        text_extents.x_bearing,
                      (rect.y + rect.height / 2) - text_extents.height / 2 -
                        text_extents.y_bearing)
      context.show_text(rect.tree.name)
      context.stroke
    end
  end

  class CairoPNGWriter < CairoWriter
    def self.accept?(output)
      Pathname.new(output).extname =~ /\A\.png\z/i
    end

    def write(rect)
      surface = Cairo::ImageSurface.new(Cairo::FORMAT_ARGB32,
                                        rect.width,
                                        rect.height)
      context = Cairo::Context.new(surface)
      do_write(context, rect)
      context.target.write_to_png(@output)
    end

    register(self)
  end

  class CairoScalableWriter < CairoWriter
    def write(rect)
      surface = @surface_class.new(@output, rect.width, rect.height)
      context = Cairo::Context.new(surface)
      do_write(context, rect)
      context.target.finish
    end
  end

  class CairoPDFWriter < CairoScalableWriter
    def self.accept?(output)
      Pathname.new(output).extname =~ /\A\.pdf\z/i
    end

    def initialize(output, config)
      super
      @surface_class = Cairo::PDFSurface
    end

    register(self)
  end

  class CairoPSWriter < CairoScalableWriter
    def self.accept?(output)
      Pathname.new(output).extname =~ /\A\.ps\z/i
    end

    def initialize(output, config)
      super
      @surface_class = Cairo::PSSurface
    end

    register(self)
  end

  class CairoSVGWriter < CairoScalableWriter
    def self.accept?(output)
      Pathname.new(output).extname =~ /\A\.svg?\z/i
    end

    def initialize(output, config)
      super
      @surface_class = Cairo::SVGSurface
    end

    register(self)
  end
end
