# -*- coding: utf-8 -*-
#
# Copyright (C) 2005-2006 Edgewall Software
# Copyright (C) 2005-2006 Christopher Lenz <cmlenz@gmx.de>
# All rights reserved.
#
# This software is licensed as described in the file COPYING, which
# you should have received as part of this distribution. The terms
# are also available at http://trac.edgewall.org/wiki/TracLicense.
#
# This software consists of voluntary contributions made by many
# individuals. For the exact contribution history, see the revision
# history and logs, available at http://trac.edgewall.org/log/.
#
# Author: Christopher Lenz <cmlenz@gmx.de>

import imp
import inspect
import os
import re
try:
    set
except NameError:
    from sets import Set as set
from StringIO import StringIO

from trac.config import default_dir
from trac.core import *
from trac.util import sorted
from trac.util.datefmt import format_date
from trac.util.html import escape, html, Markup
from trac.util.text import to_unicode
from trac.wiki.api import IWikiMacroProvider, WikiSystem
from trac.wiki.model import WikiPage
from trac.web.chrome import add_stylesheet


class WikiMacroBase(Component):
    """Abstract base class for wiki macros."""

    implements(IWikiMacroProvider)
    abstract = True

    def get_macros(self):
        """Yield the name of the macro based on the class name."""
        name = self.__class__.__name__
        if name.endswith('Macro'):
            name = name[:-5]
        yield name

    def get_macro_description(self, name):
        """Return the subclass's docstring."""
        return inspect.getdoc(self.__class__)

    def render_macro(self, req, name, content):
        raise NotImplementedError


class TitleIndexMacro(WikiMacroBase):
    """すべての Wiki ページをアルファベットのリスト形式で出力に挿入します。

    引数として、接頭辞となる文字列を許容します: 指定された場合、生成されるリストには
    ページ名が接頭辞で始まるものだけが含まれます。引数が省略された場合、
    すべてのページがリストされます。
    """

    def render_macro(self, req, name, content):
        prefix = content or None

        wiki = WikiSystem(self.env)

        return html.UL([html.LI(html.A(wiki.format_page_name(page),
                                       href=req.href.wiki(page)))
                        for page in sorted(wiki.get_pages(prefix))])


class RecentChangesMacro(WikiMacroBase):
    """最近更新されたすべてのページを最後に変更した日付で
    グループ化し、リストします。

    このマクロは、2つの引数をとります。最初の引数はプレフィックス文字列です。
    もし、プレフィックスが渡されていたら、結果のリストにはそのプレフィックスで始まるページ
    のみが、リストされます。もしこの引数が省略されると、すべてのページがリストされます。

    2番目の引数は結果リストに表示するページの数を制限するために使用します。
    例えば、5に制限した場合、
    最近更新されたページのうち新しいもの5件がリストの中に含まれます。
    """

    def render_macro(self, req, name, content):
        prefix = limit = None
        if content:
            argv = [arg.strip() for arg in content.split(',')]
            if len(argv) > 0:
                prefix = argv[0]
                if len(argv) > 1:
                    limit = int(argv[1])

        db = self.env.get_db_cnx()
        cursor = db.cursor()

        sql = 'SELECT name, ' \
              '  max(version) AS max_version, ' \
              '  max(time) AS max_time ' \
              'FROM wiki'
        args = []
        if prefix:
            sql += ' WHERE name LIKE %s'
            args.append(prefix + '%')
        sql += ' GROUP BY name ORDER BY max_time DESC'
        if limit:
            sql += ' LIMIT %s'
            args.append(limit)
        cursor.execute(sql, args)

        entries_per_date = []
        prevdate = None
        for name, version, time in cursor:
            date = format_date(time)
            if date != prevdate:
                prevdate = date
                entries_per_date.append((date, []))
            entries_per_date[-1][1].append((name, int(version)))

        wiki = WikiSystem(self.env)
        return html.DIV(
            [html.H3(date) +
             html.UL([html.LI(
            html.A(wiki.format_page_name(name), href=req.href.wiki(name)),
            ' ',
            version > 1 and 
            html.SMALL('(', html.A('diff',
                                   href=req.href.wiki(name, action='diff',
                                                      version=version)), ')') \
            or None)
                      for name, version in entries])
             for date, entries in entries_per_date])


class PageOutlineMacro(WikiMacroBase):
    """現在のwikiページの構造的なアウトラインを表示します。
    アウトラインのそれぞれの項目は一致する表題へのリンクとなります。

    このマクロは3つの任意のパラメータをとります:
    
     * 1番目の引数はアウトラインに含まれる表題の範囲（レベル）を設定することができ、
       数または数の範囲をとります。例えば、 "1" と指定した場合、アウトラインには
       トップレベルの表題のみが表示されます。 "2-3" と指定した場合、アウトラインには、
       レベル 2 とレベル 3 のすべての表題がネストしたリストとして表示されます。
       デフォルトでは、すべてのレベルの表題が表示されます。
     * 2番目の引数は、タイトルを特定するのに使われます。
       （デフォルトはタイトルなし）。
     * 3番目の引数はアウトラインのスタイルを指定します。`inline` または `pullout`
       を指定することができます（後者がデフォルトです）。`inline` スタイルでは、
       アウトラインを通常部分として整形しますが、 `pullout` スタイルでは、アウトラインを
       ボックスの中に整形します。そして、他の内容の右側に
       おかれます。
    """

    def render_macro(self, req, name, content):
        from trac.wiki.formatter import wiki_to_outline
        min_depth, max_depth = 1, 6
        title = None
        inline = 0
        if content:
            argv = [arg.strip() for arg in content.split(',')]
            if len(argv) > 0:
                depth = argv[0]
                if depth.find('-') >= 0:
                    min_depth, max_depth = [int(d) for d in depth.split('-', 1)]
                else:
                    min_depth, max_depth = int(depth), int(depth)
                if len(argv) > 1:
                    title = argv[1].strip()
                    if len(argv) > 2:
                        inline = argv[2].strip().lower() == 'inline'

        db = self.env.get_db_cnx()
        cursor = db.cursor()
        pagename = req.args.get('page') or 'WikiStart'
        page = WikiPage(self.env, pagename)

        buf = StringIO()
        if not inline:
            buf.write('<div class="wiki-toc">')
        if title:
            buf.write('<h4>%s</h4>' % escape(title))
        buf.write(wiki_to_outline(page.text, self.env, db=db,
                                  max_depth=max_depth, min_depth=min_depth))
        if not inline:
            buf.write('</div>')
        return buf.getvalue()


class ImageMacro(WikiMacroBase):
    """画像をwiki形式のテキストに組み込みます。
    
    1番目の引数は、ファイル名を指定します。ファイルの指定は添付ファイルやファイルなど
    3つの指定方法があります。
     * `module:id:file`:module には '''wiki''' または '''ticket''' が指定でき、
       ''file'' という名前の特定のwiki ページ または チケットの 添付ファイルを
       参照します。
     * `id:file`: 上記と同様ですが、id は チケットまたは wiki の簡単な指定方法
       です。
     * `file`:'file' というローカルの添付ファイルを指します。これはwiki ページまたは
       チケットの中でのみ使用できます。
    
    またファイルはリポジトリのファイルも指定できます。
    `source:file` シンタックスを使用します。 (`source:file@rev` も可能です)
    
    残りの引数は任意で、
    `<img>` 要素の 属性を設定します:
     * 数字と単位はサイズと解釈されます。
       (ex. 120, 25%)
     * `right`、`left`、`top`、`bottom` は画像の配置として
       解釈されます。
     * `nolink` は画像へのリンクを除外します。
     * `key=value` スタイルは画像の HTML 属性または CSS スタイルの
        指示として解釈されます。有効なキーは以下の通りです:
        * align, border, width, height, alt, title, longdesc, class, id
          および usemap
        * `border` は数値での指定のみ可能です。
    
    例:
    {{{
        [[Image(photo.jpg)]]                           # シンプルな指定方法
        [[Image(photo.jpg, 120px)]]                    # サイズ指定
        [[Image(photo.jpg, right)]]                    # キーワードによる配置指定
        [[Image(photo.jpg, nolink)]]                   # ソースへのリンクなし
        [[Image(photo.jpg, align=right)]]              # 属性による配置指定
    }}}
    
    他の wiki ページ、チケット、モジュールの画像を使用することができます。
    {{{
        [[Image(OtherPage:foo.bmp)]]    # 現在のモジュールが wiki の場合
        [[Image(base/sub:bar.bmp)]]     # 下位の wiki ページから
        [[Image(#3:baz.bmp)]]           # #3というチケットを指している場合
        [[Image(ticket:36:boo.jpg)]]
        [[Image(source:/images/bee.jpg)]] # リポジトリから直接指定する！
        [[Image(htdocs:foo/bar.png)]]   # プロジェクトのhtdocsディレクトリにあるファイル
    }}}
    
    ''Adapted from the Image.py macro created by Shun-ichi Goto
    <gotoh@taiyo.co.jp>''
    """

    def render_macro(self, req, name, content):
        # args will be null if the macro is called without parenthesis.
        if not content:
            return ''
        # parse arguments
        # we expect the 1st argument to be a filename (filespec)
        args = content.split(',')
        if len(args) == 0:
            raise Exception("No argument.")
        filespec = args[0]
        size_re = re.compile('[0-9]+%?$')
        attr_re = re.compile('(align|border|width|height|alt'
                             '|title|longdesc|class|id|usemap)=(.+)')
        quoted_re = re.compile("(?:[\"'])(.*)(?:[\"'])$")
        attr = {}
        style = {}
        nolink = False
        for arg in args[1:]:
            arg = arg.strip()
            if size_re.match(arg):
                # 'width' keyword
                attr['width'] = arg
                continue
            if arg == 'nolink':
                nolink = True
                continue
            if arg in ('left', 'right', 'top', 'bottom'):
                style['float'] = arg
                continue
            match = attr_re.match(arg)
            if match:
                key, val = match.groups()
                m = quoted_re.search(val) # unquote "..." and '...'
                if m:
                    val = m.group(1)
                if key == 'align':
                    style['float'] = val
                elif key == 'border':
                    style['border'] = ' %dpx solid' % int(val);
                else:
                    attr[str(key)] = val # will be used as a __call__ keyword

        # parse filespec argument to get module and id if contained.
        parts = filespec.split(':')
        url = None
        if len(parts) == 3:                 # module:id:attachment
            if parts[0] in ['wiki', 'ticket']:
                module, id, file = parts
            else:
                raise Exception("%s module can't have attachments" % parts[0])
        elif len(parts) == 2:
            from trac.versioncontrol.web_ui import BrowserModule
            try:
                browser_links = [link for link,_ in 
                                 BrowserModule(self.env).get_link_resolvers()]
            except Exception:
                browser_links = []
            if parts[0] in browser_links:   # source:path
                module, file = parts
                rev = None
                if '@' in file:
                    file, rev = file.split('@')
                url = req.href.browser(file, rev=rev)
                raw_url = req.href.browser(file, rev=rev, format='raw')
                desc = filespec
            else: # #ticket:attachment or WikiPage:attachment
                # FIXME: do something generic about shorthand forms...
                id, file = parts
                if id and id[0] == '#':
                    module = 'ticket'
                    id = id[1:]
                elif id == 'htdocs':
                    raw_url = url = req.href.chrome('site', file)
                    desc = os.path.basename(file)
                elif id in ('http', 'https', 'ftp'): # external URLs
                    raw_url = url = desc = id+':'+file
                else:
                    module = 'wiki'
        elif len(parts) == 1:               # attachment
            # determine current object
            # FIXME: should be retrieved from the formatter...
            # ...and the formatter should be provided to the macro
            file = filespec
            module, id = 'wiki', 'WikiStart'
            path_info = req.path_info.split('/',2)
            if len(path_info) > 1:
                module = path_info[1]
            if len(path_info) > 2:
                id = path_info[2]
            if module not in ['wiki', 'ticket']:
                raise Exception('Cannot reference local attachment from here')
        else:
            raise Exception('No filespec given')
        if not url: # this is an attachment
            from trac.attachment import Attachment
            attachment = Attachment(self.env, module, id, file)
            url = attachment.href(req)
            raw_url = attachment.href(req, format='raw')
            desc = attachment.description
        for key in ['title', 'alt']:
            if desc and not attr.has_key(key):
                attr[key] = desc
        if style:
            attr['style'] = '; '.join(['%s:%s' % (k, escape(v))
                                       for k, v in style.iteritems()])
        result = Markup(html.IMG(src=raw_url, **attr)).sanitize()
        if not nolink:
            result = html.A(result, href=url, style='padding:0; border:none')
        return result


class MacroListMacro(WikiMacroBase):
    """インストールされている、すべての Wiki マクロをリストします。
    もし利用可能ならばドキュメントも含みます。
    
    非必須オプションとして、特定のマクロの名前を引数として渡すことが出来ます。
    この場合、指定されたマクロのドキュメントだけを表示します。
    
    Note: このマクロは mod_python の `PythonOptimize` オプションが有効になっている
    場合は、マクロのドキュメントを表示することが出来ません!
    """

    def render_macro(self, req, name, content):
        from trac.wiki.formatter import wiki_to_html, system_message
        wiki = WikiSystem(self.env)

        def get_macro_descr():
            for macro_provider in wiki.macro_providers:
                for macro_name in macro_provider.get_macros():
                    if content and macro_name != content:
                        continue
                    try:
                        descr = to_unicode(macro_provider.get_macro_description(macro_name))
                        descr = wiki_to_html(descr or '', self.env, req)
                    except Exception, e:
                        descr = Markup(system_message(
                            "Error: Can't get description for macro %s" \
                            % macro_name, e))
                    yield (macro_name, descr)

        return html.DL([(html.DT(html.CODE('[[',macro_name,']]'),
                                 id='%s-macro' % macro_name),
                         html.DD(description))
                        for macro_name, description in get_macro_descr()])


class TracIniMacro(WikiMacroBase):
    """Trac の設定ファイルのドキュメントを生成します。

    通常、このマクロは Wiki ページ TracIni の中で使用されます。
    省略可能な引数にはコンフィグのセクションのフィルタ、
    コンフィグのオプション名のフィルタを指定できます:フィルタで指定された文字列
    で始まるコンフィグのセクションとオプション名のみが出力されます。
    """

    def render_macro(self, req, name, filter):
        from trac.config import Option
        from trac.wiki.formatter import wiki_to_html, wiki_to_oneliner
        filter = filter or ''

        sections = set([section for section, option in Option.registry.keys()
                        if section.startswith(filter)])

        return html.DIV(class_='tracini')(
            [(html.H2('[%s]' % section, id='%s-section' % section),
              html.TABLE(class_='wiki')(
                  html.TBODY([html.TR(html.TD(html.TT(option.name)),
                                      html.TD(wiki_to_oneliner(option.__doc__,
                                                               self.env)))
                              for option in Option.registry.values()
                              if option.section == section])))
             for section in sorted(sections)])


class UserMacroProvider(Component):
    """Python のソースファイルとして提供されているマクロを TracEnvironment の
    `wiki-macros` ディレクトリか、もしくはグローバルマクロディレクトリに
    追加します。
    """
    implements(IWikiMacroProvider)

    def __init__(self):
        self.env_macros = os.path.join(self.env.path, 'wiki-macros')
        self.site_macros = default_dir('macros')

    # IWikiMacroProvider methods

    def get_macros(self):
        found = []
        for path in (self.env_macros, self.site_macros):
            if not os.path.exists(path):
                continue
            for filename in [filename for filename in os.listdir(path)
                             if filename.lower().endswith('.py')
                             and not filename.startswith('__')]:
                try:
                    module = self._load_macro(filename[:-3])
                    name = module.__name__
                    if name in found:
                        continue
                    found.append(name)
                    yield name
                except Exception, e:
                    self.log.error('Failed to load wiki macro %s (%s)',
                                   filename, e, exc_info=True)

    def get_macro_description(self, name):
        return inspect.getdoc(self._load_macro(name))

    def render_macro(self, req, name, content):
        module = self._load_macro(name)
        try:
            return module.execute(req and req.hdf, content, self.env)
        except Exception, e:
            self.log.error('Wiki macro %s failed (%s)', name, e, exc_info=True)
            raise

    def _load_macro(self, name):
        for path in (self.env_macros, self.site_macros):
            macro_file = os.path.join(path, name + '.py')
            if os.path.isfile(macro_file):
                return imp.load_source(name, macro_file)
        raise TracError, 'Macro %s not found' % name
