#!/usr/bin/python

# Copyright (C) 2010 Jelmer Vernooij <jelmer@samba.org>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
# This software may be used and distributed according to the terms
# of the GNU General Public License, incorporated herein by reference.

from bzrlib.commands import Command


class cmd_close_trac_bugs(Command):
    """Close trac bugs marked as closed in this branch.

    bugtracker should be the name of a bugtracker configured in your 
    Bazaar configuration, e.g. 'bitlbee' in the case you have:

        trac_bitlbee_url = http://bugs.bitlbee.org/bitlbee/

    trac_project should be the path to your local trac environment.

    location should refer to a Bazaar branch but defaults to the current
    directory.
    """

    takes_args = ["bugtracker?", "trac_project?", "location?"]

    def run(self, bugtracker, trac_project=None, location=None):
        from bzrlib.branch import Branch
        from bzrlib.errors import BzrCommandError
        from trac.env import Environment
        from trac.ticket import Ticket
        if location is None:
            location = "."

        b = Branch.open(location)
        from bzrlib.bugtracker import tracker_registry
        trac_bt = tracker_registry.get("trac")

        config = b.get_config()

        if bugtracker is None:
            bugtracker = config.get_user_option("bugtracker")
        if bugtracker is None:
            raise BzrCommandError("No bug tracker specified and none in "
                                  "configuration")

        bt = trac_bt.get(bugtracker, b)
        if bt is None:
            raise BzrCommandError("No such trac bugtracker '%s' in configuration." %
                    bugtracker)
        if trac_project is None:
            trac_project = config.get_user_option("trac_project")
        if trac_project is None:
            raise BzrCommandError("No trac project specified and none in"
                                  "configuration.")
        env = Environment(trac_project)
        db_cnx = env.get_db_cnx()

        revids = [r for r in b.repository.get_ancestry(b.last_revision()) if r]
        for i, rev in enumerate(b.repository.get_revisions(revids)):
            if not 'bugs' in rev.properties:
                continue
            for line in rev.properties['bugs'].splitlines():
                (url, resolution) = line.rsplit(" ", 1)
                if resolution != "fixed":
                    continue
                if not url.startswith(bt._base_url):
                    continue
                # FIXME: Use bzrlib.bugtracker to extract the id
                bugid = int(url.rsplit("/")[-1])
                ticket = Ticket(env, bugid)
                if ticket['resolution'] == 'fixed':
                    # Don't attempt to close again
                    continue
                ticket['resolution'] = 'fixed'
                ticket['status'] = 'close'
                revno = b.revision_id_to_dotted_revno(rev.revision_id)
                ticket.save_changes(author=rev.committer,
                    comment="Closed in revision %s." % revno)
