# $Id: transports.tcl 665 2006-01-24 22:52:37Z aleksey $

namespace eval transport {
    variable capabilities [list tcp http_poll]
    variable disconnect quick
}

proc transport::capabilities {} {
    variable capabilities
    return $capabilities
}

namespace eval transport::proxy {
    variable capabilities [list none http]
}

proc transport::proxy::capabilities {} {
    variable capabilities
    return $capabilities
}

# Connect without proxy
namespace eval transport::proxy::none {}

proc transport::proxy::none::connect {connid server port args} {
    return [socket $server $port]
}

package require base64

# Connect with HTTP proxy
namespace eval transport::proxy::http {
    if {[catch {package require ntlm}]} {
	set auth [list basic]
    } else {
	set auth [list basic ntlm]
    }
}

proc transport::proxy::http::connect {connid server port args} {
    variable auth

    set useragent ""
    foreach {opt val} $args {
	switch -- $opt {
	    -proxyhost { set proxyhost $val }
	    -proxyport { set proxyport $val }
	    -proxyusername { set username $val }
	    -proxypassword { set password $val }
	    -proxyuseragent { set useragent $val}
	}
    }

    set sock [socket $proxyhost $proxyport]
    puts_connect_query $sock $server $port $useragent
    set code [read_proxy_answer $sock result]

    if {$code >= 200 && $code < 300} {
	while {![cequal [gets $sock] ""]} { }
	return $sock
    } elseif {$code == 407} {
	set content_length -1
	set method basic
	while {![cequal [set header [gets $sock]] ""]} {
	    switch -- [http_header_name $header] {
		proxy-authenticate {
		    set body [http_header_body $header]
		    if {[string equal -nocase -length 4 $body "NTLM"]} {
			if {[lsearch -exact $auth ntlm] >= 0} {
			    set method ntlm
			}
		    }
		}
		content-length {
		    set content_length [string trim [http_header_body $header]]
		}
		default { }
	    }
	}

	read_proxy_junk $sock $content_length
	close $sock
	
	set sock [socket $proxyhost $proxyport]
	if {[catch {connect:$method \
			$sock $server $port $useragent $username $password} \
		 result]} {
	    close $sock
	    error "HTTP proxy returned: $result"
	} else {
	    return $sock
	}
    } else {
	close $sock
	error "HTTP proxy returned: $result"
    }
}

proc transport::proxy::http::connect:basic {sock server port useragent username password} {
    set auth [string map {\n {}} \
		   [base64::encode [encoding convertto "$username:$password"]]]

    puts_connect_query $sock $server $port $useragent "Basic $auth"
    set code [read_proxy_answer $sock result]

    if {$code >= 200 && $code < 300} {
	while {![cequal [gets $sock] ""]} { }
	return $sock
    } else {
	return -code error $result
    }
}

proc transport::proxy::http::connect:ntlm {sock server port useragent username password} {
    set domain ""
    set host [info hostname]

    # if username is domain/username or domain\username
    # then set domain and username
    regexp {(\w+)[\\/](.*)} $username -> domain username

    set token [NTLM::new -domain $domain \
			 -host $host \
			 -username $username \
			 -password $password]
    set message1 [$token type1_message]

    puts_connect_query $sock $server $port $useragent "NTLM $message1"
    set code [read_proxy_answer $sock result]

    if {$code >= 200 && $code < 300} {
	while {![cequal [gets $sock] ""]} { }
	return $sock
    } elseif {$code != 407} {
	return -code error $result
    }
    
    set content_length -1
    set message2 ""
    while {![string equal [set header [gets $sock]] ""]} {
	switch -- [http_header_name $header] {
	    proxy-authenticate {
		set body [http_header_body $header]
		if {[string equal -length 5 $body "NTLM "]} {
		    set message2 [string range $body 5 end]
		}
	    }
	    content-length {
		set content_length [string trim [http_header_body $header]]
	    }
	    default { }
	}
    }

    read_proxy_junk $sock $content_length

    $token parse_type2_message -message $message2
    set message3 [$token type3_message]
    $token free

    puts_connect_query $sock $server $port $useragent "NTLM $message3"
    set code [read_proxy_answer $sock result]

    if {$code >= 200 && $code < 300} {
	while {![cequal [gets $sock] ""]} { }
	return $sock
    } else {
	return -code error $result
    }
}

proc transport::proxy::http::puts_connect_query {sock server port useragent {auth ""}} {
    fconfigure $sock -buffering line -translation auto

    puts $sock "CONNECT $server:$port HTTP/1.1"
    puts $sock "Proxy-Connection: keep-alive"
    if {$useragent != ""} {
	puts $sock "User-Agent: $useragent"
    }
    if {$auth != ""} {
	puts $sock "Proxy-Authorization: $auth"
    }
    puts $sock ""
    
}

proc transport::proxy::http::read_proxy_answer {sock resultvar} {
    variable proxy_readable
    upvar $resultvar result

    fileevent $sock readable \
	[list set [namespace current]::proxy_readable($sock) {}]
    vwait [namespace current]::proxy_readable($sock)
    fileevent $sock readable {}
    unset proxy_readable($sock)

    set result [gets $sock]
    set code [lindex [split $result { }] 1]
    
    if {[string is integer -strict $code]} {
	return $code
    } else {
	return -code error $result
    }
}

proc transport::proxy::http::read_proxy_junk {sock length} {
    fconfigure $sock -buffering none -translation binary
    if {$length != -1} {
	read $sock $length
    } else {
	read $sock
    }
}

proc transport::proxy::http::http_header_name {header} {
    set hlist [split $header ":"]
    return [string tolower [lindex $hlist 0]]
}

proc transport::proxy::http::http_header_body {header} {
    set hlist [split $header ":"]
    set body [join [lrange $hlist 1 end] ":"]
    return [string trim $body]
}


######################################################################
#
# TCP Socket Support
#
######################################################################

namespace eval transport::tcp {}

proc transport::tcp::connect {connid server port args} {
    variable lib

    set proxytype none
    foreach {opt val} $args {
	switch -- $opt {
	    -proxytype { set proxytype $val }
	}
    }

    set sock [eval [list [namespace parent]::proxy::${proxytype}::connect \
			 $connid $server $port] $args]
    fconfigure $sock -blocking 0 -buffering none \
	       -translation auto -encoding utf-8
    set lib($connid,socket) $sock

    fileevent $sock readable \
	      [list [namespace current]::inmsg $connid $sock]

    return $sock
}

proc transport::tcp::outmsg {connid msg} {
    variable lib

    if {![info exists lib($connid,socket)]} {
	::LOG "error ([namespace current]::outmsg)\
	       Cannot write to socket: socket for\
	       connection $connid doesn't exist"
	return -2
    }

    if {[catch { puts -nonewline $lib($connid,socket) $msg }]} {
	::LOG "error ([namespace current]::outmsg)\
	       Cannot write to socket: $lib($connid,socket)"
	return -2
    }
}

proc transport::tcp::disconnect {connid} {
    variable lib

    catch {
	if {[set [namespace parent]::disconnect] == "quick"} {
	    flush $lib($connid,socket)
	} else {
	    fconfigure $lib($connid,socket) -blocking 1
	    flush $lib($connid,socket)
	    vwait [namespace current]::lib($connid,socket)
	}
    }
}

proc transport::tcp::close {connid} {
    variable lib

    catch { fileevent $lib($connid,socket) readable {} }
    catch { ::close $lib($connid,socket) }

    array unset lib $connid,*
}

######################################################################
proc transport::tcp::inmsg {connid sock} {
    set msg ""
    catch { set msg [read $sock] }

    jlib::inmsg $connid $msg [eof $sock]
}

######################################################################
# TODO Cleanup
proc transport::tcp::to_compress {connid method} {
    variable lib

    set [namespace parent]::${method}::lib($connid,socket) \
	$lib($connid,socket)
    eval [list [namespace parent]::${method}::import $connid]
    set ::jlib::lib($connid,transport) $method

    array unset lib $connid,*
}

proc transport::tcp::to_tls {connid args} {
    variable lib

    set [namespace parent]::tls::lib($connid,socket) $lib($connid,socket)
    eval [list [namespace parent]::tls::tls_import $connid] $args
    set ::jlib::lib($connid,transport) tls

    array unset lib $connid,*
}


######################################################################
#
# Zlib Compressed Socket Support
#
######################################################################

if {![catch { package require zlib 1.0 }]} {
    lappend transport::capabilities compress
}

namespace eval transport::zlib {}

proc transport::zlib::connect {connid server port args} {
    variable lib

    set proxytype none
    foreach {opt val} $args {
	switch -- $opt {
	    -proxytype { set proxytype $val }
	}
    }

    set sock [eval [list [namespace parent]::proxy::${proxytype}::connect \
			 $connid $server $port] $args]

    set lib($connid,socket) $sock
    import $connid

    return $sock
}

proc transport::zlib::outmsg {connid msg} {
    variable lib

    if {![info exists lib($connid,socket)]} {
	::LOG "error ([namespace current]::outmsg)\
	       Cannot write to socket: socket for connection\
	       $connid doesn't exist"
	return -2
    }

    if {[catch { puts -nonewline $lib($connid,socket) $msg }]} {
	::LOG "error ([namespace current]::outmsg)\
	       Cannot write to socket: $lib($connid,socket)"
	return -2
    }
    flush $lib($connid,socket)
    fconfigure $lib($connid,socket) -flush output
}

proc transport::zlib::disconnect {connid} {
    variable lib

    catch {
	if {[set [namespace parent]::disconnect] == "quick"} {
	    flush $lib($connid,socket)
	    fconfigure $lib($connid,socket) -finish output
	} else {
	    fconfigure $lib($connid,socket) -blocking 1
	    flush $lib($connid,socket)
	    fconfigure $lib($connid,socket) -finish output
	    vwait [namespace current]::lib($connid,socket)
	}
    }
}

proc transport::zlib::close {connid} {
    variable lib

    catch { fileevent $lib($connid,socket) readable {} }
    catch { ::close $lib($connid,socket) }

    array unset lib $connid,*
}

######################################################################
proc transport::zlib::inmsg {connid sock} {
    set msg ""
    catch {
	fconfigure $sock -flush input
	set msg [read $sock]
    }

    jlib::inmsg $connid $msg [eof $sock]
}

######################################################################
proc transport::zlib::import {connid args} {
    variable lib

    set sock $lib($connid,socket)
    fconfigure $sock -blocking 0 -buffering none \
	       -translation auto -encoding utf-8
    zlib stream $lib($connid,socket) RDWR -output compress \
	 -input decompress

    fileevent $sock readable \
	      [list [namespace current]::inmsg $connid $sock]
}

######################################################################
#
# TLS Socket Support
#
######################################################################

if {![catch { package require tls 1.4 }]} {
    lappend transport::capabilities tls
}

namespace eval transport::tls {}

proc transport::tls::connect {connid server port args} {
    variable lib

    set proxytype none
    set tlsargs {}
    foreach {opt val} $args {
	switch -- $opt {
	    -proxytype { set proxytype $val }
	    -cacertstore {
		if {$val != ""} {
		    if {[file isdirectory $val]} {
			lappend tlsargs -cadir $val
		    } else {
			lappend tlsargs -cafile $val
		    }
		}
	    }
	    -certfile  -
	    -keyfile   {
		if {$val != ""} {
		    lappend tlsargs $opt $val
		}
	    }
	}
    }

    set sock [eval [list [namespace parent]::proxy::${proxytype}::connect \
			 $connid $server $port] $args]

    fconfigure $sock -encoding binary -translation binary

    set lib($connid,socket) $sock
    eval [list tls_import $connid] $tlsargs

    return $sock
}

proc transport::tls::outmsg {connid msg} {
    variable lib

    if {![info exists lib($connid,socket)]} {
	::LOG "error ([namespace current]::outmsg)\
	       Cannot write to socket: socket for connection\
	       $connid doesn't exist"
	return -2
    }

    if {[catch { puts -nonewline $lib($connid,socket) $msg }]} {
	::LOG "error ([namespace current]::outmsg)\
	       Cannot write to socket: $lib($connid,socket)"
	return -2
    }
}

proc transport::tls::disconnect {connid} {
    variable lib

    catch {
	if {[set [namespace parent]::disconnect] == "quick"} {
	    flush $lib($connid,socket)
	} else {
	    fconfigure $lib($connid,socket) -blocking 1
	    flush $lib($connid,socket)
	    vwait [namespace current]::lib($connid,socket)
	}
    }
}

proc transport::tls::close {connid} {
    variable lib

    catch { fileevent $lib($connid,socket) readable {} }
    catch { ::close $lib($connid,socket) }

    array unset lib $connid,*
}

######################################################################
proc transport::tls::inmsg {connid sock} {
    set msg ""
    catch { set msg [read $sock] }

    jlib::inmsg $connid $msg [eof $sock]
}

######################################################################
proc ::client:tls_callback {args} {
    return 1
}

######################################################################
proc transport::tls::tls_import {connid args} {
    variable lib

    set sock $lib($connid,socket)

    fileevent $sock readable {}
    fileevent $sock writable {}
    fconfigure $sock -blocking 1

    eval [list tls::import $sock \
	       -command [list client:tls_callback $connid] \
	       -ssl2    false \
	       -ssl3    true \
	       -tls1    true \
	       -request true \
	       -require false \
	       -server  false] $args

    if {[catch { tls::handshake $sock } tls_result]} {
	catch { ::close $sock }
	error $tls_result
    }

    fconfigure $sock -blocking 0 -buffering none \
               -translation auto -encoding utf-8

    fileevent $sock readable \
	      [list [namespace current]::inmsg $connid $sock]
}

######################################################################
# TODO Cleanup
proc transport::tls::to_compress {connid method} {
    variable lib

    set [namespace parent]::${method}::lib($connid,socket) \
	$lib($connid,socket)
    eval [list [namespace parent]::${method}::import $connid]
    set ::jlib::lib($connid,transport) $method

    array unset lib $connid,*
}

######################################################################
#
# HTTP Polling
#
######################################################################

package require sha1

namespace eval transport::http_poll {
    variable lib
    set lib(http_version) [package require http]
}

if {![catch { package require tls 1.4 }]} {
    ::http::register https 443 ::tls::socket
}

proc transport::http_poll::connect {connid server port args} {
    variable lib

    set lib($connid,polltimeout) 0
    set lib($connid,pollint) 3000
    set lib($connid,pollmin) 3000
    set lib($connid,pollmax) 30000
    set lib($connid,proxyhost) ""
    set lib($connid,proxyport) ""
    set lib($connid,proxyusername) ""
    set lib($connid,proxypassword) ""
    set lib($connid,proxyuseragent) ""
    set lib($connid,pollurl) ""
    set lib($connid,pollusekeys) 1
    set lib($connid,pollnumkeys) 100

    foreach {opt val} $args {
	switch -- $opt {
	    -polltimeout   { set lib($connid,polltimeout) $val }
	    -pollint       { set lib($connid,pollint) $val }
	    -pollmin       { set lib($connid,pollmin) $val }
	    -pollmax       { set lib($connid,pollmax) $val }
	    -pollurl       { set lib($connid,pollurl) $val }
	    -pollusekeys   { set lib($connid,pollusekeys) $val }
	    -pollnumkeys   { set lib($connid,pollnumkeys) $val }
	    -proxyhost     { set lib($connid,proxyhost) $val }
	    -proxyport     { set lib($connid,proxyport) $val }
	    -proxyusername { set lib($connid,proxyusername) $val }
	    -proxypassword { set lib($connid,proxypassword) $val }
	    -proxyuseragent { set lib($connid,proxyuseragent) $val }
	}
    }

    set lib($connid,pollwait) disconnected
    set lib($connid,polloutdata) ""
    set lib($connid,pollseskey) 0
    set lib($connid,pollid) ""
    set lib($connid,pollkeys) {}

    if {$lib($connid,proxyuseragent) != ""} {
	::http::config -useragent $lib($connid,proxyuseragent)
    }

    if {($lib($connid,proxyhost) != "") && ($lib($connid,proxyport) != "")} {
	::http::config -proxyhost $lib($connid,proxyhost) \
		       -proxyport $lib($connid,proxyport)

	if {$lib($connid,proxyusername) != ""} {
	    set auth \
		[base64::encode \
                     [encoding convertto \
			  "$lib($connid,proxyusername):$lib($connid,proxypassword)"]]
	    set lib($connid,proxyauth) [list "Proxy-Authorization" "Basic $auth"]
	} else {
	    set lib($connid,proxyauth) {}
	}
    } else {
	    set lib($connid,proxyauth) {}
    }

    if {$lib($connid,pollusekeys)} {
        # generate keys
	::HTTP_LOG "connect ($connid): generating keys"
        set seed [random 1000000000]
        set oldkey $seed
        set key_count $lib($connid,pollnumkeys)
        while { $key_count } {
            set nextkey [base64::encode [hex_decode [sha1::sha1 $oldkey]]]
            # skip the initial seed
            lappend lib($connid,pollkeys) $nextkey
            set oldkey $nextkey
            incr key_count -1
        }
    }

    set_httpwait $connid connected
}

proc transport::http_poll::outmsg {connid msg} {
    variable lib

    if {![info exists lib($connid,pollwait)]} {
	return
    }

    switch -- $lib($connid,pollwait) {
	disconnected -
	waiting -
	disconnecting { }
	default { poll $connid $msg }
    }
}

proc transport::http_poll::disconnect {connid} {
    variable lib

    if {![info exists lib($connid,pollwait)]} {
	return
    }

    switch -- $lib($connid,pollwait) {
	disconnected -
	waiting { }
	polling { set_httpwait $connid waiting }
	default { set_httpwait $connid disconnecting }
    }

    if {[set [namespace parent]::disconnect] != "quick"} {
	while {[info exists lib($connid,pollwait)] && \
		    $lib($connid,pollwait) != "disconnected"} {
	    vwait [namespace current]::lib($connid,pollwait)
	}
    }
}

proc transport::http_poll::close {connid} {
    variable lib

    set_httpwait $connid disconnected

    array unset lib $connid,*
}

######################################################################
proc transport::http_poll::inmsg {connid body} {
    if {[string length $body] > 2} {
	jlib::inmsg $connid $body 0
    }
}

######################################################################
proc ::HTTP_LOG {args} {}

######################################################################
proc transport::http_poll::set_httpwait {connid opt} {
    variable lib

    set lib($connid,pollwait) $opt
    if {$opt == "disconnected"} {
	if {[info exists lib($connid,pollid)] && \
		$lib($connid,pollid) != ""} {
	    after cancel $lib($connid,pollid)
	}
    }
}

proc transport::http_poll::process_httpreply {connid try query token} {
    variable lib

    upvar #0 $token state

    if {[set temp [::http::ncode $token]] != 200} {
	::HTTP_LOG "error (process_httpreply)\
($connid) Http returned $temp $state(status)"
	if {$try < 3} {
	    get_url $connid [expr {$try + 1}] $query
	} else {
	    set_httpwait $connid disconnected
	    jlib::emergency_disconnect $connid
	}
	::http::cleanup $token
	return
    }

    foreach {name value} $state(meta) {
	if {[string equal -nocase "Set-Cookie" $name]} {
	    ::HTTP_LOG "process_httpreply ($connid): Set-Cookie: $value"
	    set start 0
	    set end [string first ";" $value]
	    if {$end < 1} {
		set end [string length $value]
	    }
	    if {[string equal -nocase -length 3 "ID=" $value]} {
		set start 3
	    }
	    set lib($connid,pollseskey) [string range $value $start [expr {$end - 1}]]
	}
    }
    set inmsg [encoding convertfrom utf-8 $state(body)]
    ::HTTP_LOG "process_httpreply ($connid): '$inmsg'"
    ::http::cleanup $token

    if {[regexp {:0$} $lib($connid,pollseskey)] || \
	    [regexp {%3A0$} $lib($connid,pollseskey)]} {
	::HTTP_LOG "error (process_httpreply) Cookie Error"
	set_httpwait $connid disconnected
	jlib::emergency_disconnect $connid
	return
    }

    if {[string length $inmsg] > 5 } {
	set lib($connid,pollint) [expr $lib($connid,pollint) / 2]
	if {$lib($connid,pollint) < $lib($connid,pollmin)} {
	    set lib($connid,pollint) $lib($connid,pollmin)
	}
    } else {
	set lib($connid,pollint) [expr $lib($connid,pollint) * 11 / 10]
	if {$lib($connid,pollint) > $lib($connid,pollmax)} {
	    set lib($connid,pollint) $lib($connid,pollmax)
	}
    }

    inmsg $connid $inmsg

    switch -- $lib($connid,pollwait) {
	waiting { set_httpwait $connid disconnecting }
	polling { set_httpwait $connid connected }
    }
}

proc transport::http_poll::poll {connid what} {
    variable lib

    ::HTTP_LOG "poll ($connid): '$what'"

    if {![info exists lib($connid,pollwait)]} {
	set_httpwait $connid disconnected
	return
    }

    append lib($connid,polloutdata) [encoding convertto utf-8 $what]
    switch -- $lib($connid,pollwait) {
	disconnected {
	    ::HTTP_LOG "poll ($connid): DISCONNECTED"
	    return
	}
	disconnecting {
	    ::HTTP_LOG "poll ($connid): DISCONNECTING"
	    if {$lib($connid,polloutdata) == ""} {
		set_httpwait $connid disconnected
		return
	    }
	}
	waiting -
	polling {
	    ::HTTP_LOG "poll ($connid): RESCHEDULING"
	    if {[info exists lib($connid,pollid)]} {
		after cancel $lib($connid,pollid)
	    }
	    ::HTTP_LOG "poll ($connid): $lib($connid,pollint)"
	    set lib($connid,pollid) \
		[after $lib($connid,pollint) \
		       [list [namespace current]::poll $connid ""]]
	    return
	}
    }

    if {$lib($connid,pollusekeys)} {
	# regenerate 
	set firstkey [lindex $lib($connid,pollkeys) end]
	set secondkey ""
	if { [llength $lib($connid,pollkeys)] == 1} {
	    ::HTTP_LOG "poll ($connid): regenerating keys"
	    set lib($connid,pollkeys) {}
	    set seed [random 1000000000]
	    set oldkey $seed
	    set key_count $lib($connid,pollnumkeys)
	    while { $key_count } {
		set nextkey [base64::encode [hex_decode [sha1::sha1 $oldkey]]]
		# skip the initial seed
		lappend lib($connid,pollkeys) $nextkey
		set oldkey $nextkey
		incr key_count -1
	    }
	    set secondkey [lindex $lib($connid,pollkeys) end]
	}
	set l [llength $lib($connid,pollkeys)]
	set lib($connid,pollkeys) \
	    [lrange $lib($connid,pollkeys) 0 [expr {$l - 2}]]

	if {[string length $firstkey]} {
	    set firstkey ";$firstkey"
        }

        if {[string length $secondkey]} {
            set secondkey ";$secondkey"
        }

        set query "$lib($connid,pollseskey)$firstkey$secondkey,$lib($connid,polloutdata)"
    } else {
        set query "$lib($connid,pollseskey),$lib($connid,polloutdata)"
    }
    switch -- $lib($connid,pollwait) {
	disconnecting { set_httpwait $connid waiting }
	default { set_httpwait $connid polling }
    }
    ::HTTP_LOG "poll ($connid): query: '[encoding convertfrom utf-8 $query]'"

    get_url $connid 0 $query

    set lib($connid,polloutdata) ""

    if {[info exists lib($connid,pollid)]} {
        after cancel $lib($connid,pollid)
    }
    ::HTTP_LOG "poll ($connid): $lib($connid,pollint)"
    set lib($connid,pollid) \
	[after $lib($connid,pollint) \
	       [list [namespace current]::poll $connid ""]]
}

proc transport::http_poll::get_url {connid try query} {
    variable lib

    set get_url_args [list -headers $lib($connid,proxyauth)]
    if {[package vcompare 2.3.3 $lib(http_version)] <= 0} {
	lappend get_url_args -binary 1
    }

    eval [list ::http::geturl $lib($connid,pollurl) -query $query \
	       -command [list [namespace current]::process_httpreply $connid $try $query] \
	       -timeout $lib($connid,polltimeout)] $get_url_args
}

proc transport::http_poll::hex_decode {hexstring} {
    set result ""
    while { [string length $hexstring] } {
	scan [string range $hexstring 0 1] "%x" X
	regsub "^.." $hexstring "" hexstring
	set result [binary format "a*c" $result $X]
    }
    return $result
}

