/*
 *	TICKR - GTK-based Feed Reader - Copyright (C) Emmanuel Thomas-Maurin 2009-2013
 *	<manutm007@gmail.com>
 *
 * 	This program is free software: you can redistribute it and/or modify
 * 	it under the terms of the GNU General Public License as published by
 * 	the Free Software Foundation, either version 3 of the License, or
 * 	(at your option) any later version.
 *
 * 	This program is distributed in the hope that it will be useful,
 * 	but WITHOUT ANY WARRANTY; without even the implied warranty of
 * 	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * 	GNU General Public License for more details.
 *
 * 	You should have received a copy of the GNU General Public License
 * 	along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "tickr.h"

#define TIME_STR		"|%c%c:%c%c:%c%c"	/* At right = "|00:00:00" */
#define TIME_STR_LENGTH		9

void display_time(const Params *prm)
{
	TickerEnv		*env = get_ticker_env();
	PangoLayout		*p_layout;
	PangoFontDescription	*f_des;
	int			layout_width, layout_height;
	cairo_surface_t		*surf_clock;
	cairo_t			*cr;
	float			shadow_k;
	char			tmp[64];
	time_t			time2;
	gint			h_diff;

	if (prm->clock == 'l' || prm->clock == 'r') {
		if ((p_layout = pango_layout_new(gtk_widget_get_pango_context(env->win))) == NULL)
			return;
		pango_layout_set_attributes(p_layout, NULL);
		pango_layout_set_single_paragraph_mode(p_layout, TRUE);
		f_des = pango_font_description_from_string((const char *)prm->clock_font_n_s);
		pango_layout_set_font_description(p_layout, f_des);
		pango_font_description_free(f_des);

		time2 = time(NULL);
		if (prm->clock == 'l') {
			str_n_cpy(tmp, ctime(&time2) + 11, 8);
			str_n_cat(tmp, "|", 1);
		} else if (prm->clock == 'r') {
			str_n_cpy(tmp, "|", 1);
			str_n_cat(tmp, ctime(&time2) + 11, 8);
		}

		pango_layout_set_text(p_layout, tmp, -1);
		pango_layout_context_changed(p_layout);
		pango_layout_get_pixel_size2(p_layout, &layout_width, &layout_height);

		env->drwa_clock_width = get_clock_width(prm);	/* v_diff */
		/*
		 * For some fonts like 'impact bold' which give strange things
		 * v_diff = drwa_clock_width - layout_width;
		 */
		gtk_widget_set_size_request(env->drwa_clock, env->drwa_clock_width, env->drwa_height);
		h_diff = (env->drwa_height - (gint)layout_height) / 2;
		/*
		 * Create cairo image surface onto which layout will be rendered
		 */
		surf_clock = cairo_image_surface_create(CAIRO_FORMAT_ARGB32,
			env->drwa_clock_width, env->drwa_height);
		cr = cairo_create(surf_clock);
		/*
		 * Render layout
		 * (are used cairo operators ok ?)
		 */
		/* Draw background */
		cairo_set_source_rgba(cr,
			(float)prm->clock_bg_color.red / G_MAXUINT16,
			(float)prm->clock_bg_color.green / G_MAXUINT16,
			(float)prm->clock_bg_color.blue / G_MAXUINT16,
			(float)prm->clock_bg_color_alpha / G_MAXUINT16
			);
		cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
		cairo_paint(cr);
		/* Draw foreground */
		if (prm->shadow == 'y') {
			/* Draw shadow */
			if (prm->shadow_fx < 0)
				shadow_k = 1.0;
			else if (prm->shadow_fx > 10)
				shadow_k = 0.0;
			else
				shadow_k = 1.0 - (float)prm->shadow_fx / 10.0;
			cairo_set_source_rgba(cr,
				(float)prm->clock_bg_color.red * shadow_k / G_MAXUINT16,
				(float)prm->clock_bg_color.green * shadow_k / G_MAXUINT16,
				(float)prm->clock_bg_color.blue * shadow_k / G_MAXUINT16,
				(float)prm->clock_bg_color_alpha * shadow_k / G_MAXUINT16);
			pango_cairo_update_layout(cr, p_layout);
			cairo_move_to(cr, prm->shadow_offset_x, prm->shadow_offset_y);
			cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
			pango_cairo_show_layout(cr, p_layout);
		}
		/* Draw text */
		cairo_set_source_rgba(cr,
			(float)prm->clock_fg_color.red / G_MAXUINT16,
			(float)prm->clock_fg_color.green / G_MAXUINT16,
			(float)prm->clock_fg_color.blue / G_MAXUINT16,
			(float)prm->clock_fg_color_alpha / G_MAXUINT16);
		pango_cairo_update_layout(cr, p_layout);
		cairo_move_to(cr, 0, 0);
		cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
		pango_cairo_show_layout(cr, p_layout);
		/* Drawing done */
		if (p_layout != NULL)
			g_object_unref(p_layout);
		cairo_destroy(cr);
		/* Draw onto clock area
		 * (we now use cairo instead of deprecated gdk_draw_sth)
		 */
		cr = gdk_cairo_create(GDK_DRAWABLE((env->drwa_clock)->window));
		cairo_set_source_surface(cr, surf_clock, 0, h_diff);
		cairo_rectangle(cr, 0, 0, env->drwa_clock_width, env->drwa_height);
		cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
		cairo_fill(cr);
		cairo_destroy(cr);
		cairo_surface_destroy(surf_clock);
	}
}

/* Actually clock max width */
gint get_clock_width(const Params *prm)
{
	static char		clock_font_n_s_bak[FONT_MAXLEN + 1] = "";
	static gint		width = 0;
	PangoLayout		*p_layout;
	PangoFontDescription	*f_des;
	int			layout_width, layout_height, layout_max_width = 0;
	char			tmp[64];
	int			i;

	if (prm->clock == 'l' || prm->clock == 'r') {
		if (strcmp(prm->clock_font_n_s, clock_font_n_s_bak) != 0 || width == 0) {
			str_n_cpy(clock_font_n_s_bak, prm->clock_font_n_s, FONT_MAXLEN);

			p_layout = pango_layout_new(gtk_widget_get_pango_context(get_ticker_env()->win));
			pango_layout_set_attributes(p_layout, NULL);
			pango_layout_set_single_paragraph_mode(p_layout, TRUE);
			f_des = pango_font_description_from_string((const char *)prm->clock_font_n_s);
			pango_layout_set_font_description(p_layout, f_des);
			pango_font_description_free(f_des);

			for (i = '0'; i <= '9'; i++) {
				snprintf(tmp, TIME_STR_LENGTH + 1, TIME_STR, i, i, i, i, i, i);
				pango_layout_set_text(p_layout, tmp, -1);
				pango_layout_context_changed(p_layout);
				pango_layout_get_pixel_size2(p_layout, &layout_width, &layout_height);
				if (layout_width > layout_max_width)
					layout_max_width = layout_width;
			}

			if (p_layout != NULL)
				g_object_unref(p_layout);
			width = (gint)layout_max_width;
		}
	} else
		width = 0;	/* Return 0 although if clock = none, actual widget width = 1 */
	return width;
}
