/***************************************************************************
                             th-pixbufsink.c
                             ---------------
    begin                : Sun Apr 30 2006
    copyright            : (C) 2006 by Tim-Philipp Müller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "th-pixbufsink.h"

#include <gdk-pixbuf/gdk-pixbuf.h>

GST_BOILERPLATE (ThPixbufSink, th_pixbuf_sink, GstBaseSink, GST_TYPE_BASE_SINK);

static gboolean th_pixbuf_sink_start (GstBaseSink *basesink);
static gboolean th_pixbuf_sink_stop (GstBaseSink *basesink);
static gboolean th_pixbuf_sink_set_caps (GstBaseSink *basesink, GstCaps *caps);
static GstFlowReturn th_pixbuf_sink_render (GstBaseSink *bsink, GstBuffer *buf);
static GstFlowReturn th_pixbuf_sink_preroll (GstBaseSink *bsink, GstBuffer *buf);
static GdkPixbuf *th_pixbuf_sink_get_pixbuf_from_buffer (ThPixbufSink *sink,
    GstBuffer *buf);

#define SINK_CAPS \
  "video/x-raw-rgb, "             \
  "width = (int) [ 16, 4096 ], "  \
  "height = (int) [ 16, 4096 ], " \
  "bpp = (int) 24, "              \
  "depth = (int) 24, "            \
  "red_mask = (int) 16711680, "   \
  "green_mask = (int) 65280, "    \
  "blue_mask = (int) 255, "       \
  "endianness = (int) BIG_ENDIAN"

static GstStaticPadTemplate pixbufsink_sink_factory =
    GST_STATIC_PAD_TEMPLATE ("sink",
    GST_PAD_SINK,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS (SINK_CAPS));

static const GstElementDetails pixbufsink_details =
GST_ELEMENT_DETAILS ("GdkPixbuf sink",
    "Sink/Video",
    "Output GdkPixbuf structures in PAUSED and PLAYING state",
    "Tim-Philipp Müller <tim centricular net>");

static void
th_pixbuf_sink_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_set_details (element_class, &pixbufsink_details);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&pixbufsink_sink_factory));
}

static void
th_pixbuf_sink_class_init (ThPixbufSinkClass *klass)
{
  GstBaseSinkClass *basesink_class = GST_BASE_SINK_CLASS (klass);

  basesink_class->start = GST_DEBUG_FUNCPTR (th_pixbuf_sink_start);
  basesink_class->stop = GST_DEBUG_FUNCPTR (th_pixbuf_sink_stop);
  basesink_class->render = GST_DEBUG_FUNCPTR (th_pixbuf_sink_render);
  basesink_class->preroll = GST_DEBUG_FUNCPTR (th_pixbuf_sink_preroll);
  basesink_class->set_caps = GST_DEBUG_FUNCPTR (th_pixbuf_sink_set_caps);
}

static void
th_pixbuf_sink_init (ThPixbufSink *sink, ThPixbufSinkClass *klass)
{
  sink->width = 0;
  sink->height = 0;
  sink->par_n = 0;
  sink->par_d = 0;

  gst_base_sink_set_sync (GST_BASE_SINK (sink), TRUE);
  gst_base_sink_set_qos_enabled (GST_BASE_SINK (sink), FALSE);
}


static gboolean
th_pixbuf_sink_start (GstBaseSink *basesink)
{
  GST_LOG_OBJECT (basesink, "start");

  return TRUE;
}

static gboolean
th_pixbuf_sink_stop (GstBaseSink *basesink)
{
  ThPixbufSink *sink = TH_PIXBUF_SINK (basesink);

  sink->width = 0;
  sink->height = 0;
  sink->par_n = 0;
  sink->par_d = 0;

  GST_LOG_OBJECT (sink, "stop");

  return TRUE;
}

static gboolean
th_pixbuf_sink_set_caps (GstBaseSink *basesink, GstCaps *caps)
{
  ThPixbufSink *sink = TH_PIXBUF_SINK (basesink);
  GstStructure *s;

  s = gst_caps_get_structure (caps, 0);
  if (!gst_structure_get_int (s, "width", &sink->width) ||
      !gst_structure_get_int (s, "height", &sink->height)) {
    return FALSE;
  }

  if (!gst_structure_get_fraction (s, "pixel-aspect-ratio", &sink->par_n,
      &sink->par_d)) {
    sink->par_n = 1;
    sink->par_d = 1;
  }

  GST_INFO_OBJECT (sink, "width x height = %d x %d", sink->width, sink->height);
  GST_INFO_OBJECT (sink, "pixel-aspect-ratio = %d/%d", sink->par_d, sink->par_n);
  return TRUE;
}

static void
th_pixbuf_sink_pixbuf_destroy_notify (guchar *pixels, GstBuffer *buf)
{
  gst_buffer_unref (buf);
}

static GdkPixbuf *
th_pixbuf_sink_get_pixbuf_from_buffer (ThPixbufSink *sink, GstBuffer *buf)
{
  GdkPixbuf *pix = NULL;
  gint rowstride, minsize;

  g_return_val_if_fail (sink->width > 0, NULL);
  g_return_val_if_fail (sink->height > 0, NULL);

  rowstride = GST_ROUND_UP_4 (sink->width * 3);
  minsize = (rowstride * (sink->height - 1)) + (3 * sink->width);

  g_return_val_if_fail (GST_BUFFER_SIZE (buf) >= minsize, NULL);

  pix = gdk_pixbuf_new_from_data (GST_BUFFER_DATA (buf),
      GDK_COLORSPACE_RGB, FALSE, 8, sink->width, sink->height, rowstride,
      (GdkPixbufDestroyNotify) th_pixbuf_sink_pixbuf_destroy_notify,
      gst_buffer_ref (buf));

  return pix;
}

static gboolean
th_pixbuf_send_pixbuf (ThPixbufSink *sink, GstBuffer *buf, const gchar *name)
{
  GstStructure *s;
  GstMessage *msg;
  GdkPixbuf *pixbuf;

  pixbuf = th_pixbuf_sink_get_pixbuf_from_buffer (sink, buf);

  if (pixbuf == NULL)
    return FALSE;

  s = gst_structure_new (name,
      "pixbuf", GDK_TYPE_PIXBUF, pixbuf,
      "pixel-aspect-ratio", GST_TYPE_FRACTION, sink->par_d, sink->par_n,
      NULL);

  msg = gst_message_new_element (GST_OBJECT (sink), s);
  gst_element_post_message (GST_ELEMENT (sink), msg);

  g_object_unref (pixbuf); /* value in structure took a ref */

  return TRUE;
}

static GstFlowReturn
th_pixbuf_sink_preroll (GstBaseSink *basesink, GstBuffer *buf)
{
  if (th_pixbuf_send_pixbuf (TH_PIXBUF_SINK (basesink), buf, "preroll-pixbuf"))
    return GST_FLOW_OK;

  /* FIXME: post error with GST_ELEMENT_ERROR here */
  return GST_FLOW_ERROR;
}

static GstFlowReturn
th_pixbuf_sink_render (GstBaseSink *basesink, GstBuffer *buf)
{
  if (th_pixbuf_send_pixbuf (TH_PIXBUF_SINK (basesink), buf, "render-pixbuf"))
    return GST_FLOW_OK;

  /* FIXME: post error with GST_ELEMENT_ERROR here */
  return GST_FLOW_ERROR;
}
