/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test handle-mapper managers.
 * 
 * ti-handle-mapper.h:
 * Maps handle numbers to their respective data (like their names)
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-handle-mapper.h"
#include "ti-constants.h"
#include "ti-preferences.h"

struct _TIHandleMapperClass {
    GObjectClass parent;
};

G_DEFINE_TYPE (TIHandleMapper, ti_handle_mapper, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIHandleMapperPrivate {

    TIConnection* connection;
    TIPreferences* preferences;

    GtkListStore* handles_list_store;
    GtkListStore* contact_handle_list_store;
};
typedef struct _TIHandleMapperPrivate TIHandleMapperPrivate;

#define TI_HANDLE_MAPPER_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_HANDLE_MAPPER, TIHandleMapperPrivate))

/* Function prototypes */
static void _ti_handle_list_store_set_name (GtkListStore* list_store, guint handle_type, guint handle_number, const gchar* name, gboolean is_real);
static gboolean _ti_handle_list_store_get_row (GtkListStore* list_store, guint handle_type, guint handle_number, GtkTreeIter* iter);
static void _ti_handle_list_store_remove_row (GtkListStore* list_store, guint handle_type, guint handle_number);

/**
 * Drop all references to other objects.
 */
static void
ti_handle_mapper_dispose (GObject *object)
{
    TIHandleMapper* handle_mapper = TI_HANDLE_MAPPER (object);
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

    if (priv->connection != NULL)
    {
        g_object_unref (priv->connection);
        priv->connection = NULL;
    }

    if (priv->preferences != NULL)
    {
        g_object_unref (priv->preferences);
        priv->preferences = NULL;
    }

    if (priv->handles_list_store != NULL)
    {
        g_object_unref (priv->handles_list_store);
        priv->handles_list_store = NULL;
    }

    if (priv->contact_handle_list_store != NULL)
    {
        g_object_unref (priv->contact_handle_list_store);
        priv->contact_handle_list_store = NULL;
    }

    G_OBJECT_CLASS (ti_handle_mapper_parent_class)->dispose (object);
}

/**
 * Finalizes the object, marking the memory as ready for reuse
 */
/*static void
ti_handle_mapper_finalize (GObject *object)
{
    TIHandleMapper* handle_mapper = TI_HANDLE_MAPPER (object);
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

	G_OBJECT_CLASS (ti_handle_mapper_parent_class)->finalize (object);
}*/

/**
 * Class initialization.
 */
static void
ti_handle_mapper_class_init (TIHandleMapperClass* ti_handle_mapper_class)
{
	GObjectClass* gobject_class = G_OBJECT_CLASS (ti_handle_mapper_class);

	// Override base object methods
	gobject_class->dispose = ti_handle_mapper_dispose;
    //gobject_class->finalize = ti_handle_mapper_finalize;

	// Add private
	g_type_class_add_private (ti_handle_mapper_class, sizeof (TIHandleMapperPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_handle_mapper_init (TIHandleMapper* handle_mapper)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

    priv->preferences = ti_preferences_new ();

    priv->handles_list_store = gtk_list_store_new (TI_HANDLES_LIST_N_COLUMNS,
                                                   G_TYPE_UINT,     // type
                                                   G_TYPE_UINT,     // number
                                                   G_TYPE_STRING,   // name
                                                   G_TYPE_BOOLEAN); // is_real_name
    g_object_ref_sink (priv->handles_list_store);

    priv->contact_handle_list_store = gtk_list_store_new (TI_HANDLES_LIST_N_COLUMNS,
                                                          G_TYPE_UINT,    // type, always TI_CONNECTION_HANDLE_TYPE_CONTACT
                                                          G_TYPE_UINT,    // number
                                                          G_TYPE_STRING,   // name
                                                          G_TYPE_BOOLEAN); // is_real_name
    g_object_ref_sink (priv->contact_handle_list_store);
    
}

/**
 * Creates a new handle mapper
 */
TIHandleMapper*
ti_handle_mapper_new (TIConnection* connection)
{
    TIHandleMapper* handle_mapper = NULL;
    TIHandleMapperPrivate* priv = NULL;

    handle_mapper = g_object_new (TI_TYPE_HANDLE_MAPPER, NULL);
    priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);
    
    priv->connection = connection;
    g_object_ref (connection);

    return handle_mapper;
}

/**
 * Set Handle
 * Sets an unnamed handle
 */
void
ti_handle_mapper_set_handle (TIHandleMapper* handle_mapper, guint handle_type, guint handle_number)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);
    gchar* name;

    name = g_strdup_printf ("%u", handle_number);

    _ti_handle_list_store_set_name (priv->handles_list_store, handle_type, handle_number, name, FALSE);

    if (handle_type == TI_CONNECTION_HANDLE_TYPE_CONTACT)
    {
        _ti_handle_list_store_set_name (priv->contact_handle_list_store, handle_type, handle_number, name, FALSE);
    }

    g_free (name);
}

/**
 * Set Handle Name
 */
void
ti_handle_mapper_set_handle_name (TIHandleMapper* handle_mapper, guint handle_type, guint handle_number, const gchar* name)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

    _ti_handle_list_store_set_name (priv->handles_list_store, handle_type, handle_number, name, TRUE);

    if (handle_type == TI_CONNECTION_HANDLE_TYPE_CONTACT)
    {
        _ti_handle_list_store_set_name (priv->contact_handle_list_store, handle_type, handle_number, name, TRUE);
    }
}

/**
 * Set Contact Handle Name
 */
void
ti_handle_mapper_set_contact_handle_name (TIHandleMapper* handle_mapper, guint handle_number, const gchar* name)
{
    ti_handle_mapper_set_handle_name (handle_mapper, TI_CONNECTION_HANDLE_TYPE_CONTACT, handle_number, name);
}

/**
 * Set List Handle Name
 */
void
ti_handle_mapper_set_list_handle_name (TIHandleMapper* handle_mapper, guint handle_number, const gchar* name)
{
    ti_handle_mapper_set_handle_name (handle_mapper, TI_CONNECTION_HANDLE_TYPE_LIST, handle_number, name);
}

/**
 * Set Room Handle Name
 *
 */
void
ti_handle_mapper_set_room_handle_name (TIHandleMapper* handle_mapper, guint handle_number, const gchar* name)
{
    ti_handle_mapper_set_handle_name (handle_mapper, TI_CONNECTION_HANDLE_TYPE_ROOM, handle_number, name);
}

/**
 * Get Handle Name
 */
gchar*
ti_handle_mapper_get_handle_name (TIHandleMapper* handle_mapper, guint handle_type, guint handle_number)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);
    gboolean found;
    GtkTreeIter iter;
    gchar* name = NULL;
    gboolean is_real_name;
    GError* error = NULL;

    found = _ti_handle_list_store_get_row (priv->handles_list_store, handle_type, handle_number, &iter);
    if (!found)
    {
        if (ti_preferences_get_auto_inspect_handles (priv->preferences))
        {
            // Hold and inspect that handle

            ti_connection_hold_single_handle (priv->connection, handle_type, handle_number, &error);
            if (error != NULL)
            {
                goto CLEAN_UP;
            }

            name = ti_connection_inspect_single_handle (priv->connection, handle_type, handle_number, &error);
            if (error != NULL)
            {
                goto CLEAN_UP;
            }

            ti_handle_mapper_set_handle_name (handle_mapper, handle_type, handle_number, name);
        }

        goto CLEAN_UP;
    }

    gtk_tree_model_get (GTK_TREE_MODEL (priv->handles_list_store), &iter,
                        TI_HANDLES_LIST_COLUMN_NAME, &name,
                        TI_HANDLES_LIST_COLUMN_IS_REAL_NAME, &is_real_name,
                        -1);

    if (ti_preferences_get_auto_inspect_handles (priv->preferences) && !is_real_name)
    {
        // Go get its real name

        name = ti_connection_inspect_single_handle (priv->connection, handle_type, handle_number, &error);
        if (error != NULL)
        {
            goto CLEAN_UP;
        }

        gtk_list_store_set (priv->handles_list_store, &iter,
                            TI_HANDLES_LIST_COLUMN_NAME, &name,
                            TI_HANDLES_LIST_COLUMN_IS_REAL_NAME, TRUE,
                            -1);
    }

    CLEAN_UP:

    if (error != NULL)
        g_error_free (error);

    return name;
}

/**
 * Get Contact Handle Name
 */
gchar*
ti_handle_mapper_get_contact_handle_name (TIHandleMapper* handle_mapper, guint handle_number)
{
    return ti_handle_mapper_get_handle_name (handle_mapper, TI_CONNECTION_HANDLE_TYPE_CONTACT, handle_number);
}

/**
 * Get Room Handle Name
 */
gchar*
ti_handle_mapper_get_room_handle_name (TIHandleMapper* handle_mapper, guint handle_number)
{
    return ti_handle_mapper_get_handle_name (handle_mapper, TI_CONNECTION_HANDLE_TYPE_ROOM, handle_number);
}

/**
 * Get List Handle Name
 */
gchar*
ti_handle_mapper_get_list_handle_name (TIHandleMapper* handle_mapper, guint handle_number)
{
    return ti_handle_mapper_get_handle_name (handle_mapper, TI_CONNECTION_HANDLE_TYPE_LIST, handle_number);
}

/**
 * Invalidate Handle
 */
void
ti_handle_mapper_invalidate_handle (TIHandleMapper* handle_mapper, guint handle_type, guint handle_number)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

    _ti_handle_list_store_remove_row (priv->handles_list_store, handle_type, handle_number);

    if (handle_type == TI_CONNECTION_HANDLE_TYPE_CONTACT)
    {
        _ti_handle_list_store_remove_row (priv->contact_handle_list_store, handle_type, handle_number);
    }
}

/**
 * Invalidate Contact Handles
 */
void
ti_handle_mapper_invalidate_contact_handles (TIHandleMapper* handle_mapper, GArray* handle_numbers)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);
    guint i;
    guint handle_number;

    for (i = 0; i < handle_numbers->len; i++)
    {
        handle_number = g_array_index (handle_numbers, guint, i);

        _ti_handle_list_store_remove_row (priv->handles_list_store, TI_CONNECTION_HANDLE_TYPE_CONTACT, handle_number);
        _ti_handle_list_store_remove_row (priv->contact_handle_list_store, TI_CONNECTION_HANDLE_TYPE_CONTACT, handle_number);
    }
}

/**
 * Invalidate List Handles
 */
void
ti_handle_mapper_invalidate_list_handles (TIHandleMapper* handle_mapper, GArray* handle_numbers)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);
    guint i;
    guint handle_number;

    for (i = 0; i < handle_numbers->len; i++)
    {
        handle_number = g_array_index (handle_numbers, guint, i);

        _ti_handle_list_store_remove_row (priv->handles_list_store, TI_CONNECTION_HANDLE_TYPE_LIST, handle_number);
    }
}

/**
 * Set Handle List Store Name
 */
static void
_ti_handle_list_store_set_name (GtkListStore* list_store, guint handle_type, guint handle_number, const gchar* name, gboolean is_real)
{
    GtkTreeIter iter;
    gboolean found;

    found = _ti_handle_list_store_get_row (list_store, handle_type, handle_number, &iter);

    if (found)
    {
        gtk_list_store_set (list_store, &iter,
                            TI_HANDLES_LIST_COLUMN_NAME, name,
                            -1);
    }
    else
    {
        gtk_list_store_append (list_store, &iter);

        gtk_list_store_set (list_store, &iter,
                            TI_HANDLES_LIST_COLUMN_TYPE, handle_type,
                            TI_HANDLES_LIST_COLUMN_NUMBER, handle_number,
                            TI_HANDLES_LIST_COLUMN_NAME, name,
                            TI_HANDLES_LIST_COLUMN_IS_REAL_NAME, is_real,
                            -1);
    }
}

/**
 * Handle List Store Remove Row
 */
static void
_ti_handle_list_store_remove_row (GtkListStore* list_store, guint handle_type, guint handle_number)
{
    GtkTreeIter iter;
    gboolean found;

    found = _ti_handle_list_store_get_row (list_store, handle_type, handle_number, &iter);

    if (found)
    {
        gtk_list_store_remove (list_store, &iter);
    }
}

/**
 * Get Handles List Store
 */
GtkListStore*
ti_handle_mapper_get_handles_list_store (TIHandleMapper* handle_mapper)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

    g_object_ref (priv->handles_list_store);

    return priv->handles_list_store;
}

/**
 * Get Contact Handle List Store
 */
GtkListStore*
ti_handle_mapper_get_contact_handle_list_store (TIHandleMapper* handle_mapper)
{
    TIHandleMapperPrivate* priv = TI_HANDLE_MAPPER_GET_PRIVATE (handle_mapper);

    g_object_ref (priv->contact_handle_list_store);

    return priv->contact_handle_list_store;
}

/**
 * Handle List Store Get Row
 *
 * @return TRUE if the row was found and FALSE otherwise.
 */
static gboolean
_ti_handle_list_store_get_row (GtkListStore* list_store, guint handle_type, guint handle_number, GtkTreeIter* iter)
{
    gboolean ok;
    gboolean found = FALSE;
    guint row_handle_type;
    guint row_handle_number;

    ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (list_store), iter);
    while (ok && !found)
    {
        gtk_tree_model_get (GTK_TREE_MODEL (list_store), iter,
                            TI_HANDLES_LIST_COLUMN_TYPE, &row_handle_type,
                            TI_HANDLES_LIST_COLUMN_NUMBER, &row_handle_number,
                            -1);

        if (handle_type == row_handle_type && handle_number == row_handle_number)
        {
            found = TRUE;
        }
        else
        {
            ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (list_store), iter);
        }
    }

    return found;
}
