/*
 * Telapathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 * 
 * ti-dlg-get-text.c:
 * "Get Text" Dialog
 * 
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Author - Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "ti-dlg-get-text.h"
#include "ti-config.h"

#include <glade/glade.h>

G_DEFINE_TYPE (TIDlgGetText, ti_dlg_get_text, G_TYPE_OBJECT);

/**
 * Instance private data.
 */
struct _TIDlgGetTextPrivate {
    gboolean disposed;

    GtkWindow* parent;

    GladeXML* glade_xml;

    GtkWidget* dialog;

    GtkTextBuffer* text_buffer;
};
typedef struct _TIDlgGetTextPrivate TIDlgGetTextPrivate;

#define TI_DLG_GET_TEXT_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_DLG_GET_TEXT, TIDlgGetTextPrivate))

/* Function prototypes */
static void _button_ok_clicked (GtkButton *button, gpointer user_data);
static void _button_cancel_clicked (GtkButton *button, gpointer user_data);

/**
 * Drop all references to other objects.
 */
static void
ti_dlg_get_text_dispose (GObject *object)
{
    TIDlgGetText *dlg_get_text = TI_DLG_GET_TEXT (object);
    TIDlgGetTextPrivate *priv = TI_DLG_GET_TEXT_GET_PRIVATE (dlg_get_text);

    if (priv->disposed) {
        return;
    } else {
        priv->disposed = TRUE;
    }

    if (priv->glade_xml != NULL) {
        g_object_unref (priv->glade_xml);
        priv->glade_xml = NULL;
    }

    G_OBJECT_CLASS (ti_dlg_get_text_parent_class)->dispose (object);
}

/**
 * Class initialization.
 */
static void
ti_dlg_get_text_class_init (TIDlgGetTextClass *ti_dlg_get_text_class)
{
	GObjectClass *gobject_class = G_OBJECT_CLASS (ti_dlg_get_text_class);

	/* override base object methods */ 
	gobject_class->dispose = ti_dlg_get_text_dispose;
		
	/* Add private */
	g_type_class_add_private (ti_dlg_get_text_class, sizeof (TIDlgGetTextPrivate));
}

/**
 * Instance initialization.
 */
static void
ti_dlg_get_text_init (TIDlgGetText *ti_dlg_get_text)
{
    TIDlgGetTextPrivate *priv = TI_DLG_GET_TEXT_GET_PRIVATE (ti_dlg_get_text);

    priv->disposed = FALSE;
}

/**
 * Returns a new instance.
 */
TIDlgGetText*
ti_dlg_get_text_new (GtkWindow* parent, const gchar* caption) 
{
    TIDlgGetText* dialog = NULL;
    GtkFrame* frame = NULL;
    TIDlgGetTextPrivate* priv = NULL;
    gchar* glade_file_path = NULL;
    GtkTextView* text_view = NULL;

	dialog = g_object_new (TI_TYPE_DLG_GET_TEXT, NULL);
    priv = TI_DLG_GET_TEXT_GET_PRIVATE (dialog);

    priv->parent = parent;

    glade_file_path = g_strdup_printf ("%s%s", TI_DATA_DIR_PREFIX, "dlg-get-text.xml");
    priv->glade_xml = glade_xml_new (glade_file_path, NULL, NULL);
    if (priv->glade_xml == NULL)
    {
        g_critical ("Error loading glade file \"%s\".", glade_file_path);
        g_object_unref (dialog);
        dialog = NULL;
        goto CLEAN_UP;
    }

    priv->dialog = glade_xml_get_widget (priv->glade_xml, "dialog_get_text");
    g_assert (GTK_IS_DIALOG (priv->dialog));

    text_view = GTK_TEXT_VIEW (glade_xml_get_widget (priv->glade_xml, "textview_text"));
    g_assert (GTK_IS_TEXT_VIEW (text_view));
    
    priv->text_buffer = gtk_text_view_get_buffer (text_view);

    gtk_window_set_transient_for (GTK_WINDOW (priv->dialog), GTK_WINDOW (priv->parent));
    gtk_window_set_position (GTK_WINDOW (priv->dialog), GTK_WIN_POS_CENTER_ON_PARENT);

    glade_xml_signal_connect_data (priv->glade_xml, "button_ok_clicked", G_CALLBACK(_button_ok_clicked), priv->dialog);
    glade_xml_signal_connect_data (priv->glade_xml, "button_cancel_clicked", G_CALLBACK(_button_cancel_clicked), priv->dialog);

    frame = GTK_FRAME (glade_xml_get_widget (priv->glade_xml, "frame_text"));
    g_assert (GTK_IS_FRAME (frame));
    
    gtk_frame_set_label (frame, caption);

    CLEAN_UP:
    g_free (glade_file_path);

    return dialog;
}

/**
 * Run
 */
gboolean ti_dlg_get_text_run (TIDlgGetText* dlg_get_text, gchar** text)
{
    TIDlgGetTextPrivate *priv = TI_DLG_GET_TEXT_GET_PRIVATE (dlg_get_text);
    gint result;
    GtkTextIter start_iter;
    GtkTextIter end_iter;

    g_assert (text != NULL);

    gtk_widget_show_all (priv->dialog);
    result = gtk_dialog_run (GTK_DIALOG (priv->dialog));
    gtk_widget_hide_all (priv->dialog);

    gtk_text_buffer_get_start_iter  (priv->text_buffer, &start_iter);
    gtk_text_buffer_get_end_iter  (priv->text_buffer, &end_iter);
    
    *text = gtk_text_buffer_get_text (priv->text_buffer, &start_iter, &end_iter, FALSE);

    return result == GTK_RESPONSE_OK;
}

/**
 * Called when OK button is clicked.
 */
static void
_button_ok_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_OK);
}

/**
 * Called when "Cancel" button is clicked.
 */
static void
_button_cancel_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog* dialog = GTK_DIALOG (user_data);
    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}
