/*
 * hash.h:  definition of and support fns for tb_hash_t type
 *
 * Copyright (c) 2006-2007, Intel Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *   * Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above
 *     copyright notice, this list of conditions and the following
 *     disclaimer in the documentation and/or other materials provided
 *     with the distribution.
 *   * Neither the name of the Intel Corporation nor the names of its
 *     contributors may be used to endorse or promote products derived
 *     from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS
 * FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE
 * COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED
 * OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __HASH_H__
#define __HASH_H__

#define TB_HALG_SHA1    0

#ifndef SHA1_LENGTH
#define SHA1_LENGTH        20
#endif
#ifndef SHA256_LENGTH
#define SHA256_LENGTH      32
#endif

typedef union {
    uint8_t    sha1[SHA1_LENGTH];
    uint8_t    sha256[SHA256_LENGTH];
} tb_hash_t;


static inline const char *hash_alg_to_string(uint8_t hash_alg)
{
    if ( hash_alg == TB_HALG_SHA1 )
        return "TB_HALG_SHA1";
    else {
        static char buf[32];
        snprintf(buf, sizeof(buf), "unsupported (%u)", hash_alg);
        return buf;
    }
}


static inline unsigned int get_hash_size(uint8_t hash_alg)
{
    return (hash_alg == TB_HALG_SHA1) ? SHA1_LENGTH : 0;
}

extern bool are_hashes_equal(const tb_hash_t *hash1, const tb_hash_t *hash2,
                             uint8_t hash_alg);
extern bool hash_buffer(const unsigned char* buf, size_t size, tb_hash_t *hash,
                        uint8_t hash_alg);
extern bool extend_hash(tb_hash_t *hash1, const tb_hash_t *hash2,
                        uint8_t hash_alg);
extern void print_hash(const tb_hash_t *hash, uint8_t hash_alg);
extern void copy_hash(tb_hash_t *dest_hash, const tb_hash_t *src_hash,
                      uint8_t hash_alg);


#endif    /* __HASH_H__ */


/*
 * Local variables:
 * mode: C
 * c-set-style: "BSD"
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: nil
 * End:
 */
