/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "config.h"

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#include <glib-object.h>
#include <stdlib.h>
#include <string.h>

#include "tpa-media.h"

#include "tpa-media-private.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CHANNEL

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-errors.h>

/* signal enum */
enum
{
    STREAM_ADDED,
    STREAM_REMOVED,
    STREAM_STATE_CHANGED,
    LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = {0};

struct _TpaMediaPrivate {
    GObject *object;
};

/* we need to define the get_type function */
GType
tpa_media_get_type()
{
    static GType object_type = 0;

    if (!object_type) {
        static const GTypeInfo object_info = {
            sizeof(TpaIMedia),
            NULL,   /* base init */
            NULL,   /* base finalize */
        };
        object_type =
            g_type_register_static(G_TYPE_INTERFACE,
                "TpaMedia",
                &object_info, 0);
    }
    return object_type;
}

void
tpa_media_init (TpaIMedia *iface,
                gpointer data)
{
    VERBOSE ("(%p, %p)", iface, data);

    iface->list_streams = NULL;
    iface->request_streams = NULL;

    /* Interface signals */
    signals[STREAM_ADDED] =
        g_signal_new ("stream-added",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__UINT_UINT_UINT,
                      G_TYPE_NONE, 3, G_TYPE_UINT, G_TYPE_UINT, G_TYPE_UINT);

    signals[STREAM_REMOVED] =
        g_signal_new ("stream-removed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      g_cclosure_marshal_VOID__UINT,
                      G_TYPE_NONE, 1, G_TYPE_UINT);

    signals[STREAM_STATE_CHANGED] =
        g_signal_new ("stream-state-changed",
                      G_OBJECT_CLASS_TYPE (iface),
                      G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
                      0,
                      NULL, NULL,
                      tpa_marshal_VOID__UINT_UINT,
                      G_TYPE_NONE, 2, G_TYPE_UINT, G_TYPE_UINT);

    iface->priv = g_new0 (TpaIMediaPrivate, 1);
    VERBOSE ("private members %p", iface->priv);
}

void
tpa_media_finalize (GObject *obj)
{
    TpaIMedia *iface = TPA_IMEDIA (obj);
    VERBOSE ("(%p)", obj);

    if (iface->priv)
        g_free (iface->priv);
}

/**
 * tpa_streamed_media_list_streams
 *
 * Implements DBus method ListStreams
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_streamed_media_list_streams (GObject *obj,
                                 GPtrArray **ret,
                                 GError **error)
{
    TpaIMedia *iface = TPA_IMEDIA (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, ret);

    g_return_error_val_if_fail (iface->list_streams != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->list_streams (obj, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_streamed_media_request_streams
 *
 * Implements DBus method RequestStreams
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_streamed_media_request_streams (GObject *obj,
                                    guint contact_handle,
                                    const GArray *types,
                                    GArray **ret,
                                    GError **error)
{
    TpaIMedia *iface = TPA_IMEDIA (obj);
    TpaError error_code = TPA_ERROR_NONE;
    gchar *contact = NULL;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %s, %p, %p)", obj, contact, types, ret);

    g_return_error_val_if_fail (iface->request_streams != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->request_streams (obj, contact, types, ret);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_streamed_media_request_streams
 *
 * Implements DBus method RequestStreams
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 *
 * @error: Used to return a pointer to a GError detailing any error
 *         that occured, DBus will throw the error only if this
 *         function returns false.
 *
 * @return: TRUE if successful, FALSE if an error was thrown.
 */
gboolean
tpa_streamed_media_remove_streams (GObject *obj,
                                   GArray *stream_list,
                                   GError **error)
{
    TpaIMedia *iface = TPA_IMEDIA (obj);
    TpaError error_code = TPA_ERROR_NONE;

    g_return_error_val_if_fail (iface != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    VERBOSE ("(%p, %p)", obj, stream_list);

    g_return_error_val_if_fail (iface->remove_streams != NULL, error, TPA_ERROR_NOT_IMPLEMENTED);

    error_code = iface->remove_streams (obj, stream_list);

    g_return_error_val_if_fail (error_code == TPA_ERROR_NONE, error, error_code);

    return TRUE;
}

/**
 * tpa_media_signal_stream_added
 *
 * Implements DBus signal StreamAdded
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 */
void
tpa_media_signal_stream_added (GObject *obj,
                               guint stream_id,
                               guint contact_id,
                               guint stream_type)
{
    g_assert (TPA_IS_IMEDIA (obj));

    VERBOSE ("(%p, %d, %d, %d)", obj, stream_id, contact_id, stream_type);

    g_signal_emit (obj, signals[STREAM_ADDED], 0, stream_id, contact_id, stream_type);
}

/**
 * tpa_media_signal_stream_removed
 *
 * Implements DBus signal StreamRemoved
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 */
void
tpa_media_signal_stream_removed (GObject *obj,
                                 guint stream_id)
{
    g_assert (TPA_IS_IMEDIA (obj));

    VERBOSE ("(%p, %d)", obj, stream_id);

    g_signal_emit (obj, signals[STREAM_REMOVED], 0, stream_id);
}

/**
 * tpa_media_signal_stream_state_changed
 *
 * Implements DBus signal StreamStateChanged
 * on interface org.freedesktop.Telepathy.Channel.Type.StreamedMedia
 */
void
tpa_media_signal_stream_state_changed (GObject *obj,
                                       guint stream_id,
                                       guint stream_state)
{
    g_assert (TPA_IS_IMEDIA (obj));

    VERBOSE ("(%p, %d, %d)", obj, stream_id, stream_state);

    g_signal_emit (obj, signals[STREAM_STATE_CHANGED], 0, stream_id, stream_state);
}
