/*
 * Tapioca library
 * Copyright (C) 2006 INdT.
 * @author  Abner Jose de Faria Silva <abner.silva@indt.org.br>
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 * @author  Marcio Macedo <marcio.macedo@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with self library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <string.h>

#include "tpa-contact-base-priv.h"
#include "tpa-channel-target-priv.h"

#define DEBUG_DOMAIN TPA_DOMAIN_CONNECTION

#include <tapioca/base/tpa-signals-marshal.h>
#include <tapioca/base/tpa-debug.h>
#include <tapioca/base/tpa-connection-bindings.h>

struct _TpaContactBasePrivate
{
    TpaAvatar *avatar;
    gchar *alias;
    TpaContactPresence presence;
    TpaCapability capabilities;
    gchar *presence_msg;
    gboolean receive_avatar_updates;
    gboolean disposed;
    DBusGProxy *proxy_aliasing;
    DBusGProxy *proxy_avatars;
    DBusGProxy *proxy_capabilities;
    DBusGProxy *proxy_presence;
};

enum
{
    ALIAS_CHANGED,
    AVATAR_UPDATED,
    AVATAR_RECEIVED,
    CAPABILITIES_CHANGED,
    PRESENCE_UPDATED,
    LAST_SIGNAL
};

static guint tpa_contact_base_signals[LAST_SIGNAL] = { 0 };

static GObject *        tpa_contact_base_constructor        (GType type,
                                                             guint n_construct_properties,
                                                             GObjectConstructParam *construct_properties);
void                    tpa_contact_base_update_avatar      (TpaContactBase *self,
                                                             const gchar* new_token);
void                    proxy_avatar_received               (DBusGProxy *proxy,
                                                             GArray *OUT_arg1,
                                                             char * OUT_arg2,
                                                             GError *error,
                                                             gpointer userdata);

G_DEFINE_TYPE(TpaContactBase, tpa_contact_base, TPA_TYPE_CHANNEL_TARGET)

static TpaContactPresence
_str_to_presence_enum (const gchar *presence_str)
{
    if (g_str_equal(presence_str, "available"))
        return TPA_PRESENCE_AVAILABLE;
    if (g_str_equal(presence_str, "away") ||
             g_str_equal(presence_str, "brb"))
        return TPA_PRESENCE_AWAY;
    if (g_str_equal(presence_str, "xa"))
         return TPA_PRESENCE_XA;
    if (g_str_equal(presence_str, "busy") ||
             g_str_equal(presence_str, "dnd"))
        return TPA_PRESENCE_BUSY;
    if (g_str_equal(presence_str, "hidden"))
        return TPA_PRESENCE_HIDDEN;
    else
        return TPA_PRESENCE_OFFLINE;
}

static const gchar *
_presence_enum_to_str (TpaContactPresence presence)
{
    switch (presence) {
        case TPA_PRESENCE_AVAILABLE:
            return "available";
            break;
        case TPA_PRESENCE_AWAY:
            return "away";
            break;
        case TPA_PRESENCE_XA:
            return "extended away";
            break;
        case TPA_PRESENCE_BUSY:
            return "busy";
            break;
        case TPA_PRESENCE_HIDDEN:
            return "hidden";
            break;
        case TPA_PRESENCE_OFFLINE:
            return "offline";
            break;
    }
    return NULL;
}

void
proxy_avatar_received (DBusGProxy *proxy,
                       GArray *image,
                       gchar *mimetype,
                       GError *error,
                       gpointer userdata)
{
    TpaContactBase *self = TPA_CONTACT_BASE (userdata);
    const gchar *uri;

    if (error) {
            ERROR ("%s", error->message);
            g_error_free (error);
            return;
    }

    if (DEBUGGING) {
        uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
        INFO ("[avatar-received %s] %p", uri, self->priv->avatar);
    }

    g_object_set (self->priv->avatar, "image", image, "mimetype", mimetype, NULL);
    g_signal_emit (self, tpa_contact_base_signals[AVATAR_RECEIVED], 0, self->priv->avatar);
}

static void
tpa_contact_base_dispose (GObject *object)
{
    TpaContactBase *self = TPA_CONTACT_BASE(object);

    if (self->priv->disposed)
    /* If dispose did already run, return. */
       return;

    /* Make sure dispose does not run twice. */
    self->priv->disposed = TRUE;

    if(self->priv->alias)
        g_free (self->priv->alias);
    if(self->priv->avatar)
        g_object_unref (self->priv->avatar);
    if(self->priv->presence_msg)
        g_free (self->priv->presence_msg);
    self->priv->presence = TPA_PRESENCE_OFFLINE;

    G_OBJECT_CLASS (tpa_contact_base_parent_class)->dispose (object);
}

static void
tpa_contact_base_class_init (TpaContactBaseClass *klass)
{
    GObjectClass *gobject_class;
    TpaChannelTargetClass *tpa_channel_target_class;

    gobject_class = (GObjectClass *) klass;
    tpa_channel_target_class = (TpaChannelTargetClass *) (klass);
    tpa_contact_base_parent_class = g_type_class_peek_parent (klass);

    g_type_class_add_private (klass, sizeof (TpaContactBasePrivate));

    gobject_class->constructor = tpa_contact_base_constructor;

    gobject_class->dispose = tpa_contact_base_dispose;

    /* A monster GType for presence - a{u(ua{sa{sv}})} */
    klass->g_type_presence_update = (dbus_g_type_get_map ("GHashTable", G_TYPE_UINT,
                                    (dbus_g_type_get_struct ("GValueArray", G_TYPE_UINT,
                                    (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING,
                                    (dbus_g_type_get_map ("GHashTable", G_TYPE_STRING, G_TYPE_VALUE)))),
                                    G_TYPE_INVALID))));

    /* Let's create our signals */
    tpa_contact_base_signals [ALIAS_CHANGED] = g_signal_new ("alias-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        g_cclosure_marshal_VOID__STRING,
        G_TYPE_NONE,
        1,
        G_TYPE_STRING);

    tpa_contact_base_signals [CAPABILITIES_CHANGED] = g_signal_new ("capabilities-changed",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        g_cclosure_marshal_VOID__UINT,
        G_TYPE_NONE,
        1,
        G_TYPE_UINT);

    tpa_contact_base_signals [PRESENCE_UPDATED] = g_signal_new ("presence-updated",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        tpa_marshal_VOID__UINT_STRING,
        G_TYPE_NONE,
        2,
        G_TYPE_UINT,
        G_TYPE_STRING);

    tpa_contact_base_signals [AVATAR_UPDATED] = g_signal_new ("avatar-updated",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        g_cclosure_marshal_VOID__STRING,
        G_TYPE_NONE,
        1,
        G_TYPE_STRING);

    tpa_contact_base_signals [AVATAR_RECEIVED] = g_signal_new ("avatar-received",
        G_TYPE_FROM_CLASS (klass),
        G_SIGNAL_RUN_LAST | G_SIGNAL_DETAILED,
        0,
        NULL, NULL,
        g_cclosure_marshal_VOID__OBJECT,
        G_TYPE_NONE,
        1,
        TPA_TYPE_AVATAR);
}

static void
tpa_contact_base_init (TpaContactBase *self)
{
    self->priv = TPA_CONTACT_BASE_GET_PRIVATE (self);
    self->priv->avatar = NULL;
    self->priv->alias = NULL;
    self->priv->presence = TPA_PRESENCE_OFFLINE;
    self->priv->capabilities = TPA_CAPABILITY_NONE;
    self->priv->presence_msg = NULL;
    self->priv->receive_avatar_updates = FALSE;
    self->priv->disposed = FALSE;
    self->priv->proxy_aliasing = NULL;
    self->priv->proxy_avatars = NULL;
    self->priv->proxy_capabilities = NULL;
    self->priv->proxy_presence = NULL;
}

static GObject *
tpa_contact_base_constructor (GType type,
                              guint n_construct_properties,
                              GObjectConstructParam *construct_properties)
{
    GObject *obj;
    TpaContactBase *self;
    TpaObject *object;
    TpaHandle *handle;
    GArray *handles;
    GPtrArray *ret;
    GError *error = NULL;
    {
        /* Invoke parent constructor. */
        TpaContactBaseClass *klass;
        GObjectClass *parent_class;
        klass = TPA_CONTACT_BASE_CLASS (g_type_class_peek (TPA_TYPE_CONTACT_BASE));
        parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
        obj = parent_class->constructor (type,
                                         n_construct_properties,
                                         construct_properties);
    }

    VERBOSE ("(%s, %d, %p)", g_type_name (type), n_construct_properties, construct_properties);
    self = TPA_CONTACT_BASE (obj);
    object = TPA_OBJECT (self);

    self->priv->proxy_aliasing =  tpa_object_get_proxy (object, TPA_INTERFACE_ALIASING);
    self->priv->proxy_avatars = tpa_object_get_proxy (object, TPA_INTERFACE_AVATARS);
    self->priv->proxy_capabilities = tpa_object_get_proxy (object, TPA_INTERFACE_CAPABILITIES);
    self->priv->proxy_presence = tpa_object_get_proxy (object, TPA_INTERFACE_PRESENCE);

    tpa_contact_base_receive_avatar_updates (self, TRUE, "");

    /**
     * If manager do not support capabilities interface set contact capabilities
     * to support all.
     */
    if (!self->priv->proxy_capabilities)
        self->priv->capabilities = TPA_CAPABILITY_ALL;
    else {
        handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (self));
        guint id = tpa_handle_get_id (handle);

        handles = g_array_new (FALSE, FALSE, sizeof (guint));
        g_array_append_val (handles, id);

        if (!org_freedesktop_Telepathy_Connection_Interface_Capabilities_get_capabilities (self->priv->proxy_capabilities, handles, &ret, &error) ||
            error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else {
            guint i;

            for (i = 0; i < ret->len; i++) {
                GValueArray *values = g_ptr_array_index (ret, i);
                gchar *type = g_value_dup_string (g_value_array_get_nth (values, 1));
                guint flags = g_value_get_uint (g_value_array_get_nth (values, 3));

                if (g_str_equal (type, TPA_INTERFACE_STREAMED_MEDIA)) {
                    if (flags & 1 && flags & 2)
                        tpa_contact_base_capabilities_changed (self, TPA_CAPABILITY_AUDIO | TPA_CAPABILITY_VIDEO);
                    else if (flags & 1)
                        tpa_contact_base_capabilities_changed (self, TPA_CAPABILITY_AUDIO);
                    else if (flags & 2)
                        tpa_contact_base_capabilities_changed (self, TPA_CAPABILITY_VIDEO);
                }
                g_free (type);
                g_value_array_free (values);
            }
            g_ptr_array_free (ret, TRUE);
        }
    }

    VERBOSE ("return %p", obj);
    return obj;
}


/**
 * tpa_contact_base_get_avatar:
 * @self: #TpaContactBase instance
 *
 * Get contact's avatar.
 *
 * WARNING: This functions blocks until the avatar
 * is received.
 */
TpaAvatar *
tpa_contact_base_get_avatar (TpaContactBase *self)
{
    TpaHandle *handle;
    guint handle_id;
    GArray *image;
    gchar *mimetype;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!self->priv->proxy_avatars)
        return NULL;

    handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET(self));
    handle_id = tpa_handle_get_id (handle);

    if (!org_freedesktop_Telepathy_Connection_Interface_Avatars_request_avatar (self->priv->proxy_avatars, handle_id,
        &image, &mimetype, &error)
        || error) {
        ERROR ("%s", error->message);
        g_error_free (error);
    }

    g_object_set (self->priv->avatar, "image", image, "mimetype", mimetype, NULL);

    VERBOSE ("return %p", self->priv->avatar);
    return self->priv->avatar;
}

/**
 * tpa_contact_base_request_avatar:
 * @self: #TpaContactBase instance
 *
 * Request contact's avatar.
 *
 * WARNING: This functions spams threads and
 * has a limit of 32 pending request.
 */
gboolean
tpa_contact_base_request_avatar (TpaContactBase *self)
{
    TpaHandle *handle;
    guint handle_id;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!self->priv->proxy_avatars)
        return FALSE;

    handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET(self));
    handle_id = tpa_handle_get_id (handle);

    org_freedesktop_Telepathy_Connection_Interface_Avatars_request_avatar_async (self->priv->proxy_avatars, handle_id,
        proxy_avatar_received, self);

    return TRUE;
}

/**
 * tpa_contact_base_get_presence:
 * @self: #TpaContactBase instance
 * @returns: #TpaContactPresence presence.
 *
 * Get contacts presence status.
 */
TpaContactPresence
tpa_contact_base_get_presence (TpaContactBase *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %d", self->priv->presence);
    return self->priv->presence;
}

/**
 * tpa_contact_base_get_presence_as_string:
 * @self: #TpaContactBase instance
 * @returns: presence string.
 *
 * Get contacts presence status.
 */
const gchar *
tpa_contact_base_get_presence_as_string (TpaContactBase *self)
{
    const gchar *presence;
    VERBOSE ("(%p)", self);
    g_assert (self);

    presence = _presence_enum_to_str (self->priv->presence);
    VERBOSE ("return %s", presence);
    return presence;
}

/**
 * tpa_contact_base_get_presence_message:
 * @self: #TpaContactBase instance
 * @returns: presence message.
 *
 * Get contacts presence message.
 */
const gchar*
tpa_contact_base_get_presence_message (TpaContactBase *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %s", self->priv->presence_msg);
    return self->priv->presence_msg;
}

/**
 * tpa_contact_base_get_alias:
 * @self: #TpaContactBase instance
 * @returns: contact alias.
 *
 * Get contacts alias.
 */
const gchar*
tpa_contact_base_get_alias (TpaContactBase *self)
{
    GArray *handles;
    TpaHandle *handle;
    guint id;
    gchar **aliases;
    GError *error = NULL;

    VERBOSE ("(%p)", self);
    g_assert (self);

    if (!self->priv->alias) {
        handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET (self));
        id = tpa_handle_get_id (handle);
        handles = g_array_new (FALSE, FALSE, sizeof (guint));
        g_array_append_val (handles, id);
        if (!org_freedesktop_Telepathy_Connection_Interface_Aliasing_request_aliases (self->priv->proxy_aliasing, handles, &aliases, &error)
            || error) {
            ERROR ("%s", error->message);
            g_error_free (error);
        }
        else if (aliases[0]) {
            self->priv->alias = g_strdup (aliases[0]);
            g_strfreev (aliases);
        }
        g_array_free (handles, TRUE);
    }

    VERBOSE ("return %s", self->priv->alias);
    return self->priv->alias;
}

/**
 * tpa_contact_base_get_capabilities:
 * @self: #TpaContactBase instance
 * @returns: #TpaCapability capability.
 *
 * Get contacts capabilities.
 */
TpaCapability
tpa_contact_base_get_capabilities (TpaContactBase *self)
{
    VERBOSE ("(%p)", self);
    g_assert (self);

    VERBOSE ("return %d", self->priv->capabilities);
    return self->priv->capabilities;
}

const gchar *
tpa_contact_base_get_avatar_token (TpaContactBase *self)
{
    const gchar *token;

    VERBOSE ("(%p)", self);
    g_assert (self);

    token = tpa_avatar_get_token (self->priv->avatar);
    VERBOSE ("return %s", token);
    return token;
}

/**
 * tpa_contact_base_receive_avatar_update:
 * @self: #TpaContactBase instance
 * @enabled:
 * @token:
 *
 * Set wheter the contact updates or not the avatar.
 */
void
tpa_contact_base_receive_avatar_updates (TpaContactBase *self,
                                         gboolean enabled,
                                         const gchar *token)
{
    VERBOSE ("(%p, %s, %s)", self, enabled ? "TRUE" : "FALSE", token);

    if (!self->priv->proxy_avatars) {
        if (!(self->priv->proxy_avatars = tpa_object_get_proxy (TPA_OBJECT (self), TPA_INTERFACE_AVATARS))) {
            return;
        }
    }

    self->priv->receive_avatar_updates = enabled;

    if (!self->priv->avatar)
        self->priv->avatar = tpa_avatar_new (NULL, NULL, NULL);

    /*if (enabled)                                     */
    /*    tpa_contact_base_update_avatar (self, token);*/

    VERBOSE ("return");
}

/**
 * tpa_contact_base_update_avatar:
 * @self: #TpaContactBase instance
 * @new_token: new avatar token
 *
 * Updates contacts avatar token.
 */
void
tpa_contact_base_update_avatar (TpaContactBase *self,
                                const gchar *new_token)
{
    GError *error = NULL;
    TpaHandle *handle;
    guint handle_id;
    const gchar *mimetype;
    const GArray *image;

    VERBOSE ("(%p, %s)", self, new_token);
    g_assert (self);

    handle = tpa_channel_target_get_handle (TPA_CHANNEL_TARGET(self));
    handle_id = tpa_handle_get_id (handle);

    if (!new_token || strlen (new_token) == 0) {
        GArray *handle_list;
        gchar **token_list;

        handle_list = g_array_new (TRUE, TRUE, sizeof(guint));
        g_array_append_val (handle_list, handle_id);

        if (!org_freedesktop_Telepathy_Connection_Interface_Avatars_get_avatar_tokens (self->priv->proxy_avatars,
                                                                                  handle_list,
                                                                                  &token_list,
                                                                                  &error))
        {
            ERROR ("%s", error->message);
            g_error_free (error);
        }

        g_object_set (self->priv->avatar, "token", token_list[0], NULL);
        g_array_free (handle_list, TRUE);
        g_strfreev (token_list);
    }
    else
        g_object_set (self->priv->avatar, "token", new_token, NULL);

    /* clean previous avatar information */
    if ((mimetype = tpa_avatar_get_mime_type (self->priv->avatar)))
         g_free ((gchar *)mimetype);
    if ((image = tpa_avatar_get_image (self->priv->avatar)))
        g_array_free ((GArray *)image, TRUE);

    VERBOSE ("return");
}

/* Internal helpers */

void
tpa_contact_base_alias_changed (TpaContactBase *self,
                                const gchar *alias)
{
    const gchar *uri;

    VERBOSE ("(%p, %s)", self, alias);
    g_assert (self);

    if (DEBUGGING) {
        uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
        INFO ("[aliases-changed %s] %s", uri, alias);
    }

    if (self->priv->alias)
        g_free (self->priv->alias);

    self->priv->alias = g_strdup (alias);
    g_signal_emit (self, tpa_contact_base_signals[ALIAS_CHANGED], 0, alias);

    VERBOSE ("return");
}

void
tpa_contact_base_presence_updated (TpaContactBase *self,
                                   const gchar *presence,
                                   const gchar *message)
{
    const gchar *uri;
    TpaContactPresence status;

    VERBOSE ("(%p, %s, %s)", self, presence, message);
    g_assert (self);

    status = _str_to_presence_enum (presence);
    self->priv->presence = status;

    if (self->priv->presence_msg)
        g_free (self->priv->presence_msg);
    self->priv->presence_msg = g_strdup (message);

    if (DEBUGGING) {
        uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
        INFO ("[presence-update %s] %s %s", uri, presence, message);
    }

    g_signal_emit (self, tpa_contact_base_signals[PRESENCE_UPDATED],
                   0, self->priv->presence, self->priv->presence_msg);

    VERBOSE ("return");
}

void
tpa_contact_base_capabilities_changed (TpaContactBase *self,
                                       TpaCapability capability)
{
    const gchar *uri;

    VERBOSE ("(%p, %d)", self, capability);
    g_assert (self);

    self->priv->capabilities = capability;

    if (DEBUGGING) {
        uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
        INFO ("[capabilities-changed %s] %d", uri, self->priv->capabilities);
    }

    g_signal_emit (self, tpa_contact_base_signals[CAPABILITIES_CHANGED],
                   0, self->priv->capabilities);

    VERBOSE ("return");
}

void
tpa_contact_base_avatar_updated (TpaContactBase *self,
                                 const gchar *token)
{
    const gchar *uri;
    const gchar *old;
    VERBOSE ("(%p, %s)", self, token);
    g_assert (self);

    old = tpa_avatar_get_token (self->priv->avatar);
    if (!old || !g_str_equal (token, old)) {
        if (DEBUGGING) {
            uri = tpa_channel_target_get_uri (TPA_CHANNEL_TARGET (self));
            INFO ("[avatar-updated %s] %s", uri, token);
        }

        tpa_contact_base_update_avatar (self, token);
        g_signal_emit (self, tpa_contact_base_signals[AVATAR_UPDATED], 0, token);
    }

    VERBOSE ("return");
}

