/*
 * Copyright (C) 2006 INdT.
 * @author  Luiz Augusto von Dentz <luiz.dentz@indt.org.br>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdarg.h>

#include <glib.h>

#include "tpa-debug.h"

static TpaDomainFlags domain_flags = 0;
static TpaDebugLevelFlags level_flags = 0;

GDebugKey domain_keys[] = {
    { "ConnectionManager",   TPA_DOMAIN_CONNECTION_MANAGER },
    { "Connection",          TPA_DOMAIN_CONNECTION },
    { "Channel",             TPA_DOMAIN_CHANNEL },
    { "Aliasing",            TPA_DOMAIN_ALIASING },
    { "Avatars",             TPA_DOMAIN_AVATARS },
    { "Capabilities",        TPA_DOMAIN_CAPABILITIES },
    { "ContactInfo",         TPA_DOMAIN_CONTACT_INFO },
    { "Forwarding",          TPA_DOMAIN_FORWARDING },
    { "Presence",            TPA_DOMAIN_PRESENCE },
    { "Properties",          TPA_DOMAIN_PROPERTIES },
    { "Connection*",         TPA_DOMAIN_CONNECTION_IFACES },
    { 0, },
};

GDebugKey level_keys[] = {
    { "verbose", TPA_LEVEL_VERBOSE },
    { "info",  TPA_LEVEL_INFO },
    { "error", TPA_LEVEL_ERROR },
    { 0, },
};

void tpa_debug_set_flags_from_env ()
{
    guint nkeys;
    const gchar *flags_string;

    for (nkeys = 0; domain_keys[nkeys].value; nkeys++);

    flags_string = g_getenv ("TPA_DEBUG");

    if (flags_string)
        tpa_debug_set_flags (g_parse_debug_string (flags_string, domain_keys, nkeys));
    else
        tpa_debug_set_flags (TPA_DOMAIN_NONE);

    for (nkeys = 0; level_keys[nkeys].value; nkeys++);

    flags_string = g_getenv ("TPA_DEBUG_LEVEL");

    if (flags_string)
        tpa_debug_level_set_flags (g_parse_debug_string (flags_string, level_keys, nkeys));
    else
        tpa_debug_level_set_flags (TPA_LEVEL_INFO | TPA_LEVEL_ERROR);
}

void tpa_debug_set_flags (TpaDomainFlags new_flags)
{
    domain_flags |= new_flags;
}

void tpa_debug_level_set_flags (TpaDebugLevelFlags new_flags)
{
    level_flags |= new_flags;
}

gboolean tpa_debug_flag_is_set (TpaDomainFlags flag)
{
    return flag & domain_flags;
}

gboolean tpa_debug_level_flag_is_set (TpaDebugLevelFlags flag)
{
    return flag & level_flags;
}

void tpa_debug (TpaDomainFlags domain_flag,
                TpaDebugLevelFlags level_flag,
                const gchar *format,
                ...)
{
    if ((domain_flag & domain_flags) &&
        (level_flag & level_flags) &&
        format) {
        va_list args;
        va_start (args, format);
        g_logv (G_LOG_DOMAIN, G_LOG_LEVEL_DEBUG, format, args);
        va_end (args);
    }
}
