#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""
Codegeneration utilities.
"""

__author__ = "Martin Sandve Alnes"
__date__   = "2008-08-13 -- 2009-05-15"
__copyright__ = "(C) 2008-2009 Martin Sandve Alnes and Simula Resarch Laboratory"
__license__  = "GNU GPL Version 2, or (at your option) any later version"

from itertools import chain
import ufl
import ufc_utils
import SyFi

from ufl.classes import Measure
from ufl.algorithms import FormData

from dolfin_utils.wrappers import generate_dolfin_code, UFCFormNames

from sfc.common.utilities import unique
from sfc.common.output import sfc_assert, sfc_error, sfc_info, sfc_debug
#from sfc.common import ParameterDict
from sfc.common.options import default_options
from sfc.representation import ElementRepresentation, FormRepresentation, \
        CellIntegralRepresentation, ExteriorFacetIntegralRepresentation, \
        InteriorFacetIntegralRepresentation
from sfc.codegeneration.formcg import FormCG
from sfc.codegeneration.dofmapcg import DofMapCG
from sfc.codegeneration.finiteelementcg import FiniteElementCG
from sfc.codegeneration.cellintegralcg import CellIntegralCG
from sfc.codegeneration.exteriorfacetintegralcg import ExteriorFacetIntegralCG
from sfc.codegeneration.interiorfacetintegralcg import InteriorFacetIntegralCG

# version %(version)s
_header_template = r"""/*
 * %(name)s.h
 *
 * This file was automatically generated by SFC.
 *
 * http://www.fenics.org/syfi/
 *
 */

#ifndef GENERATED_SFC_CODE_%(name)s
#define GENERATED_SFC_CODE_%(name)s

#include "ufc.h"

%(includes)s

//namespace sfc
//{
%(body)s
//}

#endif
"""

# version %(version)s
_implementation_template = r"""/*
 * %(name)s.cpp
 *
 * This file was automatically generated by SFC.
 *
 * http://www.fenics.org/syfi/
 *
 */

#include "ufc.h"

#include "%(name)s.h"

%(includes)s

//namespace sfc
//{
%(body)s
//}
"""

def common_system_headers():
    return ("iostream", "cmath", "stdexcept", "cstring")

def generate_finite_element_code(ferep):
    sfc_debug("Entering generate_finite_element_code")
    classname = ferep.finite_element_classname
    cg = FiniteElementCG(ferep)
    vars = cg.generate_code_dict()
    supportcode = cg.generate_support_code()

    hcode = ufc_utils.finite_element_header % vars
    ccode = supportcode + "\n"*3 + ufc_utils.finite_element_implementation % vars
        
    includes = cg.hincludes() + cg.cincludes()

    system_headers = common_system_headers()
    local_headers = unique("%s.h" % e.finite_element_classname for e in ferep.sub_elements)

    hincludes = "\n".join('#include "%s"' % inc for inc in cg.hincludes())
    cincludes =  "\n".join('#include <%s>' % f for f in system_headers)
    cincludes += "\n"
    cincludes += "\n".join('#include "%s"' % inc for inc in chain(cg.cincludes(), local_headers))
    
    hcode = _header_template         % { "body": hcode, "name": classname, "includes": hincludes }
    ccode = _implementation_template % { "body": ccode, "name": classname, "includes": cincludes }
    sfc_debug("Leaving generate_finite_element_code")
    return classname, (hcode, ccode), includes

def generate_dof_map_code(ferep):
    sfc_debug("Entering generate_dof_map_code")
    classname = ferep.dof_map_classname
    cg = DofMapCG(ferep)
    vars = cg.generate_code_dict()
    supportcode = cg.generate_support_code()

    hcode = ufc_utils.dof_map_header % vars
    ccode = supportcode + "\n"*3 + ufc_utils.dof_map_implementation % vars
    
    includes = cg.hincludes() + cg.cincludes()
    
    system_headers = common_system_headers()
    local_headers = unique("%s.h" % e.dof_map_classname for e in ferep.sub_elements)

    hincludes = "\n".join('#include "%s"' % inc for inc in cg.hincludes())
    cincludes =  "\n".join('#include <%s>' % f for f in system_headers)
    cincludes += "\n"
    cincludes += "\n".join('#include "%s"' % inc for inc in chain(cg.cincludes(), local_headers))
    
    hcode = _header_template         % { "body": hcode, "name": classname, "includes": hincludes }
    ccode = _implementation_template % { "body": ccode, "name": classname, "includes": cincludes }
    sfc_debug("Leaving generate_dof_map_code")
    return classname, (hcode, ccode), includes

def generate_cell_integral_code(integrals, formrep):
    sfc_debug("Entering generate_cell_integral_code")
    itgrep = CellIntegralRepresentation(integrals, formrep)
    
    cg = CellIntegralCG(itgrep)
    vars = cg.generate_code_dict()
    supportcode = cg.generate_support_code()

    # TODO: Support code might be placed in unnamed namespace?
    hcode = ufc_utils.cell_integral_header % vars
    ccode = supportcode + "\n"*3 + ufc_utils.cell_integral_implementation % vars
    
    includes = cg.hincludes() + cg.cincludes()

    system_headers = common_system_headers()
    
    hincludes = "\n".join('#include "%s"' % inc for inc in cg.hincludes())
    cincludes =  "\n".join('#include <%s>' % f for f in system_headers)
    cincludes += "\n"
    cincludes += "\n".join('#include "%s"' % inc for inc in cg.cincludes())
    
    hcode = _header_template         % { "body": hcode, "name": itgrep.classname, "includes": hincludes }
    ccode = _implementation_template % { "body": ccode, "name": itgrep.classname, "includes": cincludes }
    sfc_debug("Leaving generate_cell_integral_code")
    return itgrep.classname, (hcode, ccode), includes

def generate_exterior_facet_integral_code(integrals, formrep):
    sfc_debug("Entering generate_exterior_facet_integral_code")
    itgrep = ExteriorFacetIntegralRepresentation(integrals, formrep)

    cg = ExteriorFacetIntegralCG(itgrep)
    vars = cg.generate_code_dict()
    supportcode = cg.generate_support_code()

    # TODO: Support code might be placed in unnamed namespace?
    hcode = ufc_utils.exterior_facet_integral_header % vars
    ccode = supportcode + "\n"*3 + ufc_utils.exterior_facet_integral_implementation % vars
    
    includes = cg.hincludes() + cg.cincludes()

    system_headers = common_system_headers()
    
    hincludes = "\n".join('#include "%s"' % inc for inc in cg.hincludes())
    cincludes =  "\n".join('#include <%s>' % f for f in system_headers)
    cincludes += "\n"
    cincludes += "\n".join('#include "%s"' % inc for inc in cg.cincludes())
    
    hcode = _header_template         % { "body": hcode, "name": itgrep.classname, "includes": hincludes }
    ccode = _implementation_template % { "body": ccode, "name": itgrep.classname, "includes": cincludes }
    sfc_debug("Leaving generate_exterior_facet_integral_code")
    return itgrep.classname, (hcode, ccode), includes

def generate_interior_facet_integral_code(integrals, formrep):
    sfc_debug("Entering generate_interior_facet_integral_code")
    itgrep = InteriorFacetIntegralRepresentation(integrals, formrep)
    
    cg = InteriorFacetIntegralCG(itgrep)
    vars = cg.generate_code_dict()
    supportcode = cg.generate_support_code()

    hcode = ufc_utils.interior_facet_integral_header % vars
    ccode = supportcode + "\n"*3 + ufc_utils.interior_facet_integral_implementation % vars
    
    includes = cg.hincludes() + cg.cincludes()

    system_headers = common_system_headers()
    
    hincludes = "\n".join('#include "%s"' % inc for inc in cg.hincludes())
    cincludes =  "\n".join('#include <%s>' % f for f in system_headers)
    cincludes += "\n"
    cincludes += "\n".join('#include "%s"' % inc for inc in cg.cincludes())
    
    hcode = _header_template         % { "body": hcode, "name": itgrep.classname, "includes": hincludes }
    ccode = _implementation_template % { "body": ccode, "name": itgrep.classname, "includes": cincludes }
    sfc_debug("Leaving generate_interior_facet_integral_code")
    return itgrep.classname, (hcode, ccode), includes

def generate_form_code(formrep):
    sfc_debug("Entering generate_form_code")
    cg = FormCG(formrep)
    vars = cg.generate_code_dict()
    supportcode = cg.generate_support_code()

    hcode = ufc_utils.form_header % vars
    ccode = supportcode + "\n"*3 + ufc_utils.form_implementation % vars
    
    includes = cg.hincludes() + cg.cincludes()

    system_headers = common_system_headers()
    local_headers = unique(chain(formrep.fe_names, formrep.dm_names, \
                        sorted(formrep.itg_names.values()), cg.cincludes()))

    hincludes =  "\n".join('#include "%s"' % inc for inc in cg.hincludes())
    cincludes =  "\n".join('#include <%s>' % f for f in system_headers)
    cincludes += "\n"
    cincludes += "\n".join('#include "%s.h"' % f for f in local_headers)
    
    hcode = _header_template         % { "body": hcode, "name": formrep.classname, "includes": hincludes }
    ccode = _implementation_template % { "body": ccode, "name": formrep.classname, "includes": cincludes }
    sfc_debug("Leaving generate_form_code")
    return (hcode, ccode), includes

def write_file(filename, text):
    f = open(filename, "w")
    f.write(text)
    f.close()

def write_code(classname, code):
    sfc_debug("Entering write_code")
    if isinstance(code, tuple):
        # Code is split in a header and implementation file
        hcode, ccode = code
        hname = classname + ".h"
        cname = classname + ".cpp"
        open(hname, "w").write(hcode)
        open(cname, "w").write(ccode)
        ret = (hname, cname)
    else:
        # All code is combined in a header file
        name = classname + ".h"
        open(name, "w").write(code)
        ret = (name,)
    sfc_debug("Leaving write_code")
    return ret

def compiler_input(input):
    """Map different kinds of input to a list of
    UFL elements and a list of FormData instances.
    
    The following input formats are allowed:
    - ufl.Form
    - ufl.algorithms.FormData
    - ufl.FiniteElementBase
    - list of the above
    
    Returns:
        elements, formdatas
    """
    sfc_debug("Entering compiler_input")
    sfc_assert(input, "Got no input!")
    fd = []
    fe = []
    if not isinstance(input, list):
        input = [input]
    for d in input:
        if isinstance(d, ufl.Form):
            d = d.form_data()
        if isinstance(d, FormData):
            fd.append(d)
            fe.extend(d.sub_elements)
        elif isinstance(d, ufl.FiniteElementBase):
            fe.append(d)
        else:
            sfc_error("Not a FormData or FiniteElementBase object: %s" % str(type(d)))
    fe = sorted(set(fe))
    sfc_debug("Leaving compiler_input")
    return fe, fd

dolfin_header_template = """/*
 * DOLFIN wrapper code generated by the SyFi Form Compiler.
 */

%s
"""

def generate_code(input, options = None):
    """Generate code from input and options.
    
    @param input:
        TODO
    @param options:
        TODO
    """
    sfc_debug("Entering generate_code")
    
    if options is None:
        options = default_options()
    
    ufl_elements, formdatas = compiler_input(input)
    
    filenames = []
    needed_files = []
    formnames = []
    
    # Generate UFC code for elements
    element_reps = {}
    last_nsd = None
    generated_elements = set()
    for e in ufl_elements:
        # Initialize global variables in SyFi with the right space dimension (not very nice!)
        nsd = e.cell().d
        if not nsd == last_nsd: 
            SyFi.initSyFi(nsd)
            last_nsd = nsd
        
        # Construct ElementRepresentation objects for all elements
        quad_rule = None # TODO: What to do with this one?
        assert not "Quadrature" in e.family() # No quadrature rule defined!
        if e in element_reps:
            continue

        erep = ElementRepresentation(e, quad_rule, options, element_reps)
        element_reps[e] = erep
        
        # Build flat list of all subelements
        todo = [erep]
        ereps = []
        while todo:
            erep = todo.pop()
            # Skip already generated!
            if not erep.ufl_element in generated_elements:
                generated_elements.add(erep.ufl_element)
                ereps.append(erep)
            # Queue subelements for inspection
            for subrep in erep.sub_elements:
                todo.append(subrep)

        for erep in ereps:
            # Generate code for finite_element
            classname, code, includes = generate_finite_element_code(erep)
            filenames.extend( write_code(classname, code) )
            needed_files.extend(includes)
            
            # Generate code for dof_map
            classname, code, includes = generate_dof_map_code(erep)
            filenames.extend( write_code(classname, code) )
            needed_files.extend(includes)
    
    # Generate UFC code for forms
    for formdata in formdatas:
        # Initialize global variables in SyFi with the right space dimension (not very nice!)
        nsd = formdata.geometric_dimension
        if not nsd == last_nsd: 
            SyFi.initSyFi(nsd)
            last_nsd = nsd
        
        # This object extracts and collects various information about the ufl form
        formrep = FormRepresentation(formdata, element_reps, options)
        
        # TODO: Get integrals from formrep, in case of modifications there? Or rather make sure that FormRep doesn't do any such things.
        ig = formdata.form.integral_groups()
        
        # Generate code for cell integrals
        for domain in formdata.form.domains(Measure.CELL): 
            integrals = ig[domain]
            classname, code, includes = generate_cell_integral_code(integrals, formrep)
            filenames.extend( write_code(classname, code) )
            needed_files.extend(includes)
        
        # Generate code for exterior facet integrals
        for domain in formdata.form.domains(Measure.EXTERIOR_FACET): 
            integrals = ig[domain]
            classname, code, includes = generate_exterior_facet_integral_code(integrals, formrep)
            filenames.extend( write_code(classname, code) )
            needed_files.extend(includes)
        
        # Generate code for interior facet integrals
        for domain in formdata.form.domains(Measure.INTERIOR_FACET): 
            integrals = ig[domain]
            classname, code, includes = generate_interior_facet_integral_code(integrals, formrep)
            filenames.extend( write_code(classname, code) )
            needed_files.extend(includes)
        
        # Generate code for form!
        code, includes = generate_form_code(formrep)
        filenames.extend( write_code(formrep.classname, code) )
        needed_files.extend(includes)

        # Collect classnames for use with dolfin wrappers
        namespace = "" # "sfc::"
        ufc_form_classname = namespace + formrep.classname
        ufc_finite_element_classnames = [namespace + name for name in formrep.fe_names]
        ufc_dof_map_classnames        = [namespace + name for name in formrep.dm_names]
        fn = UFCFormNames(formdata.name, formdata.function_names,
                          ufc_form_classname, ufc_finite_element_classnames, ufc_dof_map_classnames)
        formnames.append(fn)
    
    # Get other needed files:
    if needed_files:
        raise NotImplementedError("FIXME: Implement fetching non-ufc-class files like DofPtv and quadrature rule files.")

    filenames = list(unique(chain(filenames, needed_files)))
    hfilenames = [f for f in filenames if f.endswith(".h")]
    cfilenames = [f for f in filenames if f.endswith(".cpp")]

    # Generate DOLFIN wrapper code
    if options.code.dolfin_wrappers:
        if not formnames:
            print "NOT generating dolfin wrappers, missing forms!" # TODO: Generate FunctionSpaces for elements?
        else:
            prefix = options.code.prefix
            header = dolfin_header_template % "\n".join('#include "%s"' % h for h in hfilenames)
            sfc_info("Generating DOLFIN wrapper code, formnames are %s." % "\n".join(map(str,formnames)))
            dolfin_code = generate_dolfin_code(prefix, header, formnames)
            dolfin_filename = "%s.h" % prefix
            write_file(dolfin_filename, dolfin_code)
            hfilenames.append(dolfin_filename)

    sfc_debug("Leaving generate_code")
    return hfilenames, cfilenames

