/*
 * sffile.c - SoundFont chunk based file routines
 *
 * libInstPatch
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "instpatch.h"
#include "ippriv.h"
#include "i18n.h"

#define READB(ctx, var)		G_STMT_START {		\
    guint8 _temp;					\
    if (ipfile_read (ctx, &_temp, 1) != INSTP_OK)	\
	return (INSTP_FAIL);				\
    var = _temp;					\
} G_STMT_END
#define READW(ctx, var)		G_STMT_START {		\
    guint16 _temp;					\
    if (ipfile_read (ctx, &_temp, 2) != INSTP_OK)       \
	return (INSTP_FAIL);				\
    var = GUINT16_FROM_LE (_temp);			\
} G_STMT_END
#define READD(ctx, var)		G_STMT_START {		\
    guint32 _temp;					\
    if (ipfile_read (ctx, &_temp, 4) != INSTP_OK)	\
	return (INSTP_FAIL);				\
    var = GUINT32_FROM_LE (_temp);			\
} G_STMT_END
#define FSKIP(ctx, size)	G_STMT_START {		\
    if (ipfile_skip (ctx, size) != INSTP_OK)		\
	return (INSTP_FAIL);				\
} G_STMT_END


#define WRITEB(ctx, var)	G_STMT_START {		\
    if (ipfile_write (ctx, &var, 1, TRUE) != INSTP_OK)  \
	return (INSTP_FAIL);			\
} G_STMT_END
#define WRITEW(ctx, var)	G_STMT_START {		\
    guint16 _tmp = GUINT16_TO_LE (var);		\
    if (ipfile_write (ctx, &_tmp, 2, TRUE) != INSTP_OK)	\
	return (INSTP_FAIL);			\
} G_STMT_END
#define WRITED(ctx, var)	G_STMT_START {		\
    guint32 _tmp = GUINT32_TO_LE (var);		\
    if (ipfile_write (ctx, &_tmp, 4, TRUE) != INSTP_OK)	\
	return (INSTP_FAIL);			\
} G_STMT_END


typedef enum
{
  IPCTX_MODE_READ,
  IPCTX_MODE_WRITE
}
IPFileCtxMode;

typedef struct _IPChunkState
{
  IPChunkType type;		/* chunk type */
  char id[4];			/* chunk ID (only useful for custom INFO) */
  int size;			/* real (file) size of chunk */
  int vsize;			/* virtual size of chunk (for dummy data) */
  int pos;			/* current real (file) pos in chunk */
}
IPChunkState;



/* SoundFont file load/save context */
struct _IPFileContext
{
  IPFileCtxMode mode;		/* transfer direction (read/write) */
  int fhandle;			/* file handle or -1 */
  IPFileCallback *callback;	/* callback function */
  void *callback_data;		/* user ptr for callback */

  IPFileReadFunc read_func;	/* custom data read function */
  IPFileWriteFunc write_func;	/* custom data write function */
  IPFileSeekFunc seek_func;	/* custom seek function */

  void *userptr;		/* user defined pointer */

  guint file_pos;		/* position in file */
  guint file_size;		/* size of file */

  IPChunkState chunks[3];	/* Chunk states for max of 3 embedded chunks */
  int chunkcount;		/* count of active embedded chunks */

  int error_code;		/* error code */
  int parm;			/* parameter to minor error code */

  guint8 buf[IPFILE_LARGEST_CHUNK]; /* buffer for PHDR structure loads */
};

#define IPERR_MSG_BAD_CHUNK_STATE_0 "Unexpected patch file chunk state"

static int ipfile_do_callback (IPFileContext *ctx);
static void ipfile_chunk_state_add (IPFileContext *ctx, IPChunkType type,
				    const char *id, int size);
static int ipfile_start_chunk (IPFileContext *ctx, IPChunkType type,
			       IPChunkType subtype);
static int ipfile_end_chunk (IPFileContext *ctx);

static int ipfile_pdta_helper (IPFileContext *ctx, IPChunkType chunktype);

static int ipfile_load_chunk (IPFileContext *ctx, char id[4], int *size);
static int ipfile_save_chunk (IPFileContext *ctx, const char id[4], int size);
static int ipfile_read (IPFileContext *ctx, void *buf, int count);
static int ipfile_write (IPFileContext *ctx, const void *buf, int count,
			 gboolean addsize);
static void ipfile_write_dummy (IPFileContext *ctx, int count);
static int ipfile_seek (IPFileContext *ctx, int offset);
static int ipfile_error (IPFileContext *ctx, IPFileErrCode code, int parm);

/* private global data */

/* NOTE: keep in sync with IPCHUNK_* enumerations */
static char *ipfile_chunkids[] = {
  "RIFF", "LIST", "sfbk", "INFO", "sdta", "pdta", "smpl", "phdr", "pbag",
  "pmod", "pgen", "inst", "ibag", "imod", "igen", "shdr", "ifil", "isng",
  "INAM", "irom", "iver", "ICRD", "IENG", "IPRD", "ICOP", "ICMT", "ISFT"
};


/**
 * ipfile_chunk_str_to_enum:
 * @str: 4 character string to find #IPChunkType of
 *
 * Get SoundFont chunk ID enum for given string
 *
 * Returns: The chunk type enum or IPCHUNK_UNKNOWN if unknown
 */
IPChunkType
ipfile_chunk_str_to_enum (const char str[4])
{
  int i = 0;

  for (i = 0; i < sizeof (ipfile_chunkids) / sizeof (char *); i++)
    if (strncmp (str, ipfile_chunkids[i], 4) == 0)
      return (i + 1);

  return (IPCHUNK_UNKNOWN);
}

/**
 * ipfile_chunk_enum_to_str:
 * @type: Chunk ID enum to get string of
 *
 * Get SoundFont chunk ID string for given IPChunkType
 *
 * Returns: Pointer to chunk ID 4 character string or NULL if invalid chunk
 * type (NOT zero terminated and should NOT be modified).
 */
char *
ipfile_chunk_enum_to_str (IPChunkType type)
{
  if (type >= IPCHUNK_COUNT || type <= IPCHUNK_UNKNOWN) return (NULL);
  return (ipfile_chunkids[type - 1]);
}

/**
 * ipfile_chunk_size:
 * @chunktype: Type of chunk to get size of
 *
 * Get chunk size for fixed length IPChunkTypes
 *
 * Returns: Size of chunk or 0 if unrecognized chunktype or not fixed length
 */
int
ipfile_chunk_size (IPChunkType chunktype)
{
  switch (chunktype)
    {
    case IPCHUNK_PHDR:
      return (IPFILE_PHDR_SIZE);
    case IPCHUNK_IHDR:
      return (IPFILE_IHDR_SIZE);
    case IPCHUNK_SHDR:
      return (IPFILE_SHDR_SIZE);
    case IPCHUNK_PBAG:
    case IPCHUNK_IBAG:
      return (IPFILE_BAG_SIZE);
    case IPCHUNK_PMOD:
    case IPCHUNK_IMOD:
      return (IPFILE_MOD_SIZE);
    case IPCHUNK_PGEN:
    case IPCHUNK_IGEN:
      return (IPFILE_GEN_SIZE);
    case IPINFO_VERSION:
    case IPINFO_ROM_VERSION:
      return (IPFILE_VERSION_SIZE);

    default:
      return (0);
    }
}

/**
 * ipfile_context_new:
 * @mode: I/O mode, "r" for read, "w" for write
 *
 * Create a new file context
 *
 * Returns: The SoundFont file context or NULL on error
 */
IPFileContext *
ipfile_context_new (char *mode)
{
  IPFileContext *ctx;

  ctx = g_malloc0 (sizeof (IPFileContext));
  if (!ctx) return (NULL);

  /* 'w' for WRITE mode, 'r' or anything else is READ mode */
  ctx->mode = (*mode == 'w') ? IPCTX_MODE_WRITE : IPCTX_MODE_READ;
  ctx->fhandle = -1;		/* initialize file handle to "NULL" state */

  return (ctx);
}

/**
 * ipfile_context_free:
 * @ctx: SoundFont file context to free
 *
 * Free a SoundFont file context
 */
void
ipfile_context_free (IPFileContext *ctx)
{
  g_return_if_fail (ctx != NULL);
  g_free (ctx);
}

/**
 * ipfile_set_fhandle:
 * @ctx: SoundFont file context
 * @fhandle: File handle to use for I/O operations
 *
 * Set the file handle of a SoundFont file context
 */
void
ipfile_set_fhandle (IPFileContext *ctx, int fhandle)
{
  g_return_if_fail (ctx != NULL);
  ctx->fhandle = fhandle;
}

/**
 * ipfile_get_fhandle:
 * @ctx: SoundFont file context
 *
 * Get the file handle of a SoundFont file context
 *
 * Returns: The file handle or -1 if not set or ctx is NULL
 */
int
ipfile_get_fhandle (IPFileContext *ctx)
{
  g_return_val_if_fail (ctx != NULL, -1);
  return (ctx->fhandle);
}

/**
 * ipfile_set_callback:
 * @ctx: SoundFont file context
 * @callback: The callback routine to use
 * @callback_data: User definable pointer to pass to callback
 *
 * Set a SoundFont file context's callback routine, which gets called for
 * each chunk in file loading or saving
 */
void
ipfile_set_callback (IPFileContext *ctx, IPFileCallback *callback,
		     const void *callback_data)
{
  g_return_if_fail (ctx != NULL);
  g_return_if_fail (callback != NULL);
  ctx->callback = callback;
  ctx->callback_data = (void *)callback_data;
}

/**
 * ipfile_get_callback:
 * @ctx: SoundFont file context
 * @callback: Pointer to store callback in or NULL to ignore
 * @callback_data: Pointer to store user defined pointer in or NULL to ignore
 *
 * Get the callback and user pointers
 */
void
ipfile_get_callback (IPFileContext *ctx, IPFileCallback **callback,
		     void **callback_data)
{
  g_return_if_fail (ctx != NULL);

  if (callback) *callback = ctx->callback;
  if (callback_data) *callback_data = ctx->callback_data;
}

/**
 * ipfile_set_userptr:
 * @ctx: File context
 * @userptr: Value to set user definable pointer to.
 *
 * Sets the value of the user definable pointer in a file context.
 */
void
ipfile_set_userptr (IPFileContext *ctx, void *userptr)
{
  g_return_if_fail (ctx != NULL);
  ctx->userptr = userptr;
}

/**
 * ipfile_get_userptr:
 * @ctx: File context
 *
 * Sets the value of the user definable pointer in a file context.
 *
 * Returns: The user defined pointer value.
 */
void *
ipfile_get_userptr (IPFileContext *ctx)
{
  g_return_val_if_fail (ctx != NULL, NULL);
  return (ctx->userptr);
}

/**
 * ipfile_set_io_functions:
 * @ctx: File context
 * @read: Read function override (or NULL for default)
 * @write: Write function override (or NULL for default)
 * @seek: Seek function override (or NULL for default)
 *
 * Set the Input/Output override functions used for processing patch files.
 */
void
ipfile_set_io_functions (IPFileContext *ctx,
			 IPFileReadFunc read,
			 IPFileWriteFunc write,
			 IPFileSeekFunc seek)
{
  g_return_if_fail (ctx != NULL);

  ctx->read_func = read;
  ctx->write_func = write;
  ctx->seek_func = seek;
}

/**
 * ipfile_get_chunk_type:
 * @ctx: SoundFont file context
 *
 * Get the type of the current chunk being processed.
 * A convenience function, as #ipfile_get_chunk_state could be used with -1
 * as the @level: parameter.
 *
 * Returns: The type of the current chunk or IPCHUNK_UNKNOWN if not
 * processing any chunks
 */
IPChunkType
ipfile_get_chunk_type (IPFileContext *ctx)
{
  g_return_val_if_fail (ctx != NULL, IPCHUNK_UNKNOWN);

  if (ctx->chunkcount == 0) return (IPCHUNK_UNKNOWN);
  return (ipfile_chunk_str_to_enum (ctx->chunks [ctx->chunkcount - 1].id));
}

/**
 * ipfile_get_chunk_level_count:
 * @ctx: SoundFont file context
 *
 * Gets the current chunk level count (number of embedded chunks) currently
 * being processed in a SoundFont file.
 *
 * Returns: Chunk level count (0 = no chunks being processed)
 */
int
ipfile_get_chunk_level_count (IPFileContext *ctx)
{
  g_return_val_if_fail (ctx != NULL, 0);
  return (ctx->chunkcount);
}

/**
 * ipfile_get_chunk_state:
 * @ctx: SoundFont file context
 * @level: Chunk level to get state of (-1 for current chunk)
 *   should be less than the value returned from #ipfile_get_chunk_level_count
 * @type: Pointer to store IPChunkType or NULL
 * @size: Pointer to integer to store current size of chunk or NULL
 * @pos: Pointer to integer to store current position in chunk or NULL
 *
 * Get the state of a chunk being processed. The level parameter defines
 * which chunk to get the state of. Example: After saving an
 * IPINFO_NAME there will be 3 chunks (Level 0: IPCHUNK_SFBK, 1:
 * IPCHUNK_INFO, 2: IPINFO_NAME). You get the current chunk level with the
 * #ipfile_get_chunk_level_count function.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_get_chunk_state (IPFileContext *ctx, int level, IPChunkType *type,
			guint *size, guint *pos)
{
  IPChunkState *state;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (ctx->chunkcount > 0 && level < ctx->chunkcount,
			INSTP_FAIL);

  /* a negative level specifies the current chunk */
  if (level < 0) level = ctx->chunkcount - 1;
  state = &ctx->chunks[level];

  if (type) *type = state->type;
  if (size) *size = state->size;
  if (pos) *pos = state->pos;

  return (INSTP_OK);
}

/**
 * ipfile_get_file_size:
 * @ctx: SoundFont file context
 *
 * Get total size of a SoundFont file
 *
 * Returns: Total size of current SoundFont file, in bytes, or 0 if haven't
 *   loaded header yet, or ctx is NULL. When loading, the returned size is the
 *   total size of the SoundFont file; when saving, its the size written so
 *   far.
 */
int
ipfile_get_file_size (IPFileContext *ctx)
{
  g_return_val_if_fail (ctx != NULL, 0);

  return (ctx->file_size);
}

/**
 * ipfile_get_file_position:
 * @ctx: SoundFont file context
 *
 * Get current position in a SoundFont file.
 *
 * Returns: The current offset, in bytes, into the SoundFont file.
 */
int
ipfile_get_file_position (IPFileContext *ctx)
{
  g_return_val_if_fail (ctx != NULL, 0);

  return (ctx->file_pos);
}

/**
 * ipfile_get_items_left:
 * @ctx: SoundFont file context
 *
 * Get the number of items remaining in the active chunk
 *
 * Returns: Number of items remaining or -1 if no chunks being processed,
 *   not fixed length items, or ctx is NULL
 */
int
ipfile_get_items_left (IPFileContext *ctx)
{
  IPChunkState *state;
  int size;

  g_return_val_if_fail (ctx != NULL, -1);

  if (ctx->chunkcount == 0 || ctx->mode != IPCTX_MODE_READ) return (-1);

  state = &ctx->chunks[ctx->chunkcount - 1];
  if ((size = ipfile_chunk_size (state->type)) == 0) return (-1);

  return (state->size / size);
}

/**
 * ipfile_start_transfer:
 * @ctx: SoundFont file context
 *
 * Starts transfering data (saving or loading) in a SoundFont file, by
 * calling the user callback routine for each block in the file. The callback
 * routine is responsible for loading/saving the actual data.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_start_transfer (IPFileContext *ctx)
{
  struct stat st;

  g_return_val_if_fail (ctx, INSTP_FAIL);
  g_return_val_if_fail (ctx->callback, INSTP_FAIL);

  if (ctx->fhandle == -1
      && ((ctx->mode == IPCTX_MODE_READ && !ctx->read_func)
	  || (ctx->mode == IPCTX_MODE_WRITE && !ctx->write_func)))
    {
      g_critical ("File handle has not been set!");
      return (INSTP_FAIL);
    }

  /* <sfbk> load "RIFF" chunk header and "sfbk" sub ID */
  if (ipfile_start_chunk (ctx, IPCHUNK_RIFF, IPCHUNK_SFBK) != INSTP_OK)
    return (INSTP_FAIL);

  if (ctx->mode == IPCTX_MODE_READ) /* read mode? */
    {
      /* verify total size of file with RIFF chunk size */
      if (fstat (ctx->fhandle, &st) < 0)
	g_warning ("File size check failed (fstat()): %s", g_strerror (errno));
      else if (st.st_size != (off_t)(ctx->chunks [0].size +
				     IPFILE_CHUNK_SIZE))
	return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE, IPCHUNK_RIFF));

      ctx->file_size = ctx->chunks [0].size + IPFILE_CHUNK_SIZE;
    }

  /* do the callback thing */
  if (ipfile_do_callback (ctx) != INSTP_OK) return (INSTP_FAIL);

  /* <INFO> load "LIST" chunk header and "INFO" sub ID */
  if (ipfile_start_chunk (ctx, IPCHUNK_LIST, IPCHUNK_INFO) != INSTP_OK)
    return (INSTP_FAIL);

  /* do the callback thing */
  if (ipfile_do_callback (ctx) != INSTP_OK) return (INSTP_FAIL);

  /* end any run on INFO chunks */
  if (ctx->chunkcount == 3)
    if (ipfile_end_chunk (ctx) != INSTP_OK) return (INSTP_FAIL);

  if (ipfile_end_chunk (ctx) != INSTP_OK) return (INSTP_FAIL);
  /* </INFO> */

  /* <sdta> load "LIST" chunk header and "sdta" sub ID */
  if (ipfile_start_chunk (ctx, IPCHUNK_LIST, IPCHUNK_SDTA) != INSTP_OK)
    return (INSTP_FAIL);
  /* do the callback thing */
  if (ipfile_do_callback (ctx) != INSTP_OK) return (INSTP_FAIL);
  /* <smpl> load "smpl" chunk header (no sub ID) */
  if (ipfile_start_chunk (ctx, IPCHUNK_SMPL, IPCHUNK_UNKNOWN) != INSTP_OK)
    return (INSTP_FAIL);
  /* do the callback thing */
  if (ipfile_do_callback (ctx) != INSTP_OK) return (INSTP_FAIL);
  if (ipfile_end_chunk (ctx) != INSTP_OK) return (INSTP_FAIL);
  /* </smpl> */
  if (ipfile_end_chunk (ctx) != INSTP_OK) return (INSTP_FAIL);
  /* </sdta> */

  /* <pdta> load "LIST" chunk header and "pdta" sub ID */
  if (ipfile_start_chunk (ctx, IPCHUNK_LIST, IPCHUNK_PDTA) != INSTP_OK)
    return (INSTP_FAIL);

  if (ipfile_pdta_helper (ctx, IPCHUNK_PHDR) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_PBAG) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_PMOD) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_PGEN) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_IHDR) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_IBAG) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_IMOD) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_IGEN) != INSTP_OK)
    return (INSTP_FAIL);
  if (ipfile_pdta_helper (ctx, IPCHUNK_SHDR) != INSTP_OK)
    return (INSTP_FAIL);

  if (ipfile_end_chunk (ctx) != INSTP_OK) return (INSTP_FAIL);
  /* </pdta> */

  if (ipfile_end_chunk (ctx) != INSTP_OK) return (INSTP_FAIL);
  /* </sfbk> */

  return (INSTP_OK);
}

static int
ipfile_do_callback (IPFileContext *ctx)
{
  return ((*ctx->callback)(ctx, ctx->callback_data));
}

static void
ipfile_chunk_state_add (IPFileContext *ctx, IPChunkType type, const char *id,
			int size)
{
  IPChunkState *state;

  state = &ctx->chunks[ctx->chunkcount];
  state->type = type;
  state->size = size;
  state->vsize = size;
  state->pos = 0;

  if (id) strncpy (state->id, id, 4);
  else state->id[0] = '\0';

  ctx->chunkcount++;
}

static int
ipfile_start_chunk (IPFileContext *ctx, IPChunkType type, IPChunkType subtype)
{
  int size = 0;
  char id[4];

  if (ctx->mode == IPCTX_MODE_READ) /* read mode? */
    {
      if (ipfile_load_chunk (ctx, id, &size) != INSTP_OK)
	return (INSTP_FAIL);

      if (ipfile_chunk_str_to_enum (id) != type)
	return (ipfile_error (ctx, IPFILE_CHUNK_NOT_FOUND, type));
      if (size % 2)		/* all chunk sizes must be even */
	return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE, type));
    }
  else				/* write mode */
    {
      if (ipfile_save_chunk (ctx, ipfile_chunk_enum_to_str (type), 0)
	  != INSTP_OK)
	return (INSTP_FAIL);
    }

  /* use "subtype" for state->type if specified, otherwise use "type" */
  ipfile_chunk_state_add (ctx, (subtype != IPCHUNK_UNKNOWN) ? subtype : type,
			  NULL, size);
  /* chunk has subtype? */
  if (subtype != IPCHUNK_UNKNOWN)
    {
      if (ctx->mode == IPCTX_MODE_READ)	/* read mode? */
	{
	  if (ipfile_read (ctx, id, 4) != INSTP_OK)
	    return (INSTP_FAIL);

	  if (ipfile_chunk_str_to_enum (id) != subtype)
	    return (ipfile_error (ctx, IPFILE_CHUNK_NOT_FOUND, subtype));
	}
      else			/* write mode */
	{
	  if (ipfile_write (ctx, ipfile_chunk_enum_to_str (subtype), 4, TRUE)
	      != INSTP_OK)
	    return (INSTP_FAIL);
	}
    }

  return (INSTP_OK);
}

static int
ipfile_end_chunk (IPFileContext *ctx)
{
  IPChunkState *state;
  guint32 ofs, size;

  state = &ctx->chunks[ctx->chunkcount - 1];

  if (ctx->mode == IPCTX_MODE_READ) /* read mode? */
    {
      int skip = state->size - state->pos;

      if (skip > 0)		/* skip rest of chunk (if any) */
	if (ipfile_seek (ctx, skip) != INSTP_OK)
	  return (INSTP_FAIL);
    }
  else				/* in save mode, update chunk size field */
    {
      /* if its the end of an INFO sub chunk and its size is odd then make it
	 even by padding it with a zero byte */
      if (ctx->chunkcount == 3 && ctx->chunks[1].type == IPCHUNK_INFO
	  && ctx->chunks[2].size & 1)
	{
	  char zero = '\0';
	  WRITEB (ctx, zero);
	}

      ofs = -state->pos - 4;	/* ofs is to chunk size field */

      if (ofs != 0)		/* seek to chunk size field */
	if (ipfile_seek (ctx, ofs) != INSTP_OK)
	  return (INSTP_FAIL);

      /* write chunk size without adding to chunk size counter,
	 we use virtual chunk size to handle dummy sample block */
      size = GUINT32_TO_LE (state->vsize);
      if (ipfile_write (ctx, &size, 4, FALSE) != INSTP_OK)
	return (INSTP_FAIL);

      /* seek back to end of closed chunk except when it is toplevel chunk */
      if (state->size && ctx->chunkcount > 1)
	if (ipfile_seek (ctx, state->size) != INSTP_OK)
	  return (INSTP_FAIL);
    }

  ctx->chunkcount--;

  return (INSTP_OK);
}

static int
ipfile_pdta_helper (IPFileContext *ctx, IPChunkType chunktype)
{
  int size;

  /* load the chunk header (no sub ID) */
  if (ipfile_start_chunk (ctx, chunktype, IPCHUNK_UNKNOWN) != INSTP_OK)
    return (INSTP_FAIL);

  /* for read mode we do some chunk size verification */
  if (ctx->mode == IPCTX_MODE_READ)
    {
      size = ctx->chunks [ctx->chunkcount - 1].size;

      if (size % ipfile_chunk_size (chunktype))
	return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE, chunktype));

      /* FIXME! (should verify total sub chunk sizes) */
      if (size > ctx->chunks [ctx->chunkcount - 2].size)
	return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE, IPCHUNK_PDTA));
    }

  /* do the callback thing */
  if (ipfile_do_callback (ctx) != INSTP_OK) return (INSTP_FAIL);

  if (ipfile_end_chunk (ctx) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/**
 * ipfile_load_info_chunk:
 * @ctx: SoundFont file context
 * @id: Pointer to store 4 character info ID into (NOT zero terminated)
 * @size: Pointer to store size of info chunk
 *
 * Loads a SoundFont info ID and the size of the following value, which
 * is loaded with #ipfile_load_info_data. Should only be called when
 * #ipfile_get_chunk_type returns IPCHUNK_INFO. Calling this routine prior to
 * getting the info data of a previous chunk will skip that info data.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_info_chunk (IPFileContext *ctx, char *id, int *size)
{
  char chunkid[4];
  int chunksize;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (id != NULL, INSTP_FAIL);
  g_return_val_if_fail (size != NULL, INSTP_FAIL);

  if (ctx->chunkcount < 2 || ctx->chunks[1].type != IPCHUNK_INFO)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  /* INFO data from previous INFO chunk remaining? */
  if (ctx->chunkcount == 3)
    if (ipfile_end_chunk (ctx) != INSTP_OK)
      return (INSTP_FAIL);

  left = ctx->chunks[1].size - ctx->chunks[1].pos;
  if (left < IPFILE_CHUNK_SIZE)
    {
      if (left != 0)
	return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE, IPCHUNK_INFO));
      *id = '\0';
      *size = -1;
      return (INSTP_OK);
    }

  if (ipfile_load_chunk (ctx, chunkid, &chunksize) != INSTP_OK)
    return (INSTP_FAIL);

  /* check sanity of chunk size (even and not exceeding INFO block size) */
  if (chunksize % 2 || chunksize > left)
    return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE, IPCHUNK_INFO));

  ipfile_chunk_state_add (ctx, ipfile_chunk_str_to_enum (chunkid),
			  chunkid, chunksize);

  strncpy (id, chunkid, 4);
  *size = chunksize;

  return (INSTP_OK);
}

/**
 * ipfile_load_info_data:
 * @ctx: SoundFont file context
 * @buf: Buffer to store info data in
 * @size: Number of bytes of info data to store in buffer, should not
 *   exceed the number of bytes remaining in this info value
 * 
 * Load the value of a SoundFont info item from a file. Can be called
 * multiple times to load data in partial chunks (#IPINFO_COMMENT can
 * be up to 64k bytes in size).  #ipfile_load_info_chunk needs to be
 * called before this routine to get the ID and size of the
 * value. #ipfile_load_info_version can be used to load version info
 * for #IPINFO_VERSION and #IPINFO_ROM_VERSION chunks.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise (no active INFO chunk,
 *   size exceeds remaining data, etc)
 */
int
ipfile_load_info_data (IPFileContext *ctx, char *buf, int size)
{
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (buf != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || ctx->chunks[1].type != IPCHUNK_INFO)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (size <= 0 || size > left)
    {
      g_critical (IPERR_MSG_BAD_PARAM_1, "size");
      return (INSTP_FAIL);
    }

  if (ipfile_read (ctx, buf, size) != INSTP_OK)
    return (INSTP_FAIL);

  /* no more data for this info? */
  if (ctx->chunks[2].pos == ctx->chunks[2].size)
    ipfile_end_chunk (ctx);

  return (INSTP_OK);
}

/**
 * ipfile_load_info_version:
 * @ctx: SoundFont file context
 * @version: Version structure to store version in
 *
 * Load SoundFont version info. Use this routine instead of
 * #ipfile_load_info_data to load versions (#IPINFO_VERSION and
 * #IPINFO_ROM_VERSION chunks).
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_info_version (IPFileContext *ctx, IPSFontVersion *version)
{
  int left;
  int i;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (version != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || ctx->chunks [1].type != IPCHUNK_INFO)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left != 4)
    {
      /* is it indeed a version chunk? */
      if (ipfile_chunk_size (ctx->chunks[2].type) == 4)
	return (ipfile_error (ctx, IPFILE_CHUNK_BAD_SIZE,
			      ctx->chunks[2].type));
      else
	{
	  g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
	  return (INSTP_FAIL);
	}
    }

  READW (ctx, i);
  version->major = i;
  READW (ctx, i);
  version->minor = i;

  return (INSTP_OK);
}

/**
 * ipfile_load_sample_data:
 * @ctx: SoundFont file context
 * @buf: Buffer to store sample data in
 * @count: Size (in samples) to store into buffer, should not exceed
 *   number of samples remaining in sample data chunk.
 *
 * Load a block of sample data using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_SMPL before calling
 * this routine. Can be called multiple times to load sample data in
 * partial chunks.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise (IPCHUNK_SMPL not active,
 *   size exceeds remaining sample data, etc)
 */
int
ipfile_load_sample_data (IPFileContext *ctx, gint16 *buf, int count)
{
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (buf != NULL, INSTP_FAIL);

  /* should be in SMPL chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_SMPL)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  /* verify count doesn't exceed remaining data */
  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (count * 2 > left || count <= 0)
    {
      g_critical (IPERR_MSG_BAD_PARAM_1, "size");
      return (INSTP_FAIL);
    }

  if (ipfile_read (ctx, buf, count * 2) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/**
 * ipfile_load_phdr:
 * @ctx: SoundFont file context
 * @phdr: Pointer to return preset header pointer, NULL if no
 *   more preset headers left, pointer is internal and should
 *   NOT be freed
 *
 * Load a preset header using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_PHDR before calling this
 * routine.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_phdr (IPFileContext *ctx, IPFilePHDR **phdr)
{
  IPFilePHDR *p;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (phdr != NULL, INSTP_FAIL);

  /* should be in PHDR chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_PHDR)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left == 0)
    {
      *phdr = NULL;
      return (INSTP_OK);
    }

  p = (IPFilePHDR *)&ctx->buf;

  if (ipfile_read (ctx, &p->name, IPFILE_NAME_SIZE) != INSTP_OK)
    return (INSTP_FAIL);
  READW (ctx, p->psetnum);
  READW (ctx, p->bank);
  READW (ctx, p->pbagndx);
  READD (ctx, p->library);
  READD (ctx, p->genre);
  READD (ctx, p->morphology);

  *phdr = p;

  return (INSTP_OK);
}

/**
 * ipfile_load_ihdr:
 * @ctx: SoundFont file context
 * @ihdr: Pointer to return instrument header pointer, NULL if no
 *   more instrument headers left, pointer is internal and should
 *   NOT be freed
 *
 * Load an intrument header using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_IHDR before calling this
 * routine.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_ihdr (IPFileContext *ctx, IPFileIHDR **ihdr)
{
  IPFileIHDR *p;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (ihdr != NULL, INSTP_FAIL);

  /* should be in IHDR chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_IHDR)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left == 0)
    {
      *ihdr = NULL;
      return (INSTP_OK);
    }

  p = (IPFileIHDR *)&ctx->buf;

  if (ipfile_read (ctx, &p->name, IPFILE_NAME_SIZE) != INSTP_OK)
    return (INSTP_FAIL);
  READW (ctx, p->ibagndx);

  *ihdr = p;

  return (INSTP_OK);
}

/**
 * ipfile_load_shdr:
 * @ctx: SoundFont file context
 * @shdr: Pointer to return sample header pointer, NULL if no
 *   more sample headers left, pointer is internal and should
 *   NOT be freed
 *
 * Load a sample header using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_SHDR before calling this
 * routine.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_shdr (IPFileContext *ctx, IPFileSHDR **shdr)
{
  IPFileSHDR *p;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (shdr != NULL, INSTP_FAIL);

  /* should be in SHDR chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_SHDR)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left == 0)
    {
      *shdr = NULL;
      return (INSTP_OK);
    }

  p = (IPFileSHDR *)&ctx->buf;

  if (ipfile_read (ctx, &p->name, IPFILE_NAME_SIZE) != INSTP_OK)
    return (INSTP_FAIL);
  READD (ctx, p->start);
  READD (ctx, p->end);
  READD (ctx, p->loopstart);
  READD (ctx, p->loopend);
  READD (ctx, p->samplerate);
  READB (ctx, p->origpitch);
  READB (ctx, p->pitchadj);
  READW (ctx, p->samplelink);
  READW (ctx, p->sampletype);

  *shdr = p;

  return (INSTP_OK);
}

/**
 * ipfile_load_bag:
 * @ctx: SoundFont file context
 * @bag: Pointer to return a zone bag pointer, NULL if no
 *   more zone bags left, pointer is internal and should
 *   NOT be freed
 *
 * Load a zone bag using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_PBAG or IPCHUNK_IBAG
 * before calling this routine.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_bag (IPFileContext *ctx, IPFileBag **bag)
{
  IPFileBag *p;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (bag != NULL, INSTP_FAIL);

  /* should be in SHDR chunk */
  if (ctx->chunkcount != 3 || (ctx->chunks[2].type != IPCHUNK_PBAG
			       && ctx->chunks[2].type != IPCHUNK_IBAG))
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left == 0)
    {
      *bag = NULL;
      return (INSTP_OK);
    }

  p = (IPFileBag *)&ctx->buf;

  READW (ctx, p->genndx);
  READW (ctx, p->modndx);

  *bag = p;

  return (INSTP_OK);
}

/**
 * ipfile_load_mod:
 * @ctx: SoundFont file context
 * @mod: Pointer to return a modulator pointer, NULL if no
 *   more modulators left, pointer is internal and should
 *   NOT be freed
 *
 * Load a modulator using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_PMOD or IPCHUNK_IMOD
 * before calling this routine.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_mod (IPFileContext *ctx, IPFileMod **mod)
{
  IPFileMod *p;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (mod != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || (ctx->chunks[2].type != IPCHUNK_PMOD
			       && ctx->chunks[2].type != IPCHUNK_IMOD))
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left == 0)
    {
      *mod = NULL;
      return (INSTP_OK);
    }

  p = (IPFileMod *)&ctx->buf;

  READW (ctx, p->src);
  READW (ctx, p->dest);
  READW (ctx, p->amount);
  READW (ctx, p->amtsrc);
  READW (ctx, p->trans);

  *mod = p;

  return (INSTP_OK);
}

/**
 * ipfile_load_gen:
 * @ctx: SoundFont file context
 * @gen: Pointer to return a generator pointer, NULL if no
 *   more generators left, pointer is internal and should
 *   NOT be freed
 *
 * Load a generator using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_PGEN or IPCHUNK_IGEN
 * before calling this routine.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_load_gen (IPFileContext *ctx, IPFileGen **gen)
{
  IPFileGen *p;
  int left;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (gen != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || (ctx->chunks[2].type != IPCHUNK_PGEN
			       && ctx->chunks[2].type != IPCHUNK_IGEN))
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  left = ctx->chunks[2].size - ctx->chunks[2].pos;
  if (left == 0)
    {
      *gen = NULL;
      return (INSTP_OK);
    }

  p = (IPFileGen *)&ctx->buf;

  READW (ctx, p->id);

  /* check if gen->id is valid (preset or inst) and is a range unit */
  if (instp_genid_is_valid (p->id, FALSE)
      && instp_gen_info[p->id].unit == IPUNIT_RANGE)
    {				/* load the range */
      READB (ctx, p->amount.range.low);
      READB (ctx, p->amount.range.high);
    }
  else READW (ctx, p->amount.sword);

  *gen = p;

  return (INSTP_OK);
}


/**
 * ipfile_save_info:
 * @ctx: SoundFont file context
 * @id: 4 character info ID (does NOT need to be zero terminated)
 * @val: Info string value to store
 * @size: Size of info string to save
 *
 * Saves an info string to a SoundFont file. Can be called multiple
 * times for the same info id to store a value with smaller sub
 * strings. Will automatically be padded to an even number of
 * bytes. Should only be called when #ipfile_get_chunk_type returns
 * IPCHUNK_INFO.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_info (IPFileContext *ctx, const char *id,
		  const char *val, int size)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (id != NULL, INSTP_FAIL);
  g_return_val_if_fail (val != NULL, INSTP_FAIL);
  g_return_val_if_fail (size > 0, INSTP_FAIL);

  if (ctx->chunkcount < 2 || ctx->chunks[1].type != IPCHUNK_INFO)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  /* not continued info? */
  if (ctx->chunkcount < 3 || strncmp (ctx->chunks[2].id, id, 4) != 0)
    {
      if (ctx->chunkcount == 3)
	if (ipfile_end_chunk (ctx) != INSTP_OK)
	  return (INSTP_FAIL);

      if (ipfile_save_chunk (ctx, id, 0) != INSTP_OK)
	return (INSTP_FAIL);

      ipfile_chunk_state_add (ctx, ipfile_chunk_str_to_enum (id), id, 0);
    }

  if (ipfile_write (ctx, val, size, TRUE) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/**
 * ipfile_save_info_version:
 * @ctx: SoundFont file context
 * @version: Version structure to save
 *
 * Load SoundFont version info. Use this routine instead of
 * #ipfile_load_info_data to load versions (#IPINFO_VERSION and
 * #IPINFO_ROM_VERSION chunks).
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_info_version (IPFileContext *ctx, const char *id,
			  IPSFontVersion *version)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (id != NULL, INSTP_FAIL);
  g_return_val_if_fail (version != NULL, INSTP_FAIL);

  /* should be INFO block with already loaded INFO chunk */
  if (ctx->chunkcount < 2 || ctx->chunks[1].type != IPCHUNK_INFO)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  if (ctx->chunkcount == 3)
    if (ipfile_end_chunk (ctx) != INSTP_OK)
      return (INSTP_FAIL);

  if (ipfile_save_chunk (ctx, id, 4) != INSTP_OK)
    return (INSTP_FAIL);

  WRITEW (ctx, version->major);
  WRITEW (ctx, version->minor);

  return (INSTP_OK);
}

/**
 * ipfile_save_sample_data:
 * @ctx: SoundFont file context
 * @buf: Sample data to save, data should be pre-formatted for a SoundFont
 *   (16 bit little endian samples separated by 46 zero samples)
 * @count: Size (in samples) to save
 *
 * Save a block of sample data using a SoundFont file context.
 * #ipfile_get_chunk_type should return IPCHUNK_SMPL before calling
 * this routine. Can be called multiple times to save sample data in
 * partial chunks. This function is useful when wanting to store a
 * chunk of samples already formatted for SoundFonts (46 zero samples
 * after each sample), to store individual samples use
 * #ipfile_save_sample.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_sample_data (IPFileContext *ctx, gint16 *buf, int count)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (buf != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_SMPL)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  if (ipfile_write (ctx, buf, count * 2, TRUE) != INSTP_OK)
    return (INSTP_FAIL);

  return (INSTP_OK);
}

/**
 * ipfile_save_sample_dummy:
 * @ctx: File context
 * @size: Dummy sample chunk size (in samples)
 *
 * Allocates a virtual amount of sample data in the sample data chunk
 * without actually writing data. Should only be called when in the
 * #IPCHUNK_SMPL block and should not be mixed with writing of real
 * sample data.  This can be used when the sample data should be
 * handled in a non-standard way but the rest of the SoundFont should
 * be written as if it was there (say a SoundFont compression plugin
 * :)
 */
void
ipfile_save_sample_dummy (IPFileContext *ctx, guint size)
{
  g_return_if_fail (ctx != NULL);

  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_SMPL)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return;
    }

  ipfile_write_dummy (ctx, size * 2);
}

/**
 * ipfile_save_sample:
 * @ctx: SoundFont file context
 * @sample: Sample to save
 *
 * Saves a sample's data to a SoundFont. Sample data is stored at
 * current position in file along with 46 zero samples following (as
 * per SoundFont spec). Should only be called when in the
 * IPCHUNK_SMPL chunk.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_sample (IPFileContext *ctx, IPSample *sample)
{
  IPSampleStore *store;
  void *buf;
  int samsize;
  int size;
  int ofs;

  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (sample != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_SMPL)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  /* silently ignore ROM samples */
  if (sample->sampletype & IPSAMPLE_TYPE_ROM)
    return (INSTP_OK);

  /* make sure sample has some data to save */
  if (!sample->sampledata || sample->sampledata->size == 0
      || !(store = instp_sample_data_find_store
	   (sample->sampledata, 0, IPSAMPLE_STORE_FIND_FASTEST
	    | IPSAMPLE_STORE_FIND_READABLE)))
    {
      g_critical (_("Cannot save sample \"%s\""), sample->name);
      return (INSTP_FAIL);
    }

  buf = g_malloc (INSTP_SAMPLE_COPY_BUFFER_SIZE * 2);

  samsize = instp_sample_get_size (sample);
  size = INSTP_SAMPLE_COPY_BUFFER_SIZE;
  ofs = 0;
  while (ofs < samsize)
    {
      if (samsize - ofs < size) /* check for last partial fragment */
	size = samsize - ofs;

      if (instp_sample_store_read (sample->sampledata, store,
				   ofs, size, buf) != INSTP_OK)
	{
	  g_free (buf);
	  return (INSTP_FAIL);
	}

      if (ipfile_write (ctx, buf, size * 2, TRUE) != INSTP_OK)
	{
	  g_free (buf);
	  return (INSTP_FAIL);
	}

      ofs += size;
    }

  /* 46 "zero" samples following sample as per SoundFont spec */
  memset (buf, 0, 46 * 2);

  if (ipfile_write (ctx, buf, 46 * 2, TRUE) != INSTP_OK)
    {
      g_free (buf);
      return (INSTP_FAIL);
    }

  g_free (buf);

  return (INSTP_OK);
}

/**
 * ipfile_save_phdr:
 * @ctx: SoundFont file context
 * @phdr: Preset header data to save
 *
 * Save a preset header using a SoundFont file context.
 * Should only be called when in the IPCHUNK_PHDR chunk.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_phdr (IPFileContext *ctx, const IPFilePHDR *phdr)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (phdr != NULL, INSTP_FAIL);

  /* should be in PHDR chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_PHDR)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  if (ipfile_write (ctx, &phdr->name, IPFILE_NAME_SIZE, TRUE) != INSTP_OK)
    return (INSTP_FAIL);
  WRITEW (ctx, phdr->psetnum);
  WRITEW (ctx, phdr->bank);
  WRITEW (ctx, phdr->pbagndx);
  WRITED (ctx, phdr->library);
  WRITED (ctx, phdr->genre);
  WRITED (ctx, phdr->morphology);

  return (INSTP_OK);
}

/**
 * ipfile_save_ihdr:
 * @ctx: SoundFont file context
 * @ihdr: Instrument header data to save
 *
 * Save an intrument header using a SoundFont file context.
 * Should only be called when in the IPCHUNK_IHDR chunk.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_ihdr (IPFileContext *ctx, const IPFileIHDR *ihdr)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (ihdr != NULL, INSTP_FAIL);

  /* should be in IHDR chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_IHDR)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  if (ipfile_write (ctx, &ihdr->name, IPFILE_NAME_SIZE, TRUE) != INSTP_OK)
    return (INSTP_FAIL);
  WRITEW (ctx, ihdr->ibagndx);

  return (INSTP_OK);
}

/**
 * ipfile_save_shdr:
 * @ctx: SoundFont file context
 * @shdr: Sample header data to save
 *
 * Save a sample header using a SoundFont file context.
 * Should only be called when in the IPCHUNK_SHDR chunk.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_shdr (IPFileContext *ctx, const IPFileSHDR *shdr)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (shdr != NULL, INSTP_FAIL);

  /* should be in SHDR chunk */
  if (ctx->chunkcount != 3 || ctx->chunks[2].type != IPCHUNK_SHDR)
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  if (ipfile_write (ctx, &shdr->name, IPFILE_NAME_SIZE, TRUE) != INSTP_OK)
    return (INSTP_FAIL);
  WRITED (ctx, shdr->start);
  WRITED (ctx, shdr->end);
  WRITED (ctx, shdr->loopstart);
  WRITED (ctx, shdr->loopend);
  WRITED (ctx, shdr->samplerate);
  WRITEB (ctx, shdr->origpitch);
  WRITEB (ctx, shdr->pitchadj);
  WRITEW (ctx, shdr->samplelink);
  WRITEW (ctx, shdr->sampletype);

  return (INSTP_OK);
}

/**
 * ipfile_save_bag:
 * @ctx: SoundFont file context
 * @bag: Zone bag information to save
 *
 * Save a zone bag using a SoundFont file context.
 * Should only be called when in IPCHUNK_PBAG or IPCHUNK_IBAG chunks.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_bag (IPFileContext *ctx, const IPFileBag *bag)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (bag != NULL, INSTP_FAIL);

  /* should be in SHDR chunk */
  if (ctx->chunkcount != 3 || (ctx->chunks[2].type != IPCHUNK_PBAG
			       && ctx->chunks[2].type != IPCHUNK_IBAG))
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  WRITEW (ctx, bag->genndx);
  WRITEW (ctx, bag->modndx);

  return (INSTP_OK);
}

/**
 * ipfile_save_mod:
 * @ctx: SoundFont file context
 * @mod: Modulator info to save
 *
 * Save a modulator using a SoundFont file context.
 * Should only be called when in IPCHUNK_PMOD or IPCHUNK_IMOD chunks.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_mod (IPFileContext *ctx, const IPFileMod *mod)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (mod != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || (ctx->chunks[2].type != IPCHUNK_PMOD
			       && ctx->chunks[2].type != IPCHUNK_IMOD))
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  WRITEW (ctx, mod->src);
  WRITEW (ctx, mod->dest);
  WRITEW (ctx, mod->amount);
  WRITEW (ctx, mod->amtsrc);
  WRITEW (ctx, mod->trans);

  return (INSTP_OK);
}

/**
 * ipfile_save_gen:
 * @ctx: SoundFont file context
 * @gen: Generator info to save
 *
 * Save a generator using a SoundFont file context.
 * Should only be called when in the IPCHUNK_PGEN or IPCHUNK_IGEN chunks.
 *
 * Returns: INSTP_OK on success, INSTP_FAIL otherwise
 */
int
ipfile_save_gen (IPFileContext *ctx, const IPFileGen *gen)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);
  g_return_val_if_fail (gen != NULL, INSTP_FAIL);

  if (ctx->chunkcount != 3 || (ctx->chunks[2].type != IPCHUNK_PGEN
			       && ctx->chunks[2].type != IPCHUNK_IGEN))
    {
      g_critical (IPERR_MSG_BAD_CHUNK_STATE_0);
      return (INSTP_FAIL);
    }

  WRITEW (ctx, gen->id);

  /* check if gen->id is valid (preset or inst) and is a range unit */
  if (instp_genid_is_valid (gen->id, FALSE)
      && instp_gen_info[gen->id].unit == IPUNIT_RANGE)
    {				/* save the range */
      WRITEB (ctx, gen->amount.range.low);
      WRITEB (ctx, gen->amount.range.high);
    }
  else WRITEW (ctx, gen->amount.sword);

  return (INSTP_OK);
}


static int
ipfile_load_chunk (IPFileContext *ctx, char *id, int *size)
{
  if (ipfile_read (ctx, id, 4) != INSTP_OK)
    return (INSTP_FAIL);

  READD (ctx, *size);

  return (INSTP_OK);
}

static int
ipfile_save_chunk (IPFileContext *ctx, const char id[4], int size)
{
  g_return_val_if_fail (ctx != NULL, INSTP_FAIL);

  if (ipfile_write (ctx, id, 4, TRUE) != INSTP_OK)
    return (INSTP_FAIL);

  WRITED (ctx, size);

  return (INSTP_OK);
}

static int
ipfile_read (IPFileContext *ctx, void *buf, int count)
{
  int c;
  int i;

  /* use custom or default read function? */
  if (ctx->read_func) c = (*ctx->read_func)(ctx, buf, count);
  else c = read (ctx->fhandle, buf, count);

  if (c == -1)
    {
      if (ctx->read_func)
	g_critical ("Custom read function failed");
      else g_critical (_("Failed to read from file: %s"), g_strerror (errno));

      return (INSTP_FAIL);
    }

  ctx->file_pos += c;

  if (ctx->chunkcount > 0)	/* update file position */
    {
      for (i = 0; i < ctx->chunkcount; i++)
	ctx->chunks[i].pos += c;
    }

  /* custom read function should return error on short read */
  if (!ctx->read_func && c < count)
    {
      g_critical (_("File read returned %d bytes, expected %d: %s"),
		  c, count, g_strerror (errno));
      return (INSTP_FAIL);
    }

  return (INSTP_OK);
}

static int
ipfile_write (IPFileContext *ctx, const void *buf, int count, gboolean addsize)
{
  int c;
  int i;

  /* use custom or default write function? */
  if (ctx->write_func) c = (*ctx->write_func)(ctx, buf, count);
  else c = write (ctx->fhandle, buf, count);

  if (c == -1)
    {
      if (ctx->write_func)
	g_critical ("Custom read function failed");
      else g_critical (_("Failed to write to file: %s"), g_strerror (errno));

      return (INSTP_FAIL);
    }

  ctx->file_pos += c;
  if (addsize) ctx->file_size += c;

  if (ctx->chunkcount > 0)
    {
      for (i = 0; i < ctx->chunkcount; i++)
	{
	  ctx->chunks[i].pos += c;
	  if (addsize)
	    {
	      ctx->chunks[i].size += c;
	      ctx->chunks[i].vsize += c;
	    }
	}
    }

  return (INSTP_OK);
}

/* allocates for dummy data for writing dummy sample data in particular */
static void
ipfile_write_dummy (IPFileContext *ctx, int count)
{
  int i;

  if (ctx->chunkcount == 0) return;

  for (i = 0; i < ctx->chunkcount; i++)
    ctx->chunks[i].vsize += count;
}

static int
ipfile_seek (IPFileContext *ctx, int offset)
{
  int newofs;
  int i;

  if (ctx->seek_func)
    newofs = (*ctx->seek_func)(ctx, offset);
  else newofs = lseek (ctx->fhandle, offset, SEEK_CUR);

  if (newofs == -1)
    {
      if (ctx->seek_func)
	g_critical ("Custom seek function failed");
      else g_critical (_("Failed to seek %d bytes: %s"), offset,
		       g_strerror (errno));

      return (INSTP_FAIL);
    }

  ctx->file_pos += offset;

  if (ctx->chunkcount > 0)
    for (i = 0; i < ctx->chunkcount; i++)
      ctx->chunks[i].pos += offset;

  return (INSTP_OK);
}

/**
 * ipfile_ftell:
 * @ctx: SoundFont file context
 *
 * Get the current position in a SoundFont file context
 *
 * Returns: The current position in the file
 */
int
ipfile_ftell (IPFileContext *ctx)
{
  if (ctx->chunkcount != 0) return (ctx->chunks[0].pos + IPFILE_CHUNK_SIZE);
  else return (0);
}

static int
ipfile_error (IPFileContext *ctx, IPFileErrCode code, int parm)
{
  ctx->error_code = code;
  ctx->parm = parm;

  switch (code)
    {
    case IPFILE_CHUNK_NOT_FOUND:
      g_critical (_("SoundFont chunk \"%s\" (%d) not found"),
		  ipfile_chunk_enum_to_str (parm), parm);
      break;
    case IPFILE_CHUNK_BAD_SIZE:
      g_critical (_("SoundFont chunk \"%s\" (%d) has invalid size"),
		  ipfile_chunk_enum_to_str (parm), parm);
      break;
    default:
      break;
    }

  return (INSTP_FAIL);
}
