/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "ProjectItemProperties.h"
#include "ExternProvider.h"
#include "sublib/Gui.h"
#include "sublib/Utility.h"
#include "sublib/ExternButton.h"
#include "util/String.h"

// qt
#include <qlayout.h>
#include <qgroupbox.h>
#include <qpushbutton.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qfiledialog.h>
#include <qtooltip.h>
#include <qcombobox.h>
#include <qcheckbox.h>



ProjectItemProperties::ProjectItemProperties( const Project::Item& item,
  ExternProvider* p, QWidget* parent )
: super( parent, 0, true, Qt::WStyle_Customize | Qt::WStyle_Dialog |
  Qt::WStyle_NormalBorder | Qt::WStyle_Title | Qt::WStyle_SysMenu ), _p(p),
  _item(item)
{
  setCaption( _q("subcommander:bookmark properties") );

  QVBoxLayout *vbl = new QVBoxLayout(this,5,8);
  vbl->setSpacing(10);
  {
    QGroupBox* gb = new QGroupBox(1,Qt::Vertical,this);
    gb->setTitle( _q("properties: ") );
    gb->setInsideMargin(0);
    gb->setFlat(true);
    vbl->addWidget(gb);

    QGridLayout* gl = new QGridLayout(vbl,4,3);
    gl->setMargin(0);
    gl->setSpacing(4);
    gl->setColStretch( 0, 1 );
    gl->setColStretch( 1, 4 );
    {
      QLabel* tl = new QLabel( _q("bookmark type:"), this );
      QComboBox* tc = new QComboBox( false, this );
      gl->addWidget( tl, 0, 0 );
      gl->addMultiCellWidget( tc, 0, 0, 1, 2 );

      switch( _item.getType() )
      {
      case Project::Item_Trunk:
        tc->insertItem( QPixmap(getIconDir() + "BookmarkRepository.png"), _q("trunk") );
        break;
      case Project::Item_Branches:
        tc->insertItem( QPixmap(getIconDir() + "BookmarkRepository.png"), _q("branches") );
        break;
      case Project::Item_Tags:
        tc->insertItem( QPixmap(getIconDir() + "BookmarkRepository.png"), _q("tags") );
        break;
      case Project::Item_Repository:
        tc->insertItem( QPixmap(getIconDir() + "BookmarkRepository.png"), _q("repository") );
        break;
      case Project::Item_WorkingCopy:
        tc->insertItem( QPixmap(getIconDir() + "BookmarkWorkingCopy.png"), _q("working copy") );
        break;
      default:
        break;
      }

      _name      = new QLineEdit(this); 
      _source    = new QLineEdit(this);
      _ex        = new ExternButton(this);
      QLabel* ln = new QLabel( _name,   _q("&name"),    this );
      QLabel* ls = new QLabel( _source, _q("&location"), this );
      _name->setFocus();

      gl->setRowSpacing( 1, 5 );
      gl->addWidget( ln, 2, 0 );
      gl->addWidget( ls, 2, 1 );
      gl->addWidget( _name, 3, 0 ); 
      gl->addWidget( _source, 3, 1 ); 
      gl->addWidget( _ex, 3, 2 );

      connect( _name, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
      connect( _source, SIGNAL(textChanged(const QString&)), SLOT(checkOk(const QString&)) );
      connect( _ex, SIGNAL(clicked()), SLOT(selectExtern()) );

      _ex->setEnabled( _item.isWorkingCopy() );
    }

    QHBoxLayout* hu = new QHBoxLayout;
    vbl->addLayout(hu);
    {
      _update = new QCheckBox(this);
      _update->setText( _q("auto &update") );
      _update->setChecked( item.isAutoUpdate() );
      hu->addWidget(_update);
      QToolTip::add( _update, _q("automatically run update on program startup") );

      connect( _update, SIGNAL(toggled(bool)), SLOT(autoUpdate(bool)) );


      if( ! item.isWorkingCopy() )
      {
        _update->hide();
      }

      // eats extra space, so the buttons keep their size
      hu->addStretch(1); 

      _ok = new QPushButton(this);
      _ok->setEnabled(false);
      _ok->setText( _q("&Ok") );
      _ok->setDefault(true);
      hu->addWidget(_ok);
      _ca = new QPushButton(this);
      _ca->setText( _q("&Cancel") );
      hu->addWidget(_ca);

      hu->addSpacing(getSizeGripSpacing());
      
      connect( _ok, SIGNAL(clicked()), SLOT(accept()) );
      connect( _ca, SIGNAL(clicked()), SLOT(reject()) );
    }

    if( ! _item.getName().isEmpty() )
    {
      _name->setText( QString::fromUtf8(_item.getName()) );
      _name->setMinimumWidth( _name->sizeHint().width() );
    }

    if( ! _item.getSource().isEmpty() )
    {
      _source->setText( QString::fromUtf8(_item.getSource()) );
      _source->setMinimumWidth( 2*_source->sizeHint().width() );
    }
  }

  // don't resize vertically.
  setMaximumHeight( sizeHint().height() );
  //setGeometry( x(), y(), 2.5 * height(), height() );
}

ProjectItemProperties::~ProjectItemProperties()
{
}

void ProjectItemProperties::accept()
{
  _item.setName( sc::String(_name->text().utf8()) );
  _item.setSource( sc::String(_source->text().utf8()) );
  super::accept();
}

const Project::Item& ProjectItemProperties::getItem() const
{
  return _item;
}

void ProjectItemProperties::autoUpdate( bool on )
{
  _item.setAutoUpdate(on);
}

void ProjectItemProperties::checkOk( const QString& text )
{
  QString name   = _name->text();
  QString source = _source->text();

  bool allowEmptySource;
  switch( _item.getType() )
  {
  case Project::Item_Trunk:
  case Project::Item_Branches:
  case Project::Item_Tags:
    allowEmptySource = true;
    break;
  default:
    allowEmptySource = false;
  }

  if( (! name.isEmpty() && ! source.isEmpty()) || (! name.isEmpty() && allowEmptySource) )
  {
    _ok->setEnabled(true);
  }
  else
  {
    _ok->setEnabled(false);
  }

  if( ! source.isEmpty() || _item.isWorkingCopy() )
  {
    _ex->setEnabled(true);
  }
  else
  {
    _ex->setEnabled(false);
  }
}

void ProjectItemProperties::selectExtern()
{
  sc::String res;

  if( _item.isWorkingCopy() )
  {
    if( _p->selectPath( this, sc::String(_source->text().utf8()), res, ExternProvider::Dir ) )
    {
      _source->setText(QString::fromUtf8(res));
    }
  }
  else
  {
    if( _p->selectUrl( this, sc::String(_source->text().utf8()), res, ExternProvider::Dir ) )
    {
      _source->setText(QString::fromUtf8(res));
    }
  }
}
