/* ====================================================================
 * Copyright (c) 2006,      Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "LogGraphCanvasView.h"
#include "LogGraphCanvasItems.h"
#include "ActionFactory.h"
#include "sublib/Utility.h"
#include "svn/LogEntry.h"

// qt
#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qwmatrix.h>
#include <qdatetime.h>
#include <qaction.h>


class CanvasToolTip : public QToolTip
{
public:
  CanvasToolTip( QWidget* parent ) : QToolTip( parent )
  {
  }

  virtual ~CanvasToolTip()
  {
  }

  void maybeTip( const QPoint &pos );
};

///////////////////////////////////////////////////////////////////////////////

LogGraphCanvasView::LogGraphCanvasView( QWidget* parent )
: super(parent), _shiftDown(false), _sel1(NULL), _sel2(NULL)
{
  setFocusPolicy(WheelFocus);
  setMouseTracking(true);
  viewport()->setMouseTracking(true);

  new CanvasToolTip(this);

  QAction* qa = ActionFactory::createAction( afDiffDlg, this, this, SLOT(diffTrigger()) );
  qa->setEnabled(false);
  _actions.addAction( afDiffDlg, qa );

  _popup = new QPopupMenu(this);
  _actions.getAction( afDiffDlg )->addTo(_popup);
}

LogGraphCanvasView::~LogGraphCanvasView()
{
  delete _popup;
}

void LogGraphCanvasView::mousePressEvent( QMouseEvent* e )
{
  if( e->button() == Qt::RightButton )
  {
    return;
  }

  // clear old selection
  if( ! _shiftDown )
  {
    clearSelection();
  }

  QPoint pos = viewportToContents(e->pos());
  QPoint wmp = worldMatrix().invert().map(pos);

  QCanvasItemList list = canvas()->collisions(wmp);
  if( list.size() > 0 )
  {
    addSelection( dynamic_cast<LogGraphCanvasItem*>(list.first()) );
  }
  canvas()->update();
}

void LogGraphCanvasView::mouseMoveEvent( QMouseEvent* e )
{
#if 0
  static QWidget*       logtip = NULL;
  static QPoint         logpos;

  QPoint pos = viewportToContents(e->pos());
  QPoint wmp = worldMatrix().invert().map(pos);

  QCanvasItemList list = canvas()->collisions(wmp);

  for( QCanvasItemList::Iterator it = list.begin(); it != list.end(); it++ )
  {
    CanvasRev* rev = dynamic_cast<CanvasRev*>(*it);
  }
#endif
}

void LogGraphCanvasView::contextMenuEvent( QContextMenuEvent* e )
{
  _popup->exec(e->globalPos());
}

void LogGraphCanvasView::keyPressEvent( QKeyEvent* e )
{
  if( e->key() == Qt::Key_Shift )
  {
    _shiftDown = true;
  }
  else
  {
    super::keyPressEvent(e);
  }
}

void LogGraphCanvasView::keyReleaseEvent( QKeyEvent* e )
{
  if( e->key() == Qt::Key_Shift )
  {
    _shiftDown = false;
  }
  else
  {
    super::keyPressEvent(e);
  }
}

void LogGraphCanvasView::diffTrigger()
{
  emit diffRequested();
}

void LogGraphCanvasView::addSelection( LogGraphCanvasItem* item )
{
  if( _sel1 == NULL )
  {
    _sel1 = item;
    _sel1->setSelected(true);
    canvas()->setChanged(_sel1->boundingRect());
    return;
  }
  else if( _sel1 == item )
  {
    return;
  }

  // if we already have a second selection, clear the first
  // and change second to first.
  if( _sel2 != NULL && _sel2 != item )
  {
    _sel1->setSelected(false);
    canvas()->setChanged(_sel1->boundingRect());

    _sel1 = _sel2;
    _sel2 = NULL;
  }

  if( _sel2 == NULL )
  {
    _sel2 = item;
    _sel2->setSelected(true);
    canvas()->setChanged(_sel2->boundingRect());
  }

  _actions.enableAction( afDiffDlg, _sel1 && _sel2 );
  emit selectionChanged( _sel1 && _sel2 );
}

void LogGraphCanvasView::clearSelection()
{
  if( _sel1 != NULL )
  {
    _sel1->setSelected(false);
    canvas()->setChanged(_sel1->boundingRect());
    _sel1 = NULL;
  }

  if( _sel2 != NULL )
  {
    _sel2->setSelected(false);
    canvas()->setChanged(_sel2->boundingRect());
    _sel2 = NULL;
  }

  _actions.enableAction( afDiffDlg, false );
  emit selectionChanged( false );
}

const LogGraphCanvasItem* LogGraphCanvasView::getSelection1() const
{
  return _sel1;
}

const LogGraphCanvasItem* LogGraphCanvasView::getSelection2() const
{
  return _sel2;
}

const QString& LogGraphCanvasView::getRoot() const
{
  return _root;
}

void LogGraphCanvasView::setRoot( QString& root )
{
  _root = root;
}

///////////////////////////////////////////////////////////////////////////////

void CanvasToolTip::maybeTip( const QPoint &pos )
{
  LogGraphCanvasView* cv = (LogGraphCanvasView*)parentWidget();

  QPoint cpos = cv->viewportToContents(pos);
  QPoint cwmp = cv->worldMatrix().invert().map(cpos);

  QCanvasItemList list = cv->canvas()->collisions(cwmp);
  if( list.size() > 0 )
  {
    LogGraphCanvasPath* path = dynamic_cast<LogGraphCanvasPath*>(list.first());
    if( path )
    {
      QPoint p( path->rect().x(), path->rect().y() );
      QRect  pr( path->rect() );
      pr.setHeight( 20 /*PathBoxTextHeigh*/ );

      QPoint w =  cv->worldMatrix().map(p);
      QRect  wr = cv->worldMatrix().mapRect(pr);
      QPoint r = cv->contentsToViewport(w);
      QSize  s = QSize( wr.width(), wr.height() );

      tip( QRect(r,s), path->getFullName() );
    }

    LogGraphCanvasRev* rev = dynamic_cast<LogGraphCanvasRev*>(list.first());
    if( rev )
    {
      QPoint p( rev->rect().x(), rev->rect().y() );

      QPoint w  = cv->worldMatrix().map(p);
      QRect  wr = cv->worldMatrix().mapRect(rev->rect());
      QPoint r  = cv->contentsToViewport(w);
      QSize  s  = QSize( wr.width(), wr.height() );


      QString text = QString(
        "<qt><table cellpadding=0 cellspacing=0>"
        "<tr><td><nobr>%1</nobr></td></tr>")
        .arg(QString::fromUtf8(rev->getNode()->getName()));

      QDateTime date;
      date.setTime_t( (uint)apr_time_sec(rev->getNode()->getLog()->getDate()) );
      QString qdate = date.toString( getDateFormat() );

      text += QString(
        "<tr><td><nobr>r%1, %2, %3</nobr></td></tr>" )
        .arg(rev->getNode()->getLog()->getRevnumber())
        .arg(qdate)
        .arg(QString::fromUtf8(rev->getNode()->getLog()->getAuthor()));

      QString msg = QString::fromUtf8(rev->getNode()->getLog()->getMessage());
      QString cut = msg.section( '\n', 0, 14,
        QString::SectionIncludeLeadingSep | QString::SectionIncludeTrailingSep );
      QString cutmsg;
      if( cut.length() < msg.length() )
      {
        cutmsg = QString("<tr><td>%2<td></tr>").arg( _q(".... (cut)") );
      }

      text += QString(
        "<tr><td>&nbsp;</td></tr>"
        "<tr><td><pre>%1</pre></td></tr>"
        "%2"
        "</table></qt>")
        .arg(cut)
        .arg(cutmsg);

      tip( QRect(r,s), text );
    }
  }
}
